/*
 * screen.c
 *
 * $Log: screen.c,v $
 * Revision 1.13  1998/11/29  11:11:19  ctkwan
 * add video mode driver selection based on pixel size
 *
 */
static char rcsid[]="$Id: screen.c,v 1.13 1998/11/29 11:11:19 ctkwan Exp ctkwan $";

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include "hardware.h"
#include "x_window.h"
#include "memory.h"
#include "video.h"
#include "iou.h"
#include "yae.h"

static APPLE_TICK	NextVideoRefresh;

XImage *ScreenXImage;
XShmSegmentInfo	shminfo;

/*
 * frame rate control
 */
#define FRAME_INTERVAL	(65*262)
static int TargetFPS=60;
static CurrentFPS=60;
static FPSCounter=0;

/*
 * the frame rate of is fixed at 60 FPS for any
 * target speed.
 */
#define MAX_FPS	60
static  int	frameCounter=0;
static	int	frameTimeSlice[MAX_FPS];
static	int	lastTargetSpeed;

extern VideoModeDriver	c8VideoModeDriver;
extern VideoModeDriver	c16VideoModeDriver;
extern VideoModeDriver	c32VideoModeDriver;

VideoModeDriver	*currentVideoModeDriver;

void (*DrawScanlineFunction[NO_OF_SCANLINE_MODES])(scanline_t*);

scanline_t Scanline[192];
unsigned ScanlineOffsetTable[192];

ADDR ScanlineAddressTable[6][192];
ADDR *ScanlineAddressMetaTable[IOU_VIDEO_SS_MASK+1];
int UpperScanlineModeTable[IOU_VIDEO_SS_MASK+1];
int LowerScanlineModeTable[IOU_VIDEO_SS_MASK+1];

static BYTE MainMemoryPageDirty[256];
static BYTE AuxMemoryPageDirty[256];

/* Flash Text and Alt Character Set */
static int CurrentCharSet;
 
static void updateScreenMicrojob(void *arg);

static void setTimeSlice(void)
{
	int	i;
	int	targetSpeed, sliceSize;

	targetSpeed = GetTargetSpeed();
	sliceSize = targetSpeed/MAX_FPS;
	for( i = 0; i < MAX_FPS-1; i++)
	   frameTimeSlice[i] = sliceSize;

	/* let the last slice have the remainer */
	frameTimeSlice[MAX_FPS-1]=targetSpeed - (MAX_FPS-1)*sliceSize;
}

void cleanupScreenImage( DisplayContext *dc )
{
	printf("remove Shm XImage\n");
	destroyShmImage( dc, ScreenXImage,&shminfo);
}

void initTables()
{
	int	i,page_idx;

	for(i=0;i<192;i++)
	   ScanlineOffsetTable[i]=(i&7)*0x400 + ((i>>3)&7)*0x80 + (i>>6)*0x28;

	for(i=0;i<192;i++){
	   ScanlineAddressTable[0][i]=ScanlineOffsetTable[i&~7]+0x400;
	   ScanlineAddressTable[1][i]=ScanlineOffsetTable[i]+0x2000;
	   ScanlineAddressTable[2][i]=
	      ScanlineAddressTable[i>159?0:1][i];
	   ScanlineAddressTable[3][i]=ScanlineOffsetTable[i&~7]+0x800;
	   ScanlineAddressTable[4][i]=ScanlineOffsetTable[i]+0x4000;
	   ScanlineAddressTable[5][i]=
	      ScanlineAddressTable[i>159?3:4][i];
	}

	for(i=0;i<IOU_VIDEO_SS_MASK+1;i++){
	   page_idx=((i&SS_80STORE)||!(i&SS_PAGE2))? 0:3;
	   if ((i&SS_HIRES)&&!(i&SS_TEXT)){
	      /* (double) hires or mixed (double) hires */
	      if (i&SS_MIXED)
	         ScanlineAddressMetaTable[i]=ScanlineAddressTable[page_idx+2];
	      else
	         ScanlineAddressMetaTable[i]=ScanlineAddressTable[page_idx+1];
	   }
	   else
	      ScanlineAddressMetaTable[i]=ScanlineAddressTable[page_idx];

	   if (i&SS_TEXT){
	      UpperScanlineModeTable[i]=i&SS_80COL?
	         SCANLINE_MODE_TEXT80:SCANLINE_MODE_TEXT;
	   }
	   else{
	      if (i&SS_HIRES){
	         UpperScanlineModeTable[i]=i&SS_DHIRES?
	            SCANLINE_MODE_DHIRES:SCANLINE_MODE_HIRES;
	      }
	      else {
	         UpperScanlineModeTable[i]=i&SS_DHIRES?
	            SCANLINE_MODE_DLORES:SCANLINE_MODE_LORES;
	      }
	   }

	   if (i&SS_MIXED) {
	      LowerScanlineModeTable[i]=i&SS_80COL||
	         ((i&SS_DHIRES)&&!(i&SS_TEXT))?
	         SCANLINE_MODE_TEXT80:SCANLINE_MODE_TEXT;
	   }
	   else
	      LowerScanlineModeTable[i]=UpperScanlineModeTable[i];
	}

	memset(MainMemoryPageDirty,0,sizeof(MainMemoryPageDirty));
	memset(AuxMemoryPageDirty,0,sizeof(AuxMemoryPageDirty));

	for(i=0;i<192;i++) {
	   Scanline[i].state=SCANLINE_STATE_DIRTY;
	}
}

void initVideo( DisplayContext *dc )
{
	int	i;

	/* select video driver */
	if ( dc->depth > 16 )
	   currentVideoModeDriver = &c32VideoModeDriver;
	else if ( dc->depth > 8 )
	   currentVideoModeDriver = &c16VideoModeDriver;
	else
	   currentVideoModeDriver = &c8VideoModeDriver;

	/* fill in scanline function dispatch table */
	DrawScanlineFunction[SCANLINE_MODE_BLANK] =
	   currentVideoModeDriver->drawBlankScanline;
	DrawScanlineFunction[SCANLINE_MODE_TEXT] =
	   currentVideoModeDriver->drawTextScanline;
	DrawScanlineFunction[SCANLINE_MODE_LORES] =
	   currentVideoModeDriver->drawLoresScanline;
	DrawScanlineFunction[SCANLINE_MODE_HIRES] =
	   currentVideoModeDriver->drawHiresScanline;
	DrawScanlineFunction[SCANLINE_MODE_TEXT80] =
	   currentVideoModeDriver->drawText80Scanline;
	DrawScanlineFunction[SCANLINE_MODE_DLORES] =
	   currentVideoModeDriver->drawDLoresScanline;
	DrawScanlineFunction[SCANLINE_MODE_DHIRES] =
	   currentVideoModeDriver->drawDHiresScanline;
	
	ScreenXImage=(XImage*)createShmImage( dc, 560, 384,&shminfo);

	if (!ScreenXImage){
	   return;
	}

	for(i=0;i<192;i++) {
	   Scanline[i].y=i;
	   Scanline[i].mode=SCANLINE_MODE_BLANK;
	   Scanline[i].addr=0;
	   Scanline[i].scanline_addr1=ScreenXImage->data+
	      (i*2)*ScreenXImage->bytes_per_line;
	   Scanline[i].scanline_addr2=ScreenXImage->data+
	      (i*2+1)*ScreenXImage->bytes_per_line;
	}

	/* initialize offset tables */
	for(i=0;i<192;i++)
	   ScanlineOffsetTable[i]=(i&7)*0x400 + ((i>>3)&7)*0x80 + (i>>6)*0x28;

	currentVideoModeDriver->initScanlineData(dc);
	initTables();
	currentVideoModeDriver->changeCharSet(CurrentCharSet);
	(void)updateScreenMicrojob( dc );
}

int drawScreen(void)
{
	int i, lines,page;
	unsigned mode;

	mode=GetIOUSoftSwitch()&IOU_VIDEO_SS_MASK;

	/* setup video mode and address */
	for(i=0;i<192;i++){
	   Scanline[i].addr=ScanlineAddressMetaTable[mode][i];
	   if (i<160)
	      Scanline[i].mode=UpperScanlineModeTable[mode];
	   else
	      Scanline[i].mode=LowerScanlineModeTable[mode];
	   if (Scanline[i].mode!=Scanline[i].old_mode||
	      Scanline[i].addr!=Scanline[i].old_addr) {
	      Scanline[i].state=SCANLINE_STATE_DIRTY;

	      /* mode change! memory considered dirty */
	      page=Scanline[i].addr>>8;
	      MainMemoryPageDirty[page]=1;
	      if (Scanline[i].mode > SCANLINE_MODE_HIRES)
	         AuxMemoryPageDirty[page]=1;
	   }
	}

	/* check if memory written */
	for(i=0;i<192;i++) {
	   if (Scanline[i].state==SCANLINE_STATE_DIRTY)
	      continue;
	   page=Scanline[i].addr>>8;
	   if ( MainMemoryPageDirty[page] )
	      Scanline[i].state=SCANLINE_STATE_DIRTY;
	   else if ( !(GetPageLock(MainMemoryRAM[page])&PAGE_LOCK_WRITE)||
	      (GetWriteLockBrokenTime(MainMemoryRAM[page])!=
	      Scanline[i].write_lock_set)) {
	      MainMemoryPageDirty[page]=1;
	      Scanline[i].state=SCANLINE_STATE_DIRTY;
	   }
	   if (Scanline[i].mode > SCANLINE_MODE_HIRES) {
	      if ( MainMemoryPageDirty[page] )
	         Scanline[i].state=SCANLINE_STATE_DIRTY;
	      else if ( !(GetPageLock(AuxMemoryRAM[page])&PAGE_LOCK_WRITE)||
	         (GetWriteLockBrokenTime(AuxMemoryRAM[page])!=
	         Scanline[i].aux_write_lock_set)) {
	         AuxMemoryPageDirty[page]=1;
	         Scanline[i].state=SCANLINE_STATE_DIRTY;
	      }
	   }
	}

	for(lines=i=0;i<192;i++) 
	   if (Scanline[i].state==SCANLINE_STATE_DIRTY) {
	      (*DrawScanlineFunction[Scanline[i].mode])(&Scanline[i]);
	      Scanline[i].old_mode=Scanline[i].mode;
	      Scanline[i].old_addr=Scanline[i].addr;
	      lines++;
	      Scanline[i].state=SCANLINE_STATE_DRAWN;
	   }

	return lines;
}

void refreshScreen(void)
{
	int	i;

	for(i=0;i<192;i++)
	   Scanline[i].state=SCANLINE_STATE_DIRTY;
}

void updateScreen( DisplayContext *dc )
{
	int lines,i,top,bottom,n,lines_drawn,flag, page;
	static c;

 	lines=0;
	for(i=0;i<192;i++)
	   if (Scanline[i].state==SCANLINE_STATE_DRAWN)
	      lines++;

	if(lines==0){
	   return;
	}

	lines_drawn=0;
	top=0;
	while(lines_drawn<lines){
	   for(;top<192&&Scanline[top].state!=SCANLINE_STATE_DRAWN;top++);
	   for(bottom=top+1;bottom<192&&
	      Scanline[bottom].state==SCANLINE_STATE_DRAWN;bottom++);
	   bottom--;
	   n=bottom-top+1;
	   flag=lines_drawn+n==lines?True:False;

	   if (UseXShm)
	      XShmPutImage( dc->display, dc->window, dc->gc, ScreenXImage,
	         0,top*2, 0,top*2,560,n*2,False);
	   else
	      XPutImage( dc->display, dc->window, dc->gc, ScreenXImage, 0,
	         top*2, 0,top*2,560,n*2);

	   lines_drawn+=(bottom-top+1);
	   /* reset state flag */
	   for(i=top;i<=bottom;i++) {
	      Scanline[i].state=SCANLINE_STATE_CLEAN;
	      Scanline[i].old_mode=Scanline[i].mode;
	      Scanline[i].old_addr=Scanline[i].addr;
	      page=Scanline[i].addr>>8;
	      if (MainMemoryPageDirty[page]) {
	         LockAppleMemoryPage(MainMemoryRAM[page], PAGE_LOCK_WRITE);
	         MainMemoryPageDirty[page]=0;
	      }
	      Scanline[i].write_lock_set=
	         GetWriteLockSetTime(MainMemoryRAM[page]);
	      if (Scanline[i].mode > SCANLINE_MODE_HIRES ) {
	         if (AuxMemoryPageDirty[page]) {
	            LockAppleMemoryPage(AuxMemoryRAM[page], PAGE_LOCK_WRITE);
	            AuxMemoryPageDirty[page]=0;
	         }
	         Scanline[i].aux_write_lock_set=
	            GetWriteLockSetTime(AuxMemoryRAM[page]);
	      }
	   }
	   top=bottom+1;
	}
	XFlush( dc->display );

	/*
	 * seems not a good idea to wait for X server to finsh drawing
	 * this has complicated interaction with speed control
	 * resulting in a low apparent speed as the wait below is included
	 * into speed calculation.
	 */
#if 0
	/* commented out 5/8/97 */
	XSync( dc->display, False );
#endif
}

/*
 * find out all the scan line in text mode and containing
 * character in the range $40-$7F.
 */
static void invalidateFlashText(void)
{
	int	i,j;
	BYTE	*row;

	for(i=0;i<192;i++) {
	   if (Scanline[i].state != SCANLINE_STATE_DIRTY &&
	       ((Scanline[i].mode == SCANLINE_MODE_TEXT) ||
	        (Scanline[i].mode == SCANLINE_MODE_TEXT80))) {

	      j=i&~7;
	      if ( i!=j && Scanline[i].mode==Scanline[j].mode&&
	         Scanline[i].addr==Scanline[j].addr )
	         Scanline[i].state = Scanline[j].state;
	      else {
	         /* check main memory */
	         row=((BYTE*)MainMemoryRAM[Scanline[i].addr>>8]) +
	            (Scanline[i].addr&0xff);
	         for(j=0;j<40 && (row[j]<0x40||row[j]>0x7f);j++);
	      
	         if (j<40) 
	            Scanline[i].state = SCANLINE_STATE_DIRTY;
	         else if ( Scanline[i].mode== SCANLINE_MODE_TEXT80 ) {
	            row=((BYTE*)AuxMemoryRAM[Scanline[i].addr>>8]) +
	               (Scanline[i].addr&0xff);
	            for(j=0;j<40 && (row[j]<0x40||row[j]>0x7f);j++);
	            if (j<40)
	               Scanline[i].state = SCANLINE_STATE_DIRTY;
	         }
	      }
	   }
	}
}

static void updateScreenMicrojob(void *arg)
{
	int i, speed, raw_speed, target_speed, new_charset;

#if 0 ||defined(DEBUG)
	static int c;

	if (++c==60){
	   c=0;
	   printf("fps=%d\n",CurrentFPS);
	}
#endif
	target_speed = GetTargetSpeed();
	if ( target_speed != lastTargetSpeed ) {
	   setTimeSlice();
	   lastTargetSpeed = target_speed;
	}

	/* handle flash text mode and charset selection */
	new_charset=(GetIOUSoftSwitch()&SS_ALTCHAR)? 2:
	   (frameCounter<30? 0:1);

	if (new_charset != CurrentCharSet) {
	   CurrentCharSet=new_charset;
	   currentVideoModeDriver->changeCharSet(new_charset);
	   invalidateFlashText();
	}

	/*
	 * rate control: If CurrentFPS equals to TargetFPS,
	 * no frames are skipped. Otherwise, framed are skipped
	 * evenly according to the ratio of CurrentFPS to TargetFPS.
	 * For example, if CurrentFPS=30 and TargetFPS=60, then
	 * one out of every two frames will be dropped.
	 */
	FPSCounter+=CurrentFPS;
	if (FPSCounter>=TargetFPS) {
	   drawScreen();
	   updateScreen( arg );
	   FPSCounter -= TargetFPS;
	}

	/* frame increase control */
	speed=GetAverageSpeed();
	raw_speed=GetAverageRawSpeed();
#if 0
	if ( speed >= target_speed ) {
	   if ( CurrentFPS < 60 )
	      CurrentFPS++;
	}
	else{
	   if ( CurrentFPS > 5 )
	      CurrentFPS--;
	}
#else
	if (raw_speed > target_speed) { /* can do current FPS or higher */
	   CurrentFPS=(double)raw_speed/target_speed*CurrentFPS + 0.5;
	   if(CurrentFPS>TargetFPS)
	      CurrentFPS=TargetFPS;
	}
	else { /* need to slow down */
	   CurrentFPS=(double)raw_speed/target_speed*TargetFPS;
	   if (CurrentFPS<5)
	      CurrentFPS=5;	/* minimum */
	}
#endif
	
	NextVideoRefresh+=frameTimeSlice[frameCounter];
	submit_micro_job(NextVideoRefresh,updateScreenMicrojob, arg );

	if (++frameCounter==MAX_FPS)
	   frameCounter=0;
}
