/*
 * mm.c
 *
 * general memory management
 */

#include <stdio.h>
#include <stdlib.h>
#include "hardware.h"
#include "memory.h"
#include "6502.h"

#if 0
typedef	struct _AppleMemoryPageRec {
	BYTE		data[256];	
	unsigned	tag;
	int		read_page_no;
	int		write_page_no;
	int		page_id;
	unsigned	*code;
	unsigned	lock;
	APPLE_TICK	write_lock_set;
	APPLE_TICK	write_lock_broken;
} AppleMemoryPageRec;
#endif

#define	UNTAGGED(p)		((unsigned)(p)&~3)
#define	PAGE_DATA_PTR(d,l)	(BYTE*) ((unsigned)(d)|((l)&3))
#define	READ_MAPPED_APPLEMEMORYPAGE(n)\
				((AppleMemoryPage)\
	                         UNTAGGED(Read_Page_Table[n]))

#define	WRITE_MAPPED_APPLEMEMORYPAGE(n)\
				((AppleMemoryPage)\
	                         UNTAGGED(Write_Page_Table[n]))

#define	READ_MAPPED(p)		((p)->read_page_no != -1)
#define	WRITE_MAPPED(p)		((p)->write_page_no != -1)
#define	READ_PAGE_NO(p)		((p)->read_page_no)
#define	WRITE_PAGE_NO(p)	((p)->write_page_no)

#if 1
#define	CHECK(amp)	if((amp)->tag != 0xd5aa96ff) {\
			   fprintf( stderr, "Arghh!!\n" );\
	                }
#else
#define	CHECK(amp)
#endif
static	int	NewPageID = 1;

static	AppleMemoryPageRec	NullPage;

void	init_mm()
{
	int	i;

	for( i = 0; i < 256; i++ ) {
	   Read_Page_Table[i] = NullPage.data;
	   Write_Page_Table[i] = NullPage.data;
#ifdef	RUN_TIME_TRANSLATION
	   Image_Page_Table[i] = NULL;
#endif	/* RUN_TIME_TRANSLATION */
	}
	NullPage.tag = 0xd5aa96ff;
}

AppleMemoryPage GetNullPage(void)
{
	return&NullPage;
}

AppleMemoryPage AppleCreateMemoryPage(void)
{
	int		i;
	AppleMemoryPage	amp;

	amp = (AppleMemoryPage)malloc(sizeof(AppleMemoryPageRec));

	if ( !amp ) {
	   fprintf( stderr, "AppleCreateMemoryPage: out of memory\n" );
	   return amp;
	}

	for( i = 0; i < 256; i++ )
	   amp->data[i] = 0;

	amp->read_page_no = -1;
	amp->write_page_no = -1;
	amp->code = NULL;
	amp->page_id = NewPageID++;
	amp->lock = 0;
	amp->tag = 0xd5aa96ff;
	amp->write_lock_set=AppleClock;
	amp->write_lock_broken=AppleClock;
	return amp;
} 

void AppleDestroyMemoryPage( AppleMemoryPage amp )
{
	CHECK(amp);

	if ( !amp->page_id )
	   return;

	if ( READ_MAPPED( amp ) )
	   UnmapReadPage( amp->read_page_no );
	if ( WRITE_MAPPED( amp ) )
	   UnmapWritePage( amp->write_page_no );

#ifdef	RUN_TIME_TRANSLATION
	if ( amp->code )
	   DisposeCode( amp->code );
#endif
	free( amp );
}

int MapReadPage( int page_no, AppleMemoryPage amp )
{
	if ( page_no > 255 || page_no < 0 )
	   return -1;

	if ( !amp )
	   amp = &NullPage;

	if ( READ_PAGE_NO(amp) == page_no )
	   return 0;

	UnmapReadPage( page_no );

	CHECK(amp);
	if ( amp->page_id && READ_MAPPED(amp) )
	   UnmapReadPage( amp->read_page_no );

	if ( amp->page_id ) {
	   amp->read_page_no = page_no;
	   Read_Page_Table[page_no] = PAGE_DATA_PTR( amp->data,
	      amp->lock >> 2 );
#ifdef	RUN_TIME_TRANSLATION
	   Image_Page_Table[page_no] = amp->code;
#endif
	}
	else { 
	   Read_Page_Table[page_no] = PAGE_DATA_PTR( amp->data, 0 );
#ifdef	RUN_TIME_TRANSLATION
	   Image_Page_Table[page_no] = NULL;
#endif
	}
	Signal6502( SIG_6502_MEMORY );
	return 0;
}

int MapWritePage( int page_no, AppleMemoryPage amp )
{
	if ( page_no > 255 || page_no < 0 )	/* check page_no */
	   return -1;

	if ( !amp )
	   amp = &NullPage;

	/* already mapped to this page? do nothing if so */
	if ( WRITE_PAGE_NO(amp) == page_no )
	   return 0;

	UnmapWritePage( page_no );

	CHECK(amp);

	/* unmap page if it is currently mapped to other page */
	if ( amp->page_id && WRITE_MAPPED(amp) ) 
	   UnmapWritePage( amp->write_page_no );

	if ( amp->page_id ) {	/* not Null Page */
	   amp->write_page_no = page_no;
	   Write_Page_Table[page_no] = PAGE_DATA_PTR( amp->data, amp->lock );
	}
	else /* Null Page */
	   Write_Page_Table[page_no] = PAGE_DATA_PTR( amp->data, 0 );
	return 0;
}

#ifdef	RUN_TIME_TRANSLATION
int MapImagePage( int page_no, unsigned *code )
{
	AppleMemoryPage	amp;

	if ( page_no > 255 || page_no < 0 )	/* check page_no */
	   return -1;

	amp = READ_MAPPED_APPLEMEMORYPAGE( page_no );

	CHECK(amp);
	if ( !amp->page_id )	/* Null Page cannot map image */
	   return -1;

	if ( COMPILED(page_no) ) {	/* flush old image */ 
	   Signal6502( SIG_6502_MEMORY ); 
	   DisposeCode( amp->code );
	}
	   
	Image_Page_Table[page_no] = code;
	amp->code = code;

	if ( code ) /* write protect page */
	   LockAppleMemoryPage( amp, PAGE_LOCK_CODE );
	else
	   UnlockAppleMemoryPage( amp, PAGE_LOCK_CODE );
	   
	return 0;
}
#endif

int UnmapReadPage( int page_no )
{
	AppleMemoryPage	amp;

	if ( page_no > 255 || page_no < 0 )	/* check page_no */
	   return -1;

	amp = READ_MAPPED_APPLEMEMORYPAGE( page_no );
	CHECK(amp);
	if ( !amp->page_id )
	   return 0;

	amp->read_page_no = -1;
#ifdef	RUN_TIME_TRANSLATION
	if ( COMPILED(page_no) ) {	/* flush image */
	   Image_Page_Table[page_no] = NULL;
	   Signal6502( SIG_6502_MEMORY );
	}
#endif
	Read_Page_Table[page_no] = PAGE_DATA_PTR( NullPage.data, 0 );
	return 0;
}

int UnmapWritePage( page_no )
int page_no;
{
	AppleMemoryPage	amp;

	if ( page_no > 255 || page_no < 0 )
	   return -1;

	amp = WRITE_MAPPED_APPLEMEMORYPAGE( page_no );
	CHECK(amp);
	if ( !amp->page_id )
	   return 0;

	amp->write_page_no = -1;
	Write_Page_Table[page_no] = PAGE_DATA_PTR( NullPage.data, 0 );
	return 0;
}

int LockReadMappedPage( int page_no, unsigned lock )
{
	AppleMemoryPage	amp;
	unsigned	u;

	if ( page_no > 255 || page_no < 0 )
	   return -1;

	amp = READ_MAPPED_APPLEMEMORYPAGE( page_no );
	CHECK(amp);

	if ( amp->page_id )
	   return LockAppleMemoryPage( amp, lock );
	else 
	   if ( lock & PAGE_LOCK_READ ) {
	      u = (unsigned)Read_Page_Table[page_no] | ((lock >> 2) & 3);
	      Read_Page_Table[page_no] = (BYTE*)u;
	      return 0;
	   }
	return -1;
}

int LockWriteMappedPage( int page_no, unsigned lock )
{
	AppleMemoryPage	amp;
	unsigned	u;

	if ( page_no > 255 || page_no < 0 )
	   return -1;

	amp = WRITE_MAPPED_APPLEMEMORYPAGE( page_no );
	CHECK(amp);
	if ( amp->page_id )
	   return LockAppleMemoryPage( amp, lock );
	else 
	   if ( lock & ( PAGE_LOCK_WRITE | PAGE_LOCK_CODE ) ) {
	      u = (unsigned)Read_Page_Table[page_no] | (lock & 3);
	      Write_Page_Table[page_no] = (BYTE*)u;
	      return 0;
	   }
	return -1;
}

int UnlockReadMappedPage( int page_no, unsigned lock )
{
	AppleMemoryPage	amp;
	unsigned	u;

	if ( page_no > 255 || page_no < 0 )
	   return -1;

	amp = READ_MAPPED_APPLEMEMORYPAGE( page_no );
	CHECK(amp);
	if ( amp->page_id )
	   return UnlockAppleMemoryPage( amp, lock );
	else 
	   if ( lock & PAGE_LOCK_READ ) {
	      u = (unsigned)Read_Page_Table[page_no] & ~((lock>>2) & 3);
	      Read_Page_Table[page_no] = (BYTE*)u;
	      return 0;
	   }
	return -1;
}

int UnlockWriteMappedPage( int page_no, unsigned lock )
{
	AppleMemoryPage	amp;
	unsigned	u;

	if ( page_no > 255 || page_no < 0 )
	   return -1;

	amp = WRITE_MAPPED_APPLEMEMORYPAGE( page_no );
	CHECK(amp);
	if ( amp->page_id )
	   return UnlockAppleMemoryPage( amp, lock );
	else 
	   if ( lock & PAGE_LOCK_READ ) {
	      u = (unsigned)Write_Page_Table[page_no] & ~(lock & 3);
	      Read_Page_Table[page_no] = (BYTE*)u;
	      return 0;
	   }
	return -1;
}

int LockAppleMemoryPage( AppleMemoryPage amp, unsigned lock )
{
	if ( !amp->page_id )
	   return -1;

	CHECK(amp);

	lock &= ~amp->lock;
	if ( !lock )
	   return 0;

	/* record time */
	if (lock & PAGE_LOCK_WRITE) {
	   amp->write_lock_set=AppleClock;
	   amp->write_lock_broken=AppleClock;
	}

	amp->lock |= (lock & 0x0f);
	if ( READ_MAPPED( amp ) ){
	   Read_Page_Table[READ_PAGE_NO(amp)] = PAGE_DATA_PTR( amp->data, 
	      (amp->lock>>2) & 3 );
	}
	if ( WRITE_MAPPED( amp ) ){
	   Write_Page_Table[WRITE_PAGE_NO(amp)] = PAGE_DATA_PTR( amp->data, 
	      amp->lock & 3 );
	}

	return 0;
}

int UnlockAppleMemoryPage( AppleMemoryPage amp, unsigned lock )
{
	if ( !amp->page_id )
	   return -1;

	CHECK(amp);

	lock &= amp->lock;
	if ( !lock )
	   return 0;

	if (lock&PAGE_LOCK_WRITE)
	   amp->write_lock_broken=AppleClock;

	amp->lock &= ~(lock & 0x0f);
	if ( READ_MAPPED( amp ) ) { 
	   Read_Page_Table[READ_PAGE_NO(amp)] = PAGE_DATA_PTR( amp->data, 
	      (amp->lock>>2) & 3 );
	}
	if ( WRITE_MAPPED( amp ) ) { 
	   Write_Page_Table[WRITE_PAGE_NO(amp)] = PAGE_DATA_PTR( amp->data,
	      amp->lock & 3 );
	}

#ifdef	RUN_TIME_TRANSLATION
	if ( (lock & PAGE_LOCK_CODE) && amp->code ) {
	   if ( READ_PAGE_NO(amp) != -1 ) {
	      Image_Page_Table[READ_PAGE_NO(amp)] = NULL;
	      Signal6502( SIG_6502_MEMORY );
	   }
	   DisposeCode( amp->code );
	   amp->code = NULL;
	}
#endif

	return 0;
}

#if 0
unsigned GetPageLock( AppleMemoryPage amp )
{
	return amp->lock;
}

APPLE_TICK GetWriteLockSetTime( AppleMemoryPage amp )
{
	return amp->write_lock_set;
}

APPLE_TICK GetWriteLockBrokenTime( AppleMemoryPage amp )
{
	return amp->write_lock_broken;
}
#endif
