/*
 * screen.c
 */
#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include "hardware.h"
#include "memory.h"
#include "video.h"
#include "iou.h"
#include "yae.h"

static APPLE_TICK	NextVideoRefresh;

extern Display *theDisplay;
extern Window theWindow;
extern GC theGC;

XImage *ScreenXImage;
XShmSegmentInfo	shminfo;

/*
 * frame rate control
 */
#define FRAME_INTERVAL	(65*262)
static int TargetFPS=60;
static CurrentFPS=60;
static FPSCounter=0;

void (*DrawScanlineFunction[])(scanline_t*)={
	drawBlankScanline,
	drawTextScanline,
	drawLowresScanline,
	drawHiresScanline,
	drawText80Scanline,
	drawDLoresScanline,
	drawDHiresScanline,
};

scanline_t Scanline[192];
unsigned ScanlineOffsetTable[192];

ADDR ScanlineAddressTable[6][192];
ADDR *ScanlineAddressMetaTable[IOU_VIDEO_SS_MASK+1];
int UpperScanlineModeTable[IOU_VIDEO_SS_MASK+1];
int LowerScanlineModeTable[IOU_VIDEO_SS_MASK+1];

static BYTE MainMemoryPageDirty[256];
static BYTE AuxMemoryPageDirty[256];

/* Flash Text and Alt Character Set */
static int FlashTextCounter;
static int CurrentCharSet;
 
static void updateScreenMicrojob(void *arg);

void cleanupScreenImage()
{
	printf("remove Shm XImage\n");
	destroyShmImage(theDisplay,ScreenXImage,&shminfo);
}

void initTables()
{
	int	i,page_idx;

	for(i=0;i<192;i++)
	   ScanlineOffsetTable[i]=(i&7)*0x400 + ((i>>3)&7)*0x80 + (i>>6)*0x28;

	for(i=0;i<192;i++){
	   ScanlineAddressTable[0][i]=ScanlineOffsetTable[i&~7]+0x400;
	   ScanlineAddressTable[1][i]=ScanlineOffsetTable[i]+0x2000;
	   ScanlineAddressTable[2][i]=
	      ScanlineAddressTable[i>159?0:1][i];
	   ScanlineAddressTable[3][i]=ScanlineOffsetTable[i&~7]+0x800;
	   ScanlineAddressTable[4][i]=ScanlineOffsetTable[i]+0x4000;
	   ScanlineAddressTable[5][i]=
	      ScanlineAddressTable[i>159?3:4][i];
	}

	for(i=0;i<IOU_VIDEO_SS_MASK+1;i++){
	   page_idx=((i&SS_80STORE)||!(i&SS_PAGE2))? 0:3;
	   if ((i&SS_HIRES)&&!(i&SS_TEXT)){
	      /* (double) hires or mixed (double) hires */
	      if (i&SS_MIXED)
	         ScanlineAddressMetaTable[i]=ScanlineAddressTable[page_idx+2];
	      else
	         ScanlineAddressMetaTable[i]=ScanlineAddressTable[page_idx+1];
	   }
	   else
	      ScanlineAddressMetaTable[i]=ScanlineAddressTable[page_idx];

	   if (i&SS_TEXT){
	      UpperScanlineModeTable[i]=i&SS_80COL?
	         SCANLINE_MODE_TEXT80:SCANLINE_MODE_TEXT;
	   }
	   else{
	      if (i&SS_HIRES){
	         UpperScanlineModeTable[i]=i&SS_DHIRES?
	            SCANLINE_MODE_DHIRES:SCANLINE_MODE_HIRES;
	      }
	      else {
	         UpperScanlineModeTable[i]=i&SS_DHIRES?
	            SCANLINE_MODE_DLORES:SCANLINE_MODE_LORES;
	      }
	   }

	   if (i&SS_MIXED) {
	      LowerScanlineModeTable[i]=i&SS_80COL||
	         ((i&SS_DHIRES)&&!(i&SS_TEXT))?
	         SCANLINE_MODE_TEXT80:SCANLINE_MODE_TEXT;
	   }
	   else
	      LowerScanlineModeTable[i]=UpperScanlineModeTable[i];
	}

	memset(MainMemoryPageDirty,0,sizeof(MainMemoryPageDirty));
	memset(AuxMemoryPageDirty,0,sizeof(AuxMemoryPageDirty));
}

void initVideo(void)
{
	int	i;

	allocAppleColours(theDisplay);
	ScreenXImage=(XImage*)createShmImage(theDisplay,560,384,&shminfo);

	if (!ScreenXImage){
	   return;
	}

	for(i=0;i<192;i++) {
	   Scanline[i].y=i;
	   Scanline[i].mode=SCANLINE_MODE_BLANK;
	   Scanline[i].addr=0;
	   Scanline[i].scanline_addr1=ScreenXImage->data+
	      (i*2)*ScreenXImage->bytes_per_line;
	   Scanline[i].scanline_addr2=ScreenXImage->data+
	      (i*2+1)*ScreenXImage->bytes_per_line;
	}

	/* initialize offset tables */
	for(i=0;i<192;i++)
	   ScanlineOffsetTable[i]=(i&7)*0x400 + ((i>>3)&7)*0x80 + (i>>6)*0x28;

	initScanlineData();
	initTables();
	changeCharSet(CurrentCharSet);
	(void)updateScreenMicrojob(NULL);
}

int drawScreen(void)
{
	int i, lines,page;
	unsigned mode;

	mode=GetIOUSoftSwitch()&IOU_VIDEO_SS_MASK;

	/* setup video mode and address */
	for(i=0;i<192;i++){
	   Scanline[i].addr=ScanlineAddressMetaTable[mode][i];
	   if (i<160)
	      Scanline[i].mode=UpperScanlineModeTable[mode];
	   else
	      Scanline[i].mode=LowerScanlineModeTable[mode];
	   if (Scanline[i].mode!=Scanline[i].old_mode||
	      Scanline[i].addr!=Scanline[i].old_addr) {
	      Scanline[i].state=SCANLINE_STATE_DIRTY;

	      /* mode change! memory considered dirty */
	      page=Scanline[i].addr>>8;
	      MainMemoryPageDirty[page]=1;
	      if (Scanline[i].mode > SCANLINE_MODE_HIRES)
	         AuxMemoryPageDirty[page]=1;
	   }
	}

	/* check if memory written */
	for(i=0;i<192;i++) {
	   if (Scanline[i].state==SCANLINE_STATE_DIRTY)
	      continue;
	   page=Scanline[i].addr>>8;
	   if ( MainMemoryPageDirty[page] )
	      Scanline[i].state=SCANLINE_STATE_DIRTY;
	   else if ( !(GetPageLock(MainMemoryRAM[page])&PAGE_LOCK_WRITE)||
	      (GetWriteLockBrokenTime(MainMemoryRAM[page])!=
	      Scanline[i].write_lock_set)) {
	      MainMemoryPageDirty[page]=1;
	      Scanline[i].state=SCANLINE_STATE_DIRTY;
	   }
	   if (Scanline[i].mode > SCANLINE_MODE_HIRES) {
	      if ( MainMemoryPageDirty[page] )
	         Scanline[i].state=SCANLINE_STATE_DIRTY;
	      else if ( !(GetPageLock(AuxMemoryRAM[page])&PAGE_LOCK_WRITE)||
	         (GetWriteLockBrokenTime(AuxMemoryRAM[page])!=
	         Scanline[i].aux_write_lock_set)) {
	         AuxMemoryPageDirty[page]=1;
	         Scanline[i].state=SCANLINE_STATE_DIRTY;
	      }
	   }
	}

	for(lines=i=0;i<192;i++) 
	   if (Scanline[i].state==SCANLINE_STATE_DIRTY) {
	      DrawScanlineFunction[Scanline[i].mode](&Scanline[i]);
	      Scanline[i].old_mode=Scanline[i].mode;
	      Scanline[i].old_addr=Scanline[i].addr;
	      lines++;
	      Scanline[i].state=SCANLINE_STATE_DRAWN;
	   }

	return lines;
}

void refreshScreen(void)
{
	XShmPutImage(theDisplay,theWindow,theGC,ScreenXImage,0,0,0,0,
	   560,384,False);
	XFlush(theDisplay);
}

void updateScreen(void)
{
	int lines,i,top,bottom,n,lines_drawn,flag, page;
	static c;

 	lines=0;
	for(i=0;i<192;i++)
	   if (Scanline[i].state==SCANLINE_STATE_DRAWN)
	      lines++;

	if(lines==0){
	   return;
	}

	lines_drawn=0;
	top=0;
	while(lines_drawn<lines){
	   for(;top<192&&Scanline[top].state!=SCANLINE_STATE_DRAWN;top++);
	   for(bottom=top+1;bottom<192&&
	      Scanline[bottom].state==SCANLINE_STATE_DRAWN;bottom++);
	   bottom--;
	   n=bottom-top+1;
	   flag=lines_drawn+n==lines?True:False;
	   XShmPutImage(theDisplay,theWindow,theGC,ScreenXImage,0,top*2,
	      0,top*2,560,n*2,False);
	   lines_drawn+=(bottom-top+1);
	   /* reset state flag */
	   for(i=top;i<=bottom;i++) {
	      Scanline[i].state=SCANLINE_STATE_CLEAN;
	      Scanline[i].old_mode=Scanline[i].mode;
	      Scanline[i].old_addr=Scanline[i].addr;
	      page=Scanline[i].addr>>8;
	      if (MainMemoryPageDirty[page]) {
	         LockAppleMemoryPage(MainMemoryRAM[page], PAGE_LOCK_WRITE);
	         MainMemoryPageDirty[page]=0;
	      }
	      Scanline[i].write_lock_set=
	         GetWriteLockSetTime(MainMemoryRAM[page]);
	      if (Scanline[i].mode > SCANLINE_MODE_HIRES ) {
	         if (AuxMemoryPageDirty[page]) {
	            LockAppleMemoryPage(AuxMemoryRAM[page], PAGE_LOCK_WRITE);
	            AuxMemoryPageDirty[page]=0;
	         }
	         Scanline[i].aux_write_lock_set=
	            GetWriteLockSetTime(AuxMemoryRAM[page]);
	      }
	   }
	   top=bottom+1;
	}
	XFlush(theDisplay);

	/*
	 * seems not a good idea to wait for X server to finsh drawing
	 * this has complicated interaction with speed control
	 * resulting in a low apparent speed as the wait below is included
	 * into speed calculation.
	 */
#if 0
	/* commented out 5/8/97 */
	XSync(theDisplay,False);
#endif
}

static void updateScreenMicrojob(void *arg)
{
	int speed, raw_speed, target_speed, i, new_charset;

#ifdef DEBUG
	static int c;

	if (++c==60){
	   c=0;
	   printf("fps=%d\n",CurrentFPS);
	}
#endif
	/* handle flash text mode and charset selection */
	new_charset=(GetIOUSoftSwitch()&SS_ALTCHAR)? 2:
	   (FlashTextCounter<30? 0:1);

	if (new_charset != CurrentCharSet) {
	   CurrentCharSet=new_charset;
	   changeCharSet(new_charset);
	   for(i=0;i<192;i++)
	      if (Scanline[i].state != SCANLINE_STATE_DIRTY &&
	         ((Scanline[i].mode == SCANLINE_MODE_TEXT)||
	          (Scanline[i].mode == SCANLINE_MODE_TEXT80)))
	         Scanline[i].state = SCANLINE_STATE_DIRTY;
	}

	if(++FlashTextCounter>59)
	    FlashTextCounter=0;

	FPSCounter+=CurrentFPS;
	if (FPSCounter>=TargetFPS) {
	   drawScreen();
	   updateScreen();
	}

	/* frame increase control */
	speed=GetAverageSpeed();
	raw_speed=GetAverageRawSpeed();
	target_speed=GetTargetSpeed();
	if (raw_speed > target_speed) { /* cannot current FPS or higher */
	   CurrentFPS=(double)raw_speed/target_speed*CurrentFPS;
	   if(CurrentFPS>TargetFPS)
	      CurrentFPS=TargetFPS;
	}
	else { /* need to slow down */
	   CurrentFPS=(double)raw_speed/target_speed*CurrentFPS;
	   if (CurrentFPS<5)
	      CurrentFPS=5;
	}
	
	NextVideoRefresh+=FRAME_INTERVAL;
	submit_micro_job(NextVideoRefresh,updateScreenMicrojob,NULL);
}
