/*
 * scanline.c
 *
 * draw scanlines in text, lowres and hires video modes.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "memory.h"
#include "video.h"

unsigned char EvenNormalTextFontPixel[128][8][16];
unsigned char OddNormalTextFontPixel[128][8][16];
unsigned char EvenInverseTextFontPixel[128][8][16];
unsigned char OddInverseTextFontPixel[128][8][16];

unsigned char NormalText80FontPixel0[128][8][12];
unsigned char NormalText80FontPixel1[128][8][12];
unsigned char NormalText80FontPixel2[128][8][12];
unsigned char NormalText80FontPixel3[128][8][12];

unsigned char InverseText80FontPixel0[128][8][12];
unsigned char InverseText80FontPixel1[128][8][12];
unsigned char InverseText80FontPixel2[128][8][12];
unsigned char InverseText80FontPixel3[128][8][12];

/* since alternate character set does not contain flash characters */
/* there is only one set for 40 and 80 column. There are two actual */
/* character sets for the primary one. One has the flash characters */
/* normal and the other the flash characters inverse */

/* 40 column character set, one set if split into two parts. One part */
/* for the even column and the other for the odd column */
unsigned char *PrimaryEvenCharSet0[256];
unsigned char *PrimaryOddCharSet0[256];
unsigned char *PrimaryEvenCharSet1[256];
unsigned char *PrimaryOddCharSet1[256];
unsigned char *AltEvenCharSet[256];
unsigned char *AltOddCharSet[256];

/* 80 column character set, one set if split into four parts */
unsigned char *Primary80CharSet0a[256];
unsigned char *Primary80CharSet1a[256];
unsigned char *Primary80CharSet2a[256];
unsigned char *Primary80CharSet3a[256];
unsigned char *Primary80CharSet0b[256];
unsigned char *Primary80CharSet1b[256];
unsigned char *Primary80CharSet2b[256];
unsigned char *Primary80CharSet3b[256];
unsigned char *Alt80CharSet0[256];
unsigned char *Alt80CharSet1[256];
unsigned char *Alt80CharSet2[256];
unsigned char *Alt80CharSet3[256];

/* these point to character set currently being used */
unsigned char **EvenCharSet;
unsigned char **OddCharSet;
unsigned char **Text80CharSet0;
unsigned char **Text80CharSet1;
unsigned char **Text80CharSet2;
unsigned char **Text80CharSet3;

unsigned long PhaseShiftedQuad[2048][2];
unsigned long PhaseShiftedColour[16][4];

int BitShiftTable1[256];
int BitShiftTable2[256];
int BitShiftTable3[256];
int BitShiftTable4[256];

int ExpansionTable[256];
int DLoresBitTable1[16];
int DLoresBitTable2[16];
int DLoresBitTable3[16];
int DLoresBitTable4[16];

int LoresBitTable1[16];
int LoresBitTable2[16];

extern char *AppleFontBitmap[];
extern char *MouseTextFontBitmap[];

void initScanlineData(void)
{
	int i,j,k,data,bits;
	unsigned char *font_data;
	unsigned long c, c2;

	/* text pixel data initialization */

	memset(EvenNormalTextFontPixel,0,sizeof(EvenNormalTextFontPixel));
	memset(OddNormalTextFontPixel,0,sizeof(OddNormalTextFontPixel));
	memset(EvenInverseTextFontPixel,0,sizeof(EvenInverseTextFontPixel));
	memset(OddInverseTextFontPixel,0,sizeof(OddInverseTextFontPixel));

	memset(NormalText80FontPixel0,0,sizeof(NormalText80FontPixel0));
	memset(NormalText80FontPixel1,0,sizeof(NormalText80FontPixel1));
	memset(NormalText80FontPixel2,0,sizeof(NormalText80FontPixel2));
	memset(NormalText80FontPixel3,0,sizeof(NormalText80FontPixel3));
	memset(InverseText80FontPixel0,0,sizeof(InverseText80FontPixel0));
	memset(InverseText80FontPixel1,0,sizeof(InverseText80FontPixel1));
	memset(InverseText80FontPixel2,0,sizeof(InverseText80FontPixel2));
	memset(InverseText80FontPixel3,0,sizeof(InverseText80FontPixel3));

	for(i=0;i<0x80;i++) {	/* create default font data */
	   /* generate 40 column font */
	   font_data=i>0x5f? MouseTextFontBitmap[i-0x60]:AppleFontBitmap[i];
	   for(j=0;j<8;j++) {
	      data=font_data[j];
	      for(k=0;k<14;k+=2) {
	         if (data&0x01) {
	            c=AppleColourPixel[15];
	            c2=AppleColourPixel[0];
	         }
	         else {
	            c=AppleColourPixel[0];
	            c2=AppleColourPixel[15];
	         }
	         EvenNormalTextFontPixel[i][j][k]=c;
	         EvenNormalTextFontPixel[i][j][k+1]=c;
	         OddNormalTextFontPixel[i][j][k+2]=c;
	         OddNormalTextFontPixel[i][j][k+3]=c;
	         EvenInverseTextFontPixel[i][j][k]=c2;
	         EvenInverseTextFontPixel[i][j][k+1]=c2;
	         OddInverseTextFontPixel[i][j][k+2]=c2;
	         OddInverseTextFontPixel[i][j][k+3]=c2;
	         data>>=1;
	      }
	   }

	   /* generate 80 column font */
	   for(j=0;j<8;j++){
	      data=font_data[j];
	      for(k=0;k<7;k++){
	         if (data&0x01) {
	            c=AppleColourPixel[15];
	            c2=AppleColourPixel[0];
	         }
	         else {
	            c=AppleColourPixel[0];
	            c2=AppleColourPixel[15];
	         }
	         NormalText80FontPixel0[i][j][k]=c;
	         NormalText80FontPixel1[i][j][k+3]=c;
	         NormalText80FontPixel2[i][j][k+2]=c;
	         NormalText80FontPixel3[i][j][k+1]=c;
	         InverseText80FontPixel0[i][j][k]=c2;
	         InverseText80FontPixel1[i][j][k+3]=c2;
	         InverseText80FontPixel2[i][j][k+2]=c2;
	         InverseText80FontPixel3[i][j][k+1]=c2;
	         data>>=1;
	      }
	   }
	}

	/* create primary and alternate character sets */
	for(i=0;i<256;i++){
	   if (i<0x40){ /* $00-$3F inverse characters */
	      PrimaryEvenCharSet0[i]=(void*)EvenInverseTextFontPixel[i];
	      PrimaryOddCharSet0[i]=(void*)OddInverseTextFontPixel[i];
	      PrimaryEvenCharSet1[i]=(void*)EvenInverseTextFontPixel[i];
	      PrimaryOddCharSet1[i]=(void*)OddInverseTextFontPixel[i];
	      Primary80CharSet0a[i]=(void*)InverseText80FontPixel0[i];
	      Primary80CharSet1a[i]=(void*)InverseText80FontPixel1[i];
	      Primary80CharSet2a[i]=(void*)InverseText80FontPixel2[i];
	      Primary80CharSet3a[i]=(void*)InverseText80FontPixel3[i];
	      Primary80CharSet0b[i]=(void*)InverseText80FontPixel0[i];
	      Primary80CharSet1b[i]=(void*)InverseText80FontPixel1[i];
	      Primary80CharSet2b[i]=(void*)InverseText80FontPixel2[i];
	      Primary80CharSet3b[i]=(void*)InverseText80FontPixel3[i];
	   }
	   else if (i<0x80) { /* $40-$7F flash characters */
	      PrimaryEvenCharSet0[i]=(void*)EvenNormalTextFontPixel[i&0x3f];
	      PrimaryOddCharSet0[i]=(void*)OddNormalTextFontPixel[i&0x3f];
	      PrimaryEvenCharSet1[i]=(void*)EvenInverseTextFontPixel[i&0x3f];
	      PrimaryOddCharSet1[i]=(void*)OddInverseTextFontPixel[i&0x3f];
	      Primary80CharSet0a[i]=(void*)NormalText80FontPixel0[i&0x3f];
	      Primary80CharSet1a[i]=(void*)NormalText80FontPixel1[i&0x3f];
	      Primary80CharSet2a[i]=(void*)NormalText80FontPixel2[i&0x3f];
	      Primary80CharSet3a[i]=(void*)NormalText80FontPixel3[i&0x3f];
	      Primary80CharSet0b[i]=(void*)InverseText80FontPixel0[i&0x3f];
	      Primary80CharSet1b[i]=(void*)InverseText80FontPixel1[i&0x3f];
	      Primary80CharSet2b[i]=(void*)InverseText80FontPixel2[i&0x3f];
	      Primary80CharSet3b[i]=(void*)InverseText80FontPixel3[i&0x3f];
	   }
	   else if (i<0xE0) { /* $80-$DF normal characters */
	      PrimaryEvenCharSet0[i]=(void*)EvenNormalTextFontPixel[i&0x3f];
	      PrimaryOddCharSet0[i]=(void*)OddNormalTextFontPixel[i&0x3f];
	      PrimaryEvenCharSet1[i]=(void*)EvenNormalTextFontPixel[i&0x3f];
	      PrimaryOddCharSet1[i]=(void*)OddNormalTextFontPixel[i&0x3f];
	      Primary80CharSet0a[i]=(void*)NormalText80FontPixel0[i&0x3f];
	      Primary80CharSet1a[i]=(void*)NormalText80FontPixel1[i&0x3f];
	      Primary80CharSet2a[i]=(void*)NormalText80FontPixel2[i&0x3f];
	      Primary80CharSet3a[i]=(void*)NormalText80FontPixel3[i&0x3f];
	      Primary80CharSet0b[i]=(void*)NormalText80FontPixel0[i&0x3f];
	      Primary80CharSet1b[i]=(void*)NormalText80FontPixel1[i&0x3f];
	      Primary80CharSet2b[i]=(void*)NormalText80FontPixel2[i&0x3f];
	      Primary80CharSet3b[i]=(void*)NormalText80FontPixel3[i&0x3f];
	   }
	   else { /* lower case characters */
	      PrimaryEvenCharSet0[i]=(void*)EvenNormalTextFontPixel[i-0xA0];
	      PrimaryOddCharSet0[i]=(void*)OddNormalTextFontPixel[i-0xA0];
	      PrimaryEvenCharSet1[i]=(void*)EvenNormalTextFontPixel[i-0xA0];
	      PrimaryOddCharSet1[i]=(void*)OddNormalTextFontPixel[i-0xA0];
	      Primary80CharSet0a[i]=(void*)NormalText80FontPixel0[i-0xA0];
	      Primary80CharSet1a[i]=(void*)NormalText80FontPixel1[i-0xA0];
	      Primary80CharSet2a[i]=(void*)NormalText80FontPixel2[i-0xA0];
	      Primary80CharSet3a[i]=(void*)NormalText80FontPixel3[i-0xA0];
	      Primary80CharSet0b[i]=(void*)NormalText80FontPixel0[i-0xA0];
	      Primary80CharSet1b[i]=(void*)NormalText80FontPixel1[i-0xA0];
	      Primary80CharSet2b[i]=(void*)NormalText80FontPixel2[i-0xA0];
	      Primary80CharSet3b[i]=(void*)NormalText80FontPixel3[i-0xA0];
	   }
	
	   /* alternate character set */
	   if (i>=0x40&&i<=0x5f) { /* MouseText characters */
	      AltEvenCharSet[i]=(void*)EvenNormalTextFontPixel[i+0x20];
	      AltOddCharSet[i]=(void*)OddNormalTextFontPixel[i+0x20];
	      Alt80CharSet0[i]=(void*)NormalText80FontPixel0[i+0x20];
	      Alt80CharSet1[i]=(void*)NormalText80FontPixel1[i+0x20];
	      Alt80CharSet2[i]=(void*)NormalText80FontPixel2[i+0x20];
	      Alt80CharSet3[i]=(void*)NormalText80FontPixel3[i+0x20];
	   }
	   else if (i>=0x60&&i<=0x7f) { /* inverse lower case */
	      AltEvenCharSet[i]=(void*)EvenInverseTextFontPixel[i-0x20];
	      AltOddCharSet[i]=(void*)OddInverseTextFontPixel[i-0x20];
	      Alt80CharSet0[i]=(void*)InverseText80FontPixel0[i-0x20];
	      Alt80CharSet1[i]=(void*)InverseText80FontPixel1[i-0x20];
	      Alt80CharSet2[i]=(void*)InverseText80FontPixel2[i-0x20];
	      Alt80CharSet3[i]=(void*)InverseText80FontPixel3[i-0x20];
	   }
	   else { /* same as primary character set */
	      AltEvenCharSet[i]=PrimaryEvenCharSet0[i];
	      AltOddCharSet[i]=PrimaryOddCharSet0[i];
	      Alt80CharSet0[i]=Primary80CharSet0a[i];
	      Alt80CharSet1[i]=Primary80CharSet1a[i];
	      Alt80CharSet2[i]=Primary80CharSet2a[i];
	      Alt80CharSet3[i]=Primary80CharSet3a[i];
	   }
	}

	changeCharSet(0);

	/* hires pixel data initialization */
	for(i=0;i<256;i++){ /* build expansion table */
	   for(k=j=0;j<8;j++)
	      if (i&(1<<j))
	         k|=3<<(j+j);
	   ExpansionTable[i]=k;
	}

	for(i=0;i<16;i++) /* build phase shift table */
	   for(j=0;j<4;j++){
	      k=((i<<j)|(i>>(4-j)))&0x0f;
	      PhaseShiftedColour[i][j]=AppleColourPixel[k];
	   }

	for(i=0;i<2048;i++){
	   bits=i;
	   for(j=0;j<8;j++){
	      ((unsigned char*)(PhaseShiftedQuad[i]))[j]=
	         PhaseShiftedColour[bits&0xf][(j+2)&0x3];
	      bits>>=1;
	   }
	}

	for(i=0;i<256;i++){
	   BitShiftTable1[i]=
	      ExpansionTable[i&0x7f]<<((i&0x80)? 1:0);
	   BitShiftTable2[i]=
	      ExpansionTable[i&0x7f]<<((i&0x80)? 7:6);
	   BitShiftTable3[i]=
	      ExpansionTable[i&0x7f]<<((i&0x80)? 5:4);
	   BitShiftTable4[i]=
	      ExpansionTable[i&0x7f]<<((i&0x80)? 3:2);
	}

	/* Lores & Double lowres */
	for(i=0;i<16;i++){
	   j=i|i<<4;
	   j|=j<<8;
	   j|=j<<16;
	   LoresBitTable1[i]=j&0x3fff;
	   LoresBitTable2[i]=j&(0x3fff<<14);
	   DLoresBitTable1[i]=j&0x7f;
	   DLoresBitTable2[i]=j&(0x7f<<7);
	   DLoresBitTable3[i]=j&(0x7f<<14);
	   DLoresBitTable4[i]=j&(0x7f<<21);
	}
}

void drawBlankScanline(scanline_t *s)
{
	int i;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long quad;

	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	quad=AppleColourQuad[0];
	for(i=0;i<140;i++)
	   scanline_addr1[i]=scanline_addr2[i]=quad;
}

void drawTextScanline(scanline_t *s)
{
	int cline,col,offset;
	unsigned char *row_addr,c;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long *char_pixel, quad;

	cline=s->y&0x07;
	offset=cline*16;
	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	for(col=0;col<40;col+=2){
	   c=row_addr[col];
	   char_pixel=(void*)(EvenCharSet[c]+offset);
	   scanline_addr1[0]=scanline_addr2[0]=char_pixel[0];
	   scanline_addr1[1]=scanline_addr2[1]=char_pixel[1];
	   scanline_addr1[2]=scanline_addr2[2]=char_pixel[2];
	   quad=char_pixel[3];
	   c=row_addr[col+1];
	   char_pixel=(void*)(OddCharSet[c]+offset);
	   scanline_addr1[3]=scanline_addr2[3]=char_pixel[0]|quad;
	   scanline_addr1[4]=scanline_addr2[4]=char_pixel[1];
	   scanline_addr1[5]=scanline_addr2[5]=char_pixel[2];
	   scanline_addr1[6]=scanline_addr2[6]=char_pixel[3];
	   scanline_addr1 += 7;
	   scanline_addr2 += 7;
	}
}

void drawLowresScanline(scanline_t *s)
{
	int col,i,bits1, bits2;
	unsigned char *row_addr, buf[71], *bp;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long pixel;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	bp=buf;
	for(col=0;col<40;col+=4){
	   if (s->y&0x4){ /* higher nibble */
	      bits1=LoresBitTable1[row_addr[0]>>4]|
	         LoresBitTable2[row_addr[1]>>4];
	      bits2=LoresBitTable1[row_addr[2]>>4]|
	         LoresBitTable2[row_addr[3]>>4];
	   }
	   else {
	      bits1=LoresBitTable1[row_addr[0]&0xf]|
	         LoresBitTable2[row_addr[1]&0xf];
	      bits2=LoresBitTable1[row_addr[2]&0xf]|
	         LoresBitTable2[row_addr[3]&0xf];
	   }
	   bp[0]=bits1;
	   bp[1]=bits1>>8;
	   bp[2]=bits1>>16;
	   bits1=(bits2<<4)|(bits1>>24);
	   bp[3]=bits1;
	   bp[4]=bits1>>8;
	   bp[5]=bits1>>16;
	   bp[6]=bits1>>24;
	   row_addr+=4;
	   bp+=7;
	}
	buf[70]=0;

	bits1=buf[0]<<2;
	for(i=0;i<70;i++) {
	   bits1|=(buf[i+1]<<10);
	   pixel=PhaseShiftedQuad[bits1&0x7ff][0];
	   scanline_addr1[0]=scanline_addr2[0]=pixel;
	   pixel=PhaseShiftedQuad[bits1&0x7ff][1];
	   scanline_addr1[1]=scanline_addr2[1]=pixel;
	   bits1>>=8;
	   scanline_addr1+=2;
	   scanline_addr2+=2;
	}
}

void drawHiresScanline(scanline_t *s)
{
	int col,i,bits;
	unsigned char *row_addr,c1,buf[71], *bp;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long pixel;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	bp=buf;
	bits=0;
	for(col=0;col<40;col+=4){
	   c1=row_addr[col];
	   bits|=BitShiftTable1[c1];
	   bp[0]=bits;
	   bits>>=8;
	   c1=row_addr[col+1];
	   bits|=BitShiftTable2[c1];
	   bp[1]=bits;
	   bits>>=8;
	   bp[2]=bits;
	   bits>>=8;
	   c1=row_addr[col+2];
	   bits|=BitShiftTable3[c1];
	   bp[3]=bits;
	   bits>>=8;
	   bp[4]=bits;
	   bits>>=8;
	   c1=row_addr[col+3];
	   bits|=BitShiftTable4[c1];
	   bp[5]=bits;
	   bits>>=8;
	   bp[6]=bits;
	   bits>>=8;
	   bp+=7;
	}
	buf[70]=0;

	bits=buf[0]<<2;
	for(i=0;i<70;i++) {
	   bits|=(buf[i+1]<<10);
	   pixel=PhaseShiftedQuad[bits&0x7ff][0];
	   scanline_addr1[0]=scanline_addr2[0]=pixel;
	   pixel=PhaseShiftedQuad[bits&0x7ff][1];
	   scanline_addr1[1]=scanline_addr2[1]=pixel;
	   bits>>=8;
	   scanline_addr1+=2;
	   scanline_addr2+=2;
	}
}

void drawText80Scanline(scanline_t *s)
{
	int cline,col,offset;
	unsigned char *row_addr, *aux_row_addr,c;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long *char_pixel, quad;

	cline=s->y&0x07;
	offset=cline*12;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	aux_row_addr=(BYTE*)AuxMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	for(col=0;col<40;col+=2){
	   c=aux_row_addr[col];
	   char_pixel=(void*)(Text80CharSet0[c]+offset);
	   scanline_addr1[0]=scanline_addr2[0]=char_pixel[0];
	   quad=char_pixel[1];
	   c=row_addr[col];
	   char_pixel=(void*)(Text80CharSet1[c]+offset);
	   scanline_addr1[1]=scanline_addr2[1]=char_pixel[0]|quad;
	   scanline_addr1[2]=scanline_addr2[2]=char_pixel[1];
	   quad=char_pixel[2];
	   c=aux_row_addr[col+1];
	   char_pixel=(void*)(Text80CharSet2[c]+offset);
	   scanline_addr1[3]=scanline_addr2[3]=char_pixel[0]|quad;
	   scanline_addr1[4]=scanline_addr2[4]=char_pixel[1];
	   quad=char_pixel[2];
	   c=row_addr[col+1];
	   char_pixel=(void*)(Text80CharSet3[c]+offset);
	   scanline_addr1[5]=scanline_addr2[5]=char_pixel[0]|quad;
	   scanline_addr1[6]=scanline_addr2[6]=char_pixel[1];
	   scanline_addr1 += 7;
	   scanline_addr2 += 7;
	}
}

void drawDLoresScanline(scanline_t *s)
{
	int col,i,bits1,bits2;
	unsigned char *row_addr, *aux_row_addr, buf[71], *bp;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long pixel;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	aux_row_addr=(BYTE*)AuxMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	bp=buf;
	for(col=0;col<40;col+=4){
	   if (s->y&0x4){ /* higher nibble */
	      bits1=DLoresBitTable1[aux_row_addr[0]>>4]|
	         DLoresBitTable2[row_addr[0]>>4]|
	         DLoresBitTable3[aux_row_addr[1]>>4]|
	         DLoresBitTable4[row_addr[1]>>4];
	      bits2=DLoresBitTable1[aux_row_addr[2]>>4]|
	         DLoresBitTable2[row_addr[2]>>4]|
	         DLoresBitTable3[aux_row_addr[3]>>4]|
	         DLoresBitTable4[row_addr[3]>>4];
	   }
	   else {
	      bits1=DLoresBitTable1[aux_row_addr[0]&0xf]|
	         DLoresBitTable2[row_addr[0]&0xf]|
	         DLoresBitTable3[aux_row_addr[1]&0xf]|
	         DLoresBitTable4[row_addr[1]&0xf];
	      bits2=DLoresBitTable1[aux_row_addr[2]&0xf]|
	         DLoresBitTable2[row_addr[2]>>4]|
	         DLoresBitTable3[aux_row_addr[3]>>4]|
	         DLoresBitTable4[row_addr[3]>>4];
	   }
	   bp[0]=bits1;
	   bp[1]=bits1>>8;
	   bp[2]=bits1>>16;
	   bits1=(bits2<<4)|(bits1>>24);
	   bp[3]=bits1;
	   bp[4]=bits1>>8;
	   bp[5]=bits1>>16;
	   bp[6]=bits1>>24;
	   aux_row_addr+=4;
	   row_addr+=4;
	   bp+=7;
	}
	buf[70]=0;

	bits1=buf[0]<<2;
	for(i=0;i<70;i++) {
	   bits1|=(buf[i+1]<<10);
	   pixel=PhaseShiftedQuad[bits1&0x7ff][0];
	   scanline_addr1[0]=scanline_addr2[0]=pixel;
	   pixel=PhaseShiftedQuad[bits1&0x7ff][1];
	   scanline_addr1[1]=scanline_addr2[1]=pixel;
	   bits1>>=8;
	   scanline_addr1+=2;
	   scanline_addr2+=2;
	}
}

void drawDHiresScanline(scanline_t *s)
{
	int col,i,bits;
	unsigned char *row_addr, *aux_row_addr, buf[71], *bp;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long pixel;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	aux_row_addr=(BYTE*)AuxMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	bp=buf;
	for(col=0;col<40;col+=4){
	   bits=(aux_row_addr[0]&0x7f)|((row_addr[0]&0x7f)<<7);
	   bp[0]=bits;
	   bits>>=8;
	   bits|=((aux_row_addr[1]&0x7f)<<6)|((row_addr[1]&0x7f)<<13);
	   bp[1]=bits;
	   bits>>=8;
	   bp[2]=bits;
	   bits>>=8;
	   bits|=((aux_row_addr[2]&0x7f)<<4)|((row_addr[2]&0x7f)<<11);
	   bp[3]=bits;
	   bits>>=8;
	   bp[4]=bits;
	   bits>>=8;
	   bits|=((aux_row_addr[3]&0x7f)<<2)|((row_addr[3]&0x7f)<<9);
	   bp[5]=bits;
	   bits>>=8;
	   bp[6]=bits;
	   aux_row_addr+=4;
	   row_addr+=4;
	   bp+=7;
	}
	buf[70]=0;

	bits=buf[0]<<2;
	for(i=0;i<70;i++) {
	   bits|=(buf[i+1]<<10);
	   pixel=PhaseShiftedQuad[bits&0x7ff][0];
	   scanline_addr1[0]=scanline_addr2[0]=pixel;
	   pixel=PhaseShiftedQuad[bits&0x7ff][1];
	   scanline_addr1[1]=scanline_addr2[1]=pixel;
	   bits>>=8;
	   scanline_addr1+=2;
	   scanline_addr2+=2;
	}
}

void changeCharSet(int charset)
{
	switch(charset){
	case 0: /* Primary Character Set, flash text normal */ 
	   EvenCharSet=PrimaryEvenCharSet0;
	   OddCharSet=PrimaryOddCharSet0;
	   Text80CharSet0=Primary80CharSet0a;
	   Text80CharSet1=Primary80CharSet1a;
	   Text80CharSet2=Primary80CharSet2a;
	   Text80CharSet3=Primary80CharSet3a;
	   break;
	case 1: /* Primary Character Set, flash text inverse */ 
	   EvenCharSet=PrimaryEvenCharSet1;
	   OddCharSet=PrimaryOddCharSet1;
	   Text80CharSet0=Primary80CharSet0b;
	   Text80CharSet1=Primary80CharSet1b;
	   Text80CharSet2=Primary80CharSet2b;
	   Text80CharSet3=Primary80CharSet3b;
	   break;
	case 2: /* Alt Character Set */ 
	   EvenCharSet=AltEvenCharSet;
	   OddCharSet=AltOddCharSet;
	   Text80CharSet0=Alt80CharSet0;
	   Text80CharSet1=Alt80CharSet1;
	   Text80CharSet2=Alt80CharSet2;
	   Text80CharSet3=Alt80CharSet3;
	   break;
	}
}

