/*
 * memmap.c
 *
 * memory mapping
 *
 * Change log
 * $Log: memmap.c,v $
 * Revision 1.8  1997/08/06  09:49:49  ctkwan
 * deferred memory mapping change.
 *
 * Revision 1.7  1997/08/05  16:26:29  ctkwan
 * revised memory mapping.
 *
 * Revision 1.6  1997/08/04  16:34:39  ctkwan
 * add $C100-$CEFF direct mapping for read.
 *
 * Revision 1.6  1997/08/04  16:34:39  ctkwan
 * add $C100-$CEFF direct mapping for read.
 *
 * Revision 1.5  1997/08/04  11:42:00  ctkwan
 * correct SLOTCXROM switch semantics.
 *
 * Revision 1.4  1997/08/02  09:30:29  ctkwan
 * *** empty log message ***
 *
 * Revision 1.4  1997/08/02  09:30:29  ctkwan
 * *** empty log message ***
 *
 * Revision 1.3  1997/08/01  18:11:50  ctkwan
 * change Language Card switch to ALTZP.
 *
 * Revision 1.2  1997/08/01  14:24:45  ctkwan
 * *** empty log message ***
 *
 */

#include "hardware.h"
#include "memory.h"
#include "iou.h"

static char rcsid[]="$Id: memmap.c,v 1.8 1997/08/06 09:49:49 ctkwan Exp ctkwan $";

static int	  RAMReadMapValid=0;
static int	  RAMWriteMapValid=0;
static int	  TextReadMapValid=0;
static int	  TextWriteMapValid=0;
static int	  HiresReadMapValid=0;
static int	  HiresWriteMapValid=0;
static int	  LCReadMapValid=0;
static int	  LCWriteMapValid=0;

static APPLE_TICK RAMReadMapLastChange;
static APPLE_TICK RAMWriteMapLastChange;
static APPLE_TICK TextReadMapLastChange;
static APPLE_TICK TextWriteMapLastChange;
static APPLE_TICK HiresReadMapLastChange;
static APPLE_TICK HiresWriteMapLastChange;
static APPLE_TICK LCReadMapLastChange;
static APPLE_TICK LCWriteMapLastChange;

static void unmapReadRange( int, int );
static void unmapWriteRange( int, int );

static void unmapReadRange( int start, int end )
{
	int	page;

	for(page=start;page<=end;page++){
	   MapReadPage(page,NULL);
	   LockReadMappedPage(page,PAGE_LOCK_READ);
	}
}

static void unmapWriteRange( int start, int end )
{
	int	page;

	for(page=start;page<=end;page++){
	   MapWritePage(page,NULL);
	   LockWriteMappedPage(page,PAGE_LOCK_WRITE);
	}
}

/*
 * remapZP:
 *
 * set up zero page and stack memory mapping
 */
void remapZP(void)
{
	AppleMemoryPage test_page, *bank;

	test_page=(void*)((unsigned)Read_Page_Table[0]&~3);
	bank=(GetIOUSoftSwitch()&SS_ALTZP)? AuxMemoryRAM: MainMemoryRAM;
	if (test_page!=bank[0]){
#if 0 && defined(DEBUG)
	   printf("remap %s\n", bank==MainMemoryRAM? "STDZP": "ALTZP");
#endif /* DEBUG */
	   MapReadPage( 0, bank[0] );
	   MapReadPage( 1, bank[1] );
	   MapWritePage( 0, bank[0] );
	   MapWritePage( 1, bank[1] );
	}
}

/*
 * remapRAMRead:
 *
 * set up read mapping in region $200-$BFFF except text page 1 $400-$7FF
 * and hires page 1 $2000-$3FFF.
 */
void remapRAMRead(void)
{
	int i;
	AppleMemoryPage test_page, *bank;
	unsigned state;

	state=GetIOUSoftSwitch();

	if (RAMReadMapValid){
	    unmapReadRange(0x02,0x03);
	    unmapReadRange(0x08,0x1F);
	    unmapReadRange(0x40,0xBF);
	    RAMReadMapValid=0;
	    RAMReadMapLastChange=AppleClock;
	    return;
	}

	if (AppleClock-RAMReadMapLastChange<15600){
	    RAMReadMapLastChange=AppleClock;
	    return;	
	}

	bank=state&SS_RAMRD? AuxMemoryRAM : MainMemoryRAM;
	test_page=(void*)((unsigned)Read_Page_Table[2]&~3);
	if (test_page!=bank[2]){
#ifdef DEBUG
	   printf("remamp %s RAM for read at %d\n",
	      state&SS_RAMRD? "aux":"main",AppleClock);
#endif
	   MapReadPage( 2, bank[2]);
	   MapReadPage( 3, bank[3]);
	   for(i=0x08;i<0x20;i++)
	      MapReadPage( i, bank[i]);
	   for(i=0x40;i<0xc0;i++)
	      MapReadPage( i, bank[i]);
	}
	RAMReadMapValid=1;
	RAMReadMapLastChange=AppleClock;
}

/*
 * remapRAMWrite:
 *
 * set up write mapping in region $200-$BFFF except text page 1 $400-$7FF
 * and hires page 1 $2000-$3FFF.
 */
void remapRAMWrite(void)
{

	int i;
	AppleMemoryPage test_page, *bank;
	unsigned state;

	state=GetIOUSoftSwitch();

	if (RAMWriteMapValid){
	    unmapWriteRange(0x02,0x03);
	    unmapWriteRange(0x08,0x1F);
	    unmapWriteRange(0x40,0xBF);
	    RAMWriteMapValid=0;
	    RAMWriteMapLastChange=AppleClock;
	    return;
	}

	if (AppleClock-RAMWriteMapLastChange<15600){
	    RAMWriteMapLastChange=AppleClock;
	    return;	
	}

	bank=state&SS_RAMWRT? AuxMemoryRAM : MainMemoryRAM;
	test_page=(void*)((unsigned)Write_Page_Table[2]&~3);
	if (test_page!=bank[2]){
#ifdef DEBUG
           printf("remap %s RAM for write\n",state&SS_RAMWRT?"aux":"main");
#endif
	   MapWritePage( 2, bank[2]);
	   MapWritePage( 3, bank[3]);
	   for(i=0x08;i<0x20;i++)
	      MapWritePage( i, bank[i]);
	   for(i=0x40;i<0xc0;i++)
	      MapWritePage( i, bank[i]);
	}
	RAMWriteMapValid=1;
	RAMWriteMapLastChange=AppleClock;
}

/*
 * remapText:
 *
 * set up read memory mapping in text page 1 $400-$7FF.
 */
void remapTextRead(void)
{
	int i;
	AppleMemoryPage test_page, *bank;
	unsigned state;

	state=GetIOUSoftSwitch();

	if (TextReadMapValid){
	    unmapReadRange(0x04,0x07);
	    TextReadMapValid=0;
	    TextReadMapLastChange=AppleClock;
	    return;
	}

	if (AppleClock-TextReadMapLastChange<400){
	    TextReadMapLastChange=AppleClock;
	    return;
	}

	if (state & SS_80STORE)
	   bank=state&SS_PAGE2? AuxMemoryRAM : MainMemoryRAM;
	else
	   bank=state&SS_RAMRD? AuxMemoryRAM : MainMemoryRAM;

	test_page=(void*)((unsigned)Read_Page_Table[4]&~3);
	if (test_page!=bank[4]){
#ifdef DEBUG
	   printf("remap %s Text for read at %d\n",bank==AuxMemoryRAM?
	      "aux":"main",AppleClock);
#endif
	   for(i=0x04;i<0x08;i++)
	      MapReadPage( i, bank[i] );
	}

	TextReadMapValid=1;
	TextReadMapLastChange=AppleClock;
}


/*
 * remapTextWrite:
 *
 * set up write memory mapping in text page 1 $400-$7FF.
 */
void remapTextWrite(void)
{
	int i;
	AppleMemoryPage test_page, *bank;
	unsigned state;

	state=GetIOUSoftSwitch();

	if (TextWriteMapValid){
	    unmapWriteRange(0x04,0x07);
	    TextWriteMapValid=0;
	    TextWriteMapLastChange=AppleClock;
	    return;
	}

	if (AppleClock-TextWriteMapLastChange<400){
	    TextWriteMapLastChange=AppleClock;
	    return;	
	}

	if (state & SS_80STORE)
	   bank=state&SS_PAGE2? AuxMemoryRAM : MainMemoryRAM;
	else
	   bank=state&SS_RAMWRT? AuxMemoryRAM : MainMemoryRAM;

	test_page=(void*)((unsigned)Write_Page_Table[4]&~3);
	if (test_page!=bank[4]){
#ifdef DEBUG
	   printf("remap %s Text for write at %d\n",bank==AuxMemoryRAM?
	      "aux":"main",AppleClock);
#endif
	   for(i=0x04;i<0x08;i++)
	      MapWritePage( i, bank[i] );
	}
	TextWriteMapValid=1;
	TextWriteMapLastChange=AppleClock;
}


/*
 * remapHiresRead:
 *
 * set up read memory mapping in text page 1 $2000-$3FFF.
 */
void remapHiresRead(void)
{
	int i;
	AppleMemoryPage test_page, *bank;
	unsigned state;

	state=GetIOUSoftSwitch();

	if (HiresReadMapValid){
	    unmapReadRange(0x20,0x3F);
	    HiresReadMapValid=0;
	    HiresReadMapLastChange=AppleClock;
	    return;
	}

	if (AppleClock-HiresReadMapLastChange<3200){
	    HiresReadMapLastChange=AppleClock;
	    return;	
	}

	if ((state & (SS_80STORE|SS_HIRES)) == (SS_80STORE|SS_HIRES))
	   bank=state&SS_PAGE2? AuxMemoryRAM : MainMemoryRAM;
	else
	   bank=state&SS_RAMRD? AuxMemoryRAM : MainMemoryRAM;

	test_page=(void*)((unsigned)Read_Page_Table[0x20]&~3);
	if (test_page!=bank[0x20]){
#ifdef DEBUG
	   printf("remap %s Hires for read\n",bank==AuxMemoryRAM?
	      "aux":"main");
#endif
	   for(i=0x20;i<0x40;i++)
	      MapReadPage( i, bank[i] );
	}

	HiresReadMapValid=1;
	HiresReadMapLastChange=AppleClock;
}

/*
 * remapHiresWrite:
 *
 * set up write memory mapping in text page 1 $2000-$3FFF.
 */
void remapHiresWrite(void)
{
	int i;
	AppleMemoryPage test_page, *bank;
	unsigned state;

	state=GetIOUSoftSwitch();

	if (HiresWriteMapValid){
	    unmapWriteRange(0x20,0x3F);
	    HiresWriteMapValid=0;
	    HiresWriteMapLastChange=AppleClock;
	    return;
	}

	if (AppleClock-HiresWriteMapLastChange<3200){
	    HiresWriteMapLastChange=AppleClock;
	    return;	
	}

	if ((state & (SS_80STORE|SS_HIRES)) == (SS_80STORE|SS_HIRES))
	   bank=state&SS_PAGE2? AuxMemoryRAM : MainMemoryRAM;
	else
	   bank=state&SS_RAMWRT? AuxMemoryRAM : MainMemoryRAM;

	test_page=(void*)((unsigned)Write_Page_Table[0x20]&~3);
	if (test_page!=bank[0x20]){
#ifdef DEBUG
	   printf("remap %s Hires for write\n",bank==AuxMemoryRAM?
	      "aux":"main");
#endif
	   for(i=0x20;i<0x40;i++)
	      MapWritePage( i, bank[i] );
	}

	HiresWriteMapValid=1;
	HiresWriteMapLastChange=AppleClock;
}

/*
 * remapCXROM
 * set up memory mapping in region $C000-$CXXX
 */
void remapCXROM(void)
{
	int i;
	unsigned state;

	state=GetIOUSoftSwitch();

	/* $C100-$C7FF */
	if (state&SS_SLOTCXROM){
	   for(i=0xC1;i<0xC8;i++)
	      MapReadPage(i,(void*)MainMemoryROM[i]);
	}
	else {
	   for(i=0xC1;i<0xC8;i++) {
	      if (i!=0xC3||(state&SS_SLOTC3ROM)) {
	         MapReadPage(i,NULL);
	         LockReadMappedPage(i,PAGE_LOCK_READ);
	      }
	      else
	         MapReadPage(0xC3,MainMemoryROM[0xC3]);
	   }
	}

	/* $C800-$CFFF */
	if ((state&SS_SLOTCXROM)||(state&SS_EXPNROM)){ /* internal ROM */
#ifdef DEBUG
	   printf("map $C800-$CEFF internal ROM\n");
#endif
	   for(i=0xC8;i<0xCF;i++)
	      MapReadPage( i,(void*)MainMemoryROM[i]);
	}
	else{
	   for(i=0xC8;i<0xCF;i++){
	      MapReadPage(i,NULL);
	      LockReadMappedPage(i,PAGE_LOCK_READ);
	   }
	}
}

/*
 * remapLCRead:
 *
 * set up read memory mapping in region $D000-$FFFF
 */
void remapLCRead(void)
{
	int i;
	AppleMemoryPage test_page, *bank;
	unsigned state;

	state=GetIOUSoftSwitch();

	if (LCReadMapValid){
	    unmapReadRange(0xD0,0xFF);
	    LCReadMapValid=0;
	    LCReadMapLastChange=AppleClock;
	    return;
	}

	if (AppleClock-LCReadMapLastChange<4800){
	    LCReadMapLastChange=AppleClock;
	    return;	
	}

	/* map address $D000-$DFFF for read */	
	if (state & SS_LCRAMRD) {
	   bank=state & SS_ALTZP? &AuxMemoryRAM[0xC0]: &MainMemoryRAM[0xC0];
	   if (state & SS_LCBNK2)
	      bank+=16;
	}
	else
	   bank=&MainMemoryROM[0xD0];

	test_page=(void*)((unsigned)Read_Page_Table[0xD0]&~3);
	if (test_page!=bank[0x00]){
#ifdef DEBUG
	   printf("remmap $D000-$DFFF %s for read\n",
	      state&SS_LCRAM?(state&SS_ALTZP?"aux RAM":"main RAM"):"ROM");
#endif
	   for(i=0;i<0x10;i++)
	      MapReadPage(i+0xD0,bank[i]);
	}

	/* map address $E000-$FFFF */
	bank=(state & SS_LCRAMRD) ?
	   (state & SS_ALTZP? AuxMemoryRAM: MainMemoryRAM):
	   MainMemoryROM;

	test_page=(void*)((unsigned)Read_Page_Table[0xE0]&~3);
	if (test_page!=bank[0xE0]){
#ifdef DEBUG
	   printf("remmap $E000-$FFFF %s for read\n",
	      state&SS_LCRAM?(state&SS_ALTZP?"aux RAM":"main RAM"):"ROM");
#endif
	   for(i=0xE0;i<=0xFF;i++)
	      MapReadPage(i,bank[i]);
	}
	LCReadMapValid=1;
	LCReadMapLastChange=AppleClock;
}

/*
 * remapLCWrite:
 *
 * set up write memory mapping in region $D000-$FFFF
 */
void remapLCWrite(void)
{
	int i;
	AppleMemoryPage test_page, *bank;
	unsigned state;

	state=GetIOUSoftSwitch();

	if (LCWriteMapValid){
	    unmapWriteRange(0xD0,0xFF);
	    LCWriteMapValid=0;
	    LCWriteMapLastChange=AppleClock;
	    return;
	}

	if (AppleClock-LCWriteMapLastChange<4800){
	    LCWriteMapLastChange=AppleClock;
	    return;	
	}

	/* map address $D000-$DFFF for write */	
	if (state & SS_LCRAMWRT) {
	   bank=state & SS_ALTZP? &AuxMemoryRAM[0xC0]: &MainMemoryRAM[0xC0];
	   if (state & SS_LCBNK2)
	      bank+=16;
	}
	else
	   bank=NULL;
	test_page=(void*)((unsigned)Write_Page_Table[0xD0]&~3);
	if (test_page!=(bank?bank[0x00]:GetNullPage())){
#ifdef DEBUG
	   printf("remmap $D000-$DFFF %s for write\n",
	      bank?(state&SS_ALTZP?"aux RAM":"main RAM"):"ROM");
#endif
	   for(i=0;i<0x10;i++)
	      MapWritePage(i+0xD0,bank?bank[i]:NULL);
	}

	bank=(state & SS_LCRAMWRT) ?
	   (state & SS_ALTZP? &AuxMemoryRAM[0xE0]: &MainMemoryRAM[0xE0]): NULL;
	test_page=(void*)((unsigned)Write_Page_Table[0xE0]&~3);
	if (test_page!=(bank?bank[0x00]:GetNullPage())){
#ifdef DEBUG
	   printf("remmap $E000-$FFFF %s for write\n",
	      bank?(state&SS_ALTZP?"aux RAM":"main RAM"):"ROM");
#endif
	   for(i=0;i<0x20;i++)
	       MapWritePage(i+0xE0,bank?bank[i]:NULL);
	}

	LCWriteMapValid=1;
	LCWriteMapLastChange=AppleClock;
}

void mmuMicrojob(void*args)
{
	return;
	/* check segments one by one */
	if (!RAMReadMapValid)
	   remapRAMRead(); 

	if (!RAMWriteMapValid)
	   remapRAMWrite(); 

	if (!TextReadMapValid)
	   remapTextRead(); 

	if (!TextWriteMapValid)
	   remapTextWrite(); 

	if (!HiresReadMapValid)
	   remapHiresRead(); 

	if (!HiresWriteMapValid)
	   remapHiresWrite(); 

	if (!LCReadMapValid)
	   remapLCRead();

	if (!LCWriteMapValid)
	   remapLCWrite();

	submit_micro_job( AppleClock+1000, mmuMicrojob, NULL );
}
