
%define dapple

%ifndef includefile
%include "raf.ah"
%include "screen.ah"
%include "string.ah"
%include "task.ah"
%endif


;****************************************
;	Class	CPU6502
;
;	- CPU6502.Init
;	- CPU6502.Reset
;	- CPU6502.Set6502
;	- CPU6502.Set65c02
;	- CPU6502.ToString
;	- CPU6502.Line
;
;****************************************

; these functions are found inside the assembly library

; use these macros to define your own memory access functions
%ifndef dapple
%macro		getmem	0
		call	[esi+raf.get]
%endmacro

%macro		setmem	0
		call	[esi+raf.set]
%endmacro

%else

%macro		getmem	0
		call	wrapper.memget
%endmacro

%macro		setmem	0
		call	wrapper.memset
%endmacro

%endif

; the following macros implement the various addressing modes

; LDA	zp	(zeropage)
%macro		getzp	0
		getmem
		inc	bx
;		push	ebx
		mov	ebp,ebx
		mov	ebx,eax
		getmem
;		pop	ebx
		mov	ebx,ebp
%endmacro

; DEC	zp	(zeropage)
%macro		getzpw	0
		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
%endmacro

; LDA	zp,x	(zeropage,x)
%macro		getzpx	0
		getmem
		inc	bx
		add	al,[edi+cpu6502.xreg]
;		push	ebx
		mov	ebp,ebx
		mov	ebx,eax
		getmem
;		pop	ebx
		mov	ebx,ebp
%endmacro

; DEC	zp,x	(zeropage,x)
%macro		getzpxw	0
		getmem
		inc	bx
		add	al,[edi+cpu6502.xreg]
		push	ebx
		mov	ebx,eax
		getmem
%endmacro

; LDX	zp,y	(zeropage,y)
%macro		getzpy	0
		getmem
		inc	bx
		add	al,[edi+cpu6502.yreg]
;		push	ebx
		mov	ebp,ebx
		mov	ebx,eax
		getmem
;		pop	ebx
		mov	ebx,ebp
%endmacro

; LDA	abs	(absolute)
%macro		getabs	0
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
;		push	ebx
		mov	ebp,ebx
		mov	bl,ah
		mov	bh,al
		getmem
;		pop	ebx
		mov	ebx,ebp
%endmacro

; DEC	abs	(absolute)
%macro		getabsw	0
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		getmem
%endmacro

; LDA	(zp)	(zeropage indirect)
%macro		getindzp 0
		getmem
		inc	bx
;		push	ebx
		mov	ebp,ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		getmem
;		pop	ebx
		mov	ebx,ebp
%endmacro

; DEC	(zp)	(zeropage indirect)
%macro		getindzpw 0
		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		getmem
%endmacro

; LDA	(zp,x)
%macro		getindzpx 0
		getmem
		inc	bx
		add	al,[edi+cpu6502.xreg]
;		push	ebx
		mov	ebp,ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bl,ah
		mov	bh,al
		getmem
;		pop	ebx
		mov	ebx,ebp
%endmacro

; DEC	(zp,x)
%macro		getindzpxw 0
		getmem
		inc	bx
		add	al,[edi+cpu6502.xreg]
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bl,ah
		mov	bh,al
		getmem
%endmacro

cpu6502:

.version:	db	'CPU6502 V1.0',0
.versione:

; constants

.CPU6502:	equ	1;		type of CPU
.CPU65C02:	equ	2
.CPU65SC02:	equ	4

.STATEHALT:	equ	$01;		state of CPU
.STATERESET:	equ	$02
.STATENMI:	equ	$04
.STATEIRQ:	equ	$08
.STATEBRK:	equ	$10
.STATETRACE:	equ	$20
.STATEBPT:	equ	$40
.STATEGURU:	equ	$80

.IFLAG:		equ	$04;		flags
.DFLAG:		equ	$08
.BFLAG:		equ	$10
.VFLAG:		equ	$40

; struct

.areg:		equ	0
.xreg:		equ	.areg+1
.yreg:		equ	.xreg+1
.sr:		equ	.yreg+1;		status register as found with the 6502 but without N and Z
.sr2:		equ	.sr+1;			status register for N and Z as found with the 8086
.stateflags:	equ	.sr2+1
.traceflag:	equ	.stateflags+1
.sp:		equ	12
.pc:		equ	.sp+4
.cycle:		equ	.pc+4;			cycle counter
.lastcycle:	equ	.cycle+4;		cycle counter at beginning of line
.linecycles:	equ	.lastcycle+4;		number of cycles per line
.breakpoint:	equ	.linecycles+4
.waitstates:	equ	.breakpoint+4
.type:		equ	.waitstates+4
.jump:		equ	.type+4
.size:		equ	.jump+256*4


;****************************************
; 08-Dec-2001	CPU6502.Init
;
;	ESI	= pointer to cpu6502
; ==>	C	= errorflag (0)
;
;****************************************

		align	16
.init:		mov	byte [esi+cpu6502.areg],0
		mov	byte [esi+cpu6502.xreg],0
		mov	byte [esi+cpu6502.yreg],0
		mov	byte [esi+cpu6502.sr],$20
		mov	byte [esi+cpu6502.sr2],0
		mov	dword [esi+cpu6502.sp],$1ff
		mov	dword [esi+cpu6502.pc],0
		mov	byte [esi+cpu6502.stateflags],0
		mov	byte [esi+cpu6502.traceflag],0
		mov	dword [esi+cpu6502.waitstates],$158
		mov	dword [esi+cpu6502.linecycles],65
		mov	dword [esi+cpu6502.breakpoint],-1
		call	cpu6502.reset
		jmp	short .set65c02


;****************************************
; 08-Dec-2001	CPU6502.Reset
;
;	ESI	= pointer to cpu6502
; ==>	C	= errorflag (0)
;
;****************************************

		align	16
.reset:		mov	dword [esi+cpu6502.cycle],0
		mov	dword [esi+cpu6502.lastcycle],0
		or	byte [esi+cpu6502.stateflags],cpu6502.STATERESET
		clc
		ret


;****************************************
; 27-Dec-2001	CPU6502.Set6502
;
;	ESI	= pointer to cpu6502
; ==>	C	= errorflag (0)
;
;****************************************

		align	16
.set6502:	push	ecx
		push	esi
		push	edi
		mov	dword [esi+cpu6502.type],cpu6502.CPU6502
		lea	edi,[esi+cpu6502.jump]
		mov	esi,.j6502opcode
		mov	ecx,256
	rep	movsd
		pop	edi
		pop	esi
		pop	ecx
		clc
		ret


;****************************************
; 27-Dec-2001	CPU6502.Set65c02
;
;	ESI	= pointer to cpu6502
; ==>	C	= errorflag (0)
;
;****************************************

		align	16
.set65c02:	push	ecx
		push	esi
		push	edi
		mov	dword [esi+cpu6502.type],cpu6502.CPU65C02
		lea	edi,[esi+cpu6502.jump]
		mov	esi,.j65c02opcode
		mov	ecx,256
	rep	movsd
		pop	edi
		pop	esi
		pop	ecx
		clc
		ret


;****************************************
; 27-Dec-2001	CPU6502.Set65sc02
;
;	ESI	= pointer to cpu6502
; ==>	C	= errorflag (0)
;
;****************************************

		align	16
.set65sc02:	push	ecx
		push	esi
		push	edi
		mov	dword [esi+cpu6502.type],cpu6502.CPU65SC02
		lea	edi,[esi+cpu6502.jump]
		mov	esi,.j65sc02opcode
		mov	ecx,256
	rep	movsd
		pop	edi
		pop	esi
		pop	ecx
		clc
		ret


;****************************************
; 27-Dec-2001	CPU6502.ToString
;
;	ESI	= pointer to cpu6502
;	EDI	= pointer to string (\0)
; ==>	C	= errorflag (0)
;
;****************************************

		align	16
.tostring:	push	eax
		push	ebx
		push	esi
		push	edi

		mov	ebx,esi
		mov	eax,256
		mov	esi,.cputostrt
		call	_astringcopy

		lea	esi,[edi+.cpuaregt-.cputostrt]
		mov	al,[ebx+cpu6502.areg]
		call	.tostr10
		lea	esi,[edi+.cpuxregt-.cputostrt]
		mov	al,[ebx+cpu6502.xreg]
		call	.tostr10
		lea	esi,[edi+.cpuyregt-.cputostrt]
		mov	al,[ebx+cpu6502.yreg]
		call	.tostr10
		lea	esi,[edi+.cpupcregt-.cputostrt]
		mov	al,[ebx+cpu6502.pc+1]
		call	.tostr10
		lea	esi,[edi+.cpupcregt+2-.cputostrt]
		mov	al,[ebx+cpu6502.pc]
		call	.tostr10
		lea	esi,[edi+.cpuspregt-.cputostrt]
		mov	al,[ebx+cpu6502.sp]
		call	.tostr10

		lea	esi,[edi+.cpuflagst-.cputostrt]
		mov	bx,[ebx+cpu6502.sr]
		mov	ah,'-'
		mov	al,ah
		or	bh,bh
		jns	.tostr2
		mov	al,'N'
.tostr2:	mov	[esi],al
		inc	esi
		mov	al,ah
		test	bl,cpu6502.VFLAG
		je	.tostr3
		mov	al,'V'
.tostr3:	mov	[esi],al
		inc	esi
		mov	al,'0'
		test	bl,$20
		je	.tostr4
		mov	al,'1'
.tostr4:	mov	[esi],al
		inc	esi
		mov	al,ah
		test	bl,$10
		je	.tostr5
		mov	al,'B'
.tostr5:	mov	[esi],al
		inc	esi
		mov	al,ah
		test	bl,cpu6502.DFLAG
		je	.tostr6
		mov	al,'D'
.tostr6:	mov	[esi],al
		inc	esi
		mov	al,ah
		test	bl,cpu6502.IFLAG
		je	.tostr7
		mov	al,'I'
.tostr7:	mov	[esi],al
		inc	esi
		mov	al,ah
		test	bh,2
		je	.tostr8
		mov	al,'Z'
.tostr8:	mov	[esi],al
		inc	esi
		mov	al,ah
		shr	bl,1
		jnc	.tostr9
		mov	al,'C'
.tostr9:	mov	[esi],al

		pop	edi
		pop	esi
		pop	ebx
		pop	eax
		clc
		ret

		align	8
.tostr10:	mov	ah,al
		shr	al,4
		cmp	al,10
		jb	.tostr11
		add	al,'a'-'0'-10
.tostr11:	add	al,'0'
		mov	[esi],al
		inc	esi
		mov	al,$0f
		and	al,ah
		cmp	al,10
		jb	.tostr12
		add	al,'a'-'0'-10
.tostr12:	add	al,'0'
		mov	[esi],al
		ret

.cputostrt:	db	'A: $'
.cpuaregt:	db	'00 X: $'
.cpuxregt:	db	'00 Y: $'
.cpuyregt:	db	'00 PC: $'
.cpupcregt:	db	'0000 SP: $1'
.cpuspregt:	db	'00 Flags: '
.cpuflagst:	db	'NV1BDIZC',0


;****************************************

		align	16
.genflags:	mov	al,[edi+cpu6502.sr2]
		mov	ah,al
		and	al,$80;			N
		test	ah,$02
		je	.gflgs2
		or	al,$40
.gflgs2:	or	al,[edi+cpu6502.sr];	.V-BDI.C
		ret


;****************************************
; 26-Jul-2003	CPU6502.Line
;
;	ESI	= pointer to raf (see macros 'getmem' and 'setmem')
;	EDI	= pointer to cpu
; ==>	C	= errorflag
; (((	EAX	= scratch register
; (((	EBX	= pc
;
;****************************************


		align	32
; test interrupts etc
.linestate:

; 1.) test reset
		test	byte [edi+cpu6502.stateflags],cpu6502.STATERESET
		je	.linehalt
		and	byte [edi+cpu6502.stateflags],-1 - .STATEHALT - .STATERESET - .STATEGURU -.STATEBPT -.STATEBRK

; on a 65c02/65sc02 the dflag gets cleared on reset
		test	byte [edi+cpu6502.type],cpu6502.CPU65C02 + cpu6502.CPU65SC02
		je	.linereset2
		and	byte [edi+cpu6502.sr],-1 - cpu6502.DFLAG
.linereset2:	mov	ebx,$0fffd;			load new pc from reset vector
		getmem
		dec	ebx
		mov	ah,al
		getmem
		mov	bx,ax
		add	dword [edi+cpu6502.cycle],5

; 2.) test halt: processor still halted?
.linehalt:	test	byte [edi+cpu6502.stateflags],cpu6502.STATEHALT
		je	.linebrk
%ifdef dapple
		mov	dword [_screenupdate],-1
		call	_ataskswitch
%endif
		mov	[edi+cpu6502.pc],ebx
		popad
		clc
		ret

; 3.) test brk
.linebrk:	test	byte [edi+cpu6502.stateflags],cpu6502.STATEBRK
		je	.linenmi
		and	byte [edi+cpu6502.stateflags],-1 - cpu6502.STATEBRK
; on a 65c02/65sc02 the dflag gets cleared on nmi
		test	byte [edi+cpu6502.type],cpu6502.CPU65C02 + cpu6502.CPU65SC02
		je	.linebrk2
		and	byte [edi+cpu6502.sr],-1 - cpu6502.DFLAG
.linebrk2:	mov	ebx,$0ffff
		getmem
		dec	ebx
		mov	ah,al
		getmem
		mov	bx,ax
		add	dword [edi+cpu6502.cycle],7

; 4.) test nmi
.linenmi:	test	byte [edi+cpu6502.stateflags],cpu6502.STATENMI
		je	.lineirq
		and	byte [edi+cpu6502.stateflags],-1 - cpu6502.STATENMI
		mov	al,bh
		mov	ah,bl
		mov	ebx,[edi+cpu6502.sp]
		setmem
		dec	bl
		mov	al,ah
		setmem
		dec	bl
		call	cpu6502.genflags
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
; on a 65c02/65sc02 the dflag gets cleared on nmi
		test	byte [edi+cpu6502.type],cpu6502.CPU65C02 + cpu6502.CPU65SC02
		je	.linenmi2
		and	byte [edi+cpu6502.sr],-1 - cpu6502.DFLAG
.linenmi2:	mov	ebx,$0fffb
		getmem
		dec	ebx
		mov	ah,al
		getmem
		mov	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	short .lineelse

; 5.) test irq
.lineirq:	test	byte [edi+cpu6502.stateflags],cpu6502.STATEIRQ
		je	.lineelse
		test	byte [edi+cpu6502.sr],cpu6502.IFLAG
		jne	.lineelse
		and	byte [edi+cpu6502.stateflags],-1 - cpu6502.STATEIRQ
		mov	al,bh
		mov	ah,bl
		mov	ebx,[edi+cpu6502.sp]
		setmem
		dec	bl
		mov	al,ah
		setmem
		dec	bl
		call	cpu6502.genflags
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
		or	byte [cpu6502.sr],cpu6502.IFLAG
; on a 65c02/65sc02 the dflag gets cleared on nmi
		test	byte [edi+cpu6502.type],cpu6502.CPU65C02 + cpu6502.CPU65SC02
		je	.lineirq2
		and	byte [edi+cpu6502.sr],-1 - cpu6502.DFLAG
.lineirq2:	mov	ebx,$0ffff
		getmem
		dec	ebx
		mov	ah,al
		getmem
		mov	bx,ax
		add	dword [edi+cpu6502.cycle],6

; 6.) clear trace, breakpoint, guru

.lineelse:	and	byte [edi+cpu6502.stateflags],-1 - .STATETRACE - .STATEGURU

		jmp	.linebpt
;----------------------------------------

		align	16
.linebpt2:	xor	byte [edi+cpu6502.stateflags],cpu6502.STATEBPT
		test	byte [edi+cpu6502.stateflags],cpu6502.STATEBPT
		je	.linebpt3
		mov	[edi+cpu6502.pc],ebx
%ifdef dapple
		mov	esi,.linebptt2
		mov	al,bh
		call	.ghex
		mov	al,bl
		call	.ghex
		push	dword .linebptt
		call	_setmessage
		mov	dword [esp],0
		call	_cpulight
		add	esp,4
		mov	dword [_screenupdate],-1
		call	_ataskswitch
%endif
		popad
		clc
		ret


		align	16
.line:		pushad
		mov	ebx,[edi+cpu6502.pc]
		jmp	short .linewait2

		align	32
.linenext:	test	byte [edi+cpu6502.traceflag],1
		jne	.linetrace
		mov	eax,[edi+cpu6502.waitstates]
		test	eax,eax
		je	.linewait2
.linewait:	dec	eax
		jne	.linewait
.linewait2:

; test for interrupts etc
		test	byte [edi+cpu6502.stateflags],-1
		jne	near .linestate

; test for breakpoint
.linebpt:	cmp	ebx,[edi+cpu6502.breakpoint]
		je	.linebpt2

; test for line
.linebpt3:	mov	eax,[edi+cpu6502.cycle]
		sub	eax,[edi+cpu6502.lastcycle]
		cmp	eax,[edi+cpu6502.linecycles]
		jge	.linedraw

; interprete next opcode
.lineint:	mov	eax,0;		xor	eax,eax
		getmem
		inc	bx
		jmp	[edi+cpu6502.jump+eax*4]

		align	16
.linedraw:	mov	eax,[edi+cpu6502.lastcycle]
		add	eax,[edi+cpu6502.linecycles]
		mov	[edi+cpu6502.lastcycle],eax
%ifdef dapple
		call	_virtline;		draw line on screen
		test	dword [_virtrasterline],-1
		jne	.lineint
%endif
		test	byte [edi+cpu6502.traceflag],2
		jne	.linetrace
		mov	[edi+cpu6502.pc],ebx
		popad
		clc
		ret

		align	16
.linetrace:	or	byte [edi+cpu6502.stateflags],cpu6502.STATETRACE
		mov	[edi+cpu6502.pc],ebx
%ifdef dapple
		mov	esi,.linetracet2
		mov	al,bh
		call	.ghex
		mov	al,bl
		call	.ghex
		push	dword .linetracet
		call	_setmessage
		mov	dword [esp],0
		call	_cpulight
		add	esp,4
%endif
		popad
		clc
		ret

		section	.data
.linebptt:	db	'Breakpoint reached at address $'
.linebptt2:	db	'xxxx',0
.linetracet:	db	'CPU stopped at address $'
.linetracet2:	db	'xxxx because of trace mode',0
		section	.text

		align	16
.j6502opcode:	dd	.brk;		$00
		dd	.oraindzpx;	$01
		dd	.hlt;		$02	illegal
		dd	.sloindzpx;	$03	illegal
		dd	.skb04;		$04	illegal
		dd	.orazp;		$05
		dd	.aslzp;		$06
		dd	.slozp;		$07	illegal
		dd	.php;		$08
		dd	.orai;		$09
		dd	.asla;		$0a
		dd	.guru;		$0b
		dd	.skw0c;		$0c	illegal
		dd	.oraabs;	$0d
		dd	.aslabs;	$0e
		dd	.sloabs;	$0f	illegal
		dd	.bpl;		$10
		dd	.oraindzpy;	$11
		dd	.hlt;		$12	illegal
		dd	.sloindzpy;	$13	illegal
		dd	.skb14;		$14	illegal
		dd	.orazpx;	$15
		dd	.aslzpx;	$16
		dd	.slozpx;	$17	illegal
		dd	.clc;		$18
		dd	.oraabsy;	$19
		dd	.nop1a;		$1a	illegal
		dd	.sloabsy;	$1b	illegal
		dd	.skw1c;		$1c	illegal
		dd	.oraabsx;	$1d
		dd	.aslabsx;	$1e
		dd	.sloabsx;	$1f	illegal
		dd	.jsrabs;	$20
		dd	.andindzpx;	$21
		dd	.hlt;		$22	illegal
		dd	.rlaindzpx;	$23	illegal
		dd	.bitzp;		$24
		dd	.andzp;		$25
		dd	.rolzp;		$26
		dd	.rlazp;		$27	illegal
		dd	.plp;		$28
		dd	.andi;		$29
		dd	.rola;		$2a
		dd	.guru;		$2b
		dd	.bitabs;	$2c
		dd	.andabs;	$2d
		dd	.rolabs;	$2e
		dd	.rlaabs;	$2f	illegal
		dd	.bmi;		$30
		dd	.andindzpy;	$31
		dd	.hlt;		$32	illegal
		dd	.rlaindzpy;	$33	illegal
		dd	.skb34;		$34	illegal
		dd	.andzpx;	$35
		dd	.rolzpx;	$36
		dd	.rlazpx;	$37	illegal
		dd	.sec;		$38
		dd	.andabsy;	$39
		dd	.nop3a;		$3a	illegal
		dd	.rlaabsy;	$3b	illegal
		dd	.skw3c;		$3c	illegal
		dd	.andabsx;	$3d
		dd	.rolabsx;	$3e
		dd	.rlaabsx;	$3f	illegal
		dd	.rti;		$40
		dd	.eorindzpx;	$41
		dd	.hlt;		$42	illegal
		dd	.xasabsy;	$43	illegal
		dd	.skb44;		$44	illegal
		dd	.eorzp;		$45
		dd	.lsrzp;		$46
		dd	.guru;		$47
		dd	.pha;		$48
		dd	.eori;		$49
		dd	.lsra;		$4a
		dd	.alri;		$4b
		dd	.jmpabs;	$4c
		dd	.eorabs;	$4d
		dd	.lsrabs;	$4e
		dd	.guru;		$4f
		dd	.bvc;		$50
		dd	.eorindzpy;	$51
		dd	.hlt;		$52	illegal
		dd	.sreindzpy;	$53
		dd	.skb54;		$54	illegal
		dd	.eorzpx;	$55
		dd	.lsrzpx;	$56
		dd	.guru;		$57
		dd	.cli;		$58
		dd	.eorabsy;	$59
		dd	.nop5a;		$5a	illegal
		dd	.guru;		$5b
		dd	.skw5c;		$5c	illegal
		dd	.eorabsx;	$5d
		dd	.lsrabsx;	$5e
		dd	.guru;		$5f
		dd	.rts;		$60
		dd	.adcindzpx;	$61
		dd	.hlt;		$62	illegal
		dd	.guru;		$63
		dd	.skb64;		$64	illegal
		dd	.adczp;		$65
		dd	.rorzp;		$66
		dd	.guru;		$67
		dd	.pla;		$68
		dd	.adci;		$69
		dd	.rora;		$6a
		dd	.guru;		$6b
		dd	.jmpindabs;	$6c
		dd	.adcabs;	$6d
		dd	.rorabs;	$6e
		dd	.rraabs;	$6f	illegal
		dd	.bvs;		$70
		dd	.adcindzpy;	$71
		dd	.hlt;		$72	illegal
		dd	.rraindzpy;	$73	illegal
		dd	.skb74;		$74	illegal
		dd	.adczpx;	$75
		dd	.rorzpx;	$76
		dd	.guru;		$77
		dd	.sei;		$78
		dd	.adcabsy;	$79
		dd	.nop7a;		$7a	illegal
		dd	.guru;		$7b
		dd	.skw7c;		$7c	illegal
		dd	.adcabsx;	$7d
		dd	.rorabsx;	$7e
		dd	.rraabsx;	$7f	illegal
		dd	.skb80;		$80	illegal
		dd	.staindzpx;	$81
		dd	.skb82;		$82	illegal
		dd	.axsindzpx;	$83	illegal
		dd	.styzp;		$84
		dd	.stazp;		$85
		dd	.stxzp;		$86
		dd	.axszp;		$87	illegal
		dd	.dey;		$88
		dd	.guru;		$89
		dd	.txa;		$8a
		dd	.guru;		$8b
		dd	.styabs;	$8c
		dd	.staabs;	$8d
		dd	.stxabs;	$8e
		dd	.axsabs;	$8f	illegal
		dd	.bcc;		$90
		dd	.staindzpy;	$91
		dd	.hlt;		$92	illegal
		dd	.guru;		$93
		dd	.styzpx;	$94
		dd	.stazpx;	$95
		dd	.stxzpy;	$96
		dd	.axszpy;	$97	illegal
		dd	.tya;		$98
		dd	.staabsy;	$99
		dd	.txs;		$9a
		dd	.guru;		$9b
		dd	.guru;		$9c
		dd	.staabsx;	$9d
		dd	.guru;		$9e
		dd	.guru;		$9f
		dd	.ldyi;		$a0
		dd	.ldaindzpx;	$a1
		dd	.ldxi;		$a2
		dd	.laxindzpx;	$a3	illegal
		dd	.ldyzp;		$a4
		dd	.ldazp;		$a5
		dd	.ldxzp;		$a6
		dd	.laxzp;		$a7	illegal
		dd	.tay;		$a8
		dd	.ldai;		$a9
		dd	.tax;		$aa
		dd	.guru;		$ab
		dd	.ldyabs;	$ac
		dd	.ldaabs;	$ad
		dd	.ldxabs;	$ae
		dd	.laxabs;	$af	illegal
		dd	.bcs;		$b0
		dd	.ldaindzpy;	$b1
		dd	.hlt;		$b2	illegal
		dd	.laxindzpy;	$b3	illegal
		dd	.ldyzpx;	$b4
		dd	.ldazpx;	$b5
		dd	.ldxzpy;	$b6
		dd	.laxzpy;	$b7	illegal
		dd	.clv;		$b8
		dd	.ldaabsy;	$b9
		dd	.tsx;		$ba
		dd	.guru;		$bb
		dd	.ldyabsx;	$bc
		dd	.ldaabsx;	$bd
		dd	.ldxabsy;	$be
		dd	.laxabsy;	$bf	illegal
		dd	.cpyi;		$c0
		dd	.cmpindzpx;	$c1
		dd	.skbc2;		$c2	illegal
		dd	.guru;		$c3
		dd	.cpyzp;		$c4
		dd	.cmpzp;		$c5
		dd	.deczp;		$c6
		dd	.guru;		$c7
		dd	.iny;		$c8
		dd	.cmpi;		$c9
		dd	.dex;		$ca
		dd	.guru;		$cb
		dd	.cpyabs;	$cc
		dd	.cmpabs;	$cd
		dd	.decabs;	$ce
		dd	.guru;		$cf
		dd	.bne;		$d0
		dd	.cmpindzpy;	$d1
		dd	.hlt;		$d2	illegal
		dd	.dcmindzpy;	$d3	illegal
		dd	.skbd4;		$d4	illegal
		dd	.cmpzpx;	$d5
		dd	.deczpx;	$d6
		dd	.guru;		$d7
		dd	.cld;		$d8
		dd	.cmpabsy;	$d9
		dd	.nopda;		$da	illegal
		dd	.guru;		$db
		dd	.skwdc;		$dc	illegal
		dd	.cmpabsx;	$dd
		dd	.decabsx;	$de
		dd	.guru;		$df
		dd	.cpxi;		$e0
		dd	.sbcindzpx;	$e1
		dd	.skbe2;		$e2	illegal
		dd	.guru;		$e3
		dd	.cpxzp;		$e4
		dd	.sbczp;		$e5
		dd	.inczp;		$e6
		dd	.guru;		$e7
		dd	.inx;		$e8
		dd	.sbci;		$e9
		dd	.nop;		$ea
		dd	.guru;		$eb
		dd	.cpxabs;	$ec
		dd	.sbcabs;	$ed
		dd	.incabs;	$ee
		dd	.guru;		$ef
		dd	.beq;		$f0
		dd	.sbcindzpy;	$f1
		dd	.hlt;		$f2	illegal
		dd	.insindzpy;	$f3	illegal
		dd	.skbf4;		$f4	illegal
		dd	.sbczpx;	$f5
		dd	.inczpx;	$f6
		dd	.inszpx;	$f7	illegal
		dd	.sed;		$f8
		dd	.sbcabsy;	$f9
		dd	.nopfa;		$fa	illegal
		dd	.guru;		$fb
		dd	.skwfc;		$fc	illegal
		dd	.sbcabsx;	$fd
		dd	.incabsx;	$fe
		dd	.insabsx;	$ff	illegal

;----------------------------------------
		align	16
.j65c02opcode:	dd	.brk;		$00
		dd	.oraindzpx;	$01
		dd	.nop02;		$02	(65c02)
		dd	.nop03;		$03	(65c02)
		dd	.tsbzp;		$04	(65c02)
		dd	.orazp;		$05
		dd	.aslzp;		$06
		dd	.nop07;		$07	(65c02)
		dd	.php;		$08
		dd	.orai;		$09
		dd	.asla;		$0a
		dd	.nop0b;		$0b	(65c02)
		dd	.tsbabs;	$0c	(65c02)
		dd	.oraabs;	$0d
		dd	.aslabs;	$0e
		dd	.nop0f;		$0f	(65c02)
		dd	.bpl;		$10
		dd	.oraindzpy;	$11
		dd	.oraindzp;	$12	(65c02)
		dd	.guru;nop13;		$13	(65c02)
		dd	.trbzp;		$14	(65c02)
		dd	.orazpx;	$15
		dd	.aslzpx;	$16
		dd	.guru;		$17
		dd	.clc;		$18
		dd	.oraabsy;	$19
		dd	.ina;		$1a	(65c02)
		dd	.guru;nop1b;		$1b	(65c02)
		dd	.trbabs;	$1c	(65c02)
		dd	.oraabsx;	$1d
		dd	.aslabsx;	$1e
		dd	.guru;nopabs;	$1f	(65c02)
		dd	.jsrabs;	$20
		dd	.andindzpx;	$21
		dd	.guru;nop22;		$22	(65c02)
		dd	.nop23;		$23	(65c02)
		dd	.bitzp;		$24
		dd	.andzp;		$25
		dd	.rolzp;		$26
		dd	.guru;		$27
		dd	.plp;		$28
		dd	.andi;		$29
		dd	.rola;		$2a
		dd	.guru;nop2b;		$2b	(65c02)
		dd	.bitabs;	$2c
		dd	.andabs;	$2d
		dd	.rolabs;	$2e
		dd	.nop2f;		$2f	(65c02)
		dd	.bmi;		$30
		dd	.andindzpy;	$31
		dd	.andindzp;	$32	(65c02)
		dd	.guru;nop33;		$33	(65c02)
		dd	.bitzpx;	$34	(65c02)
		dd	.andzpx;	$35
		dd	.rolzpx;	$36
		dd	.guru;		$37
		dd	.sec;		$38
		dd	.andabsy;	$39
		dd	.dea;		$3a	(65c02)
		dd	.guru;nop3b;		$3b	(65c02)
		dd	.bitabsx;	$3c	(65c02)
		dd	.andabsx;	$3d
		dd	.rolabsx;	$3e
		dd	.guru;nopabs;	$3f	(65c02)
		dd	.rti;		$40
		dd	.eorindzpx;	$41
		dd	.guru;nop42;		$42	(65c02)
		dd	.nop43;		$43	(65c02)
		dd	.guru;nop44;		$44	(65c02)
		dd	.eorzp;		$45
		dd	.lsrzp;		$46
		dd	.guru;		$47
		dd	.pha;		$48
		dd	.eori;		$49
		dd	.lsra;		$4a
		dd	.nop4b;		$4b	(65c02)
		dd	.jmpabs;	$4c
		dd	.eorabs;	$4d
		dd	.lsrabs;	$4e
		dd	.guru;nopabs;	$4f	(65c02)
		dd	.bvc;		$50
		dd	.eorindzpy;	$51
		dd	.eorindzp;	$52	(65c02)
		dd	.guru;nop53;		$53	(65c02)
		dd	.nop54;		$54	(65c02)
		dd	.eorzpx;	$55
		dd	.lsrzpx;	$56
		dd	.guru;		$57	(65c02)
		dd	.cli;		$58
		dd	.eorabsy;	$59
		dd	.phy;		$5a	(65c02)
		dd	.guru;nop5b;		$5b	(65c02)
		dd	.guru;nop5c;		$5c	(65c02)
		dd	.eorabsx;	$5d
		dd	.lsrabsx;	$5e
		dd	.guru;nopabs;	$5f	(65c02)
		dd	.rts;		$60
		dd	.adcindzpx;	$61
		dd	.guru;nop62;		$62	(65c02)
		dd	.guru;		$63
		dd	.stzzp;		$64	(65c02)
		dd	.adczp;		$65
		dd	.rorzp;		$66
		dd	.guru;		$67	(65c02)
		dd	.pla;		$68
		dd	.adci;		$69
		dd	.rora;		$6a
		dd	.guru;nop6b;		$6b	(65c02)
		dd	.jmpindabs2;	$6c
		dd	.adcabs;	$6d
		dd	.rorabs;	$6e
		dd	.nop6f;		$6f	(65c02)
		dd	.bvs;		$70
		dd	.adcindzpy;	$71
		dd	.adcindzp;	$72	(65c02)
		dd	.nop73;		$73	(65c02)
		dd	.stzzpx;	$74	(65c02)
		dd	.adczpx;	$75
		dd	.rorzpx;	$76
		dd	.guru;		$77	(65c02)
		dd	.sei;		$78
		dd	.adcabsy;	$79
		dd	.ply;		$7a	(65c02)
		dd	.guru;nop7b;		$7b	(65c02)
		dd	.jmpabsx;	$7c	(65c02)
		dd	.adcabsx;	$7d
		dd	.rorabsx;	$7e
		dd	.nopfc;		$7f
		dd	.bra;		$80
		dd	.staindzpx;	$81
		dd	.guru;nop82;		$82	(65c02)
		dd	.guru;nop83;		$83	(65c02)
		dd	.styzp;		$84
		dd	.stazp;		$85
		dd	.stxzp;		$86
		dd	.guru;		$87	(65c02)
		dd	.dey;		$88
		dd	.biti;		$89	(65c02)
		dd	.txa;		$8a
		dd	.guru;nop8b;		$8b	(65c02)
		dd	.styabs;	$8c
		dd	.staabs;	$8d
		dd	.stxabs;	$8e
		dd	.guru;		$8f
		dd	.bcc;		$90
		dd	.staindzpy;	$91
		dd	.staindzp;	$92	(65c02)
		dd	.guru;nop93;		$93	(65c02)
		dd	.styzpx;	$94
		dd	.stazpx;	$95
		dd	.stxzpy;	$96
		dd	.guru;		$97	(65c02)
		dd	.tya;		$98
		dd	.staabsy;	$99
		dd	.txs;		$9a
		dd	.guru;nop9b;		$9b	(65c02)
		dd	.stzabs;	$9c	(65c02)
		dd	.staabsx;	$9d
		dd	.stzabsx;	$9e	(65c02)
		dd	.guru;		$9f
		dd	.ldyi;		$a0
		dd	.ldaindzpx;	$a1
		dd	.ldxi;		$a2
		dd	.nopa3;		$a3	(65c02)
		dd	.ldyzp;		$a4
		dd	.ldazp;		$a5
		dd	.ldxzp;		$a6
		dd	.nopa7;		$a7	(65c02)
		dd	.tay;		$a8
		dd	.ldai;		$a9
		dd	.tax;		$aa
		dd	.guru;nopab;		$ab	(65c02)
		dd	.ldyabs;	$ac
		dd	.ldaabs;	$ad
		dd	.ldxabs;	$ae
		dd	.guru;		$af
		dd	.bcs;		$b0
		dd	.ldaindzpy;	$b1
		dd	.ldaindzp;	$b2	(65c02)
		dd	.guru;nopb3;		$b3	(65c02)
		dd	.ldyzpx;	$b4
		dd	.ldazpx;	$b5
		dd	.ldxzpy;	$b6
		dd	.guru;		$b7	(65c02)
		dd	.clv;		$b8
		dd	.ldaabsy;	$b9
		dd	.tsx;		$ba
		dd	.guru;nopbb;		$bb	(65c02)
		dd	.ldyabsx;	$bc
		dd	.ldaabsx;	$bd
		dd	.ldxabsy;	$be
		dd	.guru;		$bf
		dd	.cpyi;		$c0
		dd	.cmpindzpx;	$c1
		dd	.guru;nopc2;		$c2	(65c02)
		dd	.guru;nopc3;		$c3	(65c02)
		dd	.cpyzp;		$c4
		dd	.cmpzp;		$c5
		dd	.deczp;		$c6
		dd	.guru;		$c7	(65c02)
		dd	.iny;		$c8
		dd	.cmpi;		$c9
		dd	.dex;		$ca
		dd	.guru;		$cb	(WAI)
		dd	.cpyabs;	$cc
		dd	.cmpabs;	$cd
		dd	.decabs;	$ce
		dd	.guru;		$cf
		dd	.bne;		$d0
		dd	.cmpindzpy;	$d1
		dd	.cmpindzp;	$d2	(65c02)
		dd	.nopd3;		$d3	(65c02)
		dd	.guru;nopd4;		$d4	(65c02)
		dd	.cmpzpx;	$d5
		dd	.deczpx;	$d6
		dd	.guru;		$d7	(65c02)
		dd	.cld;		$d8
		dd	.cmpabsy;	$d9
		dd	.phx;		$da	(65c02)
		dd	.guru;		$db	(STP)
		dd	.nopdc;		$dc	(65c02)
		dd	.cmpabsx;	$dd
		dd	.decabsx;	$de
		dd	.guru;		$df
		dd	.cpxi;		$e0
		dd	.sbcindzpx;	$e1
		dd	.nope2;		$e2	(65c02)
		dd	.guru;nope3;		$e3	(65c02)
		dd	.cpxzp;		$e4
		dd	.sbczp;		$e5
		dd	.inczp;		$e6
		dd	.guru;		$e7	(65c02)
		dd	.inx;		$e8
		dd	.sbci;		$e9
		dd	.nop;		$ea
		dd	.guru;nopeb;		$eb	(65c02)
		dd	.cpxabs;	$ec
		dd	.sbcabs;	$ed
		dd	.incabs;	$ee
		dd	.guru;		$ef
		dd	.beq;		$f0
		dd	.sbcindzpy;	$f1
		dd	.sbcindzp;	$f2	(65c02)
		dd	.nopf3;		$f3	(65c02)
		dd	.guru;nopf4;		$f4	(65c02)
		dd	.sbczpx;	$f5
		dd	.inczpx;	$f6
		dd	.nopf7;		$f7	(65c02)
		dd	.sed;		$f8
		dd	.sbcabsy;	$f9
		dd	.plx;		$fa	(65c02)
		dd	.guru;nopfb;		$fb	(65c02)
		dd	.nopfc;		$fc	(65c02)
		dd	.sbcabsx;	$fd
		dd	.incabsx;	$fe
		dd	.nopff;		$ff	(65c02)

;----------------------------------------
		align	16
.j65sc02opcode:	dd	.brk;		$00
		dd	.oraindzpx;	$01
		dd	.nop02;		$02	(65c02)
		dd	.nop03;		$03	(65c02)
		dd	.tsbzp;		$04	(65c02)
		dd	.orazp;		$05
		dd	.aslzp;		$06
		dd	.rmb0zp;	$07	(65SC02)
		dd	.php;		$08
		dd	.orai;		$09
		dd	.asla;		$0a
		dd	.nop0b;		$0b	(65c02)
		dd	.tsbabs;	$0c	(65c02)
		dd	.oraabs;	$0d
		dd	.aslabs;	$0e
		dd	.bbr0;		$0f	(65SC02)
		dd	.bpl;		$10
		dd	.oraindzpy;	$11
		dd	.oraindzp;	$12	(65c02)
		dd	.guru;nop13;		$13	(65c02)
		dd	.trbzp;		$14	(65c02)
		dd	.orazpx;	$15
		dd	.aslzpx;	$16
		dd	.rmb1zp;	$17	(65SC02)
		dd	.clc;		$18
		dd	.oraabsy;	$19
		dd	.ina;		$1a	(65c02)
		dd	.guru;nop1b;		$1b	(65c02)
		dd	.trbabs;	$1c	(65c02)
		dd	.oraabsx;	$1d
		dd	.aslabsx;	$1e
		dd	.bbr1;		$1f	(65SC02)
		dd	.jsrabs;	$20
		dd	.andindzpx;	$21
		dd	.guru;nop22;		$22	(65c02)
		dd	.nop23;		$23	(65c02)
		dd	.bitzp;		$24
		dd	.andzp;		$25
		dd	.rolzp;		$26
		dd	.rmb2zp;	$27	(65SC02)
		dd	.plp;		$28
		dd	.andi;		$29
		dd	.rola;		$2a
		dd	.guru;nop2b;		$2b	(65c02)
		dd	.bitabs;	$2c
		dd	.andabs;	$2d
		dd	.rolabs;	$2e
		dd	.bbr2;		$2f	(65SC02)
		dd	.bmi;		$30
		dd	.andindzpy;	$31
		dd	.andindzp;	$32	(65c02)
		dd	.guru;nop33;		$33	(65c02)
		dd	.bitzpx;	$34	(65c02)
		dd	.andzpx;	$35
		dd	.rolzpx;	$36
		dd	.rmb3zp;	$37	(65SC02)
		dd	.sec;		$38
		dd	.andabsy;	$39
		dd	.dea;		$3a	(65c02)
		dd	.guru;nop3b;		$3b	(65c02)
		dd	.bitabsx;	$3c	(65c02)
		dd	.andabsx;	$3d
		dd	.rolabsx;	$3e
		dd	.bbr3;		$3f	(65SC02)
		dd	.rti;		$40
		dd	.eorindzpx;	$41
		dd	.guru;nop42;		$42	(65c02)
		dd	.nop43;		$43	(65c02)
		dd	.guru;nop44;		$44	(65c02)
		dd	.eorzp;		$45
		dd	.lsrzp;		$46
		dd	.rmb4zp;	$47	(65SC02)
		dd	.pha;		$48
		dd	.eori;		$49
		dd	.lsra;		$4a
		dd	.nop4b;		$4b	(65c02)
		dd	.jmpabs;	$4c
		dd	.eorabs;	$4d
		dd	.lsrabs;	$4e
		dd	.bbr4;		$4f	(65SC02)
		dd	.bvc;		$50
		dd	.eorindzpy;	$51
		dd	.eorindzp;	$52	(65c02)
		dd	.guru;nop53;		$53	(65c02)
		dd	.nop54;		$54	(65c02)
		dd	.eorzpx;	$55
		dd	.lsrzpx;	$56
		dd	.rmb5zp;	$57	(65SC02)
		dd	.cli;		$58
		dd	.eorabsy;	$59
		dd	.phy;		$5a	(65c02)
		dd	.guru;nop5b;		$5b	(65c02)
		dd	.guru;nop5c;		$5c	(65c02)
		dd	.eorabsx;	$5d
		dd	.lsrabsx;	$5e
		dd	.bbr5;		$5f	(65SC02)
		dd	.rts;		$60
		dd	.adcindzpx;	$61
		dd	.guru;nop62;		$62	(65c02)
		dd	.bsrabs;	$63	(65sc02)
		dd	.stzzp;		$64	(65c02)
		dd	.adczp;		$65
		dd	.rorzp;		$66
		dd	.rmb6zp;	$67	(65SC02)
		dd	.pla;		$68
		dd	.adci;		$69
		dd	.rora;		$6a
		dd	.guru;nop6b;		$6b	(65c02)
		dd	.jmpindabs2;	$6c
		dd	.adcabs;	$6d
		dd	.rorabs;	$6e
		dd	.bbr6;		$6f	(65SC02)
		dd	.bvs;		$70
		dd	.adcindzpy;	$71
		dd	.adcindzp;	$72	(65c02)
		dd	.nop73;		$73	(65c02)
		dd	.stzzpx;	$74	(65c02)
		dd	.adczpx;	$75
		dd	.rorzpx;	$76
		dd	.rmb7zp;	$77	(65SC02)
		dd	.sei;		$78
		dd	.adcabsy;	$79
		dd	.ply;		$7a	(65c02)
		dd	.guru;nop7b;		$7b	(65c02)
		dd	.jmpabsx;	$7c	(65c02)
		dd	.adcabsx;	$7d
		dd	.rorabsx;	$7e
		dd	.bbr7;		$7f	(65SC02)
		dd	.bra;		$80
		dd	.staindzpx;	$81
		dd	.guru;nop82;		$82	(65c02)
		dd	.guru;nop83;		$83	(65c02)
		dd	.styzp;		$84
		dd	.stazp;		$85
		dd	.stxzp;		$86
		dd	.smb0zp;	$87	(65SC02)
		dd	.dey;		$88
		dd	.biti;		$89	(65c02)
		dd	.txa;		$8a
		dd	.guru;nop8b;		$8b	(65c02)
		dd	.styabs;	$8c
		dd	.staabs;	$8d
		dd	.stxabs;	$8e
		dd	.bbs0;		$8f	(65SC02)
		dd	.bcc;		$90
		dd	.staindzpy;	$91
		dd	.staindzp;	$92	(65c02)
		dd	.guru;nop93;		$93	(65c02)
		dd	.styzpx;	$94
		dd	.stazpx;	$95
		dd	.stxzpy;	$96
		dd	.smb1zp;	$97	(65SC02)
		dd	.tya;		$98
		dd	.staabsy;	$99
		dd	.txs;		$9a
		dd	.guru;nop9b;		$9b	(65c02)
		dd	.stzabs;	$9c	(65c02)
		dd	.staabsx;	$9d
		dd	.stzabsx;	$9e	(65c02)
		dd	.bbs1;		$9f	(65SC02)
		dd	.ldyi;		$a0
		dd	.ldaindzpx;	$a1
		dd	.ldxi;		$a2
		dd	.nopa3;		$a3	(65c02)
		dd	.ldyzp;		$a4
		dd	.ldazp;		$a5
		dd	.ldxzp;		$a6
		dd	.smb2zp;	$a7	(65SC02)
		dd	.tay;		$a8
		dd	.ldai;		$a9
		dd	.tax;		$aa
		dd	.guru;nopab;		$ab	(65c02)
		dd	.ldyabs;	$ac
		dd	.ldaabs;	$ad
		dd	.ldxabs;	$ae
		dd	.bbs2;		$af	(65SC02)
		dd	.bcs;		$b0
		dd	.ldaindzpy;	$b1
		dd	.ldaindzp;	$b2	(65c02)
		dd	.guru;nopb3;		$b3	(65c02)
		dd	.ldyzpx;	$b4
		dd	.ldazpx;	$b5
		dd	.ldxzpy;	$b6
		dd	.smb3zp;	$b7	(65SC02)
		dd	.clv;		$b8
		dd	.ldaabsy;	$b9
		dd	.tsx;		$ba
		dd	.guru;nopbb;		$bb	(65c02)
		dd	.ldyabsx;	$bc
		dd	.ldaabsx;	$bd
		dd	.ldxabsy;	$be
		dd	.bbs3;		$bf	(65SC02)
		dd	.cpyi;		$c0
		dd	.cmpindzpx;	$c1
		dd	.guru;nopc2;		$c2	(65c02)
		dd	.guru;nopc3;		$c3	(65c02)
		dd	.cpyzp;		$c4
		dd	.cmpzp;		$c5
		dd	.deczp;		$c6
		dd	.smb4zp;	$c7	(65SC02)
		dd	.iny;		$c8
		dd	.cmpi;		$c9
		dd	.dex;		$ca
		dd	.guru;		$cb	(WAI)
		dd	.cpyabs;	$cc
		dd	.cmpabs;	$cd
		dd	.decabs;	$ce
		dd	.bbs4;		$cf	(65SC02)
		dd	.bne;		$d0
		dd	.cmpindzpy;	$d1
		dd	.cmpindzp;	$d2	(65c02)
		dd	.nopd3;		$d3	(65c02)
		dd	.guru;nopd4;		$d4	(65c02)
		dd	.cmpzpx;	$d5
		dd	.deczpx;	$d6
		dd	.smb5zp;	$d7	(65SC02)
		dd	.cld;		$d8
		dd	.cmpabsy;	$d9
		dd	.phx;		$da	(65c02)
		dd	.guru;		$db	(STP)
		dd	.nopdc;		$dc	(65c02)
		dd	.cmpabsx;	$dd
		dd	.decabsx;	$de
		dd	.bbs5;		$df	(65SC02)
		dd	.cpxi;		$e0
		dd	.sbcindzpx;	$e1
		dd	.nope2;		$e2	(65c02)
		dd	.guru;nope3;		$e3	(65c02)
		dd	.cpxzp;		$e4
		dd	.sbczp;		$e5
		dd	.inczp;		$e6
		dd	.smb6zp;	$e7	(65SC02)
		dd	.inx;		$e8
		dd	.sbci;		$e9
		dd	.nop;		$ea
		dd	.guru;nopeb;		$eb	(65c02)
		dd	.cpxabs;	$ec
		dd	.sbcabs;	$ed
		dd	.incabs;	$ee
		dd	.bbs6;		$ef	(65SC02)
		dd	.beq;		$f0
		dd	.sbcindzpy;	$f1
		dd	.sbcindzp;	$f2	(65c02)
		dd	.nopf3;		$f3	(65c02)
		dd	.guru;nopf4;		$f4	(65c02)
		dd	.sbczpx;	$f5
		dd	.inczpx;	$f6
		dd	.smb7zp;	$f7	(65SC02)
		dd	.sed;		$f8
		dd	.sbcabsy;	$f9
		dd	.plx;		$fa	(65c02)
		dd	.guru;nopfb;		$fb	(65c02)
		dd	.nopfc;		$fc	(65c02)
		dd	.sbcabsx;	$fd
		dd	.incabsx;	$fe
		dd	.bbs7;		$ff	(65SC02)

;----------------------------------------

		align	16
.ghex:		mov	ah,al
		shr	al,4
		cmp	al,10
		jb	.ghex2
		add	al,'a'-'0'-10
.ghex2:		add	al,'0'
		mov	[esi],al
		inc	esi
		mov	al,$0f
		and	al,ah
		cmp	al,10
		jb	.ghex3
		add	al,'a'-'0'-10
.ghex3:		add	al,'0'
		mov	[esi],al
		inc	esi
		ret


.alv:		equ	16

		align	.alv
.guru:		or	byte [edi+cpu6502.stateflags],cpu6502.STATEGURU
		dec	bx;		pc = pc -1
		mov	[edi+cpu6502.pc],ebx
%ifdef dapple
		mov	esi,.guruerrm2
		call	.ghex
		mov	esi,.guruerrm3
		mov	al,bh
		call	.ghex
		mov	al,bl
		call	.ghex
		push	dword .guruerrm
		call	_setmessage
		mov	dword [esp],0
		call	_cpulight
		add	esp,4
%else
		mov	eax,.guruerrt
		call	_ataskseterror
%endif
		popad
		clc
		ret
.guruerrt:	db	'!'
		db	':#;',13
		db	'GUnbekannter 65c02-Opcode;',13
		db	'EUnknown 65c02-opcode;',13
		db	0

		section	.data
.guruerrm:	db	'Illegal instruction $'
.guruerrm2:	db	'xx hit at address $'
.guruerrm3:	db	'xxxx',0
		section	.text
;----------------------------------------
; HLT/JAM (illegal 6502)
		align	.alv
.hlt:		or	byte [edi+cpu6502.stateflags],cpu6502.STATEGURU
		dec	bx;		pc = pc -1
		mov	[edi+cpu6502.pc],ebx
%ifdef dapple
		mov	esi,.hlterrm2
		call	.ghex
		mov	esi,.hlterrm3
		mov	al,bh
		call	.ghex
		mov	al,bl
		call	.ghex
		push	dword .hlterrm
		call	_setmessage
		mov	dword [esp],0
		call	_cpulight
		add	esp,4
%else
		mov	eax,.hlterrt
		call	_ataskseterror
%endif
		popad
		clc
		ret
.hlterrt:	db	'!'
		db	':#;',13
		db	'GHLT-Befehl kann nicht ausgefhrt werden;',13
		db	'EHLT instruction cannot be executed;',13
		db	0

		section	.data
.hlterrm:	db	'HLT instruction $'
.hlterrm2:	db	'xx hit at address $'
.hlterrm3:	db	'xxxx',0
		section	.text
;----------------------------------------
		align	16
.adc:		mov	ah,[edi+cpu6502.sr]
		test	ah,cpu6502.DFLAG
		jne	.adc10
		and	byte [edi+cpu6502.sr],$0be;	Clear C and V
		shr	ah,1
		adc	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jo	.adc3
		jc	.adc2
		pop	ebx
		jmp	.linenext
		align	16
.adc2:		or	byte [edi+cpu6502.sr],1
		pop	ebx
		jmp	.linenext
		align	16
.adc3:		jc	.adc4
		or	byte [edi+cpu6502.sr],cpu6502.VFLAG
		pop	ebx
		jmp	.linenext
		align	16
.adc4:		or	byte [edi+cpu6502.sr],cpu6502.VFLAG+$1
		pop	ebx
		jmp	.linenext

		align	16
.adc10:		and	byte [edi+cpu6502.sr],$0fe;		Clear C
		mov	byte [edi+cpu6502.sr2],0;		Clear Z+N
		shr	ah,1
		adc	al,[edi+cpu6502.areg]
		daa
		mov	[edi+cpu6502.areg],al
		jnc	.adc11
		or	byte [edi+cpu6502.sr],1
.adc11:		test	al,al
		jne	.adc12
		or	byte [edi+cpu6502.sr2],$40
.adc12:		pop	ebx
		jmp	.linenext
;----------------------------------------
		align	16
.sbc:		mov	ah,[edi+cpu6502.sr]
		test	ah,cpu6502.DFLAG
		jne	.sbc10
		and	byte [edi+cpu6502.sr],$0be;	%1011 1110 .V-B DI.C
		shr	ah,1
		cmc
		sbb	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jo	.sbc3
		jnc	.sbc2
		pop	ebx
		jmp	.linenext
		align	16
.sbc2:		or	byte [edi+cpu6502.sr],1
		pop	ebx
		jmp	.linenext
		align	16
.sbc3:		jnc	.sbc4
		or	byte [edi+cpu6502.sr],cpu6502.VFLAG
		pop	ebx
		jmp	.linenext
		align	16
.sbc4:		or	byte [edi+cpu6502.sr],cpu6502.VFLAG+$1
		pop	ebx
		jmp	.linenext

		align	16
.sbc10:		and	byte [edi+cpu6502.sr],$0fe;		Clear C
		mov	byte [edi+cpu6502.sr2],0;		Clear Z + N
		shr	ah,1
		cmc
		mov	ah,[edi+cpu6502.areg]
		sbb	ah,al
		mov	al,ah
		das
		mov	[edi+cpu6502.areg],al
		jc	.sbc11
		or	byte [edi+cpu6502.sr],1
.sbc11:		test	al,al
		jne	.sbc12
		or	byte [edi+cpu6502.sr2],$40
.sbc12:		pop	ebx
		jmp	.linenext
;----------------------------------------
; 00	= BRK
		align	.alv
.brk:		dec	bx;		pc = pc -1
		or	byte [edi+cpu6502.stateflags],cpu6502.STATEBRK
%ifdef dapple
		test	byte [_cpumessageflag],-1
		je	.brk2
		push	esi
		mov	al,bh
		mov	esi,.brkm2
		call	.ghex
		mov	al,bl
		call	.ghex
		push	dword .brkm
		call	_setmessage
		add	esp,4
		pop	esi
.brk2:
%endif
		mov	ax,bx
		add	ax,2
		mov	[edi+cpu6502.pc],ax
		xchg	al,ah
		mov	ebx,[edi+cpu6502.sp]
		setmem	;			push	highbyte
		dec	bl
		mov	al,ah
		setmem	;			push	lowbyte
		dec	bl
		mov	al,[edi+cpu6502.sr2]
		mov	ah,al
		and	al,$80;			N
		and	ah,$40;			Z
		shr	ah,5
		or	al,ah
		or	byte [edi+cpu6502.sr],cpu6502.BFLAG;	set B
		or	al,[edi+cpu6502.sr];	.V-BDI.C
		or	al,cpu6502.BFLAG;	set B
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
		or	byte [edi+cpu6502.sr],cpu6502.IFLAG;	set I
		popad
		clc
		ret

		section	.data
.brkm:		db	'BRK-instruction hit at address $'
.brkm2:		db	'xxxx',0
		section	.text
;----------------------------------------
; 01	= ORA (zp,x)
		align	.alv
.oraindzpx:	getindzpx
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 02	= NOP02	(65c02)
		align	.alv
.nop02:		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 03	= SLO (ASO) (zp,x) (illegal 6502)
		align	.alv
.sloindzpx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getindzpxw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],8
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 03	= NOP03	(65c02)
		align	.alv
.nop03:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 04	= SKB04 (illegal 6502)
		align	.alv
.skb04:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 04	= TSB zp (65c02)
		align	.alv
.tsbzp:		getzpw
		and	byte [edi+cpu6502.sr2],$0bf;	clear Z
		mov	ah,al
		and	ah,[edi+cpu6502.areg]
		jne	.tsbzp2
		or	byte [edi+cpu6502.sr2],$40
.tsbzp2:	or	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 05	= ORA zp
		align	.alv
.orazp:		getzp
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 06	= ASL zp
		align	.alv
.aslzp:		getzpw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 07	= SLO (ASO) zp (illegal 6502)
		align	.alv
.slozp:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getzpw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 07	= NOP07 (65c02)
.nop07:		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
;----------------------------------------
; 07	= RMB0 zp (65SC02)
		align	.alv
.rmb0zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		and	al,$0fe
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 08	= PHP
		align	.alv
.php:		push	ebx
		mov	al,[edi+cpu6502.sr2]
		mov	ah,al
		and	al,$040;		Z
		and	ah,$080;		N
		shr	al,5
		or	al,ah
		or	al,[edi+cpu6502.sr];	.V-BDI.C
		mov	ebx,[edi+cpu6502.sp]
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 09	= ORA #
		align	.alv
.orai:		getmem
		inc	bx
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 0a	= ASL
		align	.alv
.asla:		mov	al,[edi+cpu6502.areg]
		shl	al,1
		mov	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jc	.asla2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
.asla2:		or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 0b	= NOP0b	(65c02)
		align	.alv
.nop0b:		add	dword [edi+cpu6502.cycle],1
		jmp	.linenext
;----------------------------------------
; 0c	= SKW0c (illegal 6502)
		align	.alv
.skw0c:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getabs
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 0c	= TSB abs (65c02)
		align	.alv
.tsbabs:	getabsw
		and	byte [edi+cpu6502.sr2],$0bf;	clear Z
		mov	ah,al
		and	ah,[edi+cpu6502.areg]
		jne	.tsbabs2
		or	byte [edi+cpu6502.sr2],$40
.tsbabs2:	or	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 0d	= ORA abs
		align	.alv
.oraabs:	getabs
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 0e	= ASL abs
		align	.alv
.aslabs:	getabsw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 0f	= SLO (ASO) abs (illegal 6502)
		align	.alv
.sloabs:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getabsw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 0f	= NOP0f (65c02)
		align	.alv
.nop0f:		add	bx,2
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 0f	= BBR0 (65SC02)
		align	.alv
.bbr0:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$01
		je	.bbr0b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbr0b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 10	= BPL
		align	.alv
.bpl:		test	byte [edi+cpu6502.sr2],$80
		je	.bpl2
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.bpl2:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 11	= ORA (zp),y
		align	.alv
.oraindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.oraindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.oraindzpy2:	getmem
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 12	= ORA (zp) (65c02)
		align	.alv
.oraindzp:	getindzp
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
;----------------------------------------
; 13	= SLO (ASO) (zp),y (illegal 6502)
		align	.alv
.sloindzpy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.sloindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.sloindzpy2:	getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],8
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 14	= SKB14 (illegal 6502)
		align	.alv
.skb14:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 14	= TRB zp (65c02)
		align	.alv
.trbzp:		getzpw
		and	byte [edi+cpu6502.sr2],$0bf;	clear Z
		mov	ah,al
		and	ah,[edi+cpu6502.areg]
		jne	.trbzp2
		or	byte [edi+cpu6502.sr2],$40
.trbzp2:	mov	ah,[edi+cpu6502.areg]
		not	ah
		and	al,ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 15	= ORA zp,x
		align	.alv
.orazpx:	getzpx
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 16	= ASL zp,x
		align	.alv
.aslzpx:	getzpxw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 17	= SLO (ASP) zp,x (illegal 6502)
		align	.alv
.slozpx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getzpxw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 17	= RMB1 zp (65SC2)
		align	.alv
.rmb1zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		and	al,$0fd
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 18	= CLC
		align	.alv
.clc:		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 19	= ORA abs,y
		align	.alv
.oraabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.oraabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.oraabsy2:	getmem
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 1a	= NOP1a (illegal 6502)
		align	.alv
.nop1a:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 1a	= INA (65c02)
		align	.alv
.ina:		inc	byte [edi+cpu6502.areg]
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 1b	= SLO (ASO) abs,y (illegal 6502)
		align	.alv
.sloabsy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.yreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 1c	= SKW1c (illegal 6502)
		align	.alv
.skw1c:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.skw1c2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.skw1c2:	getmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 1c	= TRB abs (65c02)
		align	.alv
.trbabs:	getabsw
		and	byte [edi+cpu6502.sr2],$0bf;	clear Z
		mov	ah,al
		and	ah,[edi+cpu6502.areg]
		jne	.trbabs2
		or	byte [edi+cpu6502.sr2],$40
.trbabs2:	mov	ah,[edi+cpu6502.areg]
		not	ah
		and	al,ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 1d	= ORA abs,x
		align	.alv
.oraabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.oraabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.oraabsx2:	getmem
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 1e	= ASL abs,x
		align	.alv
.aslabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 1f	= SLO (ASO) abs,x (illegal 6502)
		align	.alv
.sloabsx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 1f	= BBR1 (65SC02)
		align	.alv
.bbr1:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$02
		je	.bbr1b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbr1b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 20	= JSR abs
		align	.alv
.jsrabs:	getmem
		inc	bx
		mov	ah,al
		getmem
		xchg	al,ah
		push	eax;					new address
		mov	al,bh;					hi
		mov	ah,bl;					lo
		mov	ebx,[edi+cpu6502.sp]
		setmem	;					push highbyte
		dec	bl
		mov	al,ah;					push lowbyte
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 21	= AND (zp,x)
		align	.alv
.andindzpx:	getindzpx
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 23	= RLA (zp,x) (illegal 6502)
		align	.alv
.rlaindzpx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getindzpxw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],8
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 23	= NOP23 (65c02)
		align	.alv
.nop23:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 24	= BIT zp
		align	.alv
.bitzp:		getzp
		and	byte [edi+cpu6502.sr],-1 -cpu6502.VFLAG
		mov	ah,al
		and	al,cpu6502.VFLAG
		or	byte [edi+cpu6502.sr],al;		V
		mov	al,ah
		and	al,$080;				N
		test	ah,[edi+cpu6502.areg]
		jne	.bitzp2
		or	al,$040
.bitzp2:	mov	[edi+cpu6502.sr2],al
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 25	= AND zp
		align	.alv
.andzp:		getzp
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 26	= ROL zp
		align	.alv
.rolzp:		getzpw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 27	= RLA zp (illegal 6502)
		align	.alv
.rlazp:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getzpw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 27	= RMB2 zp (65SC02)
		align	.alv
.rmb2zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		and	al,$0fb
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 28	= PLP
		align	.alv
.plp:		push	ebx
		inc	byte [edi+cpu6502.sp]
		mov	ebx,[edi+cpu6502.sp]
		getmem
		mov	ah,al
		and	al,$7d;				%01111101;	.V-BDI.C
		or	al,$20
		mov	[edi+cpu6502.sr],al
		mov	al,ah
		and	al,$002;			Z
		and	ah,$080;			N
		shl	al,5
		or	al,ah
		mov	[edi+cpu6502.sr2],al
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 29	= AND #
		align	.alv
.andi:		getmem
		inc	bx
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 2a	= ROL
		align	.alv
.rola:		mov	ah,[edi+cpu6502.sr]
		mov	al,[edi+cpu6502.areg]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 2c	= BIT abs
		align	.alv
.bitabs:	getabs
		and	byte [edi+cpu6502.sr],-1 - cpu6502.VFLAG
		mov	ah,al
		and	al,cpu6502.VFLAG
		or	byte [edi+cpu6502.sr],al;		V
		mov	al,ah
		and	al,$080;				N
		test	ah,[edi+cpu6502.areg]
		jne	.bitabs2
		or	al,$040
.bitabs2:	mov	[edi+cpu6502.sr2],al
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 2d	= AND abs
		align	.alv
.andabs:	getabs
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 2e	= ROL abs
		align	.alv
.rolabs:	getabsw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 2f	= RLA abs (illegal 6502)
		align	.alv
.rlaabs:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getabsw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 2f	= NOP2f (65c02)
		align	.alv
.nop2f:		add	bx,2
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 2f	= BBR2 (65SC02)
		align	.alv
.bbr2:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$04
		je	.bbr2b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbr2b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 30	= BMI
		align	.alv
.bmi:		test	byte [edi+cpu6502.sr2],$80
		jne	.bmi2
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.bmi2:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 31	= AND (zp),y
		align	.alv
.andindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.andindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.andindzpy2:	getmem
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 32	= AND (zp) (65c02)
		align	.alv
.andindzp:	getindzp
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
;----------------------------------------
; 33	= RLA (zp),y (illegal 6502)
		align	.alv
.rlaindzpy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.rlaindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.rlaindzpy2:	getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],8
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 34	= SKB34 (illegal 6502)
		align	.alv
.skb34:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 34	= BIT zp,x (65c02)
		align	.alv
.bitzpx:	getzpx
		and	byte [edi+cpu6502.sr],-1 -cpu6502.VFLAG
		mov	ah,al
		and	al,cpu6502.VFLAG
		or	byte [edi+cpu6502.sr],al;		V
		mov	al,ah
		and	al,$080;				N
		test	ah,[edi+cpu6502.areg]
		jne	.bitzpx2
		or	al,$040;				Z
.bitzpx2:	mov	[edi+cpu6502.sr2],al
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 35	= AND zp,x
		align	.alv
.andzpx:	getzpx
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 36	= ROL zp,x
		align	.alv
.rolzpx:	getzpxw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 37	= RLA zp,x (illegal 6502)
		align	.alv
.rlazpx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getzpxw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 37	= RMB3 zp (65SC02)
		align	.alv
.rmb3zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		and	al,$0f7
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 38	= SEC
		align	.alv
.sec:		or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 39	= AND abs,y
		align	.alv
.andabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.andabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.andabsy2:	getmem
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 3a	= NOP3a (illegal 6502)
		align	.alv
.nop3a:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 3a	= DEA (65c02)
		align	.alv
.dea:		dec	byte [edi+cpu6502.areg]
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 3b	= RLA abs,y (illegal 6502)
		align	.alv
.rlaabsy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.yreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 3c	= SKW3c (illegal 6502)
		align	.alv
.skw3c:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.skw3c2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.skw3c2:	getmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 3c	= BIT abs,x (65c02)
		align	.alv
.bitabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.bitabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.bitabsx2:	getmem
		and	byte [edi+cpu6502.sr],-1 -cpu6502.VFLAG
		mov	ah,al
		and	al,cpu6502.VFLAG
		or	byte [edi+cpu6502.sr],al;		V
		mov	al,ah
		and	al,$080;				N
		test	ah,[edi+cpu6502.areg]
		jne	.bitabsx3
		or	al,$040;				Z
.bitabsx3:	mov	[edi+cpu6502.sr2],al
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 3d	= AND abs,x
		align	.alv
.andabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.andabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.andabsx2:	getmem
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 3e	= ROL abs,x
		align	.alv
.rolabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 3f	= RLA abs,x (illegal 6502)
		align	.alv
.rlaabsx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcl	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		and	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 3f	= BBR3 (65SC02)
		align	.alv
.bbr3:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$08
		je	.bbr3b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbr3b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 40	= RTI
		align	.alv
.rti:		mov	ebx,[edi+cpu6502.sp]
		inc	bl
		getmem
		mov	ah,al
		and	ah,$7d;				%01111101;	.V-BDI.C
		mov	[edi+cpu6502.sr],ah
		mov	ah,al
		and	ah,$080;			N
		and	al,$002
		shl	al,5;				Z
		or	al,ah
		mov	[edi+cpu6502.sr2],al
		inc	bl
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	[edi+cpu6502.sp],bl
		mov	bh,al
		mov	bl,ah
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 41	= EOR (zp,x)
		align	.alv
.eorindzpx:	getindzpx
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 43	= XAS abs,y (illegal 6502)
		align	.alv
.xasabsy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		adc	bh,0
		mov	al,bh
		inc	al
		and	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 43	= NOP43 (65c02)
		align	.alv
.nop43:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 44	= SKB44 (illegal 6502)
		align	.alv
.skb44:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 45	= EOR zp
		align	.alv
.eorzp:		getzp
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 46	= LSR zp
		align	.alv
.lsrzp:		getzpw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 47	= RMB4 zp (65SC02)
		align	.alv
.rmb4zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		and	al,$0ef
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 48	= PHA
		align	.alv
.pha:		push	ebx
		mov	al,[edi+cpu6502.areg]
		mov	ebx,[edi+cpu6502.sp]
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 49	= EOR #
		align	.alv
.eori:		getmem
		inc	bx
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 4a	= LSR
		align	.alv
.lsra:		mov	ah,[edi+cpu6502.sr]
		mov	al,[edi+cpu6502.areg]
		shr	ah,1
		shr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 4b	= ALR # (illegal 6502)
		align	.alv
.alri:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,[edi+cpu6502.sr]
		and	al,[edi+cpu6502.areg]
		shr	ah,1
		shr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 4b	= NOP4b (65c02)
		align	.alv
.nop4b:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 4c	= JMP abs
		align	.alv
.jmpabs:	getmem
		inc	bx
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 4d	= EOR abs
		align	.alv
.eorabs:	getabs
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 4e	= LSR abs
		align	.alv
.lsrabs:	getabsw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 4f	= BBR4 (65SC02)
		align	.alv
.bbr4:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$10
		je	.bbr4b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbr4b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 50	= BVC
		align	.alv
.bvc:		test	byte [edi+cpu6502.sr],cpu6502.VFLAG
		je	.bvc2
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.bvc2:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 51	= EOR (zp),y
		align	.alv
.eorindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.eorindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.eorindzpy2:	getmem
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 52	= EOR (zp) (65c02)
		align	.alv
.eorindzp:	getindzp
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
;----------------------------------------
; 53	= SRE (zp),y (illegal 6502)
		align	.alv
.sreindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.sreindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.sreindzpy2:	getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		setmem
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 54	= SKB54 (illegal 6502)
		align	.alv
.skb54:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 54	= NOP54 (65c02)
		align	.alv
.nop54:		inc	bx
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 55	= EOR zp,x
		align	.alv
.eorzpx:	getzpx
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 56	= LSR zp,x
		align	.alv
.lsrzpx:	getzpxw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 57	= RMB5 zp (65SC02)
		align	.alv
.rmb5zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		and	al,$0df
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 58	= CLI
		align	.alv
.cli:		and	byte [edi+cpu6502.sr],-1 - cpu6502.IFLAG
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 59	= EOR abs,y
		align	.alv
.eorabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.eorabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.eorabsy2:	getmem
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 5a	= NOP5a (illegal 6502)
		align	.alv
.nop5a:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 5a	= PHY (65c02)
		align	.alv
.phy:		push	ebx
		mov	ebx,[edi+cpu6502.sp]
		mov	al,[edi+cpu6502.yreg]
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 5c	= SKW5c (illegal 6502)
		align	.alv
.skw5c:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.skw5c2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.skw5c2:	getmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 5d	= EOR abs,x
		align	.alv
.eorabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.eorabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.eorabsx2:	getmem
		xor	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 5e	= LSR abs,x
		align	.alv
.lsrabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		shr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 5f	= BBR5 (65SC02)
		align	.alv
.bbr5:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$20
		je	.bbr5b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbr5b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 60	= RTS
		align	.alv
.rts:		mov	ebx,[edi+cpu6502.sp]
		inc	bl
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	[edi+cpu6502.sp],bl
		xchg	al,ah
		inc	ax
		mov	ebx,eax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 61	= ADC (zp,x)
		align	.alv
.adcindzpx:	getindzpxw
		add	dword [edi+cpu6502.cycle],6
		jmp	.adc
;----------------------------------------
; 63	= BSR abs (65sc02)		correct?
		align	.alv
.bsrabs:	getmem
		inc	bx
		mov	ah,al
		getmem
		xchg	al,ah
		add	ax,bx
		push	eax
		mov	al,bh
		mov	ah,bl
		mov	ebx,[edi+cpu6502.sp]
		setmem
		dec	bl
		mov	al,ah
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
		add	dword [edi+cpu6502.cycle],6; ???
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 64	= SKB64 (illegal 6502)
		align	.alv
.skb64:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 64	= STZ zp (65c02)
		align	.alv
.stzzp:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		mov	al,0;		xor	al,al
		setmem
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 65	= ADC zp
		align	.alv
.adczp:		getzpw
		add	dword [edi+cpu6502.cycle],3
		jmp	.adc
;----------------------------------------
; 66	= ROR zp
		align	.alv
.rorzp:		getzpw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 67	= RMB6 zp (65SC02)
		align	.alv
.rmb6zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		and	al,$0bf
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 68	= PLA
		align	.alv
.pla:		push	ebx
		inc	byte [edi+cpu6502.sp]
		mov	ebx,[edi+cpu6502.sp]
		getmem
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 69	= ADC #
		align	.alv
.adci:		getmem
		inc	bx
		push	ebx
		add	dword [edi+cpu6502.cycle],2
		jmp	.adc
;----------------------------------------
; 6a	= ROR
		align	.alv
.rora:		mov	ah,[edi+cpu6502.sr]
		mov	al,[edi+cpu6502.areg]
		shr	ah,1
		rcr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 6c	= JMP (abs)
		align	.alv
.jmpindabs:	getmem
		inc	bx
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		getmem
		inc	bl;			bl = 6502(!)
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
;----------------------------------------
; 6c	= JMP (abs) (65c02)
		align	.alv
.jmpindabs2:	getmem
		inc	bx
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		getmem
		inc	bx;			bx = 65c02(!)
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
;----------------------------------------
; 6d	= ADC abs
		align	.alv
.adcabs:	getabsw
		add	dword [edi+cpu6502.cycle],4
		jmp	.adc
;----------------------------------------
; 6e	= ROR abs
		align	.alv
.rorabs:	getabsw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 6f	= RRA abs (illegal 6502)
		align	.alv
.rraabs:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getabsw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		jmp	.adc
;----------------------------------------
; 6f	= NOP6f (65c02)
		align	.alv
.nop6f:		add	bx,2
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 6f	= BBR6 (65SC02)
		align	.alv
.bbr6:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$40
		je	.bbr6b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbr6b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 70	= BVS
		align	.alv
.bvs:		test	byte [edi+cpu6502.sr],cpu6502.VFLAG
		jne	.bvs2
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.bvs2:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 71	= ADC (zp),y
		align	.alv
.adcindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.adcindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.adcindzpy2:	getmem
		add	dword [edi+cpu6502.cycle],5
		jmp	.adc
;----------------------------------------
; 72	= ADC (zp) (65c02)
		align	.alv
.adcindzp:	getindzpw
		add	dword [edi+cpu6502.cycle],5
		jmp	.adc
;----------------------------------------
; 73	= RRA (zp),y (illegal 6502)
		align	.alv
.rraindzpy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.rraindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.rraindzpy2:	getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		setmem
		add	dword [edi+cpu6502.cycle],8
		jmp	.adc
;----------------------------------------
; 73	= NOP73 (65c02)
		align	.alv
.nop73:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 74	= SKB74 (illegal 6502)
		align	.alv
.skb74:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; 74	= STZ zp,x
		align	.alv
.stzzpx:	getmem
		inc	bx
		push	ebx
		add	al,[edi+cpu6502.xreg]
		mov	ebx,eax
		mov	al,0;		xor	al,al
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 75	= ADC zp,x
		align	.alv
.adczpx:	getzpxw
		add	dword [edi+cpu6502.cycle],4
		jmp	.adc
;----------------------------------------
; 76	= ROR zp,x
		align	.alv
.rorzpx:	getzpxw
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 77	= RMB7 zp (65SC02)
		align	.alv
.rmb7zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		and	al,$07f
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 78	= SEI
		align	.alv
.sei:		or	byte [edi+cpu6502.sr],cpu6502.IFLAG
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 79	= ADC abs,y
		align	.alv
.adcabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.adcabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.adcabsy2:	getmem
		add	dword [edi+cpu6502.cycle],4
		jmp	.adc
;----------------------------------------
; 7a	= NOP7a (illegal 6502)
		align	.alv
.nop7a:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 7a	= PLY (65c02)
		align	.alv
.ply:		push	ebx
		inc	byte [edi+cpu6502.sp]
		mov	ebx,[edi+cpu6502.sp]
		getmem
		mov	[edi+cpu6502.yreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 7c	= SKW7c (illegal 6502)
		align	.alv
.skw7c:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.skw7c2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.skw7c2:	getmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 7c	= JMP (abs,x) (65c02)
		align	.alv
.jmpabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		adc	bh,0
		getmem
		inc	bx
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 7d	= ADC abs,x
		align	.alv
.adcabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.adcabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.adcabsx2:	getmem
		add	dword [edi+cpu6502.cycle],4
		jmp	.adc
;----------------------------------------
; 7e	= ROR abs,x
		align	.alv
.rorabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 7f	= RRA abs,x (illegal 6502)
		align	.alv
.rraabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		mov	ah,[edi+cpu6502.sr]
		shr	ah,1
		rcr	al,1
		rcl	ah,1
		mov	[edi+cpu6502.sr],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		jmp	.adc
;----------------------------------------
; 7f	= NOP7f (65c02)
		align	.alv
.nop7f:		add	bx,2
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 7f	= BBR7 (65SC02)
		align	.alv
.bbr7:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$80
		je	.bbr7b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbr7b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 80	= SKB80 (illegal 6502)
		align	.alv
.skb80:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 80	= BRA (65c02)
		align	.alv
.bra:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 81	= STA (zp,x)
		align	.alv
.staindzpx:	getmem
		inc	bx
		add	al,[edi+cpu6502.xreg]
		push	ebx
		mov	ebx,eax
		getmem
		inc	bl
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		mov	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 82	= SKB82 (illegal 6502)
		align	.alv
.skb82:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 83	= AXS (zp,x) (illegal 6502)
		align	.alv
.axsindzpx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		add	al,[edi+cpu6502.xreg]
		push	ebx
		mov	ebx,eax
		getmem
		inc	bl
		mov	ah,al
		getmem
		mov	bl,ah
		mov	bh,al
		mov	al,[edi+cpu6502.areg]
		and	al,[edi+cpu6502.xreg]
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 84	= STY zp
		align	.alv
.styzp:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		mov	al,[edi+cpu6502.yreg]
		setmem
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 85	= STA zp
		align	.alv
.stazp:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		mov	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 86	= STX zp
		align	.alv
.stxzp:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		mov	al,[edi+cpu6502.xreg]
		setmem
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 87	= AXS zp (illegal 6502)
		align	.alv
.axszp:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		mov	al,[edi+cpu6502.areg]
		and	al,[edi+cpu6502.xreg]
		setmem
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 87	= SMB0 zp (65SC02)
		align	.alv
.smb0zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		or	al,1
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 88	= DEY
		align	.alv
.dey:		dec	byte [edi+cpu6502.yreg]
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 89	= BIT # (65c02)
		align	.alv
.biti:		getmem
		inc	bx
		and	byte [edi+cpu6502.sr],-1 -cpu6502.VFLAG
		mov	ah,al
		and	al,cpu6502.VFLAG
		or	byte [edi+cpu6502.sr],al;	V
		mov	al,ah
		and	al,$080;			N
		test	ah,[edi+cpu6502.areg]
		jne	.biti2
		or	al,$040;			Z
.biti2:		mov	[edi+cpu6502.sr2],al
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 8a	= TXA
		align	.alv
.txa:		mov	al,[edi+cpu6502.xreg]
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 8c	= STY abs
		align	.alv
.styabs:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		mov	al,[edi+cpu6502.yreg]
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 8d	= STA abs
		align	.alv
.staabs:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bh,al
		mov	bl,ah
		mov	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 8e	= STX abs
		align	.alv
.stxabs:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bh,al
		mov	bl,ah
		mov	al,[edi+cpu6502.xreg]
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 8f	= STX abs (illegal 6502)
		align	.alv
.axsabs:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bh,al
		mov	bl,ah
		mov	al,[edi+cpu6502.areg]
		and	al,[edi+cpu6502.xreg]
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 8f	= BBS0 (65SC02)
		align	.alv
.bbs0:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$01
		jne	.bbs0b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbs0b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; 90	= BCC
		align	.alv
.bcc:		test	byte [edi+cpu6502.sr],$1
		je	.bcc2
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.bcc2:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; 91	= STA (zp),y
		align	.alv
.staindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		mov	al,[edi+cpu6502.areg]
		add	bl,[edi+cpu6502.yreg]
		jnc	.staindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.staindzpy2:	setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 92	= STA (zp) (65c02)
		align	.alv
.staindzp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		mov	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 94	= STY zp,x
		align	.alv
.styzpx:	getmem
		inc	bx
		push	ebx
		add	al,[edi+cpu6502.xreg]
		mov	ebx,eax
		mov	al,[edi+cpu6502.yreg]
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 95	= STA zp,x
		align	.alv
.stazpx:	getmem
		inc	bx
		push	ebx
		add	al,[edi+cpu6502.xreg]
		mov	ebx,eax
		mov	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 96	= STX zp,y
		align	.alv
.stxzpy:	getmem
		inc	bx
		push	ebx
		add	al,[edi+cpu6502.yreg]
		mov	ebx,eax
		mov	al,[edi+cpu6502.xreg]
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 97	= AXS zp,y (illegal 6502)
		align	.alv
.axszpy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		push	ebx
		add	al,[edi+cpu6502.yreg]
		mov	ebx,eax
		mov	al,[edi+cpu6502.areg]
		and	al,[edi+cpu6502.xreg]
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 97	= SMB1 zp (65SC02)
		align	.alv
.smb1zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		or	al,2
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 98	= TYA
		align	.alv
.tya:		mov	al,[edi+cpu6502.yreg]
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 99	= STA abs,y
		align	.alv
.staabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.yreg]
		adc	al,0
		mov	bh,al
		mov	bl,ah
		mov	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 9a	= TXS
		align	.alv
.txs:		mov	al,[edi+cpu6502.xreg]
		mov	[edi+cpu6502.sp],al
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; 9c	= STZ abs (65c02)
		align	.alv
.stzabs:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bh,al
		mov	bl,ah
		mov	al,0;		xor	al,al
		setmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 9d	= STA abs,x
		align	.alv
.staabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bh,al
		mov	bl,ah
		mov	al,[edi+cpu6502.areg]
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 9e	= STZ abs,x (65c02)
		align	.alv
.stzabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bh,al
		mov	bl,ah
		mov	al,0;		xor	al,al
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; 9f	= BBS1 (65SC02)
		align	.alv
.bbs1:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$02
		jne	.bbs1b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbs1b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; a0	= LDY #
		align	.alv
.ldyi:		getmem
		inc	bx
		mov	[edi+cpu6502.yreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; a1	= LDA (zp,x)
		align	.alv
.ldaindzpx:	getindzpx
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; a2	= LDX #
		align	.alv
.ldxi:		getmem
		inc	bx
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; a3	= LAX (zp,x) (illegal 6502)
		align	.alv
.laxindzpx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getindzpx
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; a3	= NOPa3 (65c02)
		align	.alv
.nopa3:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; a4	= LDY zp
		align	.alv
.ldyzp:		getzp
		mov	[edi+cpu6502.yreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; a5	= LDA zp
		align	.alv
.ldazp:		getzp
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; a6	= LDX zp
		align	.alv
.ldxzp:		getzp
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; a7	= LAX zp (illegal 6502)
		align	.alv
.laxzp:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getzp
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; a7	= NOPa7 (65c02)
		align	.alv
.nopa7:		inc	bx
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; a7	= SMB2 zp (65SC02)
		align	.alv
.smb2zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		or	al,4
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; a8	= TAY
		align	.alv
.tay:		mov	al,[edi+cpu6502.areg]
		mov	[edi+cpu6502.yreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; a9	= LDA #
		align	.alv
.ldai:		getmem
		inc	bx
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; aa	= TAX
		align	.alv
.tax:		mov	al,[edi+cpu6502.areg]
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; ac	= LDY abs
		align	.alv
.ldyabs:	getabs
		mov	[edi+cpu6502.yreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; ad	= LDA abs
		align	.alv
.ldaabs:	getabs
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; ae	= LDX abs
		align	.alv
.ldxabs:	getabs
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; af	= LAX abs (illegal 6502)
		align	.alv
.laxabs:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getabs
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; af	= BBS2 (65SC02)
		align	.alv
.bbs2:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$04
		jne	.bbs2b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbs2b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; b0	= BCS
		align	.alv
.bcs:		test	byte [edi+cpu6502.sr],$1
		jne	.bcs2
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.bcs2:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; b1	= LDA (zp),y
		align	.alv
.ldaindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.ldaindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.ldaindzpy2:	getmem
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; b2	= LDA (zp) (65c02)
		align	.alv
.ldaindzp:	getindzp
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
;----------------------------------------
; b3	= LAX (zp),y (illegal 6502)
		align	.alv
.laxindzpy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.laxindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.laxindzpy2:	getmem
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; b4	= LDY zp,x
		align	.alv
.ldyzpx:	getzpx
		mov	[edi+cpu6502.yreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; b5	= LDA zp,x
		align	.alv
.ldazpx:	getzpx
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; b6	= LDX zp,y
		align	.alv
.ldxzpy:	getzpy
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; b7	= LAX zp,y (illegal 6502)
		align	.alv
.laxzpy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getzpy
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; b7	= SMB3 zp (65SC02)
		align	.alv
.smb3zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		or	al,8
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; b8	= CLV
		align	.alv
.clv:		and	byte [edi+cpu6502.sr],-1 -cpu6502.VFLAG
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; b9	= LDA abs,y
		align	.alv
.ldaabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.ldaabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.ldaabsy2:	getmem
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; ba	= TSX
		align	.alv
.tsx:		mov	al,[edi+cpu6502.sp]
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; bc	= LDY abs,x
		align	.alv
.ldyabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.ldyabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.ldyabsx2:	getmem
		mov	[edi+cpu6502.yreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; bd	= LDA abs,x
		align	.alv
.ldaabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.ldaabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.ldaabsx2:	getmem
		mov	[edi+cpu6502.areg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; be	= LDX abs,y
		align	.alv
.ldxabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.ldxabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.ldxabsy2:	getmem
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; bf	= LAX abs,y
		align	.alv
.laxabsy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.laxabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.laxabsy2:	getmem
		mov	[edi+cpu6502.areg],al
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; bf	= BBS0 (65SC02)
		align	.alv
.bbs3:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$08
		jne	.bbs3b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbs3b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; c0	= CPY #
		align	.alv
.cpyi:		getmem
		inc	bx
		cmp	[edi+cpu6502.yreg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cpyi2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.cpyi2:		or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; c1	= CMP (zp,x)
		align	.alv
.cmpindzpx:	getindzpx
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpindzpx2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
		align	16
.cmpindzpx2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; c2	= SKBc2 (illegal 6502)
		align	.alv
.skbc2:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; c4	= CPY zp
		align	.alv
.cpyzp:		getzp
		cmp	[edi+cpu6502.yreg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cpyzp2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
		align	16
.cpyzp2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; c5	= CMP zp
		align	.alv
.cmpzp:		getzp
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpzp2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
		align	16
.cmpzp2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; c6	= DEC zp
		align	.alv
.deczp:		getzpw
		dec	al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; c7	= SMB4 zp (65SC02)
		align	.alv
.smb4zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		or	al,$10
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; c8	= INY
		align	.alv
.iny:		inc	byte [edi+cpu6502.yreg]
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; c9	= CMP #
		align	.alv
.cmpi:		getmem
		inc	bx
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpi2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.cmpi2:		or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; ca	= DEX
		align	.alv
.dex:		dec	byte [edi+cpu6502.xreg]
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; cc	= CPY abs
		align	.alv
.cpyabs:	getabs
		cmp	[edi+cpu6502.yreg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cpyabs2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
		align	16
.cpyabs2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; cd	= CMP abs
		align	.alv
.cmpabs:	getabs
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpabs2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
		align	16
.cmpabs2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; ce	= DEC abs
		align	.alv
.decabs:	getabsw
		dec	al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; cf	= DCM abs (illegal 6502)
		align	.alv
.dcmabs:	getabsw
		dec	al
		setmem
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.dcmabs2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
		align	16
.dcmabs2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; cf	= BBS4 (65SC02)
		align	.alv
.bbs4:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$10
		jne	.bbs4b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbs4b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; d0	= BNE
		align	.alv
.bne:		test	byte [edi+cpu6502.sr2],$40
		je	.bne2
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.bne2:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; d1	= CMP (zp),y
		align	.alv
.cmpindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.cmpindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.cmpindzpy2:	getmem
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpindzpy3
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
		align	16
.cmpindzpy3:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; d2	= CMP (zp) (65c02)
		align	.alv
.cmpindzp:	getindzp
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpindzp2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	16
.cmpindzp2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
;----------------------------------------
; d3	= DCM (zp),y (illegal 6502)
		align	.alv
.dcmindzpy:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.dcmindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.dcmindzpy2:	getmem
		dec	al
		setmem
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.dcmindzpy3
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],8
		jmp	.linenext
		align	16
.dcmindzpy3:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],8
		jmp	.linenext
;----------------------------------------
; d3	= NOPd3 (65c02)
		align	.alv
.nopd3:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; d4	= SKBd4 (illegal 6502)
		align	.alv
.skbd4:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; d5	= CMP zp,x
		align	.alv
.cmpzpx:	getzpx
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpzpx2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
		align	16
.cmpzpx2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; d6	= DEC zp,x
		align	.alv
.deczpx:	getzpxw
		dec	al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; d7	= SMB5 zp (65SC02)
		align	.alv
.smb5zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		or	al,$20
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; d8	= CLD
		align	.alv
.cld:		and	byte [edi+cpu6502.sr],-1 - cpu6502.DFLAG
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; d9	= CMP abs,y
		align	.alv
.cmpabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.cmpabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.cmpabsy2:	getmem
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpabsy3
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
		align	16
.cmpabsy3:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; da	= NOPda (illegal 6502)
		align	.alv
.nopda:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; da	= PHX (65c02)
		align	.alv
.phx:		push	ebx
		mov	ebx,[edi+cpu6502.sp]
		mov	al,[edi+cpu6502.xreg]
		setmem
		dec	bl
		mov	[edi+cpu6502.sp],bl
		add	dword [edi+cpu6502.cycle],3
		pop	ebx
		jmp	.linenext
;----------------------------------------
; dc	= SKWdc (illegal 6502)
		align	.alv
.skwdc:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.skwdc2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.skwdc2:	getmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; dc	= NOPdc (65c02)
		align	.alv
.nopdc:		add	bx,2
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; dd	= CMP abs,x
		align	.alv
.cmpabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.cmpabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.cmpabsx2:	getmem
		cmp	[edi+cpu6502.areg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cmpabsx3
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
		align	16
.cmpabsx3:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; de	= DEC absx
		align	.alv
.decabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		dec	al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; df	= BBS5 (65SC02)
		align	.alv
.bbs5:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$20
		jne	.bbs5b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbs5b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; e0	= CPX #
		align	.alv
.cpxi:		getmem
		inc	bx
		cmp	[edi+cpu6502.xreg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cpxi2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.cpxi2:		or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; e1	= SBC (zp,x)
		align	.alv
.sbcindzpx:	getindzpxw
		add	dword [edi+cpu6502.cycle],6
		jmp	.sbc
;----------------------------------------
; e2	= SKBe2 (illegal 6502)
		align	.alv
.skbe2:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; e2	= NOPe2 (65c02)
		align	.alv
.nope2:		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; e4	= CPX zp
		align	.alv
.cpxzp:		getzp
		cmp	[edi+cpu6502.xreg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cpxzp2
		and	byte [edi+cpu6502.sr],255-1
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
		align	16
.cpxzp2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; e5	= SBC zp
		align	.alv
.sbczp:		getzpw
		add	dword [edi+cpu6502.cycle],3
		jmp	.sbc
;----------------------------------------
; e6	= INC zp
		align	.alv
.inczp:		getzpw
		inc	al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; e7	= SMB6 zp (65SC02)
		align	.alv
.smb6zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		or	al,$40
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; e8	= INX
		align	.alv
.inx:		inc	byte [edi+cpu6502.xreg]
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; e9	= SBC #
		align	.alv
.sbci:		getmem
		inc	bx
		push	ebx
		add	dword [edi+cpu6502.cycle],2
		jmp	.sbc
;----------------------------------------
; ea	= NOP
		align	.alv
.nop:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; ec	= CPX abs
		align	.alv
.cpxabs:	getabs
		cmp	[edi+cpu6502.xreg],al
		lahf
		mov	[edi+cpu6502.sr2],ah
		jnc	.cpxabs2
		and	byte [edi+cpu6502.sr],-1 -1
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
		align	16
.cpxabs2:	or	byte [edi+cpu6502.sr],1
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; ed	= SBC abs
		align	.alv
.sbcabs:	getabsw
		add	dword [edi+cpu6502.cycle],4
		jmp	.sbc
;----------------------------------------
; ee	= INC abs
		align	.alv
.incabs:	getabsw
		inc	al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; ef	= BBS6 (65SC02)
		align	.alv
.bbs6:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$40
		jne	.bbs6b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbs6b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; f0	= BEQ
		align	.alv
.beq:		test	byte [edi+cpu6502.sr2],$40
		jne	.beq2
		inc	bx
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
		align	16
.beq2:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; f1	= SBC (zp),y
		align	.alv
.sbcindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.sbcindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.sbcindzpy2:	getmem
		add	dword [edi+cpu6502.cycle],5
		jmp	.sbc
;----------------------------------------
; f2	= SBC (zp) (65c02)
		align	.alv
.sbcindzp:	getindzpw
		add	dword [edi+cpu6502.cycle],5
		jmp	.sbc
;----------------------------------------
; f3	= INS (zp),y (illegal 6502)
		align	.alv
.insindzpy:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		mov	ah,al
		inc	bl
		getmem
		mov	bh,al
		mov	bl,ah
		add	bl,[edi+cpu6502.yreg]
		jnc	.insindzpy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.insindzpy2:	getmem
		inc	al
		setmem
		add	dword [edi+cpu6502.cycle],8
		jmp	.sbc
;----------------------------------------
; f3	= NOPf3 (65c02)
		align	.alv
.nopf3:		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; f4	= SKBf4 (illegal 6502)
		align	.alv
.skbf4:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		inc	bx
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; f5	= SBC zp,x
		align	.alv
.sbczpx:	getzpxw
		add	dword [edi+cpu6502.cycle],4
		jmp	.sbc
;----------------------------------------
; f6	= INC zp,x
		align	.alv
.inczpx:	getzpxw
		inc	al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],6
		pop	ebx
		jmp	.linenext
;----------------------------------------
; f7	= INS zp,x (illegal 6502)
		align	.alv
.inszpx:	getzpxw
		inc	al
		setmem
		add	dword [edi+cpu6502.cycle],6
		jmp	.sbc
;----------------------------------------
; f7	= NOPf7 (65c02)
		align	.alv
.nopf7:		inc	bx
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext
;----------------------------------------
; f7	= SMB7 zp (65SC02)
		align	.alv
.smb7zp:	getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		or	al,$80
		setmem
		add	dword [edi+cpu6502.cycle],5
		pop	ebx
		jmp	.linenext
;----------------------------------------
; f8	= SED
		align	.alv
.sed:		or	byte [edi+cpu6502.sr],cpu6502.DFLAG
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; f9	= SBC abs,y
		align	.alv
.sbcabsy:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.yreg]
		jnc	.sbcabsy2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.sbcabsy2:	getmem
		add	dword [edi+cpu6502.cycle],4
		jmp	.sbc
;----------------------------------------
; fa	= NOPfa (illegal 6502)
		align	.alv
.nopfa:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		add	dword [edi+cpu6502.cycle],2
		jmp	.linenext
;----------------------------------------
; fa	= PLX (65c02)
		align	.alv
.plx:		push	ebx
		inc	byte [edi+cpu6502.sp]
		mov	ebx,[edi+cpu6502.sp]
		getmem
		mov	[edi+cpu6502.xreg],al
		or	al,al
		lahf
		mov	[edi+cpu6502.sr2],ah
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; fc	= SKWfc (illegal 6502)
		align	.alv
.skwfc:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.skwfc2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.skwfc2:	getmem
		add	dword [edi+cpu6502.cycle],4
		pop	ebx
		jmp	.linenext
;----------------------------------------
; fc	= NOPfc (65c02)
		align	.alv
.nopfc:		add	bx,2
		add	dword [edi+cpu6502.cycle],4
		jmp	.linenext
;----------------------------------------
; fd	= SBC abs,x
		align	.alv
.sbcabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		mov	bl,ah
		mov	bh,al
		add	bl,[edi+cpu6502.xreg]
		jnc	.sbcabsx2
		inc	bh
		inc	dword [edi+cpu6502.cycle]
.sbcabsx2:	getmem
		add	dword [edi+cpu6502.cycle],4
		jmp	.sbc
;----------------------------------------
; fe	= INC abs,x
		align	.alv
.incabsx:	getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		inc	al
		lahf
		mov	[edi+cpu6502.sr2],ah
		setmem
		add	dword [edi+cpu6502.cycle],7
		pop	ebx
		jmp	.linenext
;----------------------------------------
; ff	= INS abs,x (illegal 6502)
		align	.alv
.insabsx:
%ifdef dapple
		call	wrapper.cpushowillegal
%endif
		getmem
		inc	bx
		mov	ah,al
		getmem
		inc	bx
		push	ebx
		add	ah,[edi+cpu6502.xreg]
		adc	al,0
		mov	bl,ah
		mov	bh,al
		getmem
		inc	al
		setmem
		add	dword [edi+cpu6502.cycle],7
		jmp	.sbc
;----------------------------------------
; ff	= NOPff (65c02)
		align	.alv
.nopff:		add	bx,2
		add	dword [edi+cpu6502.cycle],3
		jmp	.linenext
;----------------------------------------
; ff	= BBS7 (65SC02)
		align	.alv
.bbs7:		getmem
		inc	bx
		push	ebx
		mov	ebx,eax
		getmem
		pop	ebx
		test	al,$80
		jne	.bbs7b
		inc	bx
		add	dword [edi+cpu6502.cycle],5
		jmp	.linenext
		align	.alv
.bbs7b:		getmem
		cbw
		inc	ax
		add	bx,ax
		add	dword [edi+cpu6502.cycle],6
		jmp	.linenext



%ifdef dapple

;****************************************
; wrapper code
;****************************************


		EXTERN	_dvalue1
		EXTERN	_dvalue2
		EXTERN	_dvalue3
		EXTERN	_dvalue4
		EXTERN	_dvalue5
		EXTERN	_dvalue6
		EXTERN	_dvalue7
		EXTERN	_dvalue8

		EXTERN	_applereset
		EXTERN	_appletype

		EXTERN	_setmessage

		EXTERN	_memram
		EXTERN	_memlcaux
		EXTERN	_memlcramr
		EXTERN	_memlcramw
		EXTERN	_memlcbank2
		EXTERN	_memnolc
		EXTERN	_memauxbank
		EXTERN	_memauxbankptr
		EXTERN	_memauxread
		EXTERN	_memauxwrite
		EXTERN	_memrombank
		EXTERN	_memintcxrom
		EXTERN	_memstore80
		EXTERN	_memoryread
		EXTERN	_memorywrite

		EXTERN	_virthgr
		EXTERN	_virtpage2
		EXTERN	_virtline
		EXTERN	_virtrasterline
		EXTERN	_virtwrite0400
		EXTERN	_virtwrite0400aux
		EXTERN	_virtwrite0800
		EXTERN	_virtwrite0800aux
		EXTERN	_virtwrite2000
		EXTERN	_virtwrite2000aux
		EXTERN	_virtwrite4000
		EXTERN	_virtwrite4000aux

		EXTERN	_cpumessageflag
		EXTERN	_cpulight

; type of apple
APPLEII:	equ	1
APPLEIIE:	equ	2
APPLEIIC:	equ	4

; organisation of emulated memory
MEMAUXRAM:	equ	0x10000
MEMSLOTROM:	equ	0x20000
MEMROM:		equ	0x21000
MEMINTCXROM:	equ	0x24000
MEMROM2:	equ	0x25000


wrapper:

; EAX, ECX, EDX = scratch registers


;----------------------------------------
;
;     void cpuinit()
;
;----------------------------------------

		align	16
.cpuinit:	pushad
		mov	esi,.cpu
		call	cpu6502.init
		mov	esi,.memory
		call	_arafinit
		mov	dword [esi+raf.get],.memget
		mov	dword [esi+raf.set],.memset
		mov	eax,$10000
		call	_arafsetlimit
		popad
		ret

;----------------------------------------
;	EBX	= address
; ==>	AL	= byte value

		align	32
.memget:	cmp	bh,$0c0
		jae	near .mgc000
		cmp	bh,$020
		jae	.mg2000
		cmp	bh,$04
		jae	.mg0400
		cmp	bh,$02
		jae	.mg0200
		test	byte [_memlcaux],-1
		jne	.mg0000aux
		mov	al,[_memram+ebx]
		ret
		align	8
.mg0000aux:	push	ebx
		add	ebx,[_memauxbankptr]
		mov	al,[ebx]
		pop	ebx
		ret
		align	8
.mg0200:	test	byte [_memauxread],-1
		jne	.mg0000aux
		mov	al,[_memram+ebx]
		ret
		align	8
.mg0400:	cmp	bh,$08
		jae	.mg0200
		test	byte [_memstore80],-1
		jne	.mg0400st80
		test	byte [_memauxread],-1
		jne	.mg0000aux
		mov	al,[_memram+ebx]
		ret
		align	8
.mg0400st80:	test	byte [_virtpage2],-1
		jne	.mg0400aux
		mov	al,[_memram+ebx]
		ret
		align	8
.mg0400aux:	mov	al,[_memram+ebx+MEMAUXRAM]
		ret
		align	8
.mg2000:	cmp	bh,$040
		jae	.mg0200
		test	byte [_memstore80],-1
		je	.mg0200
		test	byte [_virthgr],-1
		je	.mg0200
		test	byte [_virtpage2],-1
		jne	.mg0400aux
		mov	al,[_memram+ebx]
		ret

		align	16
.mgc000:	cmp	bh,$0d0
		jae	.mgd000
		push	eax
		push	ebx
		call	_memoryread
		pop	ebx
		mov	[esp],al
		pop	eax
		ret
		align	8
.mgd000:	test	byte [_memlcramr],-1
		je	.mgd000rom
		test	byte [_memnolc],-1
		je	.mgd000ram
		test	byte [_appletype],APPLEII
		je	.mgd000ram
.mgd000rom:	test	byte [_memrombank],-1
		jne	.mgd000rom2
		mov	al,[_memram+ebx-$0d000+MEMROM]
		ret
		align	8
.mgd000rom2:	mov	al,[_memram+ebx-$0d000+MEMROM2]
		ret
		align	8
.mgd000ram:	test	byte [_memlcaux],-1
		jne	.mgd000ram2
		cmp	bh,$0e0
		jae	.mge000ram
		test	byte [_memlcbank2],-1
		jne	.mge000ram
		mov	al,[_memram+ebx-$01000]
		ret
		align	8
.mge000ram:	mov	al,[_memram+ebx]
		ret
		align	8
.mgd000ram2:	cmp	bh,$0e0
		jae	.mge000ram2
		test	byte [_memlcbank2],-1
		jne	.mge000ram2
		push	ebx
		add	ebx,[_memauxbankptr]
		mov	al,[ebx-$1000]
		pop	ebx
		ret
		align	8
.mge000ram2:	push	ebx
		add	ebx,[_memauxbankptr]
		mov	al,[ebx]
		pop	ebx
		ret

;----------------------------------------
;	AL	= byte value
;	EBX	= address

		align	32
.memset:	cmp	bh,$0c0
		jae	near .msc000
		cmp	bh,$020
		jae	near .ms2000
		cmp	bh,$04
		jae	.ms0400
		cmp	bh,$02
		jae	.ms0200
		test	byte [_memlcaux],-1
		jne	.ms0000aux
		mov	[_memram+ebx],al
		ret
		align	8
.ms0000aux:	push	ebx
		add	ebx,[_memauxbankptr]
		mov	[ebx],al
		pop	ebx
		ret
		align	8
.ms0200:	test	byte [_memauxwrite],-1
		jne	.ms0000aux
		mov	[_memram+ebx],al
		ret
		align	8
.ms0400:	cmp	bh,$08
		jae	.ms0800
		test	byte [_memstore80],-1
		jne	.ms0400st80
		test	byte [_memauxwrite],-1
		jne	.ms0400aux
.ms0400a:	cmp	al,[_memram+ebx]
		je	.ms0400b
		mov	[_memram+ebx],al
		push	eax
		push	ebx
		call	_virtwrite0400
		pop	ebx
		pop	eax
.ms0400b:	ret
		align	8
.ms0400aux:	test	byte [_memauxbank],-1
		jne	.ms0000aux
.ms0400aux2:	cmp	al,[_memram+ebx+MEMAUXRAM]
		je	.ms0400b
		mov	[_memram+ebx+MEMAUXRAM],al
		push	eax
		push	ebx
		call	_virtwrite0400aux
		pop	ebx
		pop	eax
		ret
		align	8
.ms0400st80:	test	byte [_virtpage2],-1
		je	.ms0400a
		jmp	short .ms0400aux2
		align	8
.ms0800:	cmp	bh,$0c
		jae	.ms0200
		test	byte [_memauxwrite],-1
		jne	.ms0800aux
		cmp	al,[_memram+ebx]
		je	.ms0800b
		mov	[_memram+ebx],al
		push	eax
		push	ebx
		call	_virtwrite0800
		pop	ebx
		pop	eax
.ms0800b:	ret
		align	8
.ms0800aux:	test	byte [_memauxbank],-1
		je	.ms0000aux
		cmp	al,[_memram+ebx+MEMAUXRAM]
		je	.ms0800b
		mov	[_memram+ebx+MEMAUXRAM],al
		push	eax
		push	ebx
		call	_virtwrite0800aux
		pop	ebx
		pop	eax
		ret
		align	8
.ms2000:	cmp	bh,$040
		jae	.ms4000
		test	byte [_memstore80],-1
		jne	.ms2000st80
.ms2000a:	test	byte [_memauxwrite],-1
		jne	.ms2000aux
.ms2000b:	cmp	al,[_memram+ebx]
		je	.ms2000c
		mov	[_memram+ebx],al
		push	eax
		push	ebx
		call	_virtwrite2000
		pop	ebx
		pop	eax
.ms2000c:	ret
		align	8
.ms2000aux:	test	byte [_memauxbank],-1
		jne	.ms0000aux
.ms2000aux2:	cmp	al,[_memram+ebx+MEMAUXRAM]
		je	.ms2000c
		mov	[_memram+ebx+MEMAUXRAM],al
		push	eax
		push	ebx
		call	_virtwrite2000aux
		pop	ebx
		pop	eax
		ret
		align	8
.ms2000st80:	test	byte [_virthgr],-1
		je	.ms2000a
		test	byte [_virtpage2],-1
		je	.ms2000b
		jmp	short .ms2000aux2
		align	8
.ms4000:	cmp	bh,$60
		jae	.ms0200
		test	byte [_memauxwrite],-1
		jne	.ms4000aux
		cmp	al,[_memram+ebx]
		je	.ms4000b
		mov	[_memram+ebx],al
		push	eax
		push	ebx
		call	_virtwrite4000
		pop	ebx
		pop	eax
.ms4000b:	ret
		align	8
.ms4000aux:	test	byte [_memauxbank],-1
		je	.ms0000aux
		cmp	al,[_memram+ebx+MEMAUXRAM]
		je	.ms4000b
		mov	[_memram+ebx+MEMAUXRAM],al
		push	eax
		push	ebx
		call	_virtwrite4000aux
		pop	ebx
		pop	eax
		ret

		align	16
.msc000:	cmp	bh,$0d0
		jae	.msd000
		push	eax
		push	ebx
		call	_memorywrite
		pop	ebx
		pop	eax
		ret
		align	8
.msd000:	test	byte [_memlcramw],-1
		je	.msd000rom
		test	byte [_memnolc],-1
		je	.msd000ram
		test	byte [_appletype],APPLEII
		je	.msd000ram
.msd000rom:	ret
		align	8
.msd000ram:	test	byte [_memlcaux],-1
		jne	.msd000ram2
		cmp	bh,$0e0
		jae	.mse000ram
		test	byte [_memlcbank2],-1
		jne	.mse000ram
		mov	[_memram+ebx-$01000],al
		ret
		align	8
.mse000ram:	mov	[_memram+ebx],al
		ret
		align	8
.msd000ram2:	cmp	bh,$0e0
		jae	.mse000ram2
		test	byte [_memlcbank2],-1
		jne	.mse000ram2
		push	ebx
		add	ebx,[_memauxbankptr]
		mov	[ebx-$1000],al
		pop	ebx
		ret
		align	8
.mse000ram2:	push	ebx
		add	ebx,[_memauxbankptr]
		mov	[ebx],al
		pop	ebx
		ret


;----------------------------------------
;
;     void cpureset(void)
;
;----------------------------------------

		align	16
.cpureset:	push	esi
		mov	esi,.cpu
		call	cpu6502.reset
		pop	esi
		ret


;----------------------------------------
;
;     void cpuline(void)
;
;----------------------------------------

		align	16
.cpuline:	push	esi
		push	edi
		mov	esi,.memory
		mov	edi,.cpu
		call	cpu6502.line
		pop	edi
		pop	esi
		ret


;----------------------------------------
;
;     void cpusetpc(unsigned int address) {
;
;	pc = address & 0xffff;
;
;     } /* cpusetpc */
;
;----------------------------------------

		align	16
.cpusetpc:	xor	eax,eax
		mov	ax,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.pc],eax
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugetpc() {
;
;	return pc;
;
;     } /* cpugetpc */
;
;----------------------------------------

		align	16
.cpugetpc:	mov	eax,.cpu
		mov	eax,[eax+cpu6502.pc]
		ret


;----------------------------------------
;
;     void cpusetsp(unsigned int address) {
;
;	stack = address & 0xff;
;
;     } /* cpusetsp */
;
;----------------------------------------

		align	16
.cpusetsp:	mov	eax,$0100
		mov	al,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.sp],eax
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugetsp() {
;
;	return (stack | 0x100);
;
;     } /* cpugetsp */
;
;----------------------------------------

		align	16
.cpugetsp:	mov	eax,.cpu
		mov	eax,[eax+cpu6502.sp]
		ret


;----------------------------------------
;
;     void cpuseta(unsigned int value) {
;
;	areg = value & 0xff;
;
;     } /* cpuseta */
;
;----------------------------------------

		align	16
.cpuseta:	mov	al,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.areg],al
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugeta() {
;
;	return areg;
;
;     } /* cpugeta */
;
;----------------------------------------

		align	16
.cpugeta:	push	esi
		mov	esi,.cpu
		xor	eax,eax
		mov	al,[esi+cpu6502.areg]
		pop	esi
		ret


;----------------------------------------
;
;     void cpusetx(unsigned int value) {
;
;	xreg = value & 0xff;
;
;     } /* cpusetx */
;
;----------------------------------------

		align	16
.cpusetx:	mov	al,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.xreg],al
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugetx() {
;
;	return xreg;
;
;     } /* cpugetx */
;
;----------------------------------------

		align	16
.cpugetx:	push	esi
		mov	esi,.cpu
		xor	eax,eax
		mov	al,[esi+cpu6502.xreg]
		pop	esi
		ret


;----------------------------------------
;
;     void cpusety(unsigned int value) {
;
;	yreg = value & 0xff;
;
;     } /* cpusety */
;
;----------------------------------------

		align	16
.cpusety:	mov	al,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.yreg],al
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugety() {
;
;	return yreg;
;
;     } /* cpugety */
;
;----------------------------------------

		align	16
.cpugety:	push	esi
		mov	esi,.cpu
		xor	eax,eax
		mov	al,[esi+cpu6502.yreg]
		pop	esi
		ret


;----------------------------------------
;
;     void cpuclearstate(unsigned int value) {
;
;	value = value
;		& (STATEHALT | STATERESET | STATETRACE | STATEIRQ | STATENMI | STATEBPT | STATEGURU)
;		& (stateflags);
;
;	if (value & STATEIRQ) {
;	  setmessage("CPU IRQ cleared");
;	}
;	if (value & STATENMI) {
;	  setmessage("CPU NMI cleared");
;	}
;	if (value & STATERESET) {
;	  setmessage("CPU Reset cleared");
;	}
;	if (value & STATEHALT) {
;	  setmessage("CPU started");
;	  cpulight(1);
;	}
;	if (value & STATETRACE) {
;	  setmessage("CPU trace mode cleared");
;	}
;	if (value & STATEGURU) {	/* if it was an illegal opcode ==> reset whole machine */
;	  applereset();
;	  cpulight(1);
;	}
;	stateflags = stateflags & (~((unsigned char)value));
;
;     } /* cpuclearstate */
;
;----------------------------------------

		align	16
.cpuclearstate:	push	ebx
		push	esi
		mov	esi,.cpu
		mov	bl,[esp+4+8]
		and	bl,[esi+cpu6502.stateflags]
		test	bl,cpu6502.STATEIRQ
		je	.cpucsttnmi
		push	dword .cpucsttirqt
		call	_setmessage
		add	esp,4
.cpucsttnmi:	test	bl,cpu6502.STATENMI
		je	.cpucsttreset
		push	dword .cpucsttnmit
		call	_setmessage
		add	esp,4
.cpucsttreset:	test	bl,cpu6502.STATERESET
		je	.cpucstthalt
		push	dword .cpucsttresett
		call	_setmessage
		add	esp,4
.cpucstthalt:	test	bl,cpu6502.STATEHALT
		je	.cpucstttrace
		push	dword .cpucstthaltt
		call	_setmessage
		mov	dword [esp],-1
		call	_cpulight
		add	esp,4
.cpucstttrace:;	test	bl,cpu6502.STATETRACE		don't show message because of too many trace exceptions
;		je	.cpucsttguru
;		push	dword .cpucstttracet
;		call	_setmessage
;		add	esp,4
.cpucsttguru:	test	bl,cpu6502.STATEGURU
		je	.cpucsttguru2
		call	_applereset
		push	dword -1
		call	_cpulight
		add	esp,4
.cpucsttguru2:	not	bl
		and	[esi+cpu6502.stateflags],bl
		pop	esi
		pop	ebx
		ret

.cpucsttirqt:	db	'CPU IRQ cleared',0
.cpucsttnmit:	db	'CPU NMI cleared',0
.cpucsttresett:	db	'CPU Reset cleared',0
.cpucstthaltt:	db	'CPU started',0
.cpucstttracet:	db	'CPU trace mode cleared',0


;----------------------------------------
;
;     void cpusetstate(unsigned int value) {
;
;	value = value & (STATEHALT | STATERESET | STATETRACE | STATEIRQ | STATENMI) & (~stateflags);
;/*
;	Don't put a message on screen with Dapple, because mouse VBL just generates too many interrupts ...
;
;	if (value & STATEIRQ) {
;	  setmessage("CPU IRQ triggered");
;	}
;*/
;	if (value & STATENMI) {
;	  setmessage("CPU NMI triggered");
;	}
;	if (value & STATERESET) {
;	  setmessage("CPU Reset triggered");
;	}
;	if (value & STATEHALT) {
;	  setmessage("CPU halted");
;	  cpulight(0);
;	}
;	if (value & STATETRACE) {
;	  setmessage("CPU trace mode set");
;	}
;	  
;	stateflags = stateflags | (unsigned char)value;
;
;     } /* cpusetstate */
;
;----------------------------------------

		align	16
.cpusetstate:	push	ebx
		push	esi
		mov	bl,[esp+4+8]
		mov	esi,.cpu
		and	bl,cpu6502.STATEHALT+cpu6502.STATERESET+cpu6502.STATETRACE+cpu6502.STATENMI+cpu6502.STATEIRQ
		mov	bh,[esi+cpu6502.stateflags]
		not	bh
		and	bl,bh

%ifndef dapple
;	Don't put a message on screen with Dapple, because mouse VBL just generates too many interrupts ...
		test	bl,cpu6502.STATEIRQ
		je	.cpussttnmi
		push	dword .cpussttirqt
		call	_setmessage
		add	esp,4
%endif
.cpussttnmi:	test	bl,cpu6502.STATENMI
		je	.cpussttreset
		push	dword .cpussttnmit
		call	_setmessage
		add	esp,4
.cpussttreset:	test	bl,cpu6502.STATERESET
		je	.cpusstthalt
		push	dword .cpussttresett
		call	_setmessage
		add	esp,4
.cpusstthalt:	test	bl,cpu6502.STATEHALT
		je	.cpusstttrace
		push	dword .cpusstthaltt
		call	_setmessage
		mov	dword [esp],0
		call	_cpulight
		add	esp,4
.cpusstttrace:	test	bl,cpu6502.STATETRACE
		je	.cpusstttrace2
		push	dword .cpusstttracet
		call	_setmessage
		add	esp,4
.cpusstttrace2:
		or	[esi+cpu6502.stateflags],bl

		pop	esi
		pop	ebx
		ret

.cpussttirqt:	db	'CPU IRQ triggered',0
.cpussttnmit:	db	'CPU NMI triggered',0
.cpussttresett:	db	'CPU Reset triggered',0
.cpusstthaltt:	db	'CPU halted',0
.cpusstttracet:	db	'CPU trace mode set',0


;----------------------------------------
;
;     unsigned char cpugetstate() {
;
;	return stateflags;
;
;     } /* cpugetstate */
;
;----------------------------------------

		align	16
.cpugetstate:	push	esi
		mov	esi,.cpu
		xor	eax,eax
		mov	al,[esi+cpu6502.stateflags]
		pop	esi
		ret


;----------------------------------------
;
;     void cpufastsethalt() {
;
;	stateflags = stateflags | STATEHALT;
;	cpulight(0);
;
;     } /* cpufastsethalt */
;
;----------------------------------------

		align	16
.cpufastsethalt:
		mov	eax,.cpu
		or	byte [eax+cpu6502.stateflags],cpu6502.STATEHALT
		push	dword 0
		call	_cpulight
		add	esp,4
		ret


;----------------------------------------
;
;     void cpufastclearhalt() {
;
;	stateflags = stateflags ~(STATEHALT);
;	cpulight(1);
;
;     } /* cpufastclearhalt */
;
;----------------------------------------

		align	16
.cpufastclearhalt:
		mov	eax,.cpu
		and	byte [eax+cpu6502.stateflags],-1-cpu6502.STATEHALT
		push	dword 1
		call	_cpulight
		add	esp,4
		ret


;----------------------------------------
;
;     void cpusetcycle(unsigned int value) {
;
;	cycle = value;
;
;     } /* cpusetcycle */
;
;----------------------------------------

		align	16
.cpusetcycle:	mov	eax,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.cycle],eax
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugetcycle() {
;
;	return cycle;
;
;     } /* cpugetcycle */
;
;----------------------------------------

		align	16
.cpugetcycle:	mov	eax,.cpu
		mov	eax,[eax+cpu6502.cycle]
		ret


;----------------------------------------
;
;     void cpusetlastcycle(unsigned int value) {
;
;	lastcycle = value;
;
;     } /* cpusetlastcycle */
;
;----------------------------------------

		align	16
.cpusetlastcycle:
		mov	eax,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.lastcycle],eax
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugetlastcycle() {
;
;	return lastcycle;
;
;     } /* cpugetlastcycle */
;
;----------------------------------------

		align	16
.cpugetlastcycle:
		mov	eax,.cpu
		mov	eax,[eax+cpu6502.lastcycle]
		ret


;----------------------------------------
;
;     unsigned int cpugetlinecycle() {
;
;	return linecycles;
;
;     } /* cpugetlinecycle */
;
;----------------------------------------

		align	16
.cpugetlinecycle:
		mov	eax,.cpu
		mov	eax,[eax+cpu6502.linecycles]
		ret


;----------------------------------------
;
;     void cpusetlinecycle(unsigned int value) {
;
;	linecycles = value;
;
;     } /* cpusetlinecycle */
;
;----------------------------------------

		align	16
.cpusetlinecycle:
		mov	eax,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.linecycles],eax
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugetdelay() {
;
;	return cpudelay;
;
;     } /* cpugetdelay */
;
;----------------------------------------

		align	16
.cpugetdelay:	mov	eax,.cpu
		mov	eax,[eax+cpu6502.waitstates]
		ret


;----------------------------------------
;
;     void cpusetdelay(unsigned int value) {
;
;	cpudelay = value;
;
;     } /* cpusetdelay */
;
;----------------------------------------

		align	16
.cpusetdelay:	mov	eax,[esp+4]
		push	esi
		mov	esi,.cpu
		mov	[esi+cpu6502.waitstates],eax
		pop	esi
		ret


;----------------------------------------
;
;     unsigned int cpugetbreakpoint() {
;
;	return breakpoint;
;
;     } /* cpugetbreakpoint */
;
;----------------------------------------

		align	16
.cpugetbreakpoint:
		mov	eax,.cpu
		mov	eax,[eax+cpu6502.breakpoint]
		ret


;----------------------------------------
;
;     void cpusetbreakpoint(unsigned int value) {
;
;	if (value > 0xffff) {
;	  breakpoint = 0xffffffff;
;	}
;	else {
;	  breakpoint = value;
;	}
;	stateflags = stateflags & ~STATEBPT;
;
;     } /* cpusetbreakpoint */
;
;----------------------------------------

		align	16
.cpusetbreakpoint:
		mov	eax,[esp+4]
		push	esi
		mov	esi,.cpu
		cmp	eax,$0ffff
		jbe	.cpusbrkp2
		mov	eax,$0ffffffff
.cpusbrkp2:	mov	[esi+cpu6502.breakpoint],eax
		and	byte [esi+cpu6502.stateflags],$0ff - cpu6502.STATEBPT
		pop	esi
		ret


;----------------------------------------
;
;     void cpuclearbreakpoint() {
;
;	breakpoint = 0xffffffff;
;	stateflags = stateflags & ~STATEBPT;
;
;     } /* cpuclearbreakpoint */
;
;----------------------------------------

		align	16
.cpuclearbreakpoint:
		mov	eax,.cpu
		mov	dword [eax+cpu6502.breakpoint],$0ffffffff
		and	byte [eax+cpu6502.stateflags],$0ff - cpu6502.STATEBPT
		ret


;----------------------------------------
;
;     unsigned int cpugettracemode() {
;
;	return traceflag;
;
;     } /* cpugettracemode */
;
;----------------------------------------

		align	16
.cpugettracemode:
		push	esi
		mov	esi,.cpu
		xor	eax,eax
		mov	al,[esi+cpu6502.traceflag]
		pop	esi
		ret


;----------------------------------------
;
;     void cpusettracemode(unsigned int value) {
;
;	if (value < 3) {
;	  traceflag = value;
;	}
;
;     } /* cpusettracemode */
;
;----------------------------------------

		align	16
.cpusettracemode:
		mov	al,[esp+4]
		push	esi
		mov	esi,.cpu
		cmp	al,2
		ja	.cpusttr2
		mov	[esi+cpu6502.traceflag],al
.cpusttr2:	pop	esi
		ret


;----------------------------------------
;
;     void cpusettype (unsigned int value) {
;
;	switch (value) {
;	  case CPU6502 :
;	  case CPU65C02 :
;	  case CPU65SC02 :
;	    cputype = value;
;	    break;
;	} /* switch */
;
;     } /* cpusettype */
;
;----------------------------------------

		align	16
.cpusettype:	mov	eax,[esp+4]
		push	esi
		mov	esi,.cpu
		cmp	eax,cpu6502.CPU6502
		je	.cpustyp6502
		cmp	eax,cpu6502.CPU65C02
		je	.cpustyp65c02
		cmp	eax,cpu6502.CPU65SC02
		je	.cpustyp65sc02
		pop	esi
		ret
.cpustyp6502:	call	cpu6502.set6502
		pop	esi
		ret
.cpustyp65c02:	call	cpu6502.set65c02
		pop	esi
		ret
.cpustyp65sc02:	call	cpu6502.set65sc02
		pop	esi
		ret


;----------------------------------------
;
;     unsigned char cpugettype () {
;
;	return cputype;
;
;     } /* cpugettype */
;
;----------------------------------------

		align	16
.cpugettype:	push	esi
		mov	esi,.cpu
		xor	eax,eax
		mov	al,[esi+cpu6502.type]
		pop	esi
		ret


;----------------------------------------
;
;     void cpuwriteregs(unsigned int window);
;
;----------------------------------------

		align	16
.cpuwriteregs:	push	esi
		push	edi
		sub	esp,80
		mov	esi,.cpu
		mov	edi,esp
		call	cpu6502.tostring
		mov	esi,[esp+80+8+4]
		mov	byte [edi+cpu6502.cpuyregt+2-cpu6502.cputostrt],13
		mov	byte [edi+cpu6502.cpuspregt+2-cpu6502.cputostrt],13
		call	_astringwrite
		add	esp,80
		pop	edi
		pop	esi
		ret


;----------------------------------------
;
;/* the following function will only show that an illegal opcode was executed,
;   but will not halt the CPU */
;;
;     void cpushowillegal(unsigned int opcode) {
;	unsigned char message[80];
;
;	if (cpumessageflag) {
;	  strcpy(message, "Illegal instruction $xx hit at address $xxxx");
;	  hex8tostringf(opcode,    &message[21]);
;	  hex8tostringf((pc - 1) >> 8,   &message[40]);
;	  hex8tostringf((pc - 1) & 0xff, &message[42]);
;	  setmessage(message);
;	}
;
;     } /* cpushowillegal */
;
;----------------------------------------

;	AL	= opcode
;	BX	= pc
.cpushowillegal:
		test	byte [_cpumessageflag],-1
		je	.cpusill2
		push	eax
		push	esi
		mov	esi,.cpusillm2
		call	cpu6502.ghex
		mov	esi,.cpusillm3
		mov	al,bh
		call	cpu6502.ghex
		mov	al,bl
		call	cpu6502.ghex
		push	dword .cpusillm
		call	_setmessage
		add	esp,4
		pop	esi
		pop	eax
.cpusill2:	ret

		section	.data
.cpusillm:	db	'Illegal instruction $'
.cpusillm2:	db	'xx hit at address $'
.cpusillm3:	db	'xxxx',0
		section	.text

;----------------------------------------

		section	.bss
		alignb	32
.cpu:		resb	cpu6502.size
		alignb	32
.memory:	resb	raf.size
		section	.text

;----------------------------------------

		GLOBAL	_cpu6502init
		GLOBAL	_cpu6502reset
		GLOBAL	_cpu6502line

		GLOBAL	_cpusetpc
		GLOBAL	_cpugetpc
		GLOBAL	_cpusetsp
		GLOBAL	_cpugetsp
		GLOBAL	_cpuseta
		GLOBAL	_cpugeta
		GLOBAL	_cpusetx
		GLOBAL	_cpugetx
		GLOBAL	_cpusety
		GLOBAL	_cpugety
		GLOBAL	_cpugettype
		GLOBAL	_cpusettype
		GLOBAL	_cpuclearstate
		GLOBAL	_cpufastsethalt
		GLOBAL	_cpufastclearhalt
		GLOBAL	_cpugetstate
		GLOBAL	_cpusetstate
		GLOBAL	_cpugetcycle
		GLOBAL	_cpusetcycle
		GLOBAL	_cpugetlastcycle
		GLOBAL	_cpusetlastcycle
		GLOBAL	_cpugetlinecycle
		GLOBAL	_cpusetlinecycle
		GLOBAL	_cpugetdelay
		GLOBAL	_cpusetdelay
		GLOBAL	_cpugetbreakpoint
		GLOBAL	_cpusetbreakpoint
		GLOBAL	_cpuclearbreakpoint
		GLOBAL	_cpugettracemode
		GLOBAL	_cpusettracemode
		GLOBAL	_cpuwriteregs

;----------------------------------------
; These functions are called from within C functions
;----------------------------------------

_cpu6502init:		equ	wrapper.cpuinit
_cpu6502reset:		equ	wrapper.cpureset
_cpu6502line:		equ	wrapper.cpuline

;----------------------------------------
; These functions directly replace those formerly declared in the C file
;----------------------------------------

_cpusetpc:		equ	wrapper.cpusetpc
_cpugetpc:		equ	wrapper.cpugetpc
_cpusetsp:		equ	wrapper.cpusetsp
_cpugetsp:		equ	wrapper.cpugetsp
_cpuseta:		equ	wrapper.cpuseta
_cpugeta:		equ	wrapper.cpugeta
_cpusetx:		equ	wrapper.cpusetx
_cpugetx:		equ	wrapper.cpugetx
_cpusety:		equ	wrapper.cpusety
_cpugety:		equ	wrapper.cpugety
_cpugettype:		equ	wrapper.cpugettype
_cpusettype:		equ	wrapper.cpusettype
_cpugetstate:		equ	wrapper.cpugetstate
_cpusetstate:		equ	wrapper.cpusetstate
_cpuclearstate:		equ	wrapper.cpuclearstate
_cpufastsethalt:	equ	wrapper.cpufastsethalt
_cpufastclearhalt:	equ	wrapper.cpufastclearhalt
_cpusetcycle:		equ	wrapper.cpusetcycle
_cpugetcycle:		equ	wrapper.cpugetcycle
_cpusetlastcycle:	equ	wrapper.cpusetlastcycle
_cpugetlastcycle:	equ	wrapper.cpugetlastcycle
_cpugetlinecycle: 	equ	wrapper.cpugetlinecycle
_cpusetlinecycle: 	equ	wrapper.cpusetlinecycle
_cpugetdelay:		equ	wrapper.cpugetdelay
_cpusetdelay:		equ	wrapper.cpusetdelay
_cpugetbreakpoint:	equ	wrapper.cpugetbreakpoint
_cpusetbreakpoint:	equ	wrapper.cpusetbreakpoint
_cpuclearbreakpoint:	equ	wrapper.cpuclearbreakpoint
_cpugettracemode:	equ	wrapper.cpugettracemode
_cpusettracemode:	equ	wrapper.cpusettracemode
_cpuwriteregs:		equ	wrapper.cpuwriteregs

;<ifdef dapple
%endif
