;Apple Disk Server
;
;A program to act as a floppy disk driver server, accepting data requests from
;the PC and transmitting the required data back to the PC. Uses A2ICOMMS.
;The same binary will execute under both DOS 3.3 and ProDOS.

PIOFFSET       .EQU $49        ;$49 for regular, &87 for CL
OF_RECPACKET   .EQU $23        ;$23 for regular, $1D for CL
OF_SNDPACKET   .EQU $26        ;$26 for regular, $20 for CL

CH             .EQU $24
CV             .EQU $25
SCREENPTR      .EQU $28
COUTMODE       .EQU $32
A1             .EQU $3C        ;start address (for MOVE)
A2             .EQU $3E        ;end address (for MOVE)
A4             .EQU $42        ;destination address (for MOVE)
TEMP           .EQU $FE
STACK          .EQU $0100
DOSRWTS        .EQU $B7B5
MLI            .EQU $BF00
KEYB           .EQU $C000
DEV            .EQU $C080
TEXT           .EQU $FB2F
VTAB           .EQU $FC22
HOME           .EQU $FC58
SCROLLSCREEN   .EQU $FC70
DELAY          .EQU $FCA8
CIN            .EQU $FD0C
ACCOUT         .EQU $FDDA
COUT           .EQU $FDED
INVERSE        .EQU $FE80
NORMAL         .EQU $FE84

;Main Loop:
;
; 1. Look for data request
; 2. If no request look for ESC key pressed
; 3. If ESC key pressed, quit program
; 4. No key pressed and no block, go to 1.
; 5. Pass data request on to correct OS handler

;Both DOS 3.3 and ProDOS request handlers will accept requests of tracks,
;track/sectors and blocks.

               .ORG $1600

SERVERINIT
;Make sure A2ICOMMS is loaded in memory. Search for A2ICOMMS.
COMMSSEARCH    JSR TEXT
               LDA #$08        ;start search on page $08
               STA TEMP+1
               LDA #0
               STA TEMP
IDSEARCHPAGE   LDY #3          ;string is at offset 3
IDNEXTCHAR     LDA (TEMP),Y
               CMP IDSTRING,Y
               BNE IDNEXTPAGE
               INY
               CPY #$B         ;if Y < $B then check next character
               BCC IDNEXTCHAR
               ;if we got here, we found A2ICOMMS!
               LDA TEMP+1
               STA COMMSPAGE
               JMP SETUPSCREEN
IDNEXTPAGE     INC TEMP+1      ;no match - try next page
               LDA TEMP+1      ;if page is $C0, A2ICOMMS mustn't be loaded
               CMP #$C0
               BCC IDSEARCHPAGE
               JSR MSGOUT      ;couldn't find A2ICOMMS, quit with error msg
               .TEXT "\rAPPLE EMULATOR DISK DATA SERVER V1.0\r"
               .TEXT "COPYRIGHT (C) 1994 BY ANDREW GREGORY\r"
               .TEXT "14 WHITTINGTON AVE CARINE WA 6020\r"
               .TEXT "\r\007UNABLE TO FIND A2ICOMMS IN MEMORY!\r\000"
               JMP $03D0
IDSTRING       .TEXT "123A2ICOMMS"

;Setup status display screen.
SETUPSCREEN    JSR HOME
;border
               LDX #41         ;draw a border of '*'s
               LDA #$AA
               JSR NCHOUT
               LDY #22
DOBORDER       LDA #39
               STA CH
               LDA #$AA
               JSR COUT
               JSR COUT
               DEY
               BNE DOBORDER
               LDX #38
               LDA #$AA
               JSR NCHOUT
               LDA #$AA        ;place '*' in bottom-right corner of text screen
               STA $07F7       ;without scrolling
;title
               LDA #2          ; VTAB 3:HTAB 3
               STA CV
               JSR VTAB
               LDA #2
               STA CH
               JSR MSGOUT
               .TEXT "APPLE EMULATOR DISK DATA SERVER V1.0\000"
               LDA #4          ; VTAB 5:HTAB 3
               STA CV
               JSR VTAB
               LDA #2
               STA CH
               JSR MSGOUT
               .TEXT "COPYRIGHT (C) 1994 BY ANDREW GREGORY\000"
               LDA #5          ; VTAB 6:HTAB 3
               STA CV
               JSR VTAB
               LDA #2
               STA CH
               JSR MSGOUT
               .TEXT "14 WHITTINGTON AVE CARINE WA 6020\000"
;quit reminder
               LDA #7          ; VTAB 8:HTAB 11
               STA CV
               JSR VTAB
               LDA #10
               STA CH
               JSR MSGOUT
               .TEXT "PRESS \000"
               JSR INVERSE
               JSR MSGOUT
               .TEXT "ESC\000"
               JSR NORMAL
               JSR MSGOUT
               .TEXT " TO QUIT\000"
;request command
               LDA #11         ; VTAB 12:HTAB 3
               STA CV
               JSR VTAB
               LDA #2
               STA CH
               JSR MSGOUT
               .TEXT "COMMAND:\000"
;request parameters
               LDA #13         ; VTAB 14:HTAB 12
               STA CV
               JSR VTAB
               LDA #11
               STA CH
               JSR MSGOUT
               .TEXT "SLOT:    DRIVE:\000"

;*************
;* MAIN LOOP *
;*************
MAIN
;Check for request
CHECKREQ       LDA #PIOFFSET   ;setup PKTINFO
               STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               LDY #2          ;buffer size = 8 bytes
               LDA #8
               STA (TEMP),Y
               INY
               LDA #0
               STA (TEMP),Y
               INY
               LDA # REQBUFFER & $00FF ;buffer pointer
               STA (TEMP),Y
               INY
               LDA # REQBUFFER >> 8
               STA (TEMP),Y
               LDA #OF_RECPACKET ;Receive packet offset
               JSR CALLCOMMS
               BCS CLEARSTATUS
               JMP REQRECOK
;Request check timed out - make sure status is clear
CLEARSTATUS    LDA #11         ; VTAB 12:HTAB 12 (clear command)
               STA CV
               JSR VTAB
               LDA #11
               STA CH
               LDX #25         ; SPC(25)
               LDA #$A0
               JSR NCHOUT
               LDA #13         ; VTAB 14:HTAB 17 (clear slot)
               STA CV
               JSR VTAB
               LDA #16
               STA CH
               LDA #$A0
               JSR COUT
               LDA #26         ; HTAB 27 (clear drive)
               STA CH
               LDA #$A0
               JSR COUT
               LDA #15         ; clear line 16 (disk address)
               JSR CLEARLINE
;Check for Escape key pressed
CHECKESC       LDA KEYB
               BPL MAIN
               JSR CIN
               CMP #$9B        ;ESC key?
               BNE MAIN
               ;was the Escape key - confirm
               LDA #21         ; VTAB 22:HTAB 3
               STA CV
               JSR VTAB
               LDA #2
               STA CH
               JSR MSGOUT
               .TEXT "QUITTING. ARE YOU SURE?\000"
               JSR CIN
               AND #11011111b  ;force to uppercase
               CMP #$D9        ;'Y'?
               BNE CLRQUITMSG
               ;user really wants to quit
               JSR HOME
               JMP $03D0
;clear quit message
CLRQUITMSG     LDA #21         ; clear line 22
               JSR CLEARLINE
               JMP MAIN

;********************
;* RECEIVED REQUEST *
;********************
REQRECOK       LDA #PIOFFSET
               STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               LDY #2          ;make sure received packet was 8 bytes
               LDA (TEMP),Y
               CMP #8          ;low byte = 8?
               BNE NOTPQUERY
               INY
               LDA (TEMP),Y
               BNE NOTPQUERY   ;high byte = 0?
               LDY #7          ;check for presence query
CHECKPQ        LDA REQBUFFER,Y
               CMP PRESENCEQ,Y
               BNE NOTPQUERY
               DEY
               BPL CHECKPQ
               LDA #11         ; VTAB 12:HTAB 12
               STA CV
               JSR VTAB
               LDA #11
               STA CH
               JSR MSGOUT
               .TEXT "INSTALL CHECK\000"
               LDA #'!'        ;reply to presence query
               STA REQBUFFER+7
               LDA #OF_SNDPACKET ;Send packet offset
               JSR CALLCOMMS
               JMP MAIN
PRESENCEQ      .TEXT "RMTDISK?"

NOTPQUERY      LDA REQBUFFER+0 ;check if data will follow request
               CMP #1
               BEQ READDATA1
               JMP NODATA
READDATA1      ;read the data block
               LDA #PIOFFSET   ;setup PKTINFO
               STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               LDY #2          ;buffer size = 4096 bytes
               LDA #0
               STA (TEMP),Y
               INY
               LDA #$10
               STA (TEMP),Y
               INY
               LDA # DATABUFFER & $00FF ;buffer pointer
               STA (TEMP),Y
               INY
               LDA # DATABUFFER >> 8
               STA (TEMP),Y
               LDA #OF_RECPACKET ;Receive packet offset
               JSR CALLCOMMS
               BCC RECDATA1OK
               JMP MAIN
               ;test if track only, if so more data coming!
RECDATA1OK     LDA REQBUFFER+1
               AND #00110000b
               CMP #00100000b
               BNE NODATA
READDATA2      ;read extra data
               LDA #PIOFFSET   ;setup PKTINFO
               STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               LDY #2          ;buffer size = 2560 bytes
               LDA #0
               STA (TEMP),Y
               INY
               LDA #$A
               STA (TEMP),Y
               INY
               LDA # (DATABUFFER+$1000) & $00FF ;buffer pointer
               STA (TEMP),Y
               INY
               LDA # (DATABUFFER+$1000) >> 8
               STA (TEMP),Y
               LDA #OF_RECPACKET ;Receive packet offset
               JSR CALLCOMMS
               BCC RECDATA2OK
               JMP MAIN
;Received the entire transmission OK
RECDATA2OK
NODATA
;Unpack request
               LDA REQBUFFER+0 ;get type/command
               ASL A           ;type into Carry
               LDA #0
               ROL A           ;Carry into bit 0
               STA REQTYPE
               LDA REQBUFFER+0
               AND #$7F
               STA REQCMD
               LDA REQBUFFER+1 ;get format/slot/drive
               TAX             ;save it
               AND #7          ;get slot
               STA REQSLOT
               TXA             ;original format/slot/drive
               LSR A           ;shift slot out, drive into Carry
               LSR A
               LSR A
               LSR A
               TAX             ;save high nibble of format/slot/drive
               LDA #1          ;Drive = 1 + Carry
               ADC #0
               STA REQDRIVE
               TXA             ;high nibble of format/slot/drive
               AND #3          ;mask off format
               STA REQFMT      ;save it
               LDA REQBUFFER+2 ;copy disk address
               STA REQDISKADDR
               LDA REQBUFFER+3
               STA REQDISKADDR+1
               LDA REQBUFFER+4 ;copy error
               STA REQERROR

;********************
;* VALIDATE REQUEST *
;********************
VALIDATEREQ    LDA #PIOFFSET   ;setup pointer to PKTINFO
               STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               LDY #2          ;make sure received packet was 5 bytes
               LDA (TEMP),Y
               CMP #5          ;low byte = 5?
               BNE BADREQ
               INY
               LDA (TEMP),Y
               BNE BADREQ      ;high byte = 0?
               LDA REQCMD      ;if CMS >= 6 then BADREQ
               CMP #6
               BCS BADREQ
               LDA REQTYPE     ;if TYPE = 1 then BADREQ
               BNE BADREQ
               LDA REQSLOT     ;if SLOT = 0 or SLOT >= 8 then BADREQ
               BEQ BADREQ
               CMP #8
               BCS BADREQ
               LDA REQDRIVE    ;if DRIVE = 0 or DRIVE >= 3 then BADREQ
               BEQ BADREQ
               CMP #3
               BCS BADREQ
               LDA REQFMT      ;if FMT >= 4 then BADREQ
               CMP #4
               BCS BADREQ
               JMP DISPCMD     ;must be OK, display it and continue
BADREQ         JMP MAIN

;*******************
;* DISPLAY REQUEST *
;*******************
DISPCMD        LDA #11         ; VTAB 12:HTAB 12
               STA CV
               JSR VTAB
               LDA #11
               STA CH
               LDA REQCMD      ;get command
               CMP #0
               BEQ PRINTREAD
               CMP #1
               BEQ PRINTWRITE
               CMP #2
               BEQ PRINTREADWP
               CMP #3
               BEQ PRINTDRVON
               CMP #4
               BEQ PRINTDRVOFF
               CMP #5
               BEQ PRINTRECAL
               JMP MAIN
PRINTREAD      JSR MSGOUT
               .TEXT "READ\000"
               JMP DISPSLOTDRV
PRINTWRITE     JSR MSGOUT
               .TEXT "WRITE\000"
               JMP DISPSLOTDRV
PRINTREADWP    JSR MSGOUT
               .TEXT "READ WRITE PROTECT\000"
               JMP DISPSLOTDRV
PRINTDRVON     JSR MSGOUT
               .TEXT "DRIVE ON\000"
               JMP DISPSLOTDRV
PRINTDRVOFF    JSR MSGOUT
               .TEXT "DRIVE OFF\000"
               JMP DISPSLOTDRV
PRINTRECAL     JSR MSGOUT
               .TEXT "RECALIBRATE\000"
DISPSLOTDRV    LDA #13         ; VTAB 14:HTAB 17
               STA CV
               JSR VTAB
               LDA #16
               STA CH
               LDA REQSLOT     ;get slot
               ORA #$B0        ;convert to display character
               JSR COUT
               LDA #26         ; HTAB 27
               STA CH
               LDA REQDRIVE    ;get drive
               ORA #$B0        ;convert to display character
               JSR COUT
DISPADDR       LDA REQCMD      ;if CMD >= 2 then HANDLEREQ (skip display)
               CMP #2
               BCS HANDLEREQ
               LDA #15         ; VTAB 16:HTAB 12
               STA CV
               JSR VTAB
               LDA #11
               STA CH
               LDA REQFMT      ;get disk address format
               BEQ DISPTRKSEC
               CMP #1
               BEQ DISPBLK
               CMP #2
               BEQ DISPTRKONLY
               ;if we got here - bad track format
               JMP MAIN
DISPTRKSEC     JSR MSGOUT
               .TEXT "TRACK:\000"
               LDA REQDISKADDR
               JSR ACCOUT
               JSR MSGOUT
               .TEXT " SECTOR:\000"
               LDA REQDISKADDR+1
               JSR ACCOUT
               JMP HANDLEREQ
DISPBLK        JSR MSGOUT
               .TEXT "BLOCK:\000"
               LDA REQDISKADDR+1
               JSR ACCOUT
               LDA REQDISKADDR
               JSR ACCOUT
               JMP HANDLEREQ
DISPTRKONLY    JSR MSGOUT
               .TEXT "TRACK:\000"
               LDA REQDISKADDR
               JSR ACCOUT
               JMP HANDLEREQ

;*********************************************************************
;*                                                                   *
;* HANDLE DISK REQUEST                                               *
;*                                                                   *
;*                                                                   *
;*********************************************************************
HANDLEREQ      LDA #0          ;default to no error
               STA REQERROR
               LDA REQCMD      ;pass Read/Write Requests on the the
               CMP #0          ;disk data request handler
               BEQ HANDLEDATAREQ
               CMP #1
               BEQ HANDLEDATAREQ
               CMP #5
               BNE NOTRECAL
               LDA #0          ;seek to track zero
               STA DESTTRK
               LDA #$FF        ;set an invalid current track
               STA CURTRK
               JSR SEEKTRACK   ;do the recalibrate
               JMP TRANSREPLY
NOTRECAL       ;handle Read Write Protect/Drive On/Drive Off
               ;first, select the correct drive and set up for READING
               LDA REQSLOT     ;get slot
               ASL A
               ASL A
               ASL A
               ASL A
               TAX
               LDA DEV+$E,X    ;read mode
               LDA DEV+$C,X
               TXA
               ORA REQDRIVE
               TAY
               LDA DEV+$A-1,Y  ;select drive 1 or 2
               LDA REQCMD
               CMP #2
               BEQ READWP
               CMP #3
               BEQ DRIVEON
               CMP #4
               BEQ DRIVEOFF
               LDA #1          ;bad command - I/O error
               STA REQERROR
               JMP TRANSREPLY
READWP         LDA DEV+$F,X    ;select Read Write Protect mode
               TAY
               LDA DEV+$E,X    ;re-select Read mode
               TYA
               BPL NOTWP
               LDA #3          ;return write-protected error status
               STA REQERROR
NOTWP          JMP TRANSREPLY
DRIVEON        LDA DEV+$9,X    ;turn drive on
               JMP TRANSREPLY
DRIVEOFF       LDA DEV+$8,X    ;turn drive off
               JMP TRANSREPLY

;*********************************************************************
;*                                                                   *
;* HANDLE DISK DATA REQUEST                                          *
;*                                                                   *
;* When a data request (for Reading or Writing) has been received,   *
;* this function is called.                                          *
;* This calls the appropriate DOS 3.3 or ProDOS I/O handler to set   *
;* up the reply buffer, then sends the reply to the PC.              *
;*                                                                   *
;*********************************************************************
HANDLEDATAREQ  LDA REQFMT      ;check for invalid disk address format
               CMP #3
               BNE TESTTRKONLY
               LDA #1
               STA REQERROR
               JMP TRANSREPLY
;if track only - go to internal I/O routines
TESTTRKONLY    CMP #2
               BNE NOTTRKONLY
               JSR TRKONLYHANDLE
               JMP TRANSREPLY
;get DOS 3.3/ProDOS ID byte and call appropriate routine
NOTTRKONLY     LDA $BE00       ;get ProDOS ID byte
               CMP #$4C
               BNE CALLD33HDL
               JSR PRODOSHANDLE
               JMP TRANSREPLY
CALLD33HDL     JSR DOS33HANDLE

;************************
;*                      *
;* TRANSMIT REPLY TO PC *
;*                      *
;************************
TRANSREPLY     LDA REQCMD      ;request -> acknowledge
               ORA #$80
               STA REQBUFFER+0
               LDA REQERROR    ;copy error
               STA REQBUFFER+4
;transmit reply header
               LDA #PIOFFSET   ;setup PKTINFO
               STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               LDY #2          ;buffer size = 5 bytes
               LDA #5
               STA (TEMP),Y
               INY
               LDA #0
               STA (TEMP),Y
               INY
               LDA # REQBUFFER & $00FF ;buffer pointer
               STA (TEMP),Y
               INY
               LDA # REQBUFFER >> 8
               STA (TEMP),Y
               LDA #OF_SNDPACKET ;Send packet offset
               JSR CALLCOMMS
;check error code
               LDA REQERROR
               BNE NOREPLYDATA ;not 0 - error - no data
;transmit reply data
               LDA #PIOFFSET   ;setup PKTINFO
               STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               LDY #2

;ProDOS block: 512 bytes, DOS 3.3 Sector: 256 bytes, Track: 4096+2560
               LDA #0          ;low byte always 0
               STA (TEMP),Y
               INY
               LDX REQFMT      ;FMT is 0..2
               INX             ;make it 1..3
               LDA #$01
               DEX             ;if FMT was 0 (track/sector) high byte=$01
               BEQ SETREPBUFLEN
               LDA #$02
               DEX             ;if FMT was 1 (block) high byte=$02
               BEQ SETREPBUFLEN
               LDA #$10        ;otherwise FMT was 2 (track only) high byte=$10
SETREPBUFLEN   STA (TEMP),Y
               INY

               LDA # DATABUFFER & $00FF ;buffer pointer
               STA (TEMP),Y
               INY
               LDA # DATABUFFER >> 8
               STA (TEMP),Y
               LDA #OF_SNDPACKET ;Send packet offset
               JSR CALLCOMMS
               LDA REQFMT      ;if track only, more data to transmit
               CMP #2
               BNE NOREPLYDATA
;send extra track only data
               LDA #PIOFFSET   ;setup PKTINFO
               STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               LDY #2          ;buffer size = 2560 bytes
               LDA #0
               STA (TEMP),Y
               INY
               LDA #$A
               STA (TEMP),Y
               INY
               LDA # (DATABUFFER+$1000) & $00FF ;buffer pointer
               STA (TEMP),Y
               INY
               LDA # (DATABUFFER+$1000) >> 8
               STA (TEMP),Y
               LDA #OF_SNDPACKET ;Send packet offset
               JSR CALLCOMMS
NOREPLYDATA    JMP MAIN        ;handled request

;*************************************
;*                                   *
;* HANDLE DATA REQUEST UNDER DOS 3.3 *
;*                                   *
;*************************************
DOS33HANDLE    LDA REQCMD      ;set IOB command
               LSR A           ;Carry=0 for Read,1 for Write
               LDA #1          ;Command = 1 + Carry
               ADC #0
               STA COMMAND
               LDA REQSLOT     ;set IOB slot
               ASL A
               ASL A
               ASL A
               ASL A
               STA SLOT
               LDA REQDRIVE    ;set IOB drive
               STA DRIVE
               LDA # DATABUFFER & $00FF ;set IOB buffer
               STA DOSBUFF
               LDA # DATABUFFER >> 8
               STA DOSBUFF+1
               LDA REQFMT      ;check disk address format
               BEQ D33READTS

;***********************************
;* READ ProDOS BLOCK UNDER DOS 3.3 *
;***********************************
D33READBLK     LDA REQDISKADDR ;set IOB track (track = block/8)
               STA TEMP
               LDA REQDISKADDR+1
               STA TEMP+1
               LSR TEMP+1      ;>>3 = /8
               ROR TEMP
               LSR TEMP+1
               ROR TEMP
               LSR TEMP+1
               ROR TEMP
               LDA TEMP
               STA TRACK
               LDA REQDISKADDR ;set IOB sector
               AND #$07
               TAX
               LDA SECTORTRANS,X
               STA SECTOR
               JSR RWTS        ;READ SECTOR
               BCS D33IOERR
               INC DOSBUFF+1   ;next half of block
               LDA REQDISKADDR ;set IOB sector
               AND #$07
               TAX
               LDA SECTORTRANS+8,X
               STA SECTOR
               JSR RWTS        ;READ SECTOR
               BCS D33IOERR
               RTS
SECTORTRANS    .DB $0,$D,$B,$9,$7,$5,$3,$1,$E,$C,$A,$8,$6,$4,$2,$F

;*******************************************
;* READ DOS 3.3 TRACK/SECTOR UNDER DOS 3.3 *
;*******************************************
D33READTS      LDA REQDISKADDR ;set IOB track
               STA TRACK
               LDA REQDISKADDR+1 ;set IOB sector
               STA SECTOR
               JSR RWTS        ;READ SECTOR
               BCS D33IOERR
               RTS

D33IOERR       LDA #1
               STA REQERROR
               RTS

;Internal RWTS (Carry set on error)
RWTS           LDY # IOBTABLE & $00FF
               LDA # IOBTABLE >> 8
               JSR DOSRWTS
               PHP
               LDA #0
               STA $48
               PLP
               LDA #$FF        ;if error occurred, make sure next track seek
               BCS EXITRWTS    ; causes a re-calibration. Otherwise set the
               LDA TRACK       ; current track.
EXITRWTS       STA CURTRK
               RTS

;************************************
;*                                  *
;* HANDLE DATA REQUEST UNDER PRODOS *
;*                                  *
;************************************
PRODOSHANDLE   LDA REQCMD      ;set MLI command
               LSR A           ;Carry=0 for Read,1 for Write
               LDA #$80        ;MLI command = $80 + Carry
               ADC #0
               STA PDCOMMAND
               LDA REQDRIVE    ;Drive 1=01,2=10
               AND #1          ;      1=x1,2=x0
               EOR #1          ;      1=x0,2=x1
               ASL A
               ASL A
               ASL A
               ORA REQSLOT     ;merge slot
               ASL A
               ASL A
               ASL A
               ASL A
               STA UNIT
               LDA # DATABUFFER & $00FF ;set IOB buffer
               STA BUFFERPTR
               LDA # DATABUFFER >> 8
               STA BUFFERPTR+1
               LDA REQFMT      ;check disk address format
               BEQ PDREADTS

;**********************************
;* READ PRODOS BLOCK UNDER PRODOS *
;**********************************
PDREADBLK      LDA REQDISKADDR ;set BLOCK to read/write
               STA BLOCK
               LDA REQDISKADDR+1
               STA BLOCK+1
               JSR RWBLOCK     ;READ BLOCK
               BCS PDIOERR
               RTS

;******************************************
;* READ DOS 3.3 TRACK/SECTOR UNDER PRODOS *
;******************************************
PDREADTS       LDA REQDISKADDR ;track
               STA BLOCK
               LDA #0
               STA BLOCK+1
               ASL BLOCK       ;track<<=3
               ROL BLOCK+1
               ASL BLOCK
               ROL BLOCK+1
               ASL BLOCK
               ROL BLOCK+1
               LDX REQDISKADDR+1 ;sector
               LDA BLOCKTRANS,X
               AND #$7F
               ORA BLOCK
               STA BLOCK       ;now got block
               JSR RWBLOCK     ;READ BLOCK
               BCS PDIOERR
               LDX REQDISKADDR+1
               LDA BLOCKTRANS,X
               ASL A
               BCC HALFOK
               LDX #0          ;required sector is in second half of block
COPYHALF       LDA DATABUFFER+$100,X ;move it down to the first half
               STA DATABUFFER,X
               INX
               BNE COPYHALF
HALFOK         RTS

;Bits 2-0 are the sector offset, bit 7 is the block half specifier.
;See ProDOS 8 Technical Reference Manual, pg 174.
BLOCKTRANS     .DB $00,$07,$86,$06,$85,$05,$84,$04
               .DB $83,$03,$82,$02,$81,$01,$80,$87

PDIOERR        LDA #1
               STA REQERROR
               RTS

;Internal Read/Write Block (Carry set on error)
RWBLOCK        JSR MLI
PDCOMMAND      .DB
               .DW PDCMDLIST
               STA BLOCKERROR
               BNE RWBLOCKERR
               CLC
               LDA BLOCK       ;move block into temporary
               STA TEMP
               LDA BLOCK+1
               STA TEMP+1
               LSR TEMP+1      ;divide block by 8 to get track
               ROR TEMP
               LSR TEMP+1
               ROR TEMP
               LSR TEMP+1
               ROR TEMP
               LDA TEMP        ;set current track for next seek
               STA CURTRK
               RTS
RWBLOCKERR     SEC
               LDA #$FF        ;error occurred, make sure current track
               STA CURTRK      ; is recalibrated on next seek
               RTS

;*******************************************************************
;*                                                                 *
;* HANDLE TRACK ONLY DATA REQUEST                                  *
;*                                                                 *
;* Since an entire track has been requested, and it must be in raw *
;* form, the resident OS (DOS 3.3 or ProDOS) must be bypassed and  *
;* the Disk ][ device accessed directly.                           *
;*                                                                 *
;*******************************************************************
TRKONLYHANDLE  LDA REQCMD      ;writes are not supported
               CMP #1
               BNE READTRACK   ;not Write Request, must be a Read Request
               LDA #3          ;for Write Requests, return a Write Protected
               STA REQERROR    ; error
               RTS
READTRACK      LDA REQDISKADDR ;get required track
               STA DESTTRK
               JSR SEEKTRACK
;read $3000 (12288) bytes
               LDY #$60
               STY TEMP+1
               LDY #0
               STY TEMP
;******************************************************************************
;* IMPORTANT NOTE: FOLLOWING CODE BETWEEN THE ASTERISKS IS EXECUTION TIME
;*                 SENSITIVE. IT MUST NOT STRADDLE A PAGE BOUNDARY.
;* Numbers in the comments are: clock cycles for each instruction, and
;* cumulative clock cycles.
;******************************************************************************
WAITNIBBLE     LDA DEV+$C,X                ;4      4
               BPL WAITNIBBLE              ;2/3    6 (branch not taken)
               STA (TEMP),Y                ;6     12
               ;increment buffer pointer
               INY                         ;2     14
               BNE WAITNIBBLE              ;2/3   16 (branch not taken)
               INC TEMP+1                  ;5     21
               ;check for total number of bytes read
               LDA TEMP+1                  ;3     24
               CMP # DATABUFFER >> 8 + $30 ;2     26
               BNE WAITNIBBLE              ;2/3   29 (branch taken)
;******************************************************************************
               LDA DEV+8,X     ;turn drive off
;Now that we have $3000 bytes in the buffer, find the longest stretch of
;$FF's (self-sync bytes) and presume that to be the start/end of the track.
;Then back up about $10 bytes so we start off with some of those self-sync
;bytes, then move 6250 bytes from there down to the start of the buffer. Fill
;the rest of the buffer ($1A00-6250 = 406 bytes) with $FF's. Although a real
;Apple track can contain a maximum of 6250 disk bytes, we transmit $1A00 (6656)
;bytes to the PC to be compatible with applemu's 232960 byte disk images.

;Find those $FF's. Note that this only searches until it finds a group of more
; than $20 $FF's.
               LDY #0          ;init
               STY SSSTART
               STY SSSTART+1
               STY SSCNT
               STY TEMP
               LDA # DATABUFFER >> 8
               STA TEMP+1
SEARCHSS       LDA (TEMP),Y    ;get byte from buffer
               CMP #$FF        ;self-sync?
               BEQ FOUNDSS
CLEARCNT       LDA #0
               STA SSCNT
NEXTSS         INC TEMP        ;next byte in buffer
               BNE SEARCHSS
               INC TEMP+1
               LDA TEMP+1
               CMP # DATABUFFER >> 8 + $30
               BCC SEARCHSS
               BCS DONESEARCH
FOUNDSS        INC SSCNT
               LDA SSCNT
               CMP #$30
               BNE NEXTSS
               LDA TEMP
               STA SSSTART
               LDA TEMP+1
               STA SSSTART+1
DONESEARCH     LDA SSSTART+1   ;if zero, no large self-sync field found
               BEQ QUITSEARCH
;use Monitor move routine to copy $1A00 bytes from SSSTART to DATABUFFER
;fill in from DATABUFFER+($1A00-406) to DATABUFFER+$1A00 with $FF's.
;Fill offset $186A-$19FF.
               LDA SSSTART     ;set start address
               STA A1
               LDA SSSTART+1
               STA A1+1
               LDA #0          ;set destination address
               STA A4
               LDA # DATABUFFER >> 8
               STA A4+1
;move
               LDY #0
MOVEBUFFER     LDA (A1),Y
               STA (A4),Y
               INY
               BNE MOVEBUFFER
               INC A1+1
               INC A4+1
               LDA A4+1
               CMP # (DATABUFFER >> 8) + $19 ;only copy $1900 bytes, the last
               BNE MOVEBUFFER                ;406 bytes of $1A00 will be set to $FF
;fill
               LDA #0          ;set start of fill address
               STA A4
               LDA # (DATABUFFER >> 8) + $18
               STA A4+1
               LDY #$6A
FILLNEXTPAGE   LDA #$FF
FILLBUFFER     STA (A4),Y
               INY
               BNE FILLBUFFER
               INC A4+1
               LDA A4+1
               CMP # (DATABUFFER >> 8) + $1A
               BNE FILLNEXTPAGE
QUITSEARCH     RTS

SSSTART        .DW
SSCNT          .DB             ;count of self sync bytes

;Seek to Track in DESTTRK. Leaves drive turned on.
; If the current track is greater than 40, it will recalibrate first.
SEEKTRACK      LDA REQSLOT     ;get slot
               ASL A
               ASL A
               ASL A
               ASL A
               TAX
               STX SHIFTEDSLOT
               LDA DEV+$E,X    ;read mode
               LDA DEV+$C,X
               TXA
               ORA REQDRIVE
               TAY
               LDA DEV+$A-1,Y  ;select drive 1 or 2
               LDA DEV+$9,X    ;turn drive on
               LDA #243        ;wait 150ms
               JSR DELAY
               LDA DESTTRK
               CMP CURTRK
               BNE SKABS       ;if they're the same, just return
               RTS

;Seek Absolute
;Copied from Ultra Fast Pix by Charles H. Putney, Nibble March 1987
SKABS          LDA CURTRK
               CMP #40
               BCC SKTRACK
;Re-calibrate - set current track to 40 and seek to track 0
               LDA DESTTRK     ;save destination track
               PHA
               LDA #0          ;recalibrate to track 0
               STA DESTTRK
               LDA #40         ;pretend we're on track 40
               STA CURTRK
               JSR SKTRACK     ;and seek to 0
               PLA             ;restore original destination track
               STA DESTTRK     ;and seek to it
SKTRACK        LDA DESTTRK
               ASL A
               ASL CURTRK      ;two phases per track
               LDY #0
               STY STEPS
               SEC
               SBC CURTRK
               STA HDDIR
               BPL DIROK
               STA HDDIR
               CLC
               EOR #$FF
               ADC #1
DIROK          STA HDMOVE
               ROR CURTRK
               ROR CURTRK
               LDA HDDIR
               BCC EVENTRK
               BPL ODDIN
               BMI MOVEHEAD
EVENTRK        BPL MOVEHEAD
ODDIN          LDY #2
MOVEHEAD       LDA HDMOVE
               BEQ DONESEEK
               TAX
               CPX STEPS
               BCC STEPSOK1
               LDX STEPS
STEPSOK1       CPX #8
               BCC STEPSOK2
               LDX #7
STEPSOK2       LDA DLYTBL,X
               STA HDDLY
               LDA PHSTBL,Y
               ORA SHIFTEDSLOT
               TAX
               LDA DEV,X
WAIT1          LDA #19
WAIT2          SBC #1
               BNE WAIT2
               DEC HDDLY
               BNE WAIT1
               DEX
               LDA DEV,X
               LDA HDDIR
               BMI NXTPHASE
               INY
               CPY #4
               BNE NXTMOVE
               LDY #0
               BEQ NXTMOVE
NXTPHASE       DEY
               BPL NXTMOVE
               LDY #3
NXTMOVE        DEC HDMOVE
               INC STEPS
               JMP MOVEHEAD
DONESEEK       LDA DESTTRK
               STA CURTRK
               LDX SHIFTEDSLOT
               RTS

PHSTBL         .DB 3,5,7,1
DLYTBL         .DB $70,$6C,$68,$64,$60,$5C,$5A,$58

DESTTRK        .DB             ;destination track (0-35)
CURTRK         .DB $FF         ;current track (0-35) ($FF will force calibrate)

STEPS          .DB
HDDIR          .DB
HDMOVE         .DB
HDDLY          .DB
SHIFTEDSLOT    .DB

;Message Printer (originally by Andy Hertzfeld)
;Text to display must immediately follow the JSR calling MSGOUT and must be
;terminated with a zero. Control is returned to the address following the
;terminating zero.
MSGOUT         TSX
               LDA STACK+1,X   ;get address of text
               STA TEMP
               LDA STACK+2,X
               STA TEMP+1
               LDY #0
MSGOUT1        INC TEMP
               BNE MSGOUT2
               INC TEMP+1
MSGOUT2        LDA (TEMP),Y
               BEQ MSGOUT3
               ORA #$80
               JSR COUT
               BNE MSGOUT1
MSGOUT3        LDA TEMP+1      ;update return address on the stack
               STA STACK+2,X   ; to point after end of string data
               LDA TEMP
               STA STACK+1,X
               RTS

;Prints char in A n times where n=X register
NCHOUT         JSR COUT
               DEX
               BNE NCHOUT
               RTS

;Clears line in Accumulator
CLEARLINE      STA CV
               JSR VTAB
               LDA #1
               STA CH
               LDX #38
               LDA #$A0
               JSR NCHOUT
               RTS

;Call A2ICOMMS. Offset of vector to call must be in the Accumulator
CALLCOMMS      STA TEMP
               LDA COMMSPAGE
               STA TEMP+1
               JMP (TEMP)

;*************
;* VARIABLES *
;*************

COMMSPAGE      .DB             ;page where A2ICOMMS is loaded

;DOS 3.3 RWTS IOB Table
IOBTABLE       .DB $01
SLOT           .DB $60
DRIVE          .DB $01
VOLUME         .DB $00
TRACK          .DB $00
SECTOR         .DB $00
DVCTPTR        .DW DVCT
DOSBUFF        .DW DATABUFFER
               .DW $00
COMMAND        .DB $00
ERROR          .DB $00
LASTVOL        .DB $00
LASTSLOT       .DB $60
LASTDRIV       .DB $01
DVCT           .DB $00,$01,$EF,$D8

;ProDOS READ_BLOCK / WRITE_BLOCK Command List
PDCMDLIST      .DB 3           ;3 parameters
UNIT           .DB             ;ProDOS unit number
BUFFERPTR      .DW             ;Address of data buffer
BLOCK          .DW             ;Block to read/write
BLOCKERROR     .DB             ;Not part of regular Block I/O command list

REQBUFFER      .BLOCK 8

;unpacked Request
REQTYPE        .DB             ;0=request,1=acknowledge
REQCMD         .DB             ;0=read,1=write
REQFMT         .DB             ;0=track/sector,1=block,2=track only
REQSLOT        .DB             ;1-7
REQDRIVE       .DB             ;1-2
REQDISKADDR    .DW
REQERROR       .DB

;******************************************
;* Track buffer to read two track's worth *
;* into. (12500 bytes).                   *
;******************************************

;this must start on a page boundary
DATABUFFER     .EQU $6000

               .END
