;Read parallel port
;Received data in in the following locations:
;
;                              DATA
; Data Input Register: bit 3   bit 0
;                      bit 5   bit 1
;                      bit 6   bit 2
;                      bit 7   bit 3
;  ACK Input Register: bit 0   bit 4
;
; The data is received in these locations because they were connected via
; the parallel printer cable I have for my Apple (wired up by the dealer).
; The connections enable the Apple to read various status signals from the
; printer (I don't know if the signals -> input lines are a standard, etc).

TXTTAB         .EQU $67        ;address of start of Applesoft program
VARTAB         .EQU $69        ;address of start of Applesoft simple variables
ARYTAB         .EQU $6B        ;address of start of Applesoft array variables
STREND         .EQU $6D        ;address of end of Applesoft array variables
PGREND         .EQU $AF        ;address of end of Applesoft program
STACK          .EQU $0100      ;address of start of stack
DEV            .EQU $C080      ;base I/O address
;Applesoft ROM Routines
DELAY          .EQU $FCA8      ;create a delay
CIN            .EQU $FD0C      ;get a character from the keyboard
CROUT          .EQU $FD8E      ;output a carriage-return
COUT           .EQU $FDED      ;output a character to the screen

;******************************************************************************
;* EXTERNAL ENTRY POINTS
;******************************************************************************
;For all routines, SLOT must be set to the slot where the parallel/serial card
;is. SLOT is specified as slot*16 (or $s0).
               .ORG $800
               JMP A2ICOMMSINIT

;*****************
;* RESIDENT CODE *
;*****************
               .TEXT "A2ICOMMS" ;ID string: must be at $xx03 (xx is any page)
               JMP RECEIVEBYTE ;Receive a byte
                               ;Carry clear if byte received. Received byte in
                               ;DATA.
               JMP SENDBYTE    ;Send a byte
                               ;Byte to send in DATA.
               JMP INITCRC     ;Initialise CRC
                               ;No parameters, sets CRC
               JMP CALCCRC     ;Calculate cumulative CRC
                               ;New character in CH, sets CRC
               JMP PRESENDCRC  ;Pre-send processing on CRC
                               ;No parameters, sets CRC
               JMP VALIDCRCDATA ;Valid CRC-checksummed data test
                               ;No parameters, clears/sets Carry if CRC OK/bad
               JMP RECCLPKT    ;Receive a connectionless packet
                               ;Settings in PKT. PKT settings changed to match
                               ;successfully received packet.
               JMP SENDCLPKT   ;Send a connectionless packet
                               ;Settings in PKT.
               JMP RECPACKET   ;Receive a sequenced packet
                               ;Settings in PKTINFO. PKTINFO settings changed
                               ;to match successfully received block.
               JMP SENDPACKET  ;Send a sequenced packet
                               ;Settings in PKTINFO.
               JMP SYNCSEQ     ;Synchronise sequence #'s to zero.
;******************************************************************************
;* EXTERNAL VARIABLES
;******************************************************************************
               .ORG $840
;*** Receive/Send Byte ***
PRESNDDLY      .DB 15          ;pre-send delay ~=76us
SNDDLY         .DB 7           ;send delay ~=36us
BYTETIMEOUT    .DW $F000       ;# times to retry receiving byte (1s comp)
                               ; (NOTE: counts *up* to zero)
DATA           .DB             ;also used by CALCCRC
COMMSTYPE      .DB             ;1=parallel,2=serial
SLOT           .DB             ;slot as $s0
;*** CRC Routines ***
CRC            .DW
;also uses DATA above
;*** Receive/Send Connectionless Packet ***
;Packet Info:
;offset   description
;   0   = sequence
;   1   = type
;  2-3  = length
;  4-5  = pointer to data
PKTINFO        .BLOCK 6
PKTTIMEOUT     .DB $F          ;max # Receive Byte timeouts
;*** Receive/Send Sequenced Packet ***
MAXPKTSENDS    .DB $3          ;max # Receive CL Packet calls
;******************************************************************************
;* IMPLEMENTATION
;******************************************************************************
               .ORG $860
;******************************************************************************
;* INTERNAL VARIABLES
;******************************************************************************
;*** Receive/Send Byte ***
CNT            .DW
;*** CRC Routines ***
CRCTABLE       .DW $0000,$1081,$2102,$3183,$4204,$5285,$6306,$7387
               .DW $8408,$9489,$A50A,$B58B,$C60C,$D68D,$E70E,$F78F
;*** Receive/Send Connectionless Packet ***
MAXBUFLEN      .DW
I              .DW
PKTCNT         .DB
;*** Receive/Send Sequenced Packet ***
PKT            .BLOCK 6
ACKPKT         .BLOCK 6
ACKDATA        .DB
PKTTRIES       .DB
CURRECSEQ      .DB 0
CURSNDSEQ      .DB 0
ORIGBUFSIZE    .DW
;******************************************************************************
;* FUNCTIONS
;******************************************************************************

;*********************
;* Byte I/O Routines *
;*********************
;Receive byte over the parallel port
RECPARABYTE    LDX SLOT
               LDA BYTETIMEOUT ;initialise timeout counter
               STA CNT
               LDA BYTETIMEOUT+1
               STA CNT+1
WAITLOW        LSR DEV+4,X     ;wait for bit 4 (clock) to go low
               BCC LOOKLOWENT
               INC CNT
               BNE WAITLOW
               INC CNT+1
               BNE WAITLOW
               SEC             ;timed out
               RTS
LOOKLOWENT     LDA BYTETIMEOUT ;reset timeout counter
               STA CNT
               LDA BYTETIMEOUT+1
               STA CNT+1
LOOKLOW        LSR DEV+4,X     ;when bit 4 (clock) goes high, a nibble has
               BCS READLOW     ; been received
               INC CNT
               BNE LOOKLOW
               INC CNT+1
               BNE LOOKLOW
               SEC             ;timed out
               RTS
READLOW        LDA DEV+3,X     ;read received nibble
               AND #$E8        ;mask off unused bits
               LSR A           ;shift them down
               LSR A
               LSR A
               LSR A
               BCC NOLOWBIT3   ;bit 3 will have shifted into the Carry
               ORA #$01        ;bring it back
NOLOWBIT3      STA DATA        ;and store it
               LDA BYTETIMEOUT ;reset timeout counter
               STA CNT
               LDA BYTETIMEOUT+1
               STA CNT+1
LOOKHIGH       LSR DEV+4,X     ;when bit 4 (clock) lowers again, another nibble
               BCC READHIGH    ; has been received
               INC CNT
               BNE LOOKHIGH
               INC CNT+1
               BNE LOOKHIGH
               SEC             ;timed out
               RTS
READHIGH       LDA DEV+3,X     ;read received nibble
               AND #$E8        ;mask off unused bits
               TAX             ;X register no longer needed - use as temporary
               AND #$08        ;bit 3 needs to move to bit 4
               BEQ MERGENIBS   ;if zero merge immediately
               TXA             ;otherwise,
               ORA #$10        ;set bit 4
               AND #$F0        ;and clear bit 3
               TAX
MERGENIBS      TXA
               ORA DATA        ;merge with previous nibble
               STA DATA        ;and keep it
               CLC
               RTS

;Receive byte over serial port
RECSERIALBYTE  LDX SLOT
               LDA BYTETIMEOUT ;initialise timeout counter
               STA CNT
               LDA BYTETIMEOUT+1
               STA CNT+1
READRDR        LDA DEV+9,X     ;read ACIA status register
               AND #$08        ;test Receive Data Register full flag
               BNE READBYTE    ;not 0 - data received
               DEC CNT
               BNE READRDR
               DEC CNT+1
               BNE READRDR
               SEC             ;timed out
               RTS
READBYTE       LDA DEV+8,X
               CLC
               RTS

RECEIVEBYTE    LDA COMMSTYPE
               CMP #1          ;test for parallel type
               BNE RTESTSERIAL
               JMP RECPARABYTE
RTESTSERIAL    CMP #2          ;test for serial type
               BNE RECTYPEERROR
               JMP RECSERIALBYTE
RECTYPEERROR   SEC
               RTS

;Send a byte over the parallel port
SENDPARABYTE   LDX SLOT
               LDA #0
               STA DEV+0,X     ;make sure clock is lowered
               LDY PRESNDDLY
PSNDWAIT1      DEY
               BNE PSNDWAIT1
               LDA DATA        ;send low nibble
               AND #$0F
               ORA #$10        ;raise clock
               STA DEV+0,X     ;send it
               LDY SNDDLY
PSNDWAIT2      DEY
               BNE PSNDWAIT2
               LDA DATA        ;send high nibble
               LSR A           ;shift high nibble down to it's proper position
               LSR A           ; (this lowers the clock as well)
               LSR A
               LSR A
               STA DEV+0,X     ;send it
               LDY SNDDLY
PSNDWAIT3      DEY
               BNE PSNDWAIT3
               RTS

;Send a byte over the serial port
SENDSERBYTE    LDX SLOT
               LDA #0          ;initialise timeout counter
               STA CNT
               STA CNT+1
READTDR        LDA DEV+9,X     ;read ACIA status register
               AND #$10        ;test Transmit Data Register empty flag
               BNE WRITEBYTE   ;not 0 - register empty
               INC CNT
               BNE READTDR
               INC CNT+1
               BNE READTDR
               SEC             ;timed out
               RTS
WRITEBYTE      LDA DATA
               STA DEV+8,X
               CLC
               RTS

SENDBYTE       LDA COMMSTYPE
               CMP #1          ;test for parallel type
               BNE STESTSERIAL
               JMP SENDPARABYTE
STESTSERIAL    CMP #2          ;test for serial type
               BNE SENDTYPEERROR
               JMP SENDSERBYTE
SENDTYPEERROR  SEC
               RTS

;***********************
;* Packet I/O Routines *
;***********************

;*** CRC Stuff ***

;CCITT CRC's start as $FFFF.
INITCRC        LDA #$FF
               STA CRC
               STA CRC+1
               RTS

;Calculate CRC. Accumulator and X register trashed.
;Set DATA with data byte and CRC with cumulative CRC.
CALCCRC        LDA DATA
               CLV             ;Overflow clear - first time through
DOCRCNIBBLE    EOR CRC         ;index=(CRC^DATA)&0x000f
               AND #$0F
               ASL A           ;word-align index
               TAX             ;save it in the X register
               LSR CRC+1       ;CRC>>=4
               ROR CRC
               LSR CRC+1
               ROR CRC
               LSR CRC+1
               ROR CRC
               LSR CRC+1
               ROR CRC
               LDA CRC+1       ;CRC&=0x0fff
               AND #$0F
               EOR CRCTABLE+1,X ;CRC^=CRCTable[index]
               STA CRC+1
               LDA CRC
               EOR CRCTABLE,X
               STA CRC
               BVS CRCDONE
               BIT CRCDONE     ;BIT #$60 : sets Overflow
               LDA DATA        ;DATA>>=4
               LSR A
               LSR A
               LSR A
               LSR A
               BVS DOCRCNIBBLE
CRCDONE        RTS

;CCITT CRC's must be bit inverted (1's comp) before sending.
PRESENDCRC     LDA CRC
               EOR #$FF
               STA CRC
               LDA CRC+1
               EOR #$FF
               STA CRC+1
               RTS

;CCITT CRCs must end up as $F0B8
;Return clear Carry if CRC is valid, otherwise Carry is set.
VALIDCRCDATA   LDA CRC
               CMP #$B8
               BNE INVALIDCRC
               LDA CRC+1
               CMP #$F0
               BNE INVALIDCRC
               CLC
               RTS
INVALIDCRC     SEC
               RTS

;******************************************************************************
;*
;* RECEIVE CONNECTIONLESS PACKET and related functions
;*
;* The LENGTH field in PKT must be set before calling. It specifies the size
;* of buffer available for data storage. Received data that won't fit in the
;* buffer is discarded, but the length of the received packet is stored back in
;* LENGTH field. If data has been discarded, on return the LENGTH field will be
;* bigger than what you set it to.
;* Discarded data is not considered an error.
;*
;* Returns: Carry clear on no error, Carry set on errors.
;*          On error, the Overflow flag will be set on timeouts, clear for bad
;*          CRCs.
;*
;******************************************************************************

;Receive a connectionless packet
RECCLPKT       LDA PKT+2       ;MAXBUFLEN=sequence
               STA MAXBUFLEN
               LDA PKT+3
               STA MAXBUFLEN+1
               JSR INITCRC     ;initialise CRC
;Byte 0 - SYN character
GETSYN1        JSR RECPKTBYTE
               LDA DATA
GETSYN2        CMP #$16
               BNE GETSYN1
;Byte 1 - SOH character
               JSR RECPKTBYTE
               LDA DATA
               CMP #$01
               BNE GETSYN2
;Byte 2 - sequence, type and bits 11-8 of length
               JSR RECPKTBYTE
               JSR CALCCRC
               LDA DATA
               ASL A           ; sequence now in Carry
               ROL A           ; sequence in bit 0, type in Carry
               AND #1
               STA PKT+0       ;save sequence
               ROL A           ; type in bit 0
               AND #1
               STA PKT+1       ;save type
               LDA DATA
               AND #$0F
               STA PKT+3       ;set length (high byte)
;Byte 3 - bits 7-0 of length
               JSR RECPKTBYTE
               LDA DATA
               STA PKT+2       ;set length (low byte)
               JSR CALCCRC
;If the length is $000 correct it to $1000
               LDA PKT+2
               BNE RDATABYTES
               LDA PKT+3
               BNE RDATABYTES
               LDA #$10
               STA PKT+3
;Data bytes
RDATABYTES     LDA #0
               STA I
               STA I+1
NEXTRDATABYTE  LDA I+1         ;if I<length then RDATABYTE else GETCRC
               CMP PKT+3
               BCC RDATABYTE
               LDA I
               CMP PKT+2
               BCS GETCRC
RDATABYTE      JSR RECPKTBYTE
               JSR CALCCRC
               LDA I+1         ;if I<MAXBUFLEN then STOREDATA else RECINCI
               CMP MAXBUFLEN+1
               BCC STOREDATA
               LDA I
               CMP MAXBUFLEN
               BCS RECINCI
;add data buffer pointer and I
STOREDATA      CLC
               LDA PKT+4       ;data buffer pointer (low byte)
               ADC I
               STA RECADDR+1
               LDA PKT+5       ;data buffer pointer (high byte)
               ADC I+1
               STA RECADDR+2
;now set the buffer
               LDA DATA
RECADDR        STA $FFFF       ;self modifying code
;increment I
RECINCI        INC I
               BNE NEXTRDATABYTE
               INC I+1
               JMP NEXTRDATABYTE
;CRC bytes
GETCRC         JSR RECPKTBYTE
               JSR CALCCRC
               JSR RECPKTBYTE
               JSR CALCCRC
               JSR VALIDCRCDATA
               BCS BADCRC
;Trailer
               JSR RECPKTBYTE
               LDA DATA
               CMP #$04
               BNE BADCRC
;have received a packet with a valid CRC
               CLC             ;packet OK - clear Carry
               RTS
;have received a packet with a bad CRC
BADCRC         LDA MAXBUFLEN   ;restore original packet length (buffer size)
               STA PKT+2       ;on bad packet
               LDA MAXBUFLEN+1
               STA PKT+3
               SEC             ;bad CRC - set Carry and clear Overflow
               CLV
               RTS
;have not received a packet at all
RBTOERROR      LDA MAXBUFLEN   ;restore original packet length (buffer size)
               STA PKT+2       ;on bad packet
               LDA MAXBUFLEN+1
               STA PKT+3
               SEC             ;timeout - set Carry and set Overflow
               BIT RBTOERRRTS  ;BIT #$60 - set Overflow flag
RBTOERRRTS     RTS

;*** Receive Packet Byte ***
RECPKTBYTE     LDA PKTTIMEOUT
               STA PKTCNT
RBAGAIN        JSR RECEIVEBYTE
               BCC RECPKTBYTEOK ;Carry clear - got a byte
               DEC PKTCNT
               BNE RBAGAIN     ;not yet timed out - try again
               PLA             ;Timed out - pop return address off the stack
               PLA             ;and jump to the timeout error handler.
               JMP RBTOERROR
RECPKTBYTEOK   RTS

;******************************************************************************
;*
;* SEND CONNECTIONLESS PACKET
;*
;******************************************************************************

;Send a connectionless packet
SENDCLPKT      JSR INITCRC     ;initialise CRC
;Byte 0 - SYN character
               LDA #$16
               STA DATA
               JSR SENDBYTE
;Byte 1 - SOH character
               LDA #$01
               STA DATA
               JSR SENDBYTE
;Byte 2 - sequence, type and bits 11-8 of length
               LDA PKT+1       ;type
               LSR A
               LDA PKT+0       ;sequence
               ROR A
               ROR A
               AND #11000000b
               STA DATA
               LDA PKT+3       ;length (high byte)
               AND #$0F
               ORA DATA
               STA DATA
               JSR CALCCRC
               JSR SENDBYTE
;Byte 3 - bits 7-0 of length
               LDA PKT+2       ;length (low byte)
               STA DATA
               JSR CALCCRC
               JSR SENDBYTE
;Data bytes
               LDA #0
               STA I
               STA I+1
NEXTSDATABYTE  LDA I+1         ;if I<length then GETDATA else SENDCRC
               CMP PKT+3
               BCC GETDATA
               LDA I
               CMP PKT+2
               BCS SENDCRC
;add data buffer pointer and I
GETDATA        CLC
               LDA PKT+4       ;data buffer pointer (low byte)
               ADC I
               STA SNDADDR+1
               LDA PKT+5       ;data buffer pointer (high byte)
               ADC I+1
               STA SNDADDR+2
;now get from the buffer
SNDADDR        LDA $FFFF       ;self modifying code
               STA DATA
               JSR CALCCRC
               JSR SENDBYTE
;increment I
               INC I
               BNE NEXTSDATABYTE
               INC I+1
               JMP NEXTSDATABYTE
;CRC bytes
SENDCRC        JSR PRESENDCRC
               LDA CRC
               STA DATA
               JSR SENDBYTE
               LDA CRC+1
               STA DATA
               JSR SENDBYTE
;Trailer
               LDA #$04
               STA DATA
               JSR SENDBYTE
;Done
               RTS

;******************************************************************************
;*                                                                            *
;* PACKET PROTOCOL                                                            *
;*                                                                            *
;******************************************************************************
;Misc routines
;Copy (AAXX) to PKT
COPYTOPKT      STA FROMADDR+2
               STX FROMADDR+1
               LDX #5
COPYTOLOOP
FROMADDR       LDA $FFFF,X     ;self-modifying code
               STA PKT,X
               DEX
               BPL COPYTOLOOP
               RTS

;Copy PKT to (AAXX)
COPYFROMPKT    STA TOADDR+2
               STX TOADDR+1
               LDX #5
COPYFROMLOOP   LDA PKT,X
TOADDR         STA $FFFF,X     ;self-modifying code
               DEX
               BPL COPYFROMLOOP
               RTS

;Send specific ACK or NAK - sequence # in X
SENDACK        LDY #$06
               BNE SENDACKNAK
SENDNAK        LDY #$15
;Send an ACK or NAK - sequence # in X, ACK/NAK in Y
SENDACKNAK     STX PKT+0       ;sequence
               STY ACKCHAR
               LDA #0
               STA PKT+1       ;type is supervisory
               STA PKT+3
               LDA #1
               STA PKT+2       ;length is 1
               LDA # ACKCHAR & $FF
               STA PKT+4
               LDA # ACKCHAR >> 8
               STA PKT+5
               JSR SENDCLPKT
               RTS

ACKCHAR        .DB

;Receive sequenced packet
;Returns Carry clear on no errors, Carry set on timeout
RECPACKET      LDA PKTINFO+2   ;keep original packet length (buffer size)
               STA ORIGBUFSIZE
               LDA PKTINFO+3
               STA ORIGBUFSIZE+1
TRYRECPACKET   LDA ORIGBUFSIZE
               STA PKTINFO+2
               LDA ORIGBUFSIZE+1
               STA PKTINFO+3
               LDA # PKTINFO >> 8 ;copy PKTINFO to PKT
               LDX # PKTINFO & $FF
               JSR COPYTOPKT
               JSR RECCLPKT    ;receive a packet
               LDA # PKTINFO >> 8 ;copy PKT to PKTINFO
               LDX # PKTINFO & $FF
               JSR COPYFROMPKT
               BCC GOODCRC
               BVS TIMEOUTERR
;got a packet with a bad CRC - send a NAK
               LDX CURRECSEQ
               JSR SENDNAK
               CLC
               BCC TRYRECPACKET
;no packet
TIMEOUTERR     SEC
               RTS
;got a packet with a good CRC
GOODCRC
;check for sequence synchronisation packet
CHKSEQSYNC     LDA PKTINFO+1
               BNE CHKSEQ      ;type not 0 (supervisory)
               LDA PKTINFO+4
               STA RECSEQSYNC+1
               LDA PKTINFO+5
               STA RECSEQSYNC+2
RECSEQSYNC     LDA $FFFF       ;self-modifying
               CMP #$16
               BNE CHKSEQ      ;not SYN character
;IS a sequence sync packet - set new sequence
               LDA PKTINFO+0
               TAX             ;ACK on received sequence
               EOR #1          ;and move to next sequence
               STA CURRECSEQ
               JSR SENDACK
               JMP TRYRECPACKET ;don't let sequence sync packets to user app

;if wrong sequence must be a duplicate - send ACK
CHKSEQ         LDA PKTINFO+0
               CMP CURRECSEQ
               BEQ ACKPACKET
               TAX
               JSR SENDACK
               JMP TRYRECPACKET

ACKPACKET      LDX CURRECSEQ
               JSR SENDACK
               LDA CURRECSEQ
               EOR #1
               STA CURRECSEQ
               CLC
               RTS

;Send sequenced packet
;Returns Carry clear on no errors, Carry set on timeout
SENDPACKET
;check for sequence synchronisation packet
               LDA PKTINFO+1
               BNE INITSNDPKT  ;not type 0 (supervisory)
               LDA PKTINFO+4
               STA SNDSEQSYNC+1
               LDA PKTINFO+5
               STA SNDSEQSYNC+2
SNDSEQSYNC     LDA $FFFF       ;self-modifying
               CMP #$16
               BNE INITSNDPKT
;IS a sequence sync packet - set new sequence
               LDA PKTINFO+0
               STA CURSNDSEQ

INITSNDPKT     LDA MAXPKTSENDS ;initialise maximum retries
               STA PKTTRIES
               LDA CURSNDSEQ   ;set packet sequence to current send sequence
               STA PKTINFO+0
TRYPKTSND      SEC             ;default to unsuccessful send
               DEC PKTTRIES
               BMI STOPSEND
               LDA # PKTINFO >> 8 ;copy PKTINFO to PKT
               LDX # PKTINFO & $FF
               JSR COPYTOPKT
               JSR SENDCLPKT   ;send the packet
               LDA # PKTINFO >> 8 ;copy PKT to PKTINFO
               LDX # PKTINFO & $FF
               JSR COPYFROMPKT
               ;wait for acknowledgement
               LDA #1          ;length of 1
               STA ACKPKT+2
               LDA #0
               STA ACKPKT+3
               LDA # ACKDATA & $FF ;get pointer to ACK data
               STA ACKPKT+4
               LDA # ACKDATA >> 8
               STA ACKPKT+5
               LDA # ACKPKT >> 8 ;copy ACKPKT to PKT
               LDX # ACKPKT & $FF
               JSR COPYTOPKT
               JSR RECCLPKT
               LDA # ACKPKT >> 8 ;copy PKT to ACKPKT
               LDX # ACKPKT & $FF
               JSR COPYFROMPKT
               BCS TRYPKTSND   ;error receiving ACK - try again
               LDA ACKPKT+0    ;check sequence
               CMP CURSNDSEQ
               BNE TRYPKTSND   ;different sequences - try again
               LDA ACKPKT+1    ;check type
               BNE TRYPKTSND   ;not 0 (supervisory) - try again
               LDA ACKDATA
               CMP #$06
               BNE TRYPKTSND   ;not an ACK - try again
               ;if we got here, then everything was OK
               CLC
STOPSEND       BCS DONESEND
               LDA CURSNDSEQ   ;toggle send sequence
               EOR #1
               STA CURSNDSEQ
DONESEND       RTS

;Synchronise sequence #'s
SYNCSEQ        LDA #0
               STA PKTINFO+0   ;sequence 0
               STA PKTINFO+1   ;type 0 (supervisory)
               STA PKTINFO+3
               LDA #1
               STA PKTINFO+2   ;length 1
               LDA # SYNCHAR & $FF
               STA PKTINFO+4
               LDA # SYNCHAR >> 8
               STA PKTINFO+5
               JSR SENDPACKET
               RTS

SYNCHAR        .DB $16

;******************************************************
;* Initialisation                                     *
;*                                                    *
;* From here on may be discarded after initialisation *
;* is complete.
;*                                                    *
;******************************************************
A2ICOMMSINIT
;Display title
               JSR MSGOUT
               .TEXT "\r*** APPLE <--> IBM COMMUNICATIONS ***\r"
               .TEXT "COPYRIGHT (C) 1994 BY ANDREW GREGORY\r"
               .TEXT "14 WHITTINGTON AVENUE\r"
               .TEXT "CARINE WA 6020 AUSTRALIA\r\000"

;Get communications type
               JSR MSGOUT
               .TEXT "\rSELECT COMMUNICATIONS TYPE\r"
               .TEXT " (P)ARALLEL\r"
               .TEXT " (S)ERIAL\r"
               .TEXT "(P,S,ESC TO QUIT):\000"
GETTYPE        JSR CIN
               AND #11011111b  ;force to uppercase
               CMP #$D0        ;'P'?
               BEQ OKTYPE
               CMP #$D3        ;'S'?
               BEQ OKTYPE
               CMP #$9B        ;ESC?
               BNE GETTYPE
               RTS             ;quit
OKTYPE         JSR COUT        ;echo entered character
               AND #3
               EOR #1          ;if 'P' then A=1, if 'S' then A=2
               STA COMMSTYPE
               JSR CROUT       ;print carriage return

;Get communications port
               JSR MSGOUT
               .TEXT "\rSELECT COMMUNICATIONS SLOT\r"
               .TEXT "(1-7,ESC TO QUIT):\000"
GETSLOT        JSR CIN
               CMP #$9B        ;ESC?
               BNE TESTNUMS
               RTS             ;quit
TESTNUMS       CMP #$B1        ;<'1'?
               BCC GETSLOT
               CMP #$B8        ;>='8'?
               BCS GETSLOT
               JSR COUT        ;echo entered character
               ASL A           ;make key into slot as $s0
               ASL A
               ASL A
               ASL A
               STA SLOT
               JSR CROUT       ;print carriage return

;If serial port selected, initialise it
               LDA COMMSTYPE
               CMP #2
               BNE NOTSERIAL
               LDX SLOT
               LDA #00011111b  ;1 stop & 8 data bits, 19200 baud
               STA DEV+$B,X
               LDA #00001011b  ;no parity
               STA DEV+$A,X
NOTSERIAL

;Now reset Applesoft to load it's programs above A2ICOMMS. Figure out what page
;the A2IINITCOMMS is on and set the next page and above for Applesoft.
;Theoretically, Applesoft doesn't need to start of page boundaries (maybe the
;ProDOS BASIC System does), but it's neater on page boundaries.
;Note that this will wipe out an Applesoft program if it's in the way.
;Applesoft programs which load this driver should relocate themselves above it
;beforehand.
               LDA # ENDINITCOMMS >> 8 ;get last page used by A2ICOMMS
               CMP TXTTAB+1    ;compare with Applesoft program start page
               BCS CLEARPROG   ;if last page used >= to Applesoft program start
               JMP $03D0       ;then clear the program, otherwise we're done
;Clear the program, set start of Applesoft program space to beginning of the
;next page after the start of the A2ICOMMS initialisation routine.
CLEARPROG      LDA # A2ICOMMSINIT & $00FF ;if on exact page boundary
               BEQ ADDPAGE                ;don't need to go to next page
               LDA #1                     ;otherwise, go up to next page
ADDPAGE        CLC
               ADC # A2ICOMMSINIT >> 8
;Page now in the Accumulator, set the Applesoft pointers and restart Applesoft.
               STA TXTTAB+1
               STA VARTAB+1
               STA ARYTAB+1
               STA STREND+1
               STA PGREND+1
               LDA #0
               STA TXTTAB      ;currently zero, will fix it up below
               LDA #4
               STA VARTAB
               STA ARYTAB
               STA STREND
               STA PGREND
               LDA #0
               LDY #0          ;zero out first byte (required by Applesoft)
               STA (TXTTAB),Y  
               INY             ;zero out link fields (NEW)
               STA (TXTTAB),Y
               INY
               STA (TXTTAB),Y
               LDA #1
               STA TXTTAB
               JMP $03D0       ;Applesoft warm reset

;Message Printer (originally by Andy Hertzfeld)
;Text to display must immediately follow the JSR calling MSGOUT and must be
;terminated with a zero. Control is returned to the address following the
;terminating zero.
MSGOUT         LDA $FE         ;save temporary locations
               PHA
               LDA $FF
               PHA
               TSX
               LDA STACK+3,X   ;get address of text
               STA $FE
               LDA STACK+4,X
               STA $FF
               LDY #0
MSGOUT1        INC $FE
               BNE MSGOUT2
               INC $FF
MSGOUT2        LDA ($FE),Y
               BEQ MSGOUT3
               ORA #$80
               JSR COUT
               BNE MSGOUT1
MSGOUT3        LDA $FF         ;update return address on the stack
               STA STACK+4,X   ; to point after end of string data
               LDA $FE
               STA STACK+3,X
               PLA             ;restore temporary locations
               STA $FF
               PLA
               STA $FE
               RTS
ENDINITCOMMS

               .END
