/*

 $0000-$00FF = Zero Page
 $0100-$01FF = Stack
 $0200-$03FF = Standard RAM
 $0400-$07FF = LoRes Video Page 1
 $0800-$0CFF = LoRes Video Page 2
 $0D00-$1FFF = Standard RAM
 $2000-$3FFF = HiRes Video Page 1 / Double HiRes Page 1
 $4000-$5FFF = HiRes Video Page 2 / Double HiRes Page 2
 $6000-$BFFF = Standard RAM
 $C000-$C0FF = Hardware I/O
 $C100-$CFFF = Expansion ROM
 $D000-$DFFF = Bank switched RAM (ROM/RAM1/RAM2)
 $E000-$FFFF = ROM/RAM

 To handle zero page and stack direct memory accesses (not via the memory
 management functions), there must be a 64K chunk of memory from
 MemSeg:0000 - MemSeg:FFFF for direct reads.  All writing apart from zero
 page and CPU microcode stack accesses will go through the memory management
 functions.

 Note that due to the way the CPU is emulated, code executed in the C0-page
 will not trip any soft-switches.  Since code cannot actually be put into
 the C0-page, this should not be a problem. :-)

*/

#include <process.h>

#include <alloc.h>

#include "system.h"
#include "ram.h"
#include "cpu.h"
#include "keyboard.h"
#include "speaker.h"
#include "gameport.h"
#include "video.h"
#include "massstor.h"
#include "disk.h"
#include "pic.h"
#include "clock.h"
#include "diags.h"

/* An entry for each page in the 64K */
BYTE ( * ReadPage[ 0x0100 ] )( WORD Address );
void ( *WritePage[ 0x0100 ] )( WORD Address, BYTE Data );

/* An entry for each byte in the hardware 0xc0 page */
BYTE ( * ReadHWPage[ 0x0100 ] )( WORD Address );
void ( *WriteHWPage[ 0x0100 ] )( WORD Address, BYTE Data );

/* ACTUAL ALLOCATIONS AND SEGMENTS */
/* These are constant for the duration of the emulation */
/* Bank memory is 0x0000-0x01fff and 0xd000-0xffff */
void far *MainMem;
void far *MainBank;
void far *AuxMem;
void far *AuxBank;
void far *ROMMem;
void far *IOExpROMMem[ 7 ];
/* note all segments are set up to be 0-based in terms of Apple memory
   e.g. ROMSeg is a segment 0xd00 below the actual ROM memory, so when the
   Apple CPU accesses it between 0xd000-0xffff the correct memory will be
   accessed */
unsigned int MainSeg;
unsigned int MainBankSeg;
unsigned int AuxSeg;
unsigned int AuxBankSeg;
unsigned int ROMSeg;
unsigned int IOExpROM[ 7 ]; /* expansion ROM segments for each IO slot ROM */

/* SEGMENTS IN USE */
/* these are the current segment to use to access the memory
   they are set from the five allocated segments above */
unsigned int CurZPSSeg;        /* 0x0000-0x01ff =MainSeg/AuxSeg */
unsigned int CurMainSeg;       /* 0x0200-0xbfff =MainSeg/AuxSeg */
unsigned int CurSlotROMSeg;    /* 0xc100-0xc2ff,0xc400-0xc7ff =MainSeg/ROMSeg */
unsigned int CurAuxSlotROMSeg; /* 0xc300-0xc3ff =MainSeg/ROMSeg */
unsigned int ExpSlotSeg;       /* 0xc800-0xcfff =MainSeg/ROMSeg */
unsigned int CurBankSeg;       /* 0xd000-0xffff =MainSeg/AuxSeg */
unsigned int CurBank2Seg;      /* 0xd000-0xdfff =MainBankSeg/AuxBankSeg */
/* these are the read/write segments. They are set from the above four
   segments */
unsigned int ZPSRSeg;   /* =CurZPSSeg */
unsigned int ZPSWSeg;   /* =CurZPSSeg */
unsigned int MainRSeg;  /* =CurMainSeg */
unsigned int MainWSeg;  /* =CurMainSeg */
unsigned int BankRSeg;  /* =CurBankSeg/ROMSeg */
unsigned int BankWSeg;  /* =CurBankSeg/ROMSeg */
unsigned int Bank2RSeg; /* =CurBankSeg/CurBank2Seg/ROMSeg */
unsigned int Bank2WSeg; /* =CurBankSeg/CurBank2Seg/ROMSeg */

/**************
 *            *
 * MEMORY I/O *
 *            *
 **************/

BYTE ReadZPStk( WORD Address )
{
    return peekb( ZPSRSeg, Address );
}

void WriteZPStk( WORD Address, BYTE Data )
{
    pokeb( ZPSWSeg, Address, Data );
}

BYTE ReadRAM( WORD Address )
{
    return peekb( MainRSeg, Address );
}

void WriteRAM( WORD Address, BYTE Data )
{
    pokeb( MainWSeg, Address, Data );
}

BYTE ReadIO( WORD Address )
{
    Address &= 0x00ff;
    return ReadHWPage[ Address ]( Address );
}

void WriteIO( WORD Address, BYTE Data )
{
    Address &= 0x00ff;
    WriteHWPage[ Address ]( Address, Data );
}

/* 0xc100-0xc2ff,0xc400-0xc7ff */
BYTE ReadSlotROM( WORD Address )
{
    return peekb( CurSlotROMSeg, Address );
}

/* 0xc300-0xc3ff */
BYTE ReadAuxSlotROM( WORD Address )
{
    return peekb( CurAuxSlotROMSeg, Address );
}

/* 0xc800-0xcfff */
BYTE ReadExpSlotROM( WORD Address )
{
  return peekb( ExpSlotSeg, Address );
}

#pragma argsused
void WriteROM( WORD Address, BYTE Data )
{
}

BYTE ReadBank( WORD Address )
{
  return peekb( BankRSeg, Address );
}

void WriteBank( WORD Address, BYTE Data )
{
    if ( BankWSeg != ROMSeg )
    {
        pokeb( BankWSeg, Address, Data );
    }
}

BYTE ReadBank2( WORD Address )
{
    return peekb( Bank2RSeg, Address );
}

void WriteBank2( WORD Address, BYTE Data )
{
    if ( Bank2WSeg != ROMSeg )
    {
        pokeb( Bank2WSeg, Address, Data );
    }
}

/***************************
 *                         *
 * DO-NOTHING HARDWARE I/O *
 *                         *
 ***************************/

#pragma argsused
BYTE ReadIOBlank( WORD Address )
{
    return 0xff;
}

#pragma argsused
void WriteIOBlank( WORD Address, BYTE Data )
{
}

/*****************
 *               *
 * Apple //e IOU *
 *               *
 *****************/

BYTE IOUDisabled;

BYTE ReadIOU( WORD Address )
{
    BYTE v;

    v = 0xff;

    switch ( Address )
    {
    case 0x7e:
        /* RdIOUDis */
        v = IOUDisabled;
        break;
    }
    return v;
}

void WriteIOU( WORD Address, BYTE Data )
{
    switch ( Address )
    {
    case 0x58:
    case 0x59:
    case 0x5a:
    case 0x5b:
    case 0x5c:
    case 0x5d:
        ReadIOU( Address );
        break;
    case 0x5e:
    case 0x5f:
        /* Double High-Res Control: Pass on to video control */
        if ( IOUDisabled == 0xff )
        {
            WriteVideoControl( Address, Data );
        }
        break;
    case 0x7e:
        /* IOUDis On */
        IOUDisabled = 0xff;
        break;
    case 0x7f:
        /* IOUDis Off */
        IOUDisabled = 0x7f;
        break;
    }
}

/****************************
 *                          *
 * Alternate Apple //e ROMS *
 *                          *
 ****************************/

BYTE SlotCxROM;
BYTE SlotC3ROM;

BYTE ReadAltROMIO( WORD Address )
{
    BYTE v;

    switch ( Address )
    {
    case 0x15:
        /* Read SLOTCXROM switch */
        v = SlotCxROM;
        break;
    case 0x17:
        /* Read SLOTC3ROM switch */
        v = SlotC3ROM;
        break;
    }
    return v;
}

#pragma argsused
void WriteAltROMIO( WORD Address, BYTE Data )
{
    switch ( Address )
    {
    case 0x06:
        /* SLOTCXROM Off */
        SlotCxROM = 0x0d;
        CurSlotROMSeg = MainSeg;
        break;
    case 0x07:
        /* SLOTCXROM On */
        SlotCxROM = 0x8d;
        /* only allow slot ROM changes on a //e */
        if ( ExpSlotSeg == ROMSeg )
        {
            CurSlotROMSeg = ROMSeg;
        }
        break;
    case 0x0a:
        /* SLOTC3ROM Off */
        SlotC3ROM = 0x0d;
        break;
    case 0x0b:
        /* SLOTC3ROM On */
        SlotC3ROM = 0x8d;
        break;
    }
    if ( SlotCxROM == 0x0d && SlotC3ROM == 0x8d )
    {
        CurAuxSlotROMSeg = MainSeg;
    }
    else
    {
        CurAuxSlotROMSeg = ROMSeg;
    }
}

/*****************************
 *                           *
 * MEMORY BANK SOFT SWITCHES *
 * Slot 0 16K Language Card  *
 *                           *
 *****************************/

/*              ROM RAM1 RAM2
 *  c080 c084 - W         R
 *  c081 c085 - R         W
 *  c082 c086 - RW
 *  c083 c087 -           RW
 *  c088 c08c - W    R
 *  c089 c08d - R    W
 *  c08a c08e - RW
 *  c08b c08f -      RW
 */

WORD LastSlot0IO = 0;
BYTE RdBnk2;

BYTE ReadSlot0IO( WORD Address )
{
    BYTE v;

    v = 0xff;

    /* keep track of last bank accessed */
    if ( Address >= 0x80 && Address <= 0x83 ) RdBnk2 = 0x8d;
    if ( Address >= 0x88 && Address <= 0x8b ) RdBnk2 = 0x0d;
    /* default to write-protected RAM */
    Bank2WSeg = ROMSeg;
    BankWSeg  = ROMSeg;
    /* handle each soft-switch */
    switch ( Address )
    {
    case 0x11:
        v = RdBnk2;
        break;
    case 0x12:
        v = ( BankRSeg == ROMSeg ) ? 0x0d : 0x8d;
        break;
    case 0x80:
    case 0x83:
    case 0x84:
    case 0x87:
        Bank2RSeg = CurBank2Seg;
        BankRSeg  = CurBankSeg;
        break;
    case 0x88:
    case 0x8b:
    case 0x8c:
    case 0x8f:
        Bank2RSeg = CurBankSeg;
        BankRSeg  = CurBankSeg;
        break;
    case 0x81:
    case 0x82:
    case 0x85:
    case 0x86:
    case 0x89:
    case 0x8a:
    case 0x8d:
    case 0x8e:
        Bank2RSeg = ROMSeg;
        BankRSeg  = ROMSeg;
        break;
    }
    /* check for double read */
    if ( LastSlot0IO == Address )
    {
        /* yes - write-enable RAM */
        switch ( Address )
        {
        case 0x81:
        case 0x83:
            Bank2WSeg = CurBank2Seg;
            BankWSeg  = CurBankSeg;
            break;
        case 0x89:
        case 0x8b:
            Bank2WSeg = CurBankSeg;
            BankWSeg  = CurBankSeg;
            break;
        }
    }
    LastSlot0IO = Address; /* keep last access */
    return v;
}

#pragma argsused
void WriteSlot0IO( WORD Address, BYTE Data )
{
    /* 65C02 accesses bus twice during a write, so pretend these locations
       were just accessed */
    switch ( Address )
    {
    case 0x81:
    case 0x83:
    case 0x89:
    case 0x8b:
//        LastSlot0IO=Address;
        break;
    }
    /* treat the same as reading */
    ReadSlot0IO( Address );
}

/****************
 *              *
 * MEMORY SETUP *
 *              *
 ****************/

unsigned int GetSeg( void far *ptr )
{
    unsigned int seg;

    seg = FP_SEG( ptr );
    /* Round far* up to nearest segment boundary */
    if ( ( FP_OFF( ptr ) ) % 16 != 0 )
    {
        seg++;
    }
    return seg;
}

void InitRAM( void )
{
    int Page, Loc, fh;
    long filelen;

    MainMem = MainBank = AuxMem = AuxBank = ROMMem = NULL;
    for ( Page = 0; Page < 8; Page++ )
    {
        IOExpROMMem[ Page ] = NULL;
    }

    /* Allocate 64K+1page since the memory may not be on a segment boundary */
    MainMem = farmalloc( 0x10000UL + 16UL );
    if ( MainMem != NULL )
    {
        MainSeg = GetSeg( MainMem );
    }
    /* Allocate auxiliary 64K */
    AuxMem = farmalloc( 0x10000UL + 16UL );
    if ( AuxMem != NULL )
    {
        AuxSeg = GetSeg( AuxMem );
    }
    /* Load ROMs - correctly handle 16K //e ROM and 12K ][ / ][+ ROM */
    fh = Open( "apple.rom", O_RDONLY );
    filelen = Seek( fh, 0L, SEEK_END );
    Seek( fh, 0L, SEEK_SET );
    /* Allocate ROM storage */
    ROMMem = farmalloc( filelen + 16UL );
    if ( ROMMem != NULL )
    {
        ROMSeg = GetSeg( ROMMem );
        /* Load ROM from file */
        Read( fh, MK_FP( ROMSeg, 0 ), (unsigned int)filelen );
        /* Now correct ROM segment to be based correctly */
        switch ( filelen )
        {
          case 0x3000:
            /* ][ / ][+ 12K ROM */
            ROMSeg -= 0xd00;
            ExpSlotSeg = MainSeg;
            break;
          case 0x3f00:
            /* //e 16K ROM */
            ROMSeg -= 0xc10;
            ExpSlotSeg = ROMSeg;
            break;
          default:
            /* invalid ROM file length */
            farfree( ROMMem );
            ROMMem = NULL;
            break;
        }
    }
    Close( fh );
    /* Allocate extra bank-switched RAM (4K) */
    MainBank = farmalloc( 0x1000UL + 16UL );
    if ( MainBank != NULL )
    {
        MainBankSeg = GetSeg( MainBank );
        MainBankSeg -= 0xd00; /* correct offset */
    }
    /* Allocate auxiliary extra bank-switched RAM (4K) */
    AuxBank = farmalloc( 0x1000UL + 16UL );
    if ( AuxBank != NULL )
    {
        AuxBankSeg = GetSeg( AuxBank );
        AuxBankSeg -= 0xd00; /* correct offset */
    }

    if ( MainMem == NULL || AuxMem == NULL || MainBank == NULL ||
         AuxBank == NULL || ROMMem == NULL)
    {
        FreeRAM();
//        WriteErrorMessage( "Not enough far memory available." );
        return;
    }

    /* Setup main memory R/W handling functions */
    for ( Page = 0x00; Page < 0x002; Page++ )  ReadPage[ Page ] = ReadZPStk;
    for ( Page = 0x02; Page < 0x004; Page++ )  ReadPage[ Page ] = ReadRAM;
    for ( Page = 0x04; Page < 0x008; Page++ )  ReadPage[ Page ] = ReadVideoLo1;
    for ( Page = 0x08; Page < 0x00c; Page++ )  ReadPage[ Page ] = ReadVideoLo2;
    for ( Page = 0x0c; Page < 0x020; Page++ )  ReadPage[ Page ] = ReadRAM;
    for ( Page = 0x20; Page < 0x040; Page++ )  ReadPage[ Page ] = ReadVideoHi1;
    for ( Page = 0x40; Page < 0x060; Page++ )  ReadPage[ Page ] = ReadVideoHi2;
    for ( Page = 0x60; Page < 0x0c0; Page++ )  ReadPage[ Page ] = ReadRAM;
    for ( Page = 0xc0; Page < 0x0c1; Page++ )  ReadPage[ Page ] = ReadIO;
    for ( Page = 0xc1; Page < 0x0c3; Page++ )  ReadPage[ Page ] = ReadSlotROM;
    for ( Page = 0xc3; Page < 0x0c4; Page++ )  ReadPage[ Page ] = ReadAuxSlotROM;
    for ( Page = 0xc4; Page < 0x0c8; Page++ )  ReadPage[ Page ] = ReadSlotROM;
    for ( Page = 0xc8; Page < 0x0d0; Page++ )  ReadPage[ Page ] = ReadExpSlotROM;
    for ( Page = 0xd0; Page < 0x0e0; Page++ )  ReadPage[ Page ] = ReadBank2;
    for ( Page = 0xe0; Page < 0x100; Page++ )  ReadPage[ Page ] = ReadBank;

    for ( Page = 0x00; Page < 0x002; Page++ ) WritePage[ Page ] = WriteZPStk;
    for ( Page = 0x02; Page < 0x004; Page++ ) WritePage[ Page ] = WriteRAM;
    for ( Page = 0x04; Page < 0x008; Page++ ) WritePage[ Page ] = WriteVideoLo1;
    for ( Page = 0x08; Page < 0x00c; Page++ ) WritePage[ Page ] = WriteVideoLo2;
    for ( Page = 0x0c; Page < 0x020; Page++ ) WritePage[ Page ] = WriteRAM;
    for ( Page = 0x20; Page < 0x040; Page++ ) WritePage[ Page ] = WriteVideoHi1;
    for ( Page = 0x40; Page < 0x060; Page++ ) WritePage[ Page ] = WriteVideoHi2;
    for ( Page = 0x60; Page < 0x0c0; Page++ ) WritePage[ Page ] = WriteRAM;
    for ( Page = 0xc0; Page < 0x0c1; Page++ ) WritePage[ Page ] = WriteIO;
    for ( Page = 0xc1; Page < 0x0d0; Page++ ) WritePage[ Page ] = WriteROM;
    for ( Page = 0xd0; Page < 0x0e0; Page++ ) WritePage[ Page ] = WriteBank2;
    for ( Page = 0xe0; Page < 0x100; Page++ ) WritePage[ Page ] = WriteBank;

    /* Setup hardware page R/W handling functions */
    for ( Loc = 0x00; Loc < 0x010; Loc++ )  ReadHWPage[ Loc  ] = ReadKB;
                                            ReadHWPage[ 0x10 ] = ReadKBStrobe;
                                            ReadHWPage[ 0x11 ] = ReadSlot0IO;
                                            ReadHWPage[ 0x12 ] = ReadSlot0IO;
                                            ReadHWPage[ 0x13 ] = ReadAuxSlotIO;
                                            ReadHWPage[ 0x14 ] = ReadAuxSlotIO;
                                            ReadHWPage[ 0x15 ] = ReadAltROMIO;
                                            ReadHWPage[ 0x16 ] = ReadAuxSlotIO;
                                            ReadHWPage[ 0x17 ] = ReadAltROMIO;
                                            ReadHWPage[ 0x18 ] = ReadAuxSlotIO;
                                            ReadHWPage[ 0x19 ] = ReadIOBlank;
    for ( Loc = 0x1a; Loc < 0x020; Loc++ )  ReadHWPage[ Loc  ] = ReadVideoControl;
    for ( Loc = 0x20; Loc < 0x030; Loc++ )  ReadHWPage[ Loc  ] = ReadIOBlank;
    for ( Loc = 0x30; Loc < 0x040; Loc++ )  ReadHWPage[ Loc  ] = ReadSpeaker;
    for ( Loc = 0x40; Loc < 0x050; Loc++ )  ReadHWPage[ Loc  ] = ReadIOBlank;
    for ( Loc = 0x50; Loc < 0x058; Loc++ )  ReadHWPage[ Loc  ] = ReadVideoControl;
    for ( Loc = 0x58; Loc < 0x05e; Loc++ )  ReadHWPage[ Loc  ] = ReadIOBlank;
                                            ReadHWPage[ 0x5e ] = ReadVideoControl;
                                            ReadHWPage[ 0x5f ] = ReadVideoControl;
                                            ReadHWPage[ 0x60 ] = ReadIOBlank;
    for ( Loc = 0x61; Loc < 0x064; Loc++ )  ReadHWPage[ Loc  ] = ReadGameButtons;
    for ( Loc = 0x64; Loc < 0x066; Loc++ )  ReadHWPage[ Loc  ] = ReadGameTimer;
    for ( Loc = 0x66; Loc < 0x070; Loc++ )  ReadHWPage[ Loc  ] = ReadIOBlank;
    for ( Loc = 0x70; Loc < 0x071; Loc++ )  ReadHWPage[ Loc  ] = ResetGameTimer;
    for ( Loc = 0x71; Loc < 0x07e; Loc++ )  ReadHWPage[ Loc  ] = ReadIOBlank;
                                            ReadHWPage[ 0x7e ] = ReadIOU;
                                            ReadHWPage[ 0x7f ] = ReadVideoControl;
    for ( Loc = 0x80; Loc < 0x090; Loc++ )  ReadHWPage[ Loc  ] = ReadSlot0IO;
    for ( Loc = 0x90; Loc < 0x0a0; Loc++ )  ReadHWPage[ Loc  ] = ReadPICIO;
    for ( Loc = 0xa0; Loc < 0x0d0; Loc++ )  ReadHWPage[ Loc  ] = ReadIOBlank;
    for ( Loc = 0xd0; Loc < 0x0e0; Loc++ )  ReadHWPage[ Loc  ] = ReadMassStorIO;
    for ( Loc = 0xe0; Loc < 0x0f0; Loc++ )  ReadHWPage[ Loc  ] = ReadDiskIO;
    for ( Loc = 0xf0; Loc < 0x100; Loc++ )  ReadHWPage[ Loc  ] = ReadClockIO;

    for ( Loc = 0x00; Loc < 0x006; Loc++ ) WriteHWPage[ Loc  ] = WriteAuxSlotIO;
    for ( Loc = 0x06; Loc < 0x008; Loc++ ) WriteHWPage[ Loc  ] = WriteAltROMIO;
    for ( Loc = 0x08; Loc < 0x00a; Loc++ ) WriteHWPage[ Loc  ] = WriteAuxSlotIO;
    for ( Loc = 0x0a; Loc < 0x00c; Loc++ ) WriteHWPage[ Loc  ] = WriteAltROMIO;
    for ( Loc = 0x0c; Loc < 0x010; Loc++ ) WriteHWPage[ Loc  ] = WriteVideoControl;
    for ( Loc = 0x10; Loc < 0x020; Loc++ ) WriteHWPage[ Loc  ] = ClearKBStrobe;
    for ( Loc = 0x20; Loc < 0x030; Loc++ ) WriteHWPage[ Loc  ] = WriteIOBlank;
    for ( Loc = 0x30; Loc < 0x040; Loc++ ) WriteHWPage[ Loc  ] = WriteSpeaker;
    for ( Loc = 0x40; Loc < 0x050; Loc++ ) WriteHWPage[ Loc  ] = WriteIOBlank;
    for ( Loc = 0x50; Loc < 0x058; Loc++ ) WriteHWPage[ Loc  ] = WriteVideoControl;
    for ( Loc = 0x58; Loc < 0x07e; Loc++ ) WriteHWPage[ Loc  ] = WriteIOBlank;
    for ( Loc = 0x7e; Loc < 0x080; Loc++ ) WriteHWPage[ Loc  ] = WriteIOU;
    for ( Loc = 0x80; Loc < 0x090; Loc++ ) WriteHWPage[ Loc  ] = WriteSlot0IO;
    for ( Loc = 0x90; Loc < 0x0a0; Loc++ ) WriteHWPage[ Loc  ] = WritePICIO;
    for ( Loc = 0xa0; Loc < 0x0d0; Loc++ ) WriteHWPage[ Loc  ] = WriteIOBlank;
    for ( Loc = 0xd0; Loc < 0x0e0; Loc++ ) WriteHWPage[ Loc  ] = WriteMassStorIO;
    for ( Loc = 0xe0; Loc < 0x0f0; Loc++ ) WriteHWPage[ Loc  ] = WriteDiskIO;
    for ( Loc = 0xf0; Loc < 0x100; Loc++ ) WriteHWPage[ Loc  ] = WriteClockIO;

    /* set current memory segments */
    CurZPSSeg        = MainSeg;
    CurMainSeg       = MainSeg;
    CurSlotROMSeg    = MainSeg;
    CurAuxSlotROMSeg = ExpSlotSeg;
    CurBankSeg       = MainSeg;
    CurBank2Seg      = MainBankSeg;
    for ( Page = 0; Page < 8; Page++ )
    {
        IOExpROM[ Page ] = MainSeg;
    }

    /* set read/write memory segments */
    ZPSRSeg   = CurZPSSeg;
    ZPSWSeg   = CurZPSSeg;
    MainRSeg  = CurMainSeg;
    MainWSeg  = CurMainSeg;
    Bank2RSeg = ROMSeg;
    Bank2WSeg = ROMSeg;
    BankRSeg  = ROMSeg;
    BankWSeg  = ROMSeg;
}

void FreeRAM( void )
{
    if ( MainMem  != NULL ) farfree( MainMem  );
    if ( MainBank != NULL ) farfree( MainBank );
    if ( AuxMem   != NULL ) farfree( AuxMem   );
    if ( AuxBank  != NULL ) farfree( AuxBank  );
    if ( ROMMem   != NULL ) farfree( ROMMem   );
    MainMem = MainBank = AuxMem = AuxBank = ROMMem = NULL;
    MainSeg = MainBankSeg = AuxSeg = AuxBankSeg = ROMSeg = 0;
}
