#include <stdio.h>
#include <string.h>

#include "ram.h"

// Addressing modes
typedef enum {
  Inv,    // Invalid
  Imm,    // Immediate
  Imp,    // Implied
  Acc,    // Accumulator
  Rel,    // Relative
  Abs,    // Absolute
  AbsX,   // Absolute,X
  AbsY,   // Absolute,Y
  ZP,     // Zero page
  ZPX,    // Zero page,X
  ZPY,    // Zero page,Y
  ZPI,    // Zero page indirect
  IdrIdx, // Indirect indexed (oper),Y
  IdxIdr, // Indexed indirect (oper,X)
  Idr,    // Indirect (oper)
  AbsIdx  // Absolute index indirect
  } ModeType;

int Opcode;
ModeType MT;
FILE *fp;

char *Mnemonics[256] = {
  "BRK", "ORA", "???", "???", "TSB", "ORA", "ASL", "???",  /* 0x00 - 0x07 */
  "PHP", "ORA", "ASL", "???", "TSB", "ORA", "ASL", "???",  /* 0x08 - 0x0f */
  "BPL", "ORA", "ORA", "???", "TRB", "ORA", "ASL", "???",  /* 0x10 - 0x17 */
  "CLC", "ORA", "INC", "???", "TRB", "ORA", "ASL", "???",  /* 0x18 - 0x1f */
  "JSR", "AND", "???", "???", "BIT", "AND", "ROL", "???",  /* 0x20 - 0x27 */
  "PLP", "AND", "ROL", "???", "BIT", "AND", "ROL", "???",  /* 0x28 - 0x2f */
  "BMI", "AND", "AND", "???", "BIT", "AND", "ROL", "???",  /* 0x30 - 0x37 */
  "SEC", "AND", "DEC", "???", "BIT", "AND", "ROL", "???",  /* 0x38 - 0x3f */
  "RTI", "EOR", "???", "???", "???", "EOR", "LSR", "???",  /* 0x40 - 0x47 */
  "PHA", "EOR", "LSR", "???", "JMP", "EOR", "LSR", "???",  /* 0x48 - 0x4f */
  "BVC", "EOR", "EOR", "???", "???", "EOR", "LSR", "???",  /* 0x50 - 0x57 */
  "CLI", "EOR", "PHY", "???", "???", "EOR", "LSR", "???",  /* 0x58 - 0x5f */
  "RTS", "ADC", "???", "???", "STZ", "ADC", "ROR", "???",  /* 0x60 - 0x67 */
  "PLA", "ADC", "ROR", "???", "JMP", "ADC", "ROR", "???",  /* 0x68 - 0x6f */
  "BVS", "ADC", "ADC", "???", "STZ", "ADC", "ROR", "???",  /* 0x70 - 0x77 */
  "SEI", "ADC", "PLY", "???", "JMP", "ADC", "ROR", "???",  /* 0x78 - 0x7f */
  "BRA", "STA", "???", "???", "STY", "STA", "STX", "???",  /* 0x80 - 0x87 */
  "DEY", "BIT", "TXA", "???", "STY", "STA", "STX", "???",  /* 0x88 - 0x8f */
  "BCC", "STA", "STA", "???", "STY", "STA", "STX", "???",  /* 0x90 - 0x97 */
  "TYA", "STA", "TXS", "???", "STZ", "STA", "STZ", "???",  /* 0x98 - 0x9f */
  "LDY", "LDA", "LDX", "???", "LDY", "LDA", "LDX", "???",  /* 0xa0 - 0xa7 */
  "TAY", "LDA", "TAX", "???", "LDY", "LDA", "LDX", "???",  /* 0xa8 - 0xaf */
  "BCS", "LDA", "LDA", "???", "LDY", "LDA", "LDX", "???",  /* 0xb0 - 0xb7 */
  "CLV", "LDA", "TSX", "???", "LDY", "LDA", "LDX", "???",  /* 0xb8 - 0xbf */
  "CPY", "CMP", "???", "???", "CPY", "CMP", "DEC", "???",  /* 0xc0 - 0xc7 */
  "INY", "CMP", "DEX", "???", "CPY", "CMP", "DEC", "???",  /* 0xc8 - 0xcf */
  "BNE", "CMP", "CMP", "???", "???", "CMP", "DEC", "???",  /* 0xd0 - 0xd7 */
  "CLD", "CMP", "PHX", "???", "???", "CMP", "DEC", "???",  /* 0xd8 - 0xdf */
  "CPX", "SBC", "???", "???", "CPX", "SBC", "INC", "???",  /* 0xe0 - 0xe7 */
  "INX", "SBC", "NOP", "???", "CPX", "SBC", "INC", "???",  /* 0xe8 - 0xef */
  "BEQ", "SBC", "SBC", "???", "???", "SBC", "INC", "???",  /* 0xf0 - 0xf7 */
  "SED", "SBC", "PLX", "???", "???", "SBC", "INC", "???"   /* 0xf8 - 0xff */
};

/* Instruction sizes */
WORD InstrSize[16] = {0,1,0,0,1,2,2,2,1,1,1,1,1,1,2,2};

/* Addressing strings */
char *AddrStr[16]={
  "",    // Invalid
  "#oper ",    // Immediate
  "",    // Implied
  "Acc ",    // Accumulator
  "Rel ",    // Relative
  "Abs ",    // Absolute
  "Abs,X ",   // Absolute,X
  "Abs,Y ",   // Absolute,Y
  "ZP ",     // Zero page
  "ZP,X ",    // Zero page,X
  "ZP,Y ",    // Zero page,Y
  "(ZP) ",    // Zero page indirect
  "(ZP),Y ", // Indirect indexed (oper),Y
  "(ZP,X) ", // Indexed indirect (oper,X)
  "(Abs) ",    // Indirect (oper)
  "Abs(Abs+X) "  // Absolute index indirect
};

/* Opcode addressing modes */
ModeType AddrMode[256] = {
  Imp, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0x00 - 0x07 */
  Imp, Imm,    Acc, Inv, Abs,    Abs,  Abs,  Inv,   /* 0x08 - 0x0f */
  Rel, IdrIdx, ZPI, Inv, ZP ,    ZPX,  ZPX,  Inv,   /* 0x10 - 0x17 */
  Imp, AbsY,   Acc, Inv, Abs,    AbsX, AbsX, Inv,   /* 0x18 - 0x1f */
  Abs, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0x20 - 0x27 */
  Imp, Imm,    Acc, Inv, Abs,    Abs,  Abs,  Inv,   /* 0x28 - 0x2f */
  Rel, IdrIdx, ZPI, Inv, ZPX,    ZPX,  ZPX,  Inv,   /* 0x30 - 0x37 */
  Imp, AbsY,   Acc, Inv, AbsX,   AbsX, AbsX, Inv,   /* 0x38 - 0x3f */
  Imp, IdxIdr, Inv, Inv, Inv,    ZP ,  ZP ,  Inv,   /* 0x40 - 0x47 */
  Imp, Imm,    Acc, Inv, Abs,    Abs,  Abs,  Inv,   /* 0x48 - 0x4f */
  Rel, IdrIdx, ZPI, Inv, Inv,    ZPX,  ZPX,  Inv,   /* 0x50 - 0x57 */
  Imp, AbsY,   Imp, Inv, Inv,    AbsX, AbsX, Inv,   /* 0x58 - 0x5f */
  Imp, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0x60 - 0x67 */
  Imp, Imm,    Acc, Inv, Idr,    Abs,  Abs,  Inv,   /* 0x68 - 0x6f */
  Rel, IdrIdx, ZPI, Inv, ZPX,    ZPX,  ZPX,  Inv,   /* 0x70 - 0x77 */
  Imp, AbsY,   Imp, Inv, AbsIdx, AbsX, AbsX, Inv,   /* 0x78 - 0x7f */
  Rel, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0x80 - 0x87 */
  Imp, Imm,    Imp, Inv, Abs,    Abs,  Abs,  Inv,   /* 0x88 - 0x8f */
  Rel, IdrIdx, ZPI, Inv, ZPX,    ZPX,  ZPY,  Inv,   /* 0x90 - 0x97 */
  Imp, AbsY,   Imp, Inv, Abs,    AbsX, AbsX, Inv,   /* 0x98 - 0x9f */
  Imm, IdxIdr, Imm, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0xa0 - 0xa7 */
  Imp, Imm,    Imp, Inv, Abs,    Abs,  Abs,  Inv,   /* 0xa8 - 0xaf */
  Rel, IdrIdx, ZPI, Inv, ZPX,    ZPX,  ZPY,  Inv,   /* 0xb0 - 0xb7 */
  Imp, AbsY,   Imp, Inv, AbsX,   AbsX, AbsY, Inv,   /* 0xb8 - 0xbf */
  Imm, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0xc0 - 0xc7 */
  Imp, Imm,    Imp, Inv, Abs,    Abs,  Abs,  Inv,   /* 0xc8 - 0xcf */
  Rel, IdrIdx, ZPI, Inv, Inv,    ZPX,  ZPX,  Inv,   /* 0xd0 - 0xd7 */
  Imp, AbsY,   Imp, Inv, Inv,    AbsX, AbsX, Inv,   /* 0xd8 - 0xdf */
  Imm, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0xe0 - 0xe7 */
  Imp, Imm,    Imp, Inv, Abs,    Abs,  Abs,  Inv,   /* 0xe8 - 0xef */
  Rel, IdrIdx, ZPI, Inv, Inv,    ZPX,  ZPX,  Inv,   /* 0xf0 - 0xf7 */
  Imp, AbsY,   Imp, Inv, Inv,    AbsX, AbsX, Inv    /* 0xf8 - 0xff */
};
/* # clock cycles to process opcode */
unsigned long OpCycle[256] = {
  7, 6, 0, 0, 5, 3, 5, 0,   /* 0x00 - 0x07 */
  3, 2, 2, 0, 6, 4, 6, 0,   /* 0x08 - 0x0f */
  2, 5, 5, 0, 5, 4, 6, 0,   /* 0x10 - 0x17 */
  2, 4, 2, 0, 6, 4, 6, 0,   /* 0x18 - 0x1f */
  6, 6, 0, 0, 3, 3, 5, 0,   /* 0x20 - 0x27 */
  4, 2, 2, 0, 4, 4, 6, 0,   /* 0x28 - 0x2f */
  2, 5, 5, 0, 4, 4, 6, 0,   /* 0x30 - 0x37 */
  2, 4, 2, 0, 4, 4, 6, 0,   /* 0x38 - 0x3f */
  6, 6, 0, 0, 0, 3, 5, 0,   /* 0x40 - 0x47 */
  3, 2, 2, 0, 3, 4, 6, 0,   /* 0x48 - 0x4f */
  2, 5, 5, 0, 0, 4, 6, 0,   /* 0x50 - 0x57 */
  2, 4, 3, 0, 0, 4, 6, 0,   /* 0x58 - 0x5f */
  6, 6, 0, 0, 3, 3, 5, 0,   /* 0x60 - 0x67 */
  4, 2, 2, 0, 6, 4, 6, 0,   /* 0x68 - 0x6f */
  2, 5, 5, 0, 4, 4, 6, 0,   /* 0x70 - 0x77 */
  2, 4, 4, 0, 6, 4, 6, 0,   /* 0x78 - 0x7f */
  2, 6, 0, 0, 3, 3, 3, 0,   /* 0x80 - 0x87 */
  2, 2, 2, 0, 4, 4, 4, 0,   /* 0x88 - 0x8f */
  2, 6, 5, 0, 4, 4, 4, 0,   /* 0x90 - 0x97 */
  2, 5, 2, 0, 4, 5, 5, 0,   /* 0x98 - 0x9f */
  2, 6, 2, 0, 3, 3, 3, 0,   /* 0xa0 - 0xa7 */
  2, 2, 2, 0, 4, 4, 4, 0,   /* 0xa8 - 0xaf */
  2, 5, 5, 0, 4, 4, 4, 0,   /* 0xb0 - 0xb7 */
  2, 4, 2, 0, 4, 4, 4, 0,   /* 0xb8 - 0xbf */
  2, 6, 0, 0, 3, 3, 5, 0,   /* 0xc0 - 0xc7 */
  2, 2, 2, 0, 4, 4, 6, 0,   /* 0xc8 - 0xcf */
  2, 5, 5, 0, 0, 4, 6, 0,   /* 0xd0 - 0xd7 */
  2, 4, 3, 0, 0, 4, 6, 0,   /* 0xd8 - 0xdf */
  2, 6, 0, 0, 3, 3, 5, 0,   /* 0xe0 - 0xe7 */
  2, 2, 2, 0, 4, 4, 6, 0,   /* 0xe8 - 0xef */
  2, 5, 5, 0, 0, 4, 6, 0,   /* 0xf0 - 0xf7 */
  2, 4, 4, 0, 0, 4, 6, 0    /* 0xf8 - 0xff */
};

void MakeCPU(void)
{
  fp=fopen("cpucode.c","w");
  for (Opcode=0;Opcode<256;Opcode++) {
    /* Handle each opcode */
    MT = AddrMode[Opcode];
    fprintf(fp,"    case 0x%02.2x: /* %s %s*/\n",Opcode,Mnemonics[Opcode],AddrStr[MT]);
    if (strcmp(Mnemonics[Opcode],"JMP")!=0 &&
        strcmp(Mnemonics[Opcode],"JSR")!=0) {
      switch (MT) {
        case Abs:    fputs("      Address=ReadWord(PC);\n",fp); break;
        case AbsX:   fputs("      Address=ReadWord(PC)+X;\n",fp); break;
        case AbsY:   fputs("      Address=ReadWord(PC)+Y;\n",fp); break;
        case ZP:     fputs("      Address=(WORD)ReadByte(PC);\n",fp); break;
        case ZPX:    fputs("      Address=((WORD)ReadByte(PC)+(WORD)X)&0x00ff;\n",fp); break;
        case ZPY:    fputs("      Address=((WORD)ReadByte(PC)+(WORD)Y)&0x00ff;\n",fp); break;
        case ZPI:    fputs("      Address=ReadWord((WORD)ReadByte(PC));\n",fp); break;
        case IdrIdx: fputs("      Address=ReadWord((WORD)ReadByte(PC))+(WORD)Y;\n",fp); break;
//        case IdxIdr: fputs("      Address=ReadWord((ReadByte(PC)+X)&0x00ff);\n",fp); break;
        case IdxIdr: fputs("      Address=ReadWord((WORD)ReadByte(PC)+(WORD)X);\n",fp); break;
      }
    }
    switch (MT) {
      case Inv:
//        fprintf(fp,"      Quit=1;\n"); /* dump emulation - 6502 */
        /* NOP - 65C02 */
        break;
      case Imp:
        fprintf(fp,"      MC_%s();\n",Mnemonics[Opcode]);
        break;
      case Acc:
        fprintf(fp,"      MC_%s(A);\n",Mnemonics[Opcode]);
        break;
      case Rel:
        fputs("      Data=ReadByte(PC);\n",fp);
        fprintf(fp,"      MC_%s();\n",Mnemonics[Opcode]);
        break;
      case Imm:
        fputs("      Data=ReadByte(PC);\n",fp);
        fprintf(fp,"      MC_%s(Data);\n",Mnemonics[Opcode]);
        break;
      case Abs:
      case AbsX:
      case AbsY:
      case ZP:
      case ZPX:
      case ZPY:
      case ZPI:
      case IdrIdx:
      case IdxIdr:
      case Idr:
      case AbsIdx:
        if (strcmp(Mnemonics[Opcode],"JMP")==0) {
          switch (MT) {
            case Abs:    fputs("      PC=ReadWord(PC);\n",fp);                break;
            case Idr:    fputs("      PC=ReadWord(ReadWord(PC));\n",fp);   break;
            case AbsIdx: fputs("      PC=ReadWord(ReadWord(PC)+(WORD)X);\n",fp); break;
          }
          fputs("      PC-=2;\n",fp);
        }
        if (strcmp(Mnemonics[Opcode],"JSR")==0) {
          fputs("      PC++;\n",fp);
          fputs("      WriteByte(0x0100+S,PC>>8);\n",fp);
          fputs("      S--;\n",fp);
          fputs("      WriteByte(0x0100+S,PC&0xff);\n",fp);
          fputs("      S--;\n",fp);
          fputs("      PC=ReadWord(PC-1)-2;\n",fp); /* -2 pre-corrects for instruction size increment below */
        }
        if (strcmp(Mnemonics[Opcode],"JMP")!=0 &&
            strcmp(Mnemonics[Opcode],"JSR")!=0 &&
            strcmp(Mnemonics[Opcode],"STA")!=0 &&
            strcmp(Mnemonics[Opcode],"STX")!=0 &&
            strcmp(Mnemonics[Opcode],"STY")!=0 &&
            strcmp(Mnemonics[Opcode],"STZ")!=0) {
          fputs("      Data=ReadByte(Address);\n",fp);
        }
        if (strcmp(Mnemonics[Opcode],"JMP")!=0 &&
            strcmp(Mnemonics[Opcode],"JSR")!=0) {
          fprintf(fp,"      MC_%s(Data);\n",Mnemonics[Opcode]);
        }
        /* Handle opcodes that write to memory */
        if (strcmp(Mnemonics[Opcode],"INC")==0 ||
            strcmp(Mnemonics[Opcode],"DEC")==0 ||
            strcmp(Mnemonics[Opcode],"ASL")==0 ||
            strcmp(Mnemonics[Opcode],"LSR")==0 ||
            strcmp(Mnemonics[Opcode],"ROL")==0 ||
            strcmp(Mnemonics[Opcode],"ROR")==0 ||
            strcmp(Mnemonics[Opcode],"STA")==0 ||
            strcmp(Mnemonics[Opcode],"STX")==0 ||
            strcmp(Mnemonics[Opcode],"STY")==0 ||
            strcmp(Mnemonics[Opcode],"STZ")==0 ||
            strcmp(Mnemonics[Opcode],"TRB")==0 ||
            strcmp(Mnemonics[Opcode],"TSB")==0) {
          fputs("      WriteByte(Address,Data);\n",fp);
        }
        break;
    }
    if (*(InstrSize+MT) != 0)
      fprintf(fp,"      PC+=%d;\n",*(InstrSize+MT));
    if (*(OpCycle+Opcode) != 0)
      fprintf(fp,"      ClockTick+=%d;\n",*(OpCycle+Opcode));
    fputs("      break;\n",fp);
  }

  fclose(fp);
}

main()
{
  MakeCPU();
  return 0;
}
