#include <stdio.h>
#include <conio.h>
#include <stdarg.h>
#include <stdlib.h>
#include "ram.h"
#include "cpu.h"
#include "diags.h"
#include "system.h"
#include "video.h"

static unsigned int far *screen = (unsigned int far *)0xB8000000L;
int SX, SY;
unsigned char ATTR;

// Addressing modes
typedef enum
{
    Inv,    // Invalid
    Imm,    // Immediate
    Imp,    // Implied
    Acc,    // Accumulator
    Rel,    // Relative
    Abs,    // Absolute
    AbsX,   // Absolute,X
    AbsY,   // Absolute,Y
    ZP,     // Zero page
    ZPX,    // Zero page,X
    ZPY,    // Zero page,Y
    ZPI,    // Zero page indirect
    IdrIdx, // Indirect indexed (oper),Y
    IdxIdr, // Indexed indirect (oper,X)
    Idr,    // Indirect (oper)
    AbsIdx  // Absolute index indirect
} ModeType;

extern int KeyHit;

void GotoXY( int X, int Y )
{
    SX = X;
    SY = Y;
}

void SetAttr( unsigned char attr )
{
    ATTR = attr;
}

/* X=0-79,Y=0-24 */
void PutChar( unsigned char ch )
{
    if (SX < 0 || SX > 79 || SY < 0 || SY > 24 ) return;
    screen[ 40 * SY + SX ] = ( ( (unsigned int)ATTR ) << 8 ) |
                               ( (unsigned int)ch );
    SX++;
}

void PutString( unsigned char *str )
{
    while ( *str )
    {
        PutChar( *str++ );
    }
}

/* outputs an integer in hex */
void PutNum( long num, int digits )
{
    unsigned char n;

    while ( digits-- > 0 )
    {
        n = ( ( num >> ( digits * 4 ) ) & 0x0f ) + 48;
        if ( n > 57 )
        {
            n += 7;
        }
        PutChar( n );
    }
}

char buffer[ 255 ];

/* Opcode addressing modes */
ModeType AddrMode[ 256 ] =
{
    Imp, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0x00 - 0x07 */
    Imp, Imm,    Acc, Inv, Abs,    Abs,  Abs,  Inv,   /* 0x08 - 0x0f */
    Rel, IdrIdx, ZPI, Inv, ZP ,    ZPX,  ZPX,  Inv,   /* 0x10 - 0x17 */
    Imp, AbsY,   Acc, Inv, Abs,    AbsX, AbsX, Inv,   /* 0x18 - 0x1f */
    Abs, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0x20 - 0x27 */
    Imp, Imm,    Acc, Inv, Abs,    Abs,  Abs,  Inv,   /* 0x28 - 0x2f */
    Rel, IdrIdx, ZPI, Inv, ZPX,    ZPX,  ZPX,  Inv,   /* 0x30 - 0x37 */
    Imp, AbsY,   Acc, Inv, AbsX,   AbsX, AbsX, Inv,   /* 0x38 - 0x3f */
    Imp, IdxIdr, Inv, Inv, Inv,    ZP ,  ZP ,  Inv,   /* 0x40 - 0x47 */
    Imp, Imm,    Acc, Inv, Abs,    Abs,  Abs,  Inv,   /* 0x48 - 0x4f */
    Rel, IdrIdx, ZPI, Inv, Inv,    ZPX,  ZPX,  Inv,   /* 0x50 - 0x57 */
    Imp, AbsY,   Imp, Inv, Inv,    AbsX, AbsX, Inv,   /* 0x58 - 0x5f */
    Imp, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0x60 - 0x67 */
    Imp, Imm,    Acc, Inv, Idr,    Abs,  Abs,  Inv,   /* 0x68 - 0x6f */
    Rel, IdrIdx, ZPI, Inv, ZPX,    ZPX,  ZPX,  Inv,   /* 0x70 - 0x77 */
    Imp, AbsY,   Imp, Inv, AbsIdx, AbsX, AbsX, Inv,   /* 0x78 - 0x7f */
    Rel, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0x80 - 0x87 */
    Imp, Imm,    Imp, Inv, Abs,    Abs,  Abs,  Inv,   /* 0x88 - 0x8f */
    Rel, IdrIdx, ZPI, Inv, ZPX,    ZPX,  ZPY,  Inv,   /* 0x90 - 0x97 */
    Imp, AbsY,   Imp, Inv, Abs,    AbsX, AbsX, Inv,   /* 0x98 - 0x9f */
    Imm, IdxIdr, Imm, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0xa0 - 0xa7 */
    Imp, Imm,    Imp, Inv, Abs,    Abs,  Abs,  Inv,   /* 0xa8 - 0xaf */
    Rel, IdrIdx, ZPI, Inv, ZPX,    ZPX,  ZPY,  Inv,   /* 0xb0 - 0xb7 */
    Imp, AbsY,   Imp, Inv, AbsX,   AbsX, AbsY, Inv,   /* 0xb8 - 0xbf */
    Imm, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0xc0 - 0xc7 */
    Imp, Imm,    Imp, Inv, Abs,    Abs,  Abs,  Inv,   /* 0xc8 - 0xcf */
    Rel, IdrIdx, ZPI, Inv, Inv,    ZPX,  ZPX,  Inv,   /* 0xd0 - 0xd7 */
    Imp, AbsY,   Imp, Inv, Inv,    AbsX, AbsX, Inv,   /* 0xd8 - 0xdf */
    Imm, IdxIdr, Inv, Inv, ZP ,    ZP ,  ZP ,  Inv,   /* 0xe0 - 0xe7 */
    Imp, Imm,    Imp, Inv, Abs,    Abs,  Abs,  Inv,   /* 0xe8 - 0xef */
    Rel, IdrIdx, ZPI, Inv, Inv,    ZPX,  ZPX,  Inv,   /* 0xf0 - 0xf7 */
    Imp, AbsY,   Imp, Inv, Inv,    AbsX, AbsX, Inv    /* 0xf8 - 0xff */
};

/* # clock cycles to process opcode */
unsigned long OpCycle[ 256 ] =
{
    7, 6, 0, 0, 5, 3, 5, 0,   /* 0x00 - 0x07 */
    3, 2, 2, 0, 6, 4, 6, 0,   /* 0x08 - 0x0f */
    2, 5, 5, 0, 5, 4, 6, 0,   /* 0x10 - 0x17 */
    2, 4, 2, 0, 6, 4, 6, 0,   /* 0x18 - 0x1f */
    6, 6, 0, 0, 3, 3, 5, 0,   /* 0x20 - 0x27 */
    4, 2, 2, 0, 4, 4, 6, 0,   /* 0x28 - 0x2f */
    2, 5, 5, 0, 4, 4, 6, 0,   /* 0x30 - 0x37 */
    2, 4, 2, 0, 4, 4, 6, 0,   /* 0x38 - 0x3f */
    6, 6, 0, 0, 0, 3, 5, 0,   /* 0x40 - 0x47 */
    3, 2, 2, 0, 3, 4, 6, 0,   /* 0x48 - 0x4f */
    2, 5, 5, 0, 0, 4, 6, 0,   /* 0x50 - 0x57 */
    2, 4, 3, 0, 0, 4, 6, 0,   /* 0x58 - 0x5f */
    6, 6, 0, 0, 3, 3, 5, 0,   /* 0x60 - 0x67 */
    4, 2, 2, 0, 6, 4, 6, 0,   /* 0x68 - 0x6f */
    2, 5, 5, 0, 4, 4, 6, 0,   /* 0x70 - 0x77 */
    2, 4, 4, 0, 6, 4, 6, 0,   /* 0x78 - 0x7f */
    2, 6, 0, 0, 3, 3, 3, 0,   /* 0x80 - 0x87 */
    2, 2, 2, 0, 4, 4, 4, 0,   /* 0x88 - 0x8f */
    2, 6, 5, 0, 4, 4, 4, 0,   /* 0x90 - 0x97 */
    2, 5, 2, 0, 4, 5, 5, 0,   /* 0x98 - 0x9f */
    2, 6, 2, 0, 3, 3, 3, 0,   /* 0xa0 - 0xa7 */
    2, 2, 2, 0, 4, 4, 4, 0,   /* 0xa8 - 0xaf */
    2, 5, 5, 0, 4, 4, 4, 0,   /* 0xb0 - 0xb7 */
    2, 4, 2, 0, 4, 4, 4, 0,   /* 0xb8 - 0xbf */
    2, 6, 0, 0, 3, 3, 5, 0,   /* 0xc0 - 0xc7 */
    2, 2, 2, 0, 4, 4, 6, 0,   /* 0xc8 - 0xcf */
    2, 5, 5, 0, 0, 4, 6, 0,   /* 0xd0 - 0xd7 */
    2, 4, 3, 0, 0, 4, 6, 0,   /* 0xd8 - 0xdf */
    2, 6, 0, 0, 3, 3, 5, 0,   /* 0xe0 - 0xe7 */
    2, 2, 2, 0, 4, 4, 6, 0,   /* 0xe8 - 0xef */
    2, 5, 5, 0, 0, 4, 6, 0,   /* 0xf0 - 0xf7 */
    2, 4, 4, 0, 0, 4, 6, 0    /* 0xf8 - 0xff */
};

char *Mnemonics[ 256 ] =
{
    "BRK", "ORA", "???", "???", "TSB", "ORA", "ASL", "???",  /* 0x00 - 0x07 */
    "PHP", "ORA", "ASL", "???", "TSB", "ORA", "ASL", "???",  /* 0x08 - 0x0f */
    "BPL", "ORA", "ORA", "???", "TRB", "ORA", "ASL", "???",  /* 0x10 - 0x17 */
    "CLC", "ORA", "INC", "???", "TRB", "ORA", "ASL", "???",  /* 0x18 - 0x1f */
    "JSR", "AND", "???", "???", "BIT", "AND", "ROL", "???",  /* 0x20 - 0x27 */
    "PLP", "AND", "ROL", "???", "BIT", "AND", "ROL", "???",  /* 0x28 - 0x2f */
    "BMI", "AND", "AND", "???", "BIT", "AND", "ROL", "???",  /* 0x30 - 0x37 */
    "SEC", "AND", "DEC", "???", "BIT", "AND", "ROL", "???",  /* 0x38 - 0x3f */
    "RTI", "EOR", "???", "???", "???", "EOR", "LSR", "???",  /* 0x40 - 0x47 */
    "PHA", "EOR", "LSR", "???", "JMP", "EOR", "LSR", "???",  /* 0x48 - 0x4f */
    "BVC", "EOR", "EOR", "???", "???", "EOR", "LSR", "???",  /* 0x50 - 0x57 */
    "CLI", "EOR", "PHY", "???", "???", "EOR", "LSR", "???",  /* 0x58 - 0x5f */
    "RTS", "ADC", "???", "???", "STZ", "ADC", "ROR", "???",  /* 0x60 - 0x67 */
    "PLA", "ADC", "ROR", "???", "JMP", "ADC", "ROR", "???",  /* 0x68 - 0x6f */
    "BVS", "ADC", "ADC", "???", "STZ", "ADC", "ROR", "???",  /* 0x70 - 0x77 */
    "SEI", "ADC", "PLY", "???", "JMP", "ADC", "ROR", "???",  /* 0x78 - 0x7f */
    "BRA", "STA", "???", "???", "STY", "STA", "STX", "???",  /* 0x80 - 0x87 */
    "DEY", "BIT", "TXA", "???", "STY", "STA", "STX", "???",  /* 0x88 - 0x8f */
    "BCC", "STA", "STA", "???", "STY", "STA", "STX", "???",  /* 0x90 - 0x97 */
    "TYA", "STA", "TXS", "???", "STZ", "STA", "STZ", "???",  /* 0x98 - 0x9f */
    "LDY", "LDA", "LDX", "???", "LDY", "LDA", "LDX", "???",  /* 0xa0 - 0xa7 */
    "TAY", "LDA", "TAX", "???", "LDY", "LDA", "LDX", "???",  /* 0xa8 - 0xaf */
    "BCS", "LDA", "LDA", "???", "LDY", "LDA", "LDX", "???",  /* 0xb0 - 0xb7 */
    "CLV", "LDA", "TSX", "???", "LDY", "LDA", "LDX", "???",  /* 0xb8 - 0xbf */
    "CPY", "CMP", "???", "???", "CPY", "CMP", "DEC", "???",  /* 0xc0 - 0xc7 */
    "INY", "CMP", "DEX", "???", "CPY", "CMP", "DEC", "???",  /* 0xc8 - 0xcf */
    "BNE", "CMP", "CMP", "???", "???", "CMP", "DEC", "???",  /* 0xd0 - 0xd7 */
    "CLD", "CMP", "PHX", "???", "???", "CMP", "DEC", "???",  /* 0xd8 - 0xdf */
    "CPX", "SBC", "???", "???", "CPX", "SBC", "INC", "???",  /* 0xe0 - 0xe7 */
    "INX", "SBC", "NOP", "???", "CPX", "SBC", "INC", "???",  /* 0xe8 - 0xef */
    "BEQ", "SBC", "SBC", "???", "???", "SBC", "INC", "???",  /* 0xf0 - 0xf7 */
    "SED", "SBC", "PLX", "???", "???", "SBC", "INC", "???"   /* 0xf8 - 0xff */
};

char AddrStr[ 16 ][ 20 ] =
{
    "???",
    "#nn",
    "",
    "A",
    "$nnnn",
    "$nnnn",
    "$nnnn,X",
    "$nnnn,Y",
    "$nn",
    "$nn,X",
    "$nn,Y",
    "($nn)",
    "($nn),Y",
    "($nn,X)",
    "($nnnn)",
    "($nnnn,X)"
};

struct {
    char *Fmt;
    int  Len;
    int  IsMem;
} OperandFormat[ 16 ] =
{
    { "???      "    , 0, 0 },
    { "#$%02.2X     ", 1, 0 },
    { "         "    , 0, 0 },
    { "A        "    , 0, 0 },
    { "$%04.4X    "  , 2, 1 },
    { "$%04.4X    "  , 2, 1 },
    { "$%04.4X,X  "  , 2, 1 },
    { "$%04.4X,Y  "  , 2, 1 },
    { "$%02.2X      ", 1, 1 },
    { "$%02.2X,X    ", 1, 1 },
    { "$%02.2X,Y    ", 1, 1 },
    { "($%02.2X)    ", 1, 1 },
    { "($%02.2X),Y  ", 1, 1 },
    { "($%02.2X,X)  ", 1, 1 },
    { "($%04.4X)  "  , 2, 1 },
    { "($%04.4X,X)"  , 2, 1 }
};

/* Print the opcodes to stdout */
void DumpOpcodes( void )
{
    int Loop;

    for ( Loop = 0; Loop < 256; Loop++ )
    {
        printf( "%02.2x\t%s\t%s\t%d\n", Loop,
                                        Mnemonics[ Loop ],
                                        AddrStr[ AddrMode[ Loop ] ],
                                        OpCycle[ Loop ] );
    }
}

int StopExec = 0;

BYTE DiagReadByte( WORD Address )
{
    if ( ( Address & 0xff00 ) == 0xc000 )
    {
        return peekb( MainSeg, Address );
    }
    return ReadByte( Address );
}

WORD DiagReadWord( WORD Address )
{
    return DiagReadByte( Address ) | ( DiagReadByte( Address + 1 ) << 8 );
}

/* 80-column output */
void OutputCurrentState80col( void )
{
    int opcode;
    ModeType MT;
    unsigned int operand, Key;

    if ( PC == 0x0303 ) StopExec = 1;
    if ( PC == 0xc652 ) StopExec = 1;
    if ( PC == 0xec9a ) StopExec = 1;

    opcode = peekb( MainRSeg, PC );
    MT = *( AddrMode + opcode );
    /* Output current opcode */
    switch ( OperandFormat[ MT ].Len )
    {
    case 1:
        operand=peekb( MainRSeg, PC + 1 );
        break;
    case 2:
        operand=peek ( MainRSeg, PC + 1 );
        break;
    }
    if ( AddrMode[ opcode ] == Rel )
    {
        operand = PC + 1;
        operand += (signed char)peekb( MainRSeg, PC + 1 );
        operand++;
    }
    SetAttr( 0x0d );
    GotoXY( 41, 1 );
    PutNum( PC, 4 );
    PutChar( ':' );
    PutString( Mnemonics[ opcode ] );
    PutChar( ' ' );
    sprintf( buffer, OperandFormat[ MT ].Fmt, operand );
    GotoXY( 50, 1 );
    PutString( buffer );
    GotoXY( 49, 2 );
    if ( OperandFormat[ MT ].IsMem )
    {
        sprintf( buffer, "=$%02.2X", peekb( MainRSeg, operand ) );
        PutString( buffer );
    }
    else
    {
        PutString( "    " );
    }
    sprintf( buffer, "P=$%02.2X ->",
      ( Negative  << 7 ) |
      ( Overflow  << 6 ) |
      ( 1         << 5 ) |
      ( Break     << 4 ) |
      ( Decimal   << 3 ) |
      ( Interrupt << 2 ) |
      ( Zero      << 1 ) |
      ( Carry     << 0 ) );
    GotoXY( 41, 3 );
    PutString( buffer );

    GotoXY( 50, 3 );
    PutString( Negative  ? "N=1" : "N=0" );
    GotoXY( 50, 4 );
    PutString( Overflow  ? "V=1" : "V=0" );
    GotoXY( 50, 5 );
    PutString( Break     ? "B=1" : "B=0" );
    GotoXY( 50, 6 );
    PutString( Decimal   ? "D=1" : "D=0" );
    GotoXY( 50, 7 );
    PutString( Interrupt ? "I=1" : "I=0" );
    GotoXY( 50, 8 );
    PutString( Zero      ? "Z=1" : "Z=0" );
    GotoXY( 50, 9 );
    PutString( Carry     ? "C=1" : "C=0" );

    sprintf( buffer, "S=$%02.2X", S );
    GotoXY( 41, 4 );
    PutString( buffer );
    sprintf( buffer, "A=$%02.2X", A );
    GotoXY( 41, 5 );
    PutString( buffer );
    sprintf( buffer, "X=$%02.2X", X );
    GotoXY( 41, 6 );
    PutString( buffer );
    sprintf( buffer, "Y=$%02.2X", Y );
    GotoXY( 41, 7 );
    PutString( buffer );

    if ( StopExec )
    {
        while ( !kbhit() );
        Key = getch();
        if ( Key == 'R' ){
            StopExec = 0;
        }
    }
}

WORD nextStop = 0xffff;

/* 1-line output */
void OutputCurrentState( void )
{
    int opcode;
    ModeType MT;
    unsigned int operand, Key;

    GotoXY( 0, 24 );
    SetAttr( 0x0d );

//    if ( PC == 0x0303 ) StopExec = 1;
//    if ( PC == 0xc652 ) StopExec = 1;
//    if ( PC == 0xec9a ) StopExec = 1;
    if ( PC == 0x2000 )
    {
//      StopExec = 1;
//      DoDebug = 0;
    }
    if ( PC != 0xffff && PC == nextStop )
    {
        StopExec = 1;
        nextStop = 0xffff;
    }

    opcode = DiagReadByte( PC );
    MT = *( AddrMode + opcode );
    /* Output current opcode */
    switch ( OperandFormat[ MT ].Len )
    {
    case 1:
        operand = DiagReadByte( PC + 1 );
        break;
    case 2:
        operand = DiagReadWord( PC + 1 );
        break;
    }
    if ( AddrMode[ opcode ] == Rel )
    {
        operand = PC + 1;
        operand += (signed char)DiagReadByte( PC + 1 );
        operand++;
    }
    PutNum( PC, 4 );
    PutChar( ':' );
    PutString( Mnemonics[ opcode ] );
//    PutChar( ' ' );
    sprintf( buffer, OperandFormat[ MT ].Fmt, operand );
    PutString( buffer );
    sprintf( buffer, "$%02.2X", DiagReadByte( operand ) );
    SetAttr( 0x0c );
    if ( OperandFormat[ MT ].IsMem )
    {
        PutString( buffer );
    }
    else
    {
        *buffer = ' ';
        PutString( buffer );
    }
    SetAttr( 0x0d );
    sprintf( buffer, "P=%02.2X",
      ( Negative  << 7 ) |
      ( Overflow  << 6 ) |
      ( 1         << 5 ) |
      ( Break     << 4 ) |
      ( Decimal   << 3 ) |
      ( Interrupt << 2 ) |
      ( Zero      << 1 ) |
      ( Carry     << 0 ) );
    PutString( buffer );

//    PutString( Negative  ? "N=1" : "N=0" );
//    PutString( Overflow  ? "V=1" : "V=0" );
//    PutString( Break     ? "B=1" : "B=0" );
//    PutString( Decimal   ? "D=1" : "D=0" );
//    PutString( Interrupt ? "I=1" : "I=0" );
//    PutString( Zero      ? "Z=1" : "Z=0" );
//    PutString( Carry     ? "C=1" : "C=0" );

    SetAttr( 0x0c);
    sprintf( buffer, "S=%02.2X", S );
    PutString( buffer );
    SetAttr( 0x0d);
    sprintf( buffer, "A=%02.2X", A );
    PutString( buffer );
    SetAttr( 0x0c);
    sprintf( buffer, "X=%02.2X", X );
    PutString( buffer );
    SetAttr( 0x0d);
    sprintf( buffer, "Y=%02.2X", Y );
    PutString( buffer );

    if ( StopExec )
    {
        while ( !kbhit() );
        Key = getch();
        if ( Key == 'D')
        {
            DoDebug = 0;
        }
        if ( Key == 'R')
        {
            StopExec = 0;
        }
        if ( Key == 'J' && opcode==0x20)
        {
            nextStop = PC + 3;
            StopExec = 0;
        }
        if ( Key == 'B' )
        {
            StopExec = 1;
        }
    }
}
