#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#include "a2icomms.h"

#define BCD2RAW( n ) ( n ) / 16 * 10 + ( n ) % 16

static struct A2ICommsParms CurParms;

static int baseport;

/* Fix system clock by re-setting with CMOS RTC clock */
/* Returns 0 if OK, !0 on error */
static int FixClock( void )
{
    union REGS regs;

    /* Get CMOS clock */
    regs.h.ah = 0x02;
    int86( 0x1a, &regs, &regs );
    if ( regs.x.cflag == 0 )
    {
        /* Convert RTC BCD format to MS-DOS raw format */
        regs.h.ch = BCD2RAW( regs.h.ch );
        regs.h.cl = BCD2RAW( regs.h.cl );
        regs.h.dh = BCD2RAW( regs.h.dh );
        /* Set MS-DOS system clock */
        regs.h.ah = 0x2d;
        regs.h.dl = 0;
        int86( 0x21, &regs, &regs );
        if ( regs.h.al != 0 )
        {
            return 2; /* DOS couldn't or wouldn't set the clock */
        }
    }
    else
    {
        return 1; /* Couldn't read the RTC */
    }
    return 0;
}

/* entries set to zero do not alter current settings */
void SetA2ICommsParms( struct A2ICommsParms *parms )
{
    if ( parms->porttype       ) CurParms.porttype       = parms->porttype;
    if ( parms->portnum        ) CurParms.portnum        = parms->portnum;
    if ( parms->presenddelay   ) CurParms.presenddelay   = parms->presenddelay;
    if ( parms->senddelay      ) CurParms.senddelay      = parms->senddelay;
    if ( parms->rectimeout     ) CurParms.rectimeout     = parms->rectimeout;
    if ( parms->packettimeout  ) CurParms.packettimeout  = parms->packettimeout;
    if ( parms->maxpacketsends ) CurParms.maxpacketsends = parms->maxpacketsends;

    /* Setup for each port type */
    switch ( CurParms.porttype )
    {
    case A2I_PARALLEL_TYPE:
        baseport = 0x378;
        break;
    case A2I_SERIAL_TYPE:
        baseport = 0x3f8;
        /* 2400 baud, 8 data bits, 1 stop bit, no parity */
        /* 2400 baud (divisor = 115200/2400 = 48) */
        outportb( baseport + 3, inportb( baseport + 3 ) | 0x80 ); /* set Divisor Latch Bit */
        outportb( baseport + 0, 48 );
        outportb( baseport + 1, 0 );
        /* Set Line Control Register (also clears DLB) */
        /* 00000011b */
        outportb( baseport + 3, 3 );
        break;
    }
}

/*******************************************
 *                                         *
 * ~1microsecond resolution delay routines *
 *                                         *
 * Thank you Borland!                      *
 *                                         *
 *******************************************/

/* NOTE: readtimer, timer_init, and usdelay are slight modifications of the
   functions found in the C Run-Time Library file DELAY.CAS */

static unsigned long multiplier;

static void near dummy ( void ) {}

/* returns complement of counter in timer 0. Complement is used so the
   reported count appears to go up (the timer counts down). */
static unsigned near readtimer( void )
{
    asm pushf                    /* Save interrupt flag */
    asm cli                      /* Disable interrupts */
    asm mov  al,0                /* Latch timer 0 */
    asm out  0x43, al
        dummy();                 /* Waste some time */
    asm in   al, 0x40            /* Counter --> bx */
    asm mov  bl, al              /* LSB in BL */
        dummy();                 /* Waste some time */
    asm in   al, 0x40
    asm mov  bh, al              /* MSB in BH */
    asm not  bx                  /* Need ascending counter */
    asm popf                     /* Restore interrupt flag */
    return( _BX );
}

/* Figure out what mode timer 0 is in. It should be either mode 2 or 3. In
   mode 3 it counts down by two twice and should always be even. Mode 2
   counts can be either odd or even. If the count is sampled 100 times and
   an odd value appears, then take it as mode 2. */
static void init_timer( void )
{
    int i;

    /* assume mode 3 */
    multiplier = 2L;
    for ( i = 0; i < 100; i++ )
    {
        if ( ( readtimer() & 1 ) == 0 ) /* readtimer() returns complement */
        {
          /* mode 2 */
          multiplier = 1L;
          return;
        }
    }
}

/* usdelay (microsecond delay) - wait for specified number of microseconds */
static void _CType usdelay( unsigned long microseconds )
{
    unsigned long stop;
    unsigned cur, prev;

    stop = ( prev = readtimer() ) + ( microseconds * 1000L / 838L * multiplier );

    while ( ( cur = readtimer() ) < stop )
    {
        if ( cur < prev ) /* Check for timer wraparound */
        {
            if ( stop < 0x10000L )
            {
                break;
            }
            stop -= 0x10000L;
        }
        prev = cur;
    }
}

/******************
 *                *
 * INITIALISATION *
 *                *
 ******************/

void InitA2IComms( void )
{
    struct A2ICommsParms InitParms;

    init_timer();
    InitParms.porttype = A2I_PARALLEL_TYPE; /* use a Parallel Port */
    InitParms.portnum = 1; /* use port 1 (i.e. LPT1) */
    InitParms.presenddelay = 400UL;   /* microsecond length of phase 1 */
    InitParms.senddelay = 75L;      /* microsecond length of phases 2 & 3 */
    InitParms.rectimeout = 0xf000UL; /* # retries for each nibble receive phase */
    InitParms.packettimeout = 0x0000fUL; /* # retries for each byte */
    InitParms.maxpacketsends = 8; /* max # times to send packet before giving up */
    SetA2ICommsParms( &InitParms );
}

/************
 *          *
 * BYTE I/O *
 *          *
 ************/

/* returns -1 on timeout */
int ReceiveParallelByte( void )
{
    unsigned long cnt;
    unsigned char nibble, data;

    /* wait for clock to lower */
    cnt = CurParms.rectimeout;
    do
    {
        nibble = ( inportb( baseport + 1 ) >> 3 ) ^ 0x10;
        if ( --cnt == 0 )
        {
            return -1;
        }
    } while ( ( nibble & 0x10 ) == 0x10 );

    /* wait for clock to rise */
    cnt = CurParms.rectimeout;
    do
    {
        nibble = ( inportb( baseport + 1 ) >> 3 ) ^ 0x10;
        if ( --cnt == 0 )
        {
            return -1;
        }
    } while ( ( nibble & 0x10 ) == 0x00 );

    /* get low nibble */
    nibble = ( ( inportb( baseport + 1 ) >> 3 ) ^ 0x10 ) & 0x0f;
    data = nibble;

    /* wait for clock to drop */
    cnt = CurParms.rectimeout;
    do
    {
        nibble = ( inportb( baseport + 1 ) >> 3 ) ^ 0x10;
        if ( --cnt == 0 )
        {
            return -1;
        }
    } while ( ( nibble & 0x10 ) == 0x10 );

    /* get high nibble */
    nibble = ( ( inportb( baseport + 1 ) >> 3 ) ^ 0x10 ) & 0x0f;
    data |= nibble << 4;
    return data;
}

/* returns -1 on timeout */
int ReceiveSerialByte( void )
{
    unsigned long cnt;

    /* wait for byte */
    cnt = CurParms.rectimeout;
    while ( --cnt > 0 )
    {
        /* check Data Ready bit in Line Status Register */
        if ( inportb( baseport + 5 ) & 1 )
        {
            /* some data has been received */
            return inportb( baseport + 0 );
        }
    }
    return -1;
}

int ReceiveByte( void )
{
    int retval;

    switch ( CurParms.porttype )
    {
    case A2I_PARALLEL_TYPE:
        retval = ReceiveParallelByte();
        break;
    case A2I_SERIAL_TYPE:
        retval = ReceiveSerialByte();
        break;
    }
    return retval;
}

void SendParallelByte( unsigned char byte )
{
    /* lower clock */
    outportb( baseport + 0, 0 );
    /* pre-send delay */
    usdelay( CurParms.presenddelay );
    /* send low nibble with clock raised */
    outportb( baseport + 0, ( byte & 0x0f ) | 0x10 );
    /* send delay */
    usdelay( CurParms.senddelay );
    /* send high nibble with clock lowered */
    outportb( baseport + 0, ( byte >> 4 ) & 0x0f );
    /* send delay */
    usdelay( CurParms.senddelay );
}

void SendSerialByte( unsigned char byte )
{
    /* pre-send delay */
    usdelay( CurParms.presenddelay );
    /* send byte */
    while ( ( inportb( baseport + 5 ) & 0x20 ) == 0 ); /* wait for transmit reg. to empty */
    outportb( baseport + 0, byte );
}

void SendByte( unsigned char byte )
{
    switch ( CurParms.porttype )
    {
    case A2I_PARALLEL_TYPE:
        SendParallelByte( byte );
        break;
    case A2I_SERIAL_TYPE:
        SendSerialByte( byte );
        break;
    }
}

/****************************
 *                          *
 * PACKET HANDLING ROUTINES *
 *                          *
 ****************************/

/**********************
 *                    *
 * CRC-CCITT Routines *
 *                    *
 **********************/

static unsigned int CRCTable[ 16 ] =
{
    0x0000, 0x1081, 0x2102, 0x3183, 0x4204, 0x5285, 0x6306, 0x7387,
    0x8408, 0x9489, 0xa50a, 0xb58b, 0xc60c, 0xd68d, 0xe70e, 0xf78f
};

void InitCRC( unsigned int *CRC )
{
    *CRC = 0xffff;
}

/* Residue = 0xf0b8 */
void CalcCRC( unsigned int *CRC, int Ch )
{
    int idx;
    idx = ( ( *CRC ^ ( Ch >> 0 ) ) & 0x000f );
    *CRC = ( ( *CRC >> 4 ) & 0x0fff ) ^ CRCTable[ idx ];
    idx = ( ( *CRC ^ ( Ch >> 4 ) ) & 0x000f );
    *CRC = ( ( *CRC >> 4 ) & 0x0fff ) ^ CRCTable[ idx ];
}

void PreSendCRC( unsigned int *CRC )
{
    *CRC = ~*CRC;
}

/* Assume CRC has been calculated on received data AND CRC. Return 0 if
   invalid CRC, non-zero otherwise */
int ValidCRCData( unsigned int *CRC  )
{
    return ( *CRC == 0xf0b8 );
}

/**************
 *            *
 * PACKET I/O *
 *            *
 **************/

#define RECEIVE_PACKET_BYTE     \
  cnt = CurParms.packettimeout; \
  do                            \
  {                             \
      byte = ReceiveByte();     \
      if ( --cnt == 0 )         \
      {                         \
          i = 1;                \
          goto QuitReceive;     \
      }                         \
  } while ( byte == -1 );

/* Receive packet in connectionles (best-try) mode
   Returns 0 if OK, 1 on timeout, 2 on error
   pkt->data must be set to a buffer of at least pkt->length bytes before
   calling. The remaining fields are filled in by the function, and will be
   valid only if the function returns 0. pkt->length is set to the length of
   the received packet, however, there will only a maximum of maxbuflen bytes
   actually in the buffer, where maxbuflen was the orignal setting of
   pkt->length.
   NOTE: It is VERY important to correctly set pkt->length and pkt->data
   before calling. */
int ReceiveConnectionlessPacket( struct PacketInfo *pkt )
{
    unsigned char SYN = 0x16;
    unsigned char SOH = 0x01;
    unsigned char EOT = 0x04;
    unsigned int CRC;
    int maxbuflen, byte, i;
    unsigned long cnt;

    /* Disable interrupts */
    /* I noticed during testing that sometimes up to 20 bytes would be 'missing'
       during a transmission from the Apple. Suspecting that interrupts on the
       PC were causing this loss, I disabled interrupts, and it worked!
       Unfortunately, disabling interrupts caused the system clock to lose time,
       so I fixed it by writing a small routine to read the CMOS RTC and set the
       DOS clock from that. Now the only problem is that it will only work on
       systems with a CMOS RTC (286 or better). Just can't win. */
    disable();
    /* Init */
    maxbuflen = pkt->length;
    InitCRC( &CRC );
    /* Header - receive SYN and SOH in sequence */
    do
    {
        /* Byte 0 - SYN character */
        RECEIVE_PACKET_BYTE
        if ( byte == SYN )
        {
            /* Byte 1 - SOH character */
            RECEIVE_PACKET_BYTE
        }
        else
        {
            byte = -1;
        }
    } while ( byte != SOH );
    /* Byte 2 - sequence, type, and bits 11-8 of length */
    RECEIVE_PACKET_BYTE
    CalcCRC( &CRC, byte );
    pkt->seq    = ( byte & 0x80 ) ? 1 : 0;
    pkt->type   = ( byte & 0x40 ) ? 1 : 0;
    pkt->length = ( byte & 0x0f ) << 8;
    /* Byte 3 - bits 7-0 of length */
    RECEIVE_PACKET_BYTE
    CalcCRC( &CRC, byte );
    pkt->length |= ( byte & 0xff );
    /* Correct a length of 0x000 to 0x1000 */
    if ( pkt->length == 0x000 )
    {
        pkt->length = 0x1000;
    }
    /* Data bytes */
    for ( i = 0; i < pkt->length; i++ )
    {
        RECEIVE_PACKET_BYTE
        CalcCRC( &CRC, byte );
        if ( i < maxbuflen )
        {
            pkt->data[ i ] = byte;
        }
    }
    /* CRC */
    RECEIVE_PACKET_BYTE
    CalcCRC( &CRC, byte );
    RECEIVE_PACKET_BYTE
    CalcCRC( &CRC, byte );
    /* Check CRC */
    i = 0;
    if ( !ValidCRCData( &CRC ) )
    {
        i = 2;
    }
    /* Check trailer */
    RECEIVE_PACKET_BYTE
    if ( byte != EOT )
    {
        i = 2; /* no trailer - data error */
    }

    QuitReceive:
    /* Re-enable interrupts and fix system clock */
    enable();
    FixClock();
    /* if bad packet, restore original packet length (buffer size) */
    if ( i != 0 )
    {
        pkt->length = maxbuflen;
    }
    return i;
}

/* Send packet in connectionless (best-try) mode */
/* all pkt members should be appropriately filled in before calling */
void SendConnectionlessPacket( struct PacketInfo *pkt )
{
    unsigned char SYN = 0x16;
    unsigned char SOH = 0x01;
    unsigned char EOT = 0x04;
    unsigned int CRC;
    int byte, i;

    InitCRC( &CRC );
    /* Byte 0 - SYN character */
    SendByte( SYN );
    /* Byte 1 - SOH character */
    SendByte( SOH );
    /* Byte 2 - sequence, type, and bits 11-8 of length */
    byte  = ( pkt->seq  ) ? 0x80 : 0x00;
    byte |= ( pkt->type ) ? 0x40 : 0x00;
    byte |= ( pkt->length >> 8 ) & 0x0f;
    CalcCRC( &CRC, byte );
    SendByte( byte );
    /* Byte 3 - bits 7-0 of length */
    byte = pkt->length & 0xff;
    CalcCRC( &CRC, byte );
    SendByte( byte );
    /* Data bytes */
    for ( i = 0; i < pkt->length; i++ )
    {
        byte = pkt->data[ i ];
        CalcCRC( &CRC, byte );
        SendByte( byte );
    }
    /* CRC */
    PreSendCRC( &CRC );
    byte = CRC & 0xff;
    SendByte( byte );
    byte = ( CRC >> 8 ) & 0xff;
    SendByte( byte );
    /* Trailer */
    SendByte( EOT );
}

/*******************
 *                 *
 * PACKET PROTOCOL *
 *                 *
 *******************/

void SendACKNAK( unsigned char ACKChar, int seq )
{
    struct PacketInfo ACKPkt;

    ACKPkt.type   = 0;    /* Supervisory */
    ACKPkt.seq    = seq;
    ACKPkt.length = 1;    /* Single byte */
    ACKPkt.data   = &ACKChar; /* NAK byte */
    SendConnectionlessPacket( &ACKPkt );
}

#define SendACK( seq ) SendACKNAK( 0x06, seq )
#define SendNAK( seq ) SendACKNAK( 0x15, seq )

/* Receive sequenced packet */
/* Returns 0 OK or 1 on timeout */
int ReceivePacket( struct PacketInfo *pkt )
{
    static int CurRecSeq = 0;
    int origbufsize;
    int err;

    /* keep original packet length (buffer size) */
    origbufsize = pkt->length;
    do
    {
      /* while packets are received with CRC errors, NAK them */
      pkt->length = origbufsize;
      while ( ( err = ReceiveConnectionlessPacket( pkt ) ) == 2 )
      {
          SendNAK( CurRecSeq );
      }

      if ( err == 1 )
      {
          return 1; /* timeout */
      }

      /* err must be 0 - received a good packet */

      /* if sequence sync packet - set new sequence */
      if ( pkt->type == 0 && *pkt->data == 0x16 )
      {
          CurRecSeq = pkt->seq;
          SendACK( CurRecSeq ); /* ACK received packet */
          CurRecSeq ^= 1; /* next sequence # */
      }
      else
      {
          /* if wrong sequence must be a duplicate - send ACK */
          if ( pkt->seq != CurRecSeq )
          {
              SendACK( pkt->seq );
          }
      }
    } while ( pkt->seq != CurRecSeq );

    SendACK( CurRecSeq ); /* ACK received packet */
    CurRecSeq ^= 1; /* next sequence # */
    return 0;
}

/* Send sequenced packet */
/* Returns 0 OK, 1 timeout */
int SendPacket( struct PacketInfo *pkt )
{
    static int CurSndSeq = 0;
    unsigned char ACK;
    struct PacketInfo ACKPkt;
    int cnt = CurParms.maxpacketsends;
    int err;

    /* if sequence sync packet - set new sequence */
    if ( pkt->type == 0 && *pkt->data == 0x16 )
    {
        CurSndSeq = pkt->seq;
    }

    /* Keep on sending packet until timeout or ACK received */
    if ( cnt < 1 )
    {
        cnt = 1; /* try at least once */
    }
    pkt->seq = CurSndSeq;
    while  (--cnt >= 0 )
    {
        /* send the packet */
        SendConnectionlessPacket( pkt );
        /* wait for acknowledgement */
        ACKPkt.length = 1;
        ACKPkt.data   = &ACK;
        err = ReceiveConnectionlessPacket( &ACKPkt );
        /* check acknowledgement for sequence # */
        if ( ( ACKPkt.seq != CurSndSeq ) || /* different sequence */
             ( ACKPkt.type != 0 ) ||        /* not supervisory */
             ( *ACKPkt.data != 0x06 ) )     /* not an ACK */
        {
            err = 1;
        }
        if ( err == 0 )
        {
            cnt = 0; /* ACK'd - exit immediately */
        }
    }
    if ( err == 0 )
    {
        /* no errors - good block */
        CurSndSeq ^= 1; /* next sequence # */
        return 0;
    }
    return 1;
}

int SynchroniseSequences( void )
{
  unsigned char SYN = 0x16;
  struct PacketInfo pkt;

    pkt.type = 0; /* supervisory */
    pkt.seq  = 0; /* synchronise to sequence 0 */
    pkt.length = 1; /* one byte */
    pkt.data = &SYN;
    return SendPacket( &pkt );
}
