Procedure FreeSymbolTable;
Begin
  If Ok(SymbolTable) then VirtualFree(SymbolTable,0,MEM_RELEASE);
  SymbolNum   := 0;
  SymbolTable := Nil
End;

Procedure DebugInitialize;
Var Loop        : Integer;
    Filename    : Array[0..Pred(MAX_PATH)] of Char;
    SymbolAlloc : Integer;
    InFile      : pFile;
    LastValue   : Word;
    Value       : DWord;
    Name        : Array[0..13]  of Char;
    Line        : Array[0..255] of Char;
    NewTable    : SymbolPtr;
Begin
  // CLEAR THE BREAKPOINT AND WATCH TABLES
  FillChar(Breakpoint,BREAKPOINTS * SizeOf(BpRec),#0);
  Loop := 0;
  While Loop < WATCHES do Begin
    Watch[Loop] := -1;
    Inc(Loop)
  End;
  // READ IN THE SYMBOL TABLE
  StrCopy(Filename,ProgDir);
  StrCat(Filename,'APPLE2E.SYM');
  SymbolAlloc := 0;
  InFile      := fOpen(Filename,'rt');
  LastValue   := 0;
  If Ok(InFile) then Begin
    While not Ok(fEof(Infile)) do Begin
      // READ IN THE NEXT LINE, AND MAKE SURE IT IS SORTED CORRECTLY IN
      // VALUE ORDER
      Value     := 0;
      Name      := '';
      fScanF(InFile,'%x %13s',@Value,Name);
      fGets(Line,255,InFile);
      If Ok(Value) then Begin
        If Value < LastValue then Begin
          MessageBox(GetDesktopWindow(),'The symbol file is not sorted correctly.  Symbols will not be loaded.',
                     TITLE,MB_ICONEXCLAMATION OR MB_SETFOREGROUND);
          FreeSymbolTable();
          Exit
        End Else Begin
          // IF OUR CURRENT SYMBOL TABLE IS NOT BIG ENOUGH TO HOLD THIS
          // ADDITIONAL SYMBOL, THEN ALLOCATE A BIGGER TABLE AND COPY THE
          // CURRENT DATA ACROSS
          If not Ok(SymbolTable) or (SymbolAlloc <= SymbolNum) then Begin
            Inc(SymbolAlloc,8192 Div SizeOf(SymbolRec));
            Newtable := SymbolPtr(VirtualAlloc(Nil,SymbolAlloc * SizeOf(SymbolRec),MEM_COMMIT,PAGE_READWRITE));
            If Ok(NewTable) then Begin
              If Ok(SymbolTable) then Begin
                CopyMemory(NewTable,SymbolTable,SymbolNum * SizeOf(SymbolRec));
                VirtualFree(SymbolTable,0,MEM_RELEASE)
              End;
              SymbolTable := NewTable
            End Else Begin
              MessageBox(GetDesktopWindow(),'There is not enough memory available to load '+
                                            'the symbol file.',
                         TITLE,MB_ICONEXCLAMATION OR MB_SETFOREGROUND);
              FreeSymbolTable()
            End
          End;
          // SAVE THE NEW SYMBOL IN THE SYMBOL TABLE
          If Ok(SymbolTable) then Begin
            SymbolPtr(DWord(SymbolTable) + DWord(SymbolNum * SizeOf(SymbolRec)))^.Value := Word(Value AND $FFFF);
            StrLCopy(SymbolPtr(DWord(SymbolTable) + DWord(SymbolNum * SizeOf(SymbolRec)))^.Name,Name,12);
            SymbolPtr(DWord(SymbolTable) + DWord(SymbolNum * SizeOf(SymbolRec)))^.Name[13] := #0;
            Inc(SymbolNum)
          End;
          LastValue := Word(Value)
        End
      End
    End;
    fClose(InFile)
  End;
  // CREATE A FONT FOR THE DEBUGGING SCREEN
  DebugFont := CreateFont(15,0,0,0,FW_MEDIUM,0,0,0,OEM_CHARSET,OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,
                          DEFAULT_QUALITY,FIXED_PITCH OR 4 OR FF_MODERN,'Courier New')
End;

Function GetSymbol( Address : Word; Bytes : Integer ) : LPTSTR;
Var LowLimit  : Integer;
    HighLimit : Integer;
    Curr      : Integer;
    Diff      : Integer;
Begin
  // PERFORM A BINARY SEARCH THROUGH THE SYMBOL TABLE LOOKING FOR A VALUE
  // MATCHING THIS ADDRESS
  LowLimit  := -1;
  HighLimit := SymbolNum;
  Curr      := Symbolnum Shr 1;
  Repeat
    Diff := Integer(Address) - Integer(LPWORD(DWord(@SymbolTable.Value) + Curr * SizeOf(SymbolRec))^);
    If Diff < 0 then Begin
      HighLimit := Curr;
      Curr := LowLimit + ((Curr - Lowlimit) Shr 1)
    End Else If Diff > 0 then Begin
      LowLimit := Curr;
      Curr := Curr + ((HighLimit + 1 - Curr) Shr 1)
    End Else Begin
      Result := LPTSTR(DWord(@SymbolTable.Name) + Curr * SizeOf(SymbolRec));
      Exit
    End
  Until not ((Curr > LowLimit) and (Curr < HighLimit));
  // IF THERE IS NO SYMBOL FOR THIS ADDRESS, THEN JUST RETURN A STRING
  // CONTAINING THE ADDRESS NUMBER
  Case Bytes of
    2 : StrFmt(Sym_Buffer,'$%.2X',[Address]);
    3 : StrFmt(Sym_Buffer,'$%.4X',[Address])
    Else Sym_Buffer[0] := #0
  End;
  Result := LPTSTR(@Sym_Buffer)
End;

Function InternalSingleStep : Bool;
Var _Result : Bool;
Begin
  _Result := false;
  Try
    Inc(ProfileData[LPBYTE(DWord(Mem) + Regs.PC)^]);
    CpuExecute(DWORD(Abs(Ord(StepLine))));
    DWORD(_Result) := 1
  Except                      // (EXCEPTION_EXECUTE_HANDLER) berprfen
    _result := false
  End;
  Result := _Result
End;

Function CheckJump( TargetAddress : Word ) : Bool;
Var SavedPC : Word;
    _Result : Bool;
Begin
  SavedPC := Regs.PC;
  InternalSingleStep();
  DWORD(_Result) := Abs(Ord(Regs.PC = TargetAddress));
  Regs.PC := SavedPC;
  DWORD(Result) := Abs(Ord(_Result))
End;

Procedure GetTargets( Intermediate,Final : LPINT );
Var AddrMode   : Integer;
    Argument8  : Byte;
    Argument16 : Word;
Begin
  Intermediate^ := -1;
  Final^        := -1;
  Addrmode   := Instruction[LPBYTE(DWord(Mem) + Regs.PC)^].AddrMode;
  Argument8  := LPBYTE(DWord(Mem) + Succ(Regs.PC))^;
  Argument16 := LPWORD(DWord(Mem) + Succ(Regs.PC))^;
  Case Addrmode of
    ADDR_ABS      : Final^ := Argument16;
    ADDR_ABSIINDX : Begin
                      Inc(Argument16,Regs.X);
                      Intermediate^ := Argument16;
                      Final^        := LPWORD(DWord(Mem) + Intermediate^)^
                    End;
    ADDR_ABSX     : Begin
                      Inc(Argument16,Regs.X);
                      Final^ := Argument16
                    End;
    ADDR_ABSY     : Begin
                      Inc(Argument16,Regs.Y);
                      Final^ := Argument16
                    End;
    ADDR_IABS     : Begin
                      Intermediate^ := Argument16;
                      Final^        := LPWORD(DWord(Mem) + Intermediate^)^
                    End;
    ADDR_INDX     : Begin
                      Inc(Argument8,Regs.X);
                      Intermediate^ := Argument8;
                      Final^        := LPWORD(DWord(Mem) + Intermediate^)^
                    End;
    ADDR_INDY     : Begin
                      Intermediate^ := Argument8;
                      Final^        := LPWORD(DWord(Mem) + Intermediate^)^ + Regs.Y
                    End;
    ADDR_IZPG     : Begin
                      Intermediate^ := Argument8;
                      Final^        := LPWORD(DWord(Mem) + Intermediate^)^
                    End;
    ADDR_ZPG      : Final^ := Argument8;
    ADDR_ZPGX     : Final^ := Argument8 + Regs.X;
    ADDR_ZPGY     : Final^ := Argument8 + Regs.Y
  End;
  If (Final^ >= 0) and (not Ok(StrComp(Instruction[LPBYTE(DWord(Mem) + Regs.PC)^].Mnemonic,'JMP')) or
                        not Ok(StrComp(Instruction[LPBYTE(DWord(Mem) + Regs.PC)^].Mnemonic,'JSR'))) then Final^ := -1
End;

Function CheckBreakpoint( Address : Word; Memory : Bool ) : Bool;
Var Target     : Array[0..2] of Integer;
    TargetNum  : Integer;
    TargetAddr : Word;
    Slot       : Integer;
Begin
  Target[0] := Address;
  Target[1] := -1;
  Target[2] := -1;
  If Ok(Memory) then GetTargets(@Target[1],@Target[2]);
  If Ok(Memory) then TargetNum := 3 Else TargetNum := 1;
  While Ok(TargetNum) do Begin
    Dec(TargetNum);
    If Target[TargetNum] >= 0 then Begin
      TargetAddr := WORD(Target[TargetNum] AND $FFFF);
      Slot       := 0;
      While Slot < BREAKPOINTS do Begin
        If Ok(Breakpoint[Slot].Length) and Ok(Breakpoint[Slot].Enabled) and (Breakpoint[Slot].Address <= TargetAddr)
                                       and (Breakpoint[Slot].Address + Breakpoint[Slot].Length > TargetAddr) then Begin
          DWORD(Result) := 1;
          Exit
        End;
        Inc(Slot)
      End
    End
  End;
  Result := false
End;

Function DrawDisassembly( DC : hDC; Line : Integer; Offset : Word; Text : LPTSTR ) : Word;
Var AddressText : Array[0..39] of Char;
    BytesText   : Array[0..9]  of Char;
    FullText    : Array[0..49] of Char;
    Inst        : Byte;
    AddrMode    : Integer;
    Bytes       : Word;
    Address     : Word;
    Loop        : Integer;
    LineRect    : tRect;
    BP          : Bool;
Begin
  AddressText := '';
  BytesText   := '';
  FullText    := '';
  Inst        := LPBYTE(DWord(Mem) + Offset)^;
  Addrmode    := Instruction[Inst].AddrMode;
  Bytes       := AddressMode[AddrMode].Bytes;
  // BUILD A STRING CONTAINING THE TARGET ADDRESS OR SYMBOL
  If Ok(AddressMode[AddrMode].Format[0]) then Begin
    Address := LPWORD(DWord(Mem) + Succ(Offset))^;
    If Bytes = 2 then Address := Address AND $FF;
    If AddrMode = ADDR_REL then Address := Offset + 2 + ShortInt(Address);
    If Ok(StrPos(AddressMode[AddrMode].Format,'%s')) then Begin
      StrFmt(AddressText,AddressMode[AddrMode].Format,[GetSymbol(Address,Bytes)])
    End Else Begin
      StrFmt(AddressText,AddressMode[AddrMode].Format,[Address])
    End;
    If (AddrMode = ADDR_REL) and (Offset = Regs.PC) and CheckJump(Address) then Begin
      If Address > Offset then StrCat(AddressText,' '#$19)
                          Else StrCat(AddressText,' '#$18)
    End
  End;
  // BUILD A STRING CONTAINING THE ACTUAL BYTES THAT MAKE UP THIS
  // INSTRUCTION
  Loop := 0;
  While Loop < Bytes do Begin
    StrFmt(LPTSTR(@BytesText[StrLen(BytesText)]),'%.2X',[LPBYTE(DWord(Mem) + Offset + Loop)^]);
    Inc(Loop)
  End;
  While StrLen(BytesText) < 6 do StrCat(BytesText,' ');
  // PUT TOGETHER ALL OF THE DIFFERENT ELEMENTS THAT WILL MAKE UP THE LINE
  StrFmt(Fulltext,'%.4X  %s  %-9s %s %s',[Offset,BytesText,GetSymbol(Offset,0),Instruction[Inst].Mnemonic,AddressText]);
  If Ok(Text) then StrCopy(Text,Fulltext);
  // DRAW THE LINE
  If Ok(DC) then Begin
    LineRect.Left   := 0;
    LineRect.Top    := Line Shl 4;
    LineRect.Right  := SCREENSPLIT1 - 14;
    LineRect.Bottom := LineRect.Top + 16;
    DWORD(BP) := Abs(Ord(UsingBP and CheckBreakpoint(Offset,Offset = Regs.PC)));
    If Offset = Regs.PC then SetTextColor(DC,Color[ColorScheme][COLOR_INSTBKG])
                        Else If Ok(BP) then SetTextColor(DC,Color[ColorScheme][COLOR_INSTBP])
                                       Else SetTextColor(DC,Color[ColorScheme][COLOR_INSTTEXT]);
    If Offset = Regs.PC then If Ok(BP) then SetBkColor(DC,Color[ColorScheme][COLOR_INSTBP])
                                       Else SetBkColor(DC,Color[ColorScheme][COLOR_INSTTEXT])
                        Else SetBkColor(DC,Color[ColorScheme][COLOR_INSTBKG]);
    ExtTextOut(DC,12,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@Linerect,Fulltext,StrLen(Fulltext),Nil)
  End;
  Result := Bytes
End;

Procedure DrawFlags( DC : hDC; Line : Integer; Value : Word; Text : LPTSTR );
Var Mnemonic : Array[0..8] of Char;
    Fulltext : Array[0..1] of Char;
    LineRect : tRect;
    Loop     : Integer;
Begin
  Mnemonic := 'NVRBDIZC';
  Fulltext := '?';
  If Ok(DC) then Begin
    LineRect.Left   := SCREENSPLIT1 + 63;
    LineRect.Top    := Line Shl 4;
    LineRect.Right  := SCREENSPLIT1 + 72;
    LineRect.Bottom := LineRect.Top + 16;
    SetBkColor(DC,Color[ColorScheme][COLOR_DATABKG])
  End;
  Loop := 8;
  While Ok(Loop) do Begin
    Dec(Loop);
    If Ok(DC) then Begin
      Fulltext[0] := Mnemonic[Loop];
      If Ok(Value AND 1) then SetTextColor(DC,Color[ColorScheme][COLOR_DATATEXT])
                         Else SetTextColor(DC,Color[ColorScheme][COLOR_STATIC]);
      ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@Linerect,Fulltext,1,Nil);
      Dec(LineRect.Left,9);
      Dec(LineRect.Right,9)
    End;
    If not Ok(Value AND 1) then Mnemonic[Loop] := '.';
    Value := Value Shr 1
  End;
  If Ok(Text) then StrCopy(Text,Mnemonic)
End;

Procedure OutputTraceLine;
Var DisAssembly : Array[0..49] of Char;
    Flags       : Array[0..8]  of Char;
Begin
  DrawDisassembly(hDC(0),0,Regs.PC,Disassembly);
  DrawFlags(hDC(0),0,Regs.PS,Flags);
  fPrintF(TraceFile,'a=%02x x=%02x y=%02x sp=%03x ps=%s   %s'#0,
          Regs.A,Regs.X,Regs.Y,Regs.SP,LPCTSTR(@Flags),LPCTSTR(@Disassembly))
End;

Procedure ComputeTopOffset( CenterOffset : Word );
Var Invalid  : Bool;
    InstOfs  : Array[0..$2F] of Word;
    CurrOfs  : Word;
    CurrNum  : Integer;
    AddrMode : Integer;
Begin
  TopOffset := CenterOffset - $30;
  Repeat
    Invalid := false;
    CurrOfs := TopOffset;
    CurrNum := 0;
    Repeat
      AddrMode := Instruction[LPBYTE(DWORD(Mem) + CurrOfs)^].Addrmode;
      If (AddrMode >= 1) and (AddrMode <= 3) then Begin
        DWORD(Invalid) := 1
      End Else Begin
        InstOfs[CurrNum] := CurrOfs;
        Inc(CurrNum);
        Inc(CurrOfs,AddressMode[AddrMode].Bytes)
      End
    Until not (not Invalid and (CurrOfs < CenterOffset));
    If Invalid then Inc(TopOffset) Else If CurrNum > (SOURCELINES Shr 1) then Begin
      TopOffset := InstOfs[CurrNum - (SOURCELINES Shr 1)]
    End
  Until not Invalid
End;

Procedure DrawStack( DC : hDC; Line : Integer );
Var CurrAddr : Word;
    Loop     : Integer;
    LineRect : tRect;
    OutText  : Array[0..7] of Char;
Begin
  CurrAddr := Regs.sp;
  Loop     := 0;
  While Loop < STACKLINES do Begin
    Inc(CurrAddr);
    LineRect.Left   := SCREENSPLIT1;
    LineRect.Top    := (Loop + Line) Shl 4;
    LineRect.Right  := SCREENSPLIT1 + 40;
    LineRect.Bottom := LineRect.Top + 16;
    SetTextColor(DC,Color[ColorScheme][COLOR_STATIC]);
    SetBkColor(DC,Color[Colorscheme][COLOR_DATABKG]);
    OutText := '';
    If CurrAddr <= $1FF then Begin
      StrFmt(OutText,'%.4X',[CurrAddr]);
    End;
    ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,OutText,StrLen(OutText),Nil);
    LineRect.Left   := SCREENSPLIT1 + 40;
    LineRect.Right  := SCREENSPLIT2;
    SetTextColor(DC,Color[ColorScheme][COLOR_DATATEXT]);
    If CurrAddr <= $1FF then Begin
      StrFmt(OutText,'%.2X',[LPBYTE(DWord(Mem) + CurrAddr)^]);
    End;
    ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,OutText,StrLen(OutText),Nil);
    Inc(Loop)
  End
End;

Procedure DrawTargets( DC : hDC; Line : Integer );
Var Address    : Array[0..1] of Integer;
    Loop       : Integer;
    AddressStr : Array[0..7] of Char;
    ValueStr   : Array[0..7] of Char;
    LineRect   : tRect;
Begin
  GetTargets(@Address[0],@Address[1]);
  Loop := 2;
  While Ok(Loop) do Begin
    Dec(Loop);
    If (Address[Loop] >= $C000) and (Address[Loop] <= $C0FF) then Address[Loop] := -1;
    AddressStr := '';
    ValueStr   := '';
    If Address[Loop] >= 0 then Begin
      StrFmt(AddressStr,'%.4X',[Address[Loop]]);
      If Ok(Loop) then StrFmt(ValueStr,'%.2X',[LPBYTE(DWord(Mem) + Address[Loop])^])
                  Else StrFmt(ValueStr,'%.4X',[LPWORD(DWord(Mem) + Address[Loop])^])
    End;
    LineRect.Left   := SCREENSPLIT1;
    LineRect.Top    := (Line + Loop) Shl 4;
    LineRect.Right  := SCREENSPLIT1 + 40;
    LineRect.Bottom := LineRect.Top + 16;
    SetTextColor(DC,Color[ColorScheme][COLOR_STATIC]);
    SetBkColor(DC,Color[ColorScheme][COLOR_DATABKG]);
    ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,AddressStr,StrLen(AddressStr),Nil);
    LineRect.Left  := SCREENSPLIT1 + 40;
    LineRect.Right := SCREENSPLIT2;
    SetTextColor(DC,Color[ColorScheme][COLOR_DATATEXT]);
    ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,ValueStr,StrLen(ValueStr),Nil)
  End
End;

Procedure DrawRegister( DC : hDC; Line : Integer; Name : PChar; Bytes : Integer; Value : Word );
Var LineRect : TRect;
    ValueStr : Array[0..7] of Char;
Begin
  LineRect.Left   := SCREENSPLIT1;
  LineRect.Top    := Line Shl 4;
  LineRect.Right  := SCREENSPLIT1 + 40;
  LineRect.Bottom := LineRect.Top + 16;
  SetTextColor(DC,Color[ColorScheme][COLOR_STATIC]);
  SetBkColor(DC,Color[ColorScheme][COLOR_DATABKG]);
  ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,Name,StrLen(Name),Nil);
  If Bytes = 2 then Begin
    StrFmt(ValueStr,'%.4X',[Value]);
    LineRect.Left  := SCREENSPLIT1 + 40
  End Else Begin
    StrFmt(ValueStr,'%.2X',[Value]);
    LineRect.Left  := SCREENSPLIT1 + 54
  End;
  LineRect.Right := SCREENSPLIT2;
  SetTextColor(DC,Color[ColorScheme][COLOR_DATATEXT]);
  ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,ValueStr,StrLen(ValueStr),Nil)
End;

Procedure DrawBreakpoints( DC : hDC; Line : Integer );
Var LineRect : tRect;
    Fulltext : Array[0..15] of Char;
    Loop     : Integer;
Begin
  LineRect.Left   := SCREENSPLIT2;
  LineRect.Top    := Line Shl 4;
  LineRect.Right  := 560;
  LineRect.Bottom := LineRect.Top + 16;
  Fulltext := 'Breakpoints';
  SetTextColor(DC,Color[ColorScheme][COLOR_STATIC]);
  SetBkColor(DC,Color[ColorScheme][COLOR_DATABKG]);
  Loop := 0;
  Repeat
    ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,Fulltext,StrLen(Fulltext),Nil);
    Inc(LineRect.Top,16);
    Inc(LineRect.Bottom,16);
    If (Loop < BREAKPOINTS) and Ok(Breakpoint[Loop].Length) then Begin
      StrFmt(Fulltext,'%d: %.4X',[Succ(Loop),Breakpoint[Loop].Address]);
      If Breakpoint[Loop].Length > 1 then
        StrFmt(Fulltext + StrLen(Fulltext),'-%.4X',[Breakpoint[Loop].Address + Breakpoint[Loop].Length - 1]);
      If Breakpoint[Loop].Enabled then SetTextColor(DC,Color[ColorScheme][COLOR_BPDATA])
                                  Else SetTextColor(DC,Color[ColorScheme][COLOR_STATIC])
    End Else Fulltext[0] := #0;
    Inc(Loop)
  Until Loop > BREAKPOINTS
End;

Procedure DrawWatches( DC : hDC; Line : Integer );
Var LineRect : tRect;
    OutStr   : Array[0..15] of Char;
    Loop     : Integer;
Begin
  LineRect.Left   := SCREENSPLIT2;
  LineRect.Top    := (Line Shl 4);
  LineRect.Right  := 560;
  LineRect.Bottom := LineRect.Top + 16;
  Outstr := 'Watches';
  SetTextColor(DC,Color[ColorScheme][COLOR_STATIC]);
  SetBkColor(DC,Color[ColorScheme][COLOR_DATABKG]);
  ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@Linerect,OutStr,StrLen(OutStr),Nil);
  LineRect.Right := SCREENSPLIT2 + 64;
  Loop := 0;
  While Loop < WATCHES do Begin
    If Watch[Loop] >= 0 then StrFmt(OutStr,'%d: %.4X',[Succ(Loop),Watch[Loop]])
                        Else OutStr[0] := #0;
    Inc(LineRect.Top,16);
    Inc(LineRect.Bottom,16);
    ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@Linerect,OutStr,StrLen(OutStr),Nil);
    Inc(Loop)
  End;
  LineRect.Left   := SCREENSPLIT2 + 64;
  LineRect.Top    := (Line Shl 4);
  LineRect.Right  := 560;
  LineRect.Bottom := LineRect.Top + 16;
  SetTextColor(DC,Color[ColorScheme][COLOR_DATATEXT]);
  Loop := 0;
  While Loop < WATCHES do Begin
    If Watch[Loop] >= 0 then StrFmt(OutStr,'%.2X',[LPBYTE(DWord(Mem) + Watch[Loop])^])
                        Else OutStr[0] := #0;
    Inc(LineRect.Top,16);
    Inc(LineRect.Bottom,16);
    ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@Linerect,OutStr,StrLen(OutStr),Nil);
    Inc(Loop)
  End
End;

Procedure DrawMemory( DC : hDC; Line : Integer );
Var LineRect : tRect;
    Fulltext : Array[0..15] of Char;
    CurrAddr : Word;
    Loop     : Integer;
    Loop2    : Integer;
Begin
  LineRect.Left   := SCREENSPLIT2;
  LineRect.Top    := (Line Shl 4);
  LineRect.Right  := 560;
  LineRect.Bottom := LineRect.Top + 16;
  StrFmt(Fulltext,'Mem at %.4X',[MemoryDump]);
  SetTextColor(DC,Color[ColorScheme][COLOR_STATIC]);
  SetBkColor(DC,Color[ColorScheme][COLOR_DATABKG]);
  CurrAddr := MemoryDump;
  Loop     := 0;
  Repeat
    ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@Linerect,Fulltext,StrLen(Fulltext),Nil);
    Inc(LineRect.Top,16);
    Inc(LineRect.Bottom,16);
    Fulltext[0] := #0;
    If Loop < 4 then For Loop2 := 0 to 3 do Begin
      If (CurrAddr >= $C000) and (CurrAddr <= $C0FF) then Begin
        StrCopy(Fulltext + StrLen(Fulltext),'IO ')
      End Else Begin
        StrFmt(Fulltext + StrLen(Fulltext),'%.2X ',[LPBYTE(DWord(MemBank) + CurrAddr)^])
      End;
      Inc(CurrAddr)
    End;
    SetTextColor(dc,color[colorscheme][COLOR_DATATEXT]);
    Inc(Loop)
  Until Loop > 4
End;

{$O-}
Procedure DrawCommandLine( DC : hDC; Line : Integer );
Var LineRect : tRect;
Var Title    : Boolean;
    Text     : LPTSTR;
Begin
  Title := CommandString[Line][0] = ' ';
  SetTextColor(DC,Color[ColorScheme][COLOR_COMMAND]);
  SetBkColor(DC,0);
  LineRect.Left   := 0;
  LineRect.Top    := 368 - (Line Shl 4);
  LineRect.Right  := 12;
  LineRect.Bottom := LineRect.Top + 16;
  If not Title then Begin
    ExtTextOut(DC,1,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,'>',1,Nil);
    LineRect.Left := 12
  End;
  LineRect.Right := 560;
  Text := LPTSTR(@CommandString[Line][Ord(Title)]);
  ExtTextOut(DC,LineRect.Left,LineRect.Top,ETO_CLIPPED OR ETO_OPAQUE,@LineRect,Text,Length(Text),Nil)
End;

Procedure DebugDisplay( DrawBackground : Bool );
Var DC           : hDC;
    ViewportRect : tRect;
    Line         : Integer;
    Offset       : Word;
Begin
  DC := FrameGetDC();
  SelectObject(DC,DebugFont);
  SetTextAlign(DC,TA_TOP OR TA_LEFT);
  // DRAW THE BACKGROUND
  If DrawBackground then Begin
    ViewportRect.Left   := 0;
    ViewportRect.Top    := 0;
    ViewportRect.Right  := SCREENSPLIT1 - 14;
    ViewportRect.Bottom := 304;
    SetBkColor(dc,color[colorscheme][COLOR_INSTBKG]);
    ExtTextOut(DC,0,0,ETO_OPAQUE,@ViewportRect,'',0,Nil);
    ViewportRect.Left   := SCREENSPLIT1 - 14;
    ViewportRect.Right  := 560;
    SetBkColor(DC,Color[ColorScheme][COLOR_DATABKG]);
    ExtTextOut(DC,0,0,ETO_OPAQUE,@ViewportRect,'',0,Nil)
  End;
  // DRAW DISASSEMBLED LINES
  Line   := 0;
  Offset := TopOffset;
  While Line < SOURCELINES do Begin
    Inc(Offset,DrawDisassembly(DC,Line,Offset,Nil));
    Inc(Line)
  End;
  // DRAW THE DATA AREA
  DrawStack(DC,0);
  DrawTargets(DC,10);
  DrawRegister(DC,13,'A' ,1,Regs.a);
  DrawRegister(DC,14,'X' ,1,Regs.x);
  DrawRegister(DC,15,'Y' ,1,Regs.y);
  DrawRegister(DC,16,'PC',2,Regs.pc);
  DrawRegister(DC,17,'SP',2,Regs.sp);
  DrawFlags(DC,18,Regs.ps,Nil);
  If UsingBP      then DrawBreakpoints(DC,0);
  If UsingWatches then DrawWatches(DC,7);
  If UsingMemDump then DrawMemory(DC,14);
  // DRAW THE COMMAND LINE
  Line := COMMANDLINES;
  While Ok(Line) do Begin
    Dec(Line);
    DrawCommandLine(DC,Line)
  End;
  FrameReleaseDC()
End;

Procedure DebugContinueStepping;
Begin
  If Ok(StepCount) then Begin
    If Ok(Tracefile) then OutputTraceLine();
    LastPC := Regs.PC;
    InternalSingleStep();
    If (Regs.PC = StepUntil) or CheckBreakpoint(Regs.PC,true) then Begin
      StepCount := 0
    End Else If StepCount > 0 then Dec(StepCount)
  End;
  If Ok(StepCount) then Begin
    Inc(StepsTaken);
    If not Ok(StepsTaken AND $FFFF) then If StepsTaken = $10000 then VideoRedrawScreen()
                                                                Else VideoRefreshScreen()
  End Else Begin
    Mode := MODE_DEBUG;
    FrameRefreshStatus(DRAW_TITLE);
    If (StepStart < Regs.PC) and (StepStart + 3 >= Regs.PC) then
      Inc(TopOffset,AddressMode[Instruction[LPBYTE(DWord(Mem)+TopOffset)^].AddrMode].Bytes)
    Else
      ComputeTopOffset(Regs.PC);
    DebugDisplay(BOOL(DWORD(Abs(Ord(StepsTaken >= $10000)))));
    StepsTaken := 0
  End
End;

Procedure DebugProcessChar( Ch : Char );
Var DC     : hDC;
    Length : Integer;
Begin
  If (Mode = MODE_STEPPING) and (Ch = #$1B) then StepCount := 0;
  If Mode <> MODE_DEBUG then Exit;
  If (Ch = ' ') and not Ok(CommandString[0][0]) then Exit;
  If (Ch >= #32) and (Ch <= #126) then Begin
    Ch := UpCase(Ch);
    Length := StrLen(CommandString[0]);
    If Length < 68 then Begin
      CommandString[0][Length]       := Ch;
      CommandString[0][Succ(Length)] := #0
    End;
    DC := FrameGetDC();
    DrawCommandLine(DC,0);
    FrameReleaseDC()
  End
End;

Procedure WriteProfileData;
Var Filename : Array[0..Pred(MAX_PATH)] of Char;
    _File    : pFile;
    MaxValue : DWord;
    MaxItem  : DWord;
    Loop     : DWord;
Begin
  StrCopy(Filename,ProgDir);
  StrCat(Filename,'Profile.txt');
  _File := fOpen(Filename,'wt');
  If Ok(_File) then Begin
    Repeat
      MaxValue := 0;
      For Loop := 0 to 255 do If ProFileData[Loop] > MaxValue then Begin
        MaxValue := ProFileData[Loop];
        MaxItem  := Loop
      End;
      If Ok(MaxValue) then Begin
        fPrintF(_File,'%9u  %02X  %s'#0,MaxValue,MaxItem,LPCTSTR(@Instruction[MaxItem].Mnemonic));
        ProFileData[MaxItem] := 0
      End
    Until not Ok(MaxValue);
    fclose(_File)
  End
End;

Procedure DebugBegin;
Begin
  If CpuEmType = CPU_FASTPAGING then MemSetFastPaging(false);
  If not Ok(MemBank) then MemBank := Mem;
  Mode := MODE_DEBUG;
  FrameRefreshStatus(DRAW_TITLE);
  If Ok(Apple2e) then Begin
    AddressMode[INVALID2].Bytes := 2;
    AddressMode[INVALID3].Bytes := 3
  End Else Begin
    AddressMode[INVALID2].Bytes := 1;
    AddressMode[INVALID3].Bytes := 1
  End;
  ComputeTopOffset(Regs.PC);
  DebugDisplay(true)
End;

Procedure DebugEnd;
Begin
  If Ok(ProFiling) then WriteProfileData();
  If Ok(TraceFile) then Begin
    fclose(TraceFile);
    TraceFile := Nil
  End
End;

Procedure DebugDestroy;
Begin
  DebugEnd();
  DeleteObject(DebugFont);
  FreeSymbolTable()
End;

Function DisplayError( ErrorText : LPTSTR ) : Bool;
Begin
  Result := false
End;

Function ExecuteCommand( Args : Integer ) : Bool;
Var Name        : LPTSTR;
    Found       : Integer;
    _Function   : CmdFunction;
    Length      : Integer;
    Loop        : Integer;
Begin
  Name := StrTok(CommandString[0],' ,-=');
  If not Ok(Name) then Name := CommandString[0];
  Found     := 0;
  _Function := Nil;
  Length    := StrLen(Name);
  Loop      := 0;
  While (Loop < COMMANDS) and (Name[0] >= Command[Loop].Name[0]) do Begin
    If not Ok(StrLComp(Name,Command[Loop].Name,Length)) then Begin
      _Function := Command[Loop].Funct;
      If not Ok(StrComp(Name,Command[Loop].Name)) then Begin
        Found := 1;
        Loop  := COMMANDS
      End Else Inc(Found)
    End;
    Inc(Loop)
  End;
  If Found > 1 then Begin
    Result := DisplayError('Ambiguous command');
    Exit
   End Else If Ok(@_Function) then Begin
    Result := _Function(Args);
    Exit
  End Else Result := DisplayError('Illegal command')
End;

Function ParseCommandString : Integer;
Var Args    : Integer;
    CurrPtr : LPTSTR;
    EndPtr  : LPTSTR;
    Length  : Word;
    More    : Boolean;
    Loop    : Integer;
Begin
  Args    := 0;
  CurrPtr := CommandString[0];
  While Ok(CurrPtr^) do Begin
    EndPtr := Nil;
    Length := StrLen(CurrPtr);
    StrTok(CurrPtr,' ,-=');
    StrLCopy(Arg[Args].Str,CurrPtr,11);
    Arg[Args].Str[11] := #0;
    Arg[Args].Val1    := Word(StrToul(CurrPtr,@EndPtr,16) AND $FFFF);
    If Ok(EndPtr) then
      If EndPtr^ = 'L' then Begin
        Arg[Args].Val2 := Word(StrToul(LPTSTR(DWord(@EndPtr)+1),@EndPtr,16) AND $FFFF);
        If Ok(EndPtr) and Ok(EndPtr^) then Arg[Args].Val2 := 0
      End Else Begin
        Arg[Args].Val2 := 0;
        If Ok(EndPtr^) then Arg[Args].Val1 := 0
      End
    Else Arg[Args].Val2 := 0;
    More := Ok(CurrPtr^) and (Length > StrLen(CurrPtr));
    Inc(Args,Ord(More));
    Inc(CurrPtr,StrLen(CurrPtr) + Ord(More))
  End;
  Loop := Args;
  While Loop < Pred(MAXARGS - 1) do Begin
    Inc(Loop);
    Arg[Loop].Str[0] := #0;
    Arg[Loop].Val1   := 0;
    Arg[Loop].Val2   := 0
  End;
  Result := Args
End;

Function CmdLineUp( Args : Integer ) : Bool;
Var SavedOffset : Word;
    NewOffset   : Word;
Begin
  SavedOffset := TopOffset;
  ComputeTopOffset(TopOffset);
  NewOffset := TopOffset;
  While NewOffset < SavedOffset do Begin
    TopOffset := NewOffset;
    Inc(NewOffset,AddressMode[Instruction[LPBYTE(DWord(Mem) + NewOffset)^].AddrMode].Bytes);
  End;
  TopOffset := MIN(TopOffset,Pred(SavedOffset));
  DWORD(Result) := 1
End;

Function CmdLineDown( Args : Integer ) : Bool;
Begin
  Inc(TopOffset,AddressMode[Instruction[LPBYTE(DWord(Mem) + TopOffset)^].AddrMode].Bytes);
  DWORD(Result) := 1
End;

Function CmdPageUp( Args : Integer) : Bool;
Var Loop : Integer;
Begin
  Loop := 0;
  While Loop < SOURCELINES do Begin
    Inc(Loop);
    CmdLineUp(Args)
  End;
  DWORD(Result) := 1
End;

Function CmdPageDown( Args : Integer ) : Bool;
Var Loop : Integer;
Begin
  Loop := 0;
  While Loop < SOURCELINES do Begin
    Inc(Loop);
    CmdLineDown(Args)
  End;
  DWORD(Result) := 1
End;

Procedure DebugProcessCommand( KeyCode : Integer );
Var NeedsCmdRefresh  : Bool;
    NeedsFullRefresh : Bool;
    Length           : Integer;
    Loop             : Integer;
    DC               : hDC;
Begin
  If Mode <> MODE_DEBUG then Exit;
  If Ok(ViewIngOutput) then Begin
    DebugDisplay(BOOL(DWORD(1)));
    ViewIngOutput := false;
    Exit
  End;
  NeedsCmdRefresh  := false;
  NeedsFullRefresh := false;
  If (KeyCode = VK_SPACE) and Ok(CommandString[0][0]) then Exit;
  If (KeyCode = VK_BACK) then Begin
    Length := StrLen(commandstring[0]);
    If Ok(Length) then CommandString[0][Length - 1] := #0;
    DWORD(NeedsCmdRefresh) := 1
  End Else If KeyCode = VK_RETURN then Begin
    If not Ok(CommandString[0][0]) and (CommandString[1][0] <> ' ') then StrCopy(CommandString[0],CommandString[1]);
    Loop := COMMANDLINES - 1;
    While Ok(Loop) do Begin
      Dec(Loop);
      StrCopy(CommandString[Succ(Loop)],CommandString[Loop])
    End;
    DWORD(NeedsCmdRefresh)  := COMMANDLINES;
    NeedsFullRefresh := ExecuteCommand(ParseCommandString());
    CommandString[0][0] := #0
  End Else Case KeyCode of
    VK_SPACE : NeedsFullRefresh := CmdTrace(0);
    VK_PRIOR : NeedsFullRefresh := CmdPageUp(0);
    VK_NEXT  : NeedsFullRefresh := CmdPageDown(0);
    VK_UP    : NeedsFullRefresh := CmdLineUp(0);
    VK_DOWN  : NeedsFullRefresh := CmdLineDown(0);
  End;
  If NeedsFullRefresh then Begin
    DebugDisplay(false)
  End Else If NeedsCmdRefresh and not ViewIngOutput and ((Mode = MODE_DEBUG) or (Mode = MODE_STEPPING)) then Begin
    DC := FrameGetDC();
    While NeedsCmdRefresh do Begin
      Dec(NeedsCmdRefresh);
      DrawCommandLine(DC,Abs(Ord(NeedsCmdRefresh)))
    End;
    FrameReleaseDC();
  End
End;

Function GetAddress( Symbol : LPCTSTR ) : Word;
Var Loop : Integer;
Begin
  Loop := SymbolNum;
  While Ok(Loop) do Begin
    Dec(Loop);
    If not Ok(StrComp(LPTSTR(DWord(@SymbolTable^.Name)+SizeOf(SymbolRec)*Loop),Symbol)) then
      Result := Word(DWord(@SymbolTable^.Value)+SizeOf(SymbolRec)*Loop)
  End;
  Result := 0
End;

Function DisplayHelp( Funct : CmdFunction ) : Bool;
Begin
  Result := false
End;

Function CmdBreakpointAdd( Args : Integer ) : Bool;
Var AddedOne : Bool;
    Loop     : Integer;
    FreeSlot : Integer;
Begin
  If not Ok(Args) then Begin
    Args := 1;
    Arg[Args].Val1 := Regs.PC
  End;
  AddedOne := false;
  Loop     := 0;
  While Loop < Args do Begin
    Inc(Loop);
    If Ok(Arg[Loop].Val1) or (Arg[Loop].Str[0] = '0') or Ok(GetAddress(Arg[Loop].Str)) then Begin
      If not Ok(Arg[Loop].Val1) then Arg[Loop].Val1 := GetAddress(Arg[Loop].Str);
      // FIND A FREE SLOT FOR THIS NEW BREAKPOINT
      FreeSlot := 0;
      While (FreeSlot < BREAKPOINTS) and Ok(Breakpoint[FreeSlot].Length) do Inc(FreeSlot);
      If (Freeslot >= BREAKPOINTS) and not Ok(AddedOne) then Begin
        Result := DisplayError('All breakpoint slots are currently in use.');
        Exit
      End;
      // ADD THE BREAKPOINT
      If FreeSlot < BREAKPOINTS then Begin
        Breakpoint[FreeSlot].Address := Arg[Loop].Val1;
        If Ok(Arg[Loop].Val2) then Breakpoint[FreeSlot].Length := MIN($10000 - Arg[Loop].Val1,Arg[Loop].Val2)
                              Else Breakpoint[FreeSlot].Length := 1;
        DWORD(Breakpoint[FreeSlot].Enabled) := 1;
        DWORD(AddedOne) := 1;
        DWORD(UsingBP)  := 1
      End
    End
  End;
  If not Ok(AddedOne) then Begin
    Result := DisplayHelp(CmdBreakpointAdd);
    Exit
  End;
  DWORD(Result) := 1
End;

Function CmdBreakpointClear(Args : Integer) : Bool;
Var Loop     : Integer;
    UsedSlot : Integer;
Begin
  // CHECK FOR ERRORS
  If not Ok(Args) then Begin
    Result := DisplayHelp(CmdBreakpointClear);
    Exit
  End;
  If not Ok(UsingBP) then Begin
    Result := DisplayError('There are no breakpoints defined.');
    Exit
  End;
  // CLEAR EACH BREAKPOINT IN THE LIST
  While Ok(Args) do Begin
    If not Ok(StrComp(Arg[Args].Str,'*')) then Begin
      Loop := BREAKPOINTS;
      While Ok(Loop) do Begin
        Dec(Loop);
        Breakpoint[Loop].Length := 0
      End
    End Else If (Arg[Args].Val1 >= 1) and (Arg[Args].Val1 <= BREAKPOINTS) then
      Breakpoint[Arg[Args].Val1 - 1].Length := 0;
    Dec(Args)
  End;
  // IF THERE ARE NO MORE BREAKPOINTS DEFINED, DISABLE THE BREAKPOINT
  // FUNCTIONALITY AND ERASE THE BREAKPOINT DISPLAY FROM THE SCREEN
  UsedSlot := 0;
  While (UsedSlot < BREAKPOINTS) and not Ok(Breakpoint[UsedSlot].Length) do Inc(UsedSlot);
  If UsedSlot >= BREAKPOINTS then Begin
    UsingBP := false;
    DebugDisplay(true);
    Result := false
  End Else DWORD(Result) := 1
End;

Function CmdBreakpointDisable(Args : Integer) : Bool;
Var Loop : Integer;
Begin
  // CHECK FOR ERRORS
  If not Ok(Args) then Begin
    Result := DisplayHelp(CmdBreakpointDisable);
    Exit
  End;
  If not Ok(UsingBP) then Begin
    Result := DisplayError('There are no breakpoints defined.');
    Exit
  End;
  // DISABLE EACH BREAKPOINT IN THE LIST
  While Ok(Args) do Begin
    If not Ok(StrComp(Arg[Args].Str,'*')) then Begin
      Loop := BREAKPOINTS;
      While Ok(Loop) do Begin
        Dec(Loop);
        Breakpoint[Loop].Enabled := false
      End
    End Else If (Arg[Args].Val1 >= 1) and (Arg[Args].Val1 <= BREAKPOINTS) then
      Breakpoint[Arg[Args].Val1 - 1].Enabled := false;
    Dec(Args)
  End;
  DWord(Result) := 1
End;

Function CmdBreakpointEnable(Args : Integer) : Bool;
Var Loop : Integer;
Begin
  // CHECK FOR ERRORS
  If not Ok(Args) then Begin
    Result := DisplayHelp(CmdBreakpointEnable);
    Exit
  End;
  If not Ok(Usingbp) then Begin
    Result := DisplayError('There are no breakpoints defined.');
    Exit
  End;
  // ENABLE EACH BREAKPOINT IN THE LIST
  While Ok(Args) do Begin
    If not Ok(StrComp(Arg[Args].Str,'*')) then Begin
      Loop := BREAKPOINTS;
      While Ok(Loop) do Begin
        Dec(Loop);
        DWORD(Breakpoint[Loop].Enabled) := 1
      End
    End Else If (Arg[Args].Val1 >= 1) and (Arg[Args].Val1 <= BREAKPOINTS) then
      DWORD(Breakpoint[Arg[Args].Val1 - 1].Enabled) := 1;
    Dec(Args)
  End;
  DWORD(Result) := 1
End;

Function CmdSetupBenchmark(Args : Integer) : Bool;
Begin
  CpuSetupBenchmark();
  ComputeTopOffset(Regs.pc);
  DWORD(Result) := 1
End;

Function CmdBlackWhite(Args : Integer) : Bool;
Begin
  ColorScheme := 1;
  DebugDisplay(BOOL(DWORD(1)));
  Result := false
End;

Function CmdColor(Args : Integer) : Bool;
Begin
  ColorScheme := 0;
  DebugDisplay(BOOL(DWORD(1)));
  Result := false
End;

Function CmdMemoryDump(Args : Integer) : Bool;
Begin
  If not Ok(Args) or ((Arg[1].Str[0] <> '0') and not Ok(Arg[1].Val1) and not Ok(GetAddress(Arg[1].Str))) then Begin
    Result := DisplayHelp(CmdMemoryDump);
    Exit
  End;
  MemoryDump := Arg[1].Val1;
  If not Ok(MemoryDump) then MemoryDump := GetAddress(Arg[1].Str);
  DWORD(UsingMemDump) := 1;
  DWORD(Result) := 1
End;

Function CmdExtBenchmark(Args : Integer) : Bool;
Var CurrTime : DWord;
Begin
  DebugEnd();
  Mode := MODE_RUNNING;
  FrameRefreshStatus(DRAW_TITLE);
  VideoRedrawScreen();
  CurrTime := GetTickCount();
  While Ord(ExtBench = GetTickCount()) <> CurrTime do KeybQueueKeypress(Ord(' '),BOOL(DWORD(1)));
  DWORD(ResetTiming) := 1;
  Result := false
End;

Function CmdGo(Args : Integer) : Bool;
Begin
  StepCount := -1;
  StepLine  := false;
  StepStart := Regs.pc;
  If Ok(Args) then StepUntil := Arg[1].Val1 Else StepUntil := -1;
  If not Ok(StepUntil) then StepUntil := GetAddress(Arg[1].Str);
  Mode := MODE_STEPPING;
  FrameRefreshStatus(DRAW_TITLE);
  Result := false
End;

Function CmdInput(Args : Integer) : Bool;
Begin
  If not Ok(Args) or ((Arg[1].Str[0] <> '0') and not Ok(Arg[1].Val1) and not Ok(GetAddress(Arg[1].Str))) then Begin
    Result := DisplayHelp(CmdInput);
    Exit
  End;
  If not Ok(Arg[1].Val1) then Arg[1].Val1 := GetAddress(Arg[1].Str);
  IoRead[Arg[1].Val1 AND $FF](Regs.pc,Arg[1].Val1 AND $FF,0,0);
  Result := false
End;

Function CmdInternalCodeDump(Args : Integer) : Bool;
Var Filename     : Array[0..Pred(MAX_PATH)] of Char;
    _File        : tHandle;
    CodeLength   : DWord;
    CodePtr      : LPBYTE;
    BytesWritten : DWord;
Begin
  If not Ok(Args) or ((Arg[1].Str[0] <> '0') and not Ok(Arg[1].Val1) and not Ok(GetAddress(Arg[1].Str))) then Begin
    Result := DisplayHelp(CmdInternalCodeDump);
    Exit
  End;
  If not Ok(Arg[1].Val1) then Arg[1].Val1 := GetAddress(Arg[1].Str);
  StrCopy(Filename,ProgDir);
  StrCat(Filename,'Output.bin');
  _File := CreateFile(Filename,GENERIC_WRITE,0,PSecurityAttributes(Nil),CREATE_ALWAYS,
                      FILE_ATTRIBUTE_NORMAL OR FILE_FLAG_SEQUENTIAL_SCAN,0);
  If _File <> INVALID_HANDLE_VALUE then Begin
    CodeLength := 0;
    CodePtr    := Nil;
    CpuGetCode(Arg[1].Val1,@CodePtr,@CodeLength);
    If Ok(CodePtr) and Ok(CodeLength) then Begin
      BytesWritten := 0;
      WriteFile(_File,CodePtr,CodeLength,Byteswritten,Nil)
    End;
    CloseHandle(_File)
  End;
  Result := false
End;

Function CmdInternalMemoryDump(Args : Integer) : Bool;
Var Filename     : Array[0..Pred(MAX_PATH)] of Char;
    _File        : tHandle;
    BytesWritten : DWord;
Begin
  StrCopy(Filename,ProgDir);
  StrCat(Filename,'Output.bin');
  _File := CreateFile(Filename,GENERIC_WRITE,0,PSecurityAttributes(Nil),CREATE_ALWAYS,
                      FILE_ATTRIBUTE_NORMAL OR FILE_FLAG_SEQUENTIAL_SCAN,0);
  If _File <> INVALID_HANDLE_VALUE then Begin
    If CpuEmtype = CPU_COMPILING then WriteFile(_File,Mem,$30000,BytesWritten,0)
                                 Else WriteFile(_File,Mem,$10000,BytesWritten,0);
    CloseHandle(_File)
  End;
  Result := false
End;

Function CmdFeedKey(Args : Integer) : Bool;
Begin
  If Ok(Args) then If Ok(Arg[1].Val1) then KeybQueueKeypress(Arg[1].Val1,BOOL(DWORD(1)))
                                      Else KeybQueueKeypress(Ord(Arg[1].Str[0]),BOOL(DWORD(1)))
                Else KeybQueueKeypress(Ord(' '),BOOL(DWORD(1)));
  Result := false
End;

Function CmdCodeDump(Args : Integer) : Bool;
Begin
  If not Ok(Args) or ((Arg[1].Str[0] <> '0') and not Ok(Arg[1].Val1) and not Ok(GetAddress(Arg[1].Str))) then Begin
    Result := DisplayHelp(CmdCodeDump);
    Exit
  End;
  TopOffset := Arg[1].Val1;
  If not Ok(TopOffset) then TopOffset := GetAddress(Arg[1].Str);
  DWORD(Result) := 1
End;

Function CmdMemoryEnter(Args : Integer) : Bool;
Var Address : Word;
Begin
  If (Args < 2) or ((Arg[1].Str[0] <> '0') and not Ok(Arg[1].Val1) and not Ok(GetAddress(Arg[1].Str)))
                or ((Arg[2].Str[0] <> '0') and not Ok(Arg[2].Val1)) then Begin
    Result := DisplayHelp(CmdMemoryEnter);
    Exit
  End;
  Address := Arg[1].Val1;
  If not Ok(Address) then Address := GetAddress(Arg[1].Str);
  While Args >= 2 do Begin
    LPBYTE(DWord(Membank)  + Address + Args - 2)^ := BYTE(Arg[Args].Val1);
    LPBYTE(DWord(MemDirty) + (Address Shr 8))^    := 1;
    Dec(Args)
  End;
  DWORD(Result) := 1
End;

Function CmdMemoryFill(Args : Integer) : Bool;
Var Address : Word;
    Bytes   : Word;
Begin
  If not Ok(Args) or ((Arg[1].Str[0] <> '0') and not Ok(Arg[1].Val1) and not Ok(GetAddress(Arg[1].Str))) then Begin
    Result := DisplayHelp(CmdMemoryFill);
    Exit
  End;
  If Ok(Arg[1].Val1) then Address := Arg[1].Val1 Else Address := GetAddress(Arg[1].Str);
  Bytes := MAX(1,Arg[1].Val2);
  While Ok(Bytes) do Begin
    Dec(Bytes);
    If (Address < $C000) or (Address > $C0FF) then LPBYTE(DWord(MemBank) + Address)^ := BYTE(Arg[2].Val1 AND $FF);
    Inc(Address)
  End;
  DWORD(Result) := 1
End;

Function CmdOutput(Args : Integer) : Bool;
Begin
  If not Ok(Args) or ((Arg[1].Str[0] <> '0') and not Ok(Arg[1].Val1) and not Ok(GetAddress(Arg[1].Str))) then Begin
    Result := DisplayHelp(CmdInput);
    Exit
  End;
  If not Ok(Arg[1].Val1) then Arg[1].Val1 := GetAddress(Arg[1].Str);
  IoWrite[Arg[1].Val1 AND $FF](Regs.pc,Arg[1].Val1 AND $FF,1,Arg[2].Val1 AND $FF);
  DWORD(Result) := 1
End;

Function CmdProfile(Args : Integer) : Bool;
Begin
  FillChar(ProfileData,256 * SizeOf(DWORD),#0);
  DWORD(Profiling) := 1;
  Result := false
End;

Function CmdRegisterSet(Args : Integer) : Bool;
Begin
  If (Args = 2) and (Arg[1].Str[0] = 'P') and (Arg[2].Str[0] = 'L') then Begin
    Regs.pc := LastPC
  End Else If (Args < 2) or ((Arg[2].Str[0] <> '0') and not Ok(Arg[2].Val1)) then Begin
    Result := DisplayHelp(CmdMemoryEnter);
    Exit
  End Else Case Arg[1].Str[0] of
    'A' : Regs.A  := BYTE(Arg[2].Val1 AND $FF);
    'P' : Regs.PC := Arg[2].Val1;
    'S' : Regs.SP := $100 OR (Arg[2].Val1 AND $FF);
    'X' : Regs.X  := BYTE(Arg[2].Val1 AND $FF);
    'Y' : Regs.Y  := BYTE(Arg[2].Val1 AND $FF)
    Else Begin
      Result := DisplayHelp(CmdMemoryEnter);
      Exit
    End
  End;
  ComputeTopOffset(Regs.pc);
  DWORD(Result) := 1
End;

Function CmdFlagSet(Args : Integer) : Bool;
Var Loop : Integer;
Begin
  Loop := 0;
  While Loop < 8 do If Flagname[Loop] = Arg[0].Str[1] then Break Else Inc(Loop);
  If Loop < 8 then If Arg[0].Str[0] = 'R' then Regs.PS := Regs.PS AND NOT (1 Shl Loop)Else Regs.PS := Regs.PS OR (1 Shl Loop);
  DWORD(Result) := 1
End;

Function CmdTrace(Args : Integer) : Bool;
Begin
  If Ok(Args) then StepCount := Arg[1].Val1 Else StepCount := 1;
  StepLine  := false;
  StepStart := Regs.PC;
  StepUntil := -1;
  Mode := MODE_STEPPING;
  FrameRefreshStatus(DRAW_TITLE);
  DebugContinueStepping();
  Result := false
End;

Function CmdTraceFile(Args : Integer) : Bool;
Var Filename : Array[0..Pred(MAX_PATH)] of Char;
Begin
  If Ok(Tracefile) then fClose(TraceFile);
  StrCopy(Filename,ProgDir);
  If Ok(Args) and Ok(Arg[1].Str[0]) then StrCat(Filename,Arg[1].Str)
                                    Else StrCat(Filename,'Trace.txt');
  TraceFile := fOpen(Filename,'wt');
  Result := false
End;

Function CmdTraceLine(Args : Integer) : Bool;
Begin
  If Ok(Args) then StepCount := Arg[1].Val1 Else StepCount := 1;
  DWORD(StepLine) := 1;
  StepStart := Regs.PC;
  StepUntil := -1;
  Mode := MODE_STEPPING;
  FrameRefreshStatus(DRAW_TITLE);
  DebugContinueStepping();
  Result := false
End;

Function CmdWatchAdd(Args : Integer) : Bool;
Var AddedOne : Bool;
    Loop     : Integer;
    FreeSlot : Integer;
Begin
  If not Ok(Args) then Begin
    Result :=DisplayHelp(CmdWatchAdd);
    Exit
  End;
  Addedone := false;
  Loop     := 0;
  While Loop < Args do Begin
    Inc(Loop);
    If Ok(Arg[Loop].Val1) or (Arg[Loop].Str[0] = '0') or Ok(GetAddress(Arg[Loop].Str)) then Begin
      If not Ok(Arg[Loop].Val1) then Arg[Loop].Val1 := GetAddress(Arg[Loop].Str);
      // FIND A FREE SLOT FOR THIS NEW WATCH
      FreeSlot := 0;
      While (FreeSlot < WATCHES) and (Watch[FreeSlot] >= 0) do Inc(FreeSlot);
      If (FreeSlot >= WATCHES) and not Ok(AddedOne) then Begin
        Result := DisplayError('All watch slots are currently in use.');
        Exit
      End;
      // VERIFY THAT THE WATCH IS NOT ON AN I/O LOCATION
      If (Arg[Loop].Val1 >= $C000) and (Arg[Loop].Val1 <= $C0FF) then Begin
        Result := DisplayError('You may not watch an I/O location.');
        Exit
      End;
      // ADD THE WATCH
      If FreeSlot < WATCHES then Begin
        Watch[FreeSlot] := Arg[Loop].Val1;
        DWORD(AddedOne) := 1;
        DWORD(UsingWatches) := 1
      End
    End
  End;
  If not Ok(Addedone) then Result := DisplayHelp(CmdWatchAdd) Else DWORD(Result) := 1
End;

Function CmdWatchClear(Args : Integer) : Bool;
Var Loop     : Integer;
    UsedSlot : Integer;
Begin
  // CHECK FOR ERRORS
  If not Ok(Args) then Begin
    Result := DisplayHelp(CmdWatchAdd);
    Exit
  End;
  If not Ok(UsingWatches) then Begin
    Result := DisplayError('There are no watches defined.');
    Exit
  End;
  // CLEAR EACH WATCH IN THE LIST
  While Ok(Args) do Begin
    If not Ok(StrComp(Arg[Args].Str,'*')) then Begin
      Loop := WATCHES;
      While Ok(Loop) do Begin
        Dec(Loop);
        Watch[Loop] := -1
      End
    End Else If (Arg[Args].Val1 >= 1) and (Arg[Args].Val1 <= WATCHES) then Watch[Pred(Arg[Args].Val1)] := -1;
    Dec(Args)
  End;
  // IF THERE ARE NO MORE WATCHES DEFINED, ERASE THE WATCH DISPLAY
  Usedslot := 0;
  While (UsedSlot < WATCHES) and (Watch[UsedSlot] < 0) do Inc(UsedSlot);
  If UsedSlot >= WATCHES then Begin
    UsingWatches := false;
    DebugDisplay(BOOL(DWORD(1)));
    Result := false
  End Else DWORD(Result) := 1
End;

Function CmdZap(Args : Integer) : Bool;
Var Loop : Integer;
Begin
  Loop := AddressMode[Instruction[LPBYTE(DWord(Mem) + Regs.PC)^].AddrMode].Bytes;
  While Ok(Loop) do Begin
    Dec(Loop);
    LPBYTE(DWord(Mem) + Regs.PC + Loop)^ := $EA
  End;
  DWORD(Result) := 1
End;

Function CmdViewOutput(Args : Integer) : Bool;
Begin
  VideoRedrawScreen();
  DWORD(ViewingOutput) := 1;
  Result := false
End;





