/*
 * Apple iiPod - Appleblossom Interface for iPodLinux
 * Copyright (C) 2005 Jonathan Bettencourt (jonrelay)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../a2includes.h"
#include "../a2colors.h"
#include "ifvars.h"
#include "nano-X.h"

unsigned char a2_text_buffer[1920];
unsigned char a2_gr_buffer[1920];
unsigned char a2_hgr_buffer[15360];
unsigned char a2_hgr_temp[7680];

extern void a2_draw_char_40(GR_WINDOW_ID wid, GR_GC_ID gc, unsigned char ch, int x, int y);
extern void a2_draw_char_80(GR_WINDOW_ID wid, GR_GC_ID gc, unsigned char ch, int x, int y);
extern void a2_draw_keyboard(GR_WINDOW_ID wid, GR_GC_ID gc);

GR_COLOR a2_get_color(int i)
{
	int c = a2_color_palette[a2_selected_palette][i];
	return GR_RGB( ((c & 0xFF0000) >> 16), ((c & 0xFF00) >> 8), (c & 0xFF) );
}

void a2_draw_lr(GR_WINDOW_ID wid, GR_GC_ID gc, unsigned char ch, int x, int y, int w)
{
	GrSetGCForeground(gc, a2_get_color(ch & 0xF));
	GrFillRect(wid, gc, x, y, w, 3);
	GrSetGCForeground(gc, a2_get_color((ch & 0xF0) >> 4));
	GrFillRect(wid, gc, x, y+3, w, 2);
}

void a2_draw_hr(GR_WINDOW_ID wid, GR_GC_ID gc, unsigned char ch, int x, int y, int stretch)
{
	GrSetGCForeground(gc, a2_get_color((ch & 0xF0) >> 4));
	GrPoint(wid, gc, x, y);
	GrSetGCForeground(gc, a2_get_color(ch & 0xF));
	GrPoint(wid, gc, x+1, y);
	if (stretch) GrPoint(wid, gc, x+2, y);
}

void a2_draw_text_buffer(GR_WINDOW_ID wid, GR_GC_ID gc, int dblwidth, int mixed)
{
	int r, c, a;
	int nc = (dblwidth?80:40);
	for (r=(mixed?20:0); r<24; r++) {
		for (c=0; c<nc; c++) {
			a = (r*nc)+c;
			if (dblwidth) {
				a2_draw_char_80(wid, gc, a2_text_buffer[a], c<<1, r*5);
			} else {
				a2_draw_char_40(wid, gc, a2_text_buffer[a], c<<2, r*5);
			}
		}
	}
}

void a2_draw_gr_buffer(GR_WINDOW_ID wid, GR_GC_ID gc, int dblwidth, int mixed)
{
	int r, c, a;
	int nc = (dblwidth?80:40);
	int nr = (mixed?20:24);
	for (r=0; r<nr; r++) {
		for (c=0; c<nc; c++) {
			a = (r*nc)+c;
			a2_draw_lr(wid, gc, a2_gr_buffer[a], c<<((!dblwidth)+1), r*5, ((!dblwidth)<<1)+2);
		}
	}
}

void a2_draw_hgr_buffer(GR_WINDOW_ID wid, GR_GC_ID gc, int mixed)
{
	int r, c, a, rc;
	int rr = 0;
	int nr = (mixed?160:192);
	for (r=0; r<nr; r++) {
		switch (r % 8) {
		case 0: case 2: case 4: case 5: case 7:
			for (c=0; c<70; c++) {
				a = (r*70)+c;
				rc = (c<<1);
				a2_draw_hr(wid, gc, a2_hgr_buffer[a], (rc + rc/7), rr, ((rc%7)>=5));
			}
			rr++;
			break;
		default:
			break;
		}
	}
}

void a2_draw_video(GR_WINDOW_ID wid, GR_GC_ID gc)
{
	/* First get the video mode */
	/*
		0	40-Column Text Page 1
		1	40-Column Text Page 2
		2	80-Column Text
		3	Split Single Lo-Res with 40-Column Text Page 1
		4	Split Single Lo-Res with 40-Column Text Page 2
		5	Split Single Lo-Res with 80-Column Text
		6	Single Lo-Res Page 1
		7	Single Lo-Res Page 2
		8	Split Single Lo-Res with Hi-Res Bit Patterns with 40-Column Text Page 1
		9	Split Single Lo-Res with Hi-Res Bit Patterns with 40-Column Text Page 2
		10	Single Lo-Res with Hi-Res Bit Patterns Page 1
		11	Single Lo-Res with Hi-Res Bit Patterns Page 2
		12	Split Double Lo-Res with 80-Column Text
		13	Double Lo-Res
		14	Split Single Hi-Res with 40-Column Text Page 1
		15	Split Single Hi-Res with 40-Column Text Page 2
		16	Split Single Hi-Res with 80-Column Text
		17	Single Hi-Res Page 1
		18	Single Hi-Res Page 2
		19	Split Double Hi-Res with 80-Column Text
		20	Double Hi-Res
	*/
	int vm = a2_video_big_mode();
	/* Clear the screen because some mode switches leave garbage */
	GrSetGCForeground(gc, BLACK);
	GrFillRect(wid, gc, 0, 0, 160, 128);
	/* Get the text screen if we need it */
	switch (vm) {
	case 0: case 1: case 2: case 3: case 4: case 5:
		a2_memcpy_text_buffer(a2_text_buffer, vm%3);
		break;
	case 8: case 9: case 14: case 15:
		a2_memcpy_text_buffer(a2_text_buffer, vm%2);
		break;
	case 12: case 16: case 19:
		a2_memcpy_text_buffer(a2_text_buffer, 2);
		break;
	}
	/* Get the lo-res screen if we need it */
	switch (vm) {
	case 3: case 5: case 6: case 8: case 10:
		a2_memcpy_text_buffer(a2_gr_buffer, 0);
		break;
	case 4: case 7: case 9: case 11:
		a2_memcpy_text_buffer(a2_gr_buffer, 1);
		break;
	case 12: case 13:
		a2_memcpy_text_buffer(a2_gr_buffer, 2);
		a2_decode_dlr_buffer(a2_gr_buffer);
		break;
	}
	/* Get the hi-res screen if we need it */
	switch (vm) {
	case 14: case 16: case 17:
		a2_memcpy_hires_buffer(a2_hgr_temp, 0);
		a2_decode_hgr_buffer(a2_hgr_buffer, a2_hgr_temp);
		break;
	case 15: case 18:
		a2_memcpy_hires_buffer(a2_hgr_temp, 1);
		a2_decode_hgr_buffer(a2_hgr_buffer, a2_hgr_temp);
		break;
	case 19: case 20:
		a2_memcpy_hires_buffer(a2_hgr_buffer, 2);
		a2_decode_dhr_buffer(a2_hgr_buffer, a2_hgr_buffer);
		break;
	}
	/* Draw the main part of the screen */
	switch (vm) {
	case 0: case 1:
		a2_draw_text_buffer(wid, gc, 0, 0);
		break;
	case 2:
		a2_draw_text_buffer(wid, gc, 1, 0);
		break;
	case 3: case 4: case 5: case 8: case 9:
		a2_draw_gr_buffer(wid, gc, 0, 1);
		break;
	case 6: case 7: case 10: case 11:
		a2_draw_gr_buffer(wid, gc, 0, 0);
		break;
	case 12:
		a2_draw_gr_buffer(wid, gc, 1, 1);
		break;
	case 13:
		a2_draw_gr_buffer(wid, gc, 1, 0);
		break;
	case 14: case 15: case 16: case 19:
		a2_draw_hgr_buffer(wid, gc, 1);
		break;
	case 17: case 18: case 20:
		a2_draw_hgr_buffer(wid, gc, 0);
		break;
	}
	/* Draw the lower four lines of text in mixed mode */
	switch (vm) {
	case 3: case 4: case 8: case 9: case 14: case 15:
		a2_draw_text_buffer(wid, gc, 0, 1);
		break;
	case 5: case 12: case 16: case 19:
		a2_draw_text_buffer(wid, gc, 1, 1);
		break;
	}
	/* In full hi-res graphics move the "keyboard" to the side of the screen */
	/*
	switch (vm) {
	case 17: case 18: case 20:
		if (!a2_keyboard_orientation) {
			a2_keyboard_orientation = 1;
			a2_draw_keyboard(wid, gc);
		}
		break;
	default:
		if (a2_keyboard_orientation) {
			a2_keyboard_orientation = 0;
			a2_draw_keyboard(wid, gc);
		}
		break;
	}
	*/
}

void a2_do_draw(void)
{
	a2_draw_video(a2_bufwid, a2_gc);
	a2_draw_keyboard(a2_bufwid, a2_gc);
	GrCopyArea(a2_wid, a2_gc, 0, 0, 160, 128, a2_bufwid, 0, 0, MWROP_SRCCOPY);
}



