/*
 * Appleblossom - Portable Open-Source Apple IIe Emulator
 * Copyright (C) 2005 Jonathan Bettencourt (jonrelay)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "a2vars.h"
#include "a2colors.h"

int a2_video_mode(void)
{
	/* Here's how it goes down:
	   0 - 40-Column Text
	   1 - 80-Column Text
	   2 - Single Lo-Res
	   3 - Double Lo-Res
	   4 - Lo-Res w/ Hi-Res Bit Patterns
	   5 - Single Hi-Res
	   6 - Double Hi-Res
	*/
	if (!A2_GRAPHICS) {
		if (A2_80VIDEO) {
			return 1;
		} else {
			return 0;
		}
	} else if (A2_HIRES) {
		if (A2_ANN3 || (!A2_80VIDEO)) {
			return 5;
		} else {
			return 6;
		}
	} else if (A2_ANN3) {
		return 2;
	} else if (A2_80VIDEO) {
		return 3;
	} else {
		return 4;
	}
}

int a2_video_split_mode(void)
{
	/*
	   0 - Full Screen
	   1 - Split w/ 40 Column
	   2 - Split w/ 80 Column
	*/
	if ((!A2_GRAPHICS) || (!A2_MIXED)) {
		return 0;
	} else if (A2_80VIDEO) {
		return 2;
	} else {
		return 1;
	}
}

int a2_video_page(void)
{
	/*
	   0 - Main Page 1, Single-Width
	   1 - Main Page 2, Single-Width
	   2 - Auxiliary & Main Page 1, Double-Width
	*/
	if (A2_80VIDEO) {
		return ((A2_ANN3 && A2_GRAPHICS) ? 0 : 2);
	} else if (A2_PAGE2) {
		return 1;
	} else {
		return 0;
	}
}

int a2_video_split_page(void)
{
	/*
	   0 - Main Page 1, 40-Column
	   1 - Main Page 2, 40-Column
	   2 - Auxiliary & Main Page 1, 80-Column
	*/
	if (A2_80VIDEO) {
		return 2;
	} else if (A2_PAGE2) {
		return 1;
	} else {
		return 0;
	}
}

int a2_video_big_mode(void)
{
	if (!A2_GRAPHICS) {
		if (A2_80VIDEO) return 2;
		if (A2_PAGE2) return 1;
		return 0;
	} else if (A2_HIRES) {
		if (A2_80VIDEO && !A2_ANN3) {
			if (A2_MIXED) return 19;
			return 20;
		} else if (A2_MIXED) {
			if (A2_80VIDEO) return 16;
			if (A2_PAGE2) return 15;
			return 14;
		} else {
			if (A2_80VIDEO) return 17;
			if (A2_PAGE2) return 18;
			return 17;
		}
	} else if (A2_ANN3) {
		if (A2_MIXED) {
			if (A2_80VIDEO) return 5;
			if (A2_PAGE2) return 4;
			return 3;
		} else {
			if (A2_80VIDEO) return 6;
			if (A2_PAGE2) return 7;
			return 6;
		}
	} else {
		if (A2_MIXED) {
			if (A2_80VIDEO) return 12;
			if (A2_PAGE2) return 9;
			return 8;
		} else {
			if (A2_80VIDEO) return 13;
			if (A2_PAGE2) return 11;
			return 10;
		}
	}
}

#define a2_text_base(r) (0x0400|(((r)&0x7)<<7)|(((r)&0x18)<<2)|((r)&0x18))
#define a2_hires_base(r) (0x2000|(((r)&0x07)<<10)|(((r)&0x38)<<4)|(((r)&0xC0)>>1)|(((r)&0xC0)>>3))

/* #defined as macros above
unsigned short a2_text_base(int row)
{
	return (  0x0400 | ((row & 0x07) << 7) | ((row & 0x18) << 2) | (row & 0x18)  );
}

unsigned short a2_hires_base(int row)
{
	return (  0x2000 | ((row & 0x07) << 10) | ((row & 0x38) << 7) | ((row & 0xC0) >> 1) | ((row & 0xC0) >> 3)  );
}
*/

unsigned char __bit_reverse(unsigned char b)
{
	unsigned char nibs[] = {0x00, 0x08, 0x04, 0x0C, 0x02, 0x0A, 0x06, 0x0E, 0x01, 0x09, 0x05, 0x0D, 0x03, 0x0B, 0x07, 0x0F};
	return ((nibs[(b & 0x0F)]<<4)|(nibs[(b & 0xF0)>>4]));
}

char a2_char_strip_fmt(unsigned char ch)
{
	switch (ch & 0xE0) {
	case 0xA0:
	case 0xC0:
	case 0xE0:
		if (ch == 0xFF) return '#';
		return (ch & 0x7F);
		break;
	case 0x40:
		if (A2_ALTCHAR) {
			return '?';
		}
	case 0x80:
	case 0x00:
		return ('@' + (ch & 0x1F));
		break;
	case 0x60:
		if (A2_ALTCHAR) {
			if (ch == 0x7F) return '#';
			return ch;
		}
	case 0x20:
		return (' ' + (ch & 0x1F));
		break;
	}
}

void a2_memcpy_text_buffer(unsigned char * buf, int page)
{
	int row, col;
	unsigned char * p = buf;
	unsigned short a;
	for (row=0; row<24; row++) {
		a = a2_text_base(row);
		if (page==1) a+=0x400;
		if (page==2) {
			for (col=0; col<40; col++) {
				*p++ = A2_AUX[a];
				*p++ = A2_MAIN[a++];
			}
		} else {
			for (col=0; col<40; col++) {
				*p++ = A2_MAIN[a++];
			}
		}
	}
}

void a2_memcpy_hires_buffer(unsigned char * buf, int page)
{
	int row, col;
	unsigned char * p = buf;
	unsigned short a;
	for (row=0; row<192; row++) {
		a = a2_hires_base(row);
		if (page==1) a+=0x2000;
		if (page==2) {
			for (col=0; col<40; col++) {
				*p++ = A2_AUX[a];
				*p++ = A2_MAIN[a++];
			}
		} else {
			for (col=0; col<40; col++) {
				*p++ = A2_MAIN[a++];
			}
		}
	}
}

void a2_decode_dlr_buffer(unsigned char * buf)
{
	int i;
	unsigned char * p = buf;
	for (i=0; i<960; i++) {
		*p = (  (((*p) << 1) & 0xEE) | (((*p) & 0x88) >> 3)  );
		p+=2;
	}
}

unsigned char __hgr_pxl(int bp, int cg)
{
	unsigned char pixels[] = {0x0, 0x0, 0xC, 0x9, 0x3, 0x6, 0xF, 0xF};
	return pixels[(bp << 1) | (!!cg)];
}

unsigned char __hgr_wrd_pxl(int bp, int cg1, int cg2)
{
	unsigned char weird_pixels[] = {0x0, 0x0, 0xD, 0xE, 0x7, 0x2, 0x1, 0x8};
	if ((cg1==cg2)||(!bp)) return __hgr_pxl(bp,cg1);
	return weird_pixels[(bp << 1) | (!!cg1)];
}

unsigned char __hgr_wrd_pxl2(int bp, int cg2)
{
	return __hgr_wrd_pxl(((bp&4)>>1)|(bp&1), !!(bp&2), !!cg2);
}

void a2_decode_hgr_buffer(unsigned char * dbuf, unsigned char * sbuf)
{
	/* This one is unique in that it requires a buffer bigger than */
	/* the one it's taking in, due to the color bit stuff and all  */
	/* that weird standard-hi-res mode crap.                       */
	int i;
	unsigned long a;
	unsigned char * src = sbuf;
	unsigned char * dest = dbuf;
	for (i=0; i<1920; i++) {
		a = __bit_reverse(*src++);
		a = (a << 8) | __bit_reverse(*src++);
		a = (a << 8) | __bit_reverse(*src++);
		a = (a << 8) | __bit_reverse(*src++);
		*dest++ =      ( __hgr_pxl( ((a & 0xC0000000) >> 30), (a & 0x01000000) ) << 4 ) |
		               ( __hgr_pxl( ((a & 0x30000000) >> 28), (a & 0x01000000) )      ) ;
		*dest++ =      ( __hgr_pxl( ((a & 0x0C000000) >> 26), (a & 0x01000000) ) << 4 ) |
		          ( __hgr_wrd_pxl2( ((a & 0x03800000) >> 23), (a & 0x00010000) )      ) ;
		*dest++ =      ( __hgr_pxl( ((a & 0x00600000) >> 21), (a & 0x00010000) ) << 4 ) |
		               ( __hgr_pxl( ((a & 0x00180000) >> 19), (a & 0x00010000) )      ) ;
		*dest++ =      ( __hgr_pxl( ((a & 0x00060000) >> 17), (a & 0x00010000) ) << 4 ) |
		               ( __hgr_pxl( ((a & 0x0000C000) >> 14), (a & 0x00000100) )      ) ;
		*dest++ =      ( __hgr_pxl( ((a & 0x00003000) >> 12), (a & 0x00000100) ) << 4 ) |
		               ( __hgr_pxl( ((a & 0x00000C00) >> 10), (a & 0x00000100) )      ) ;
		*dest++ = ( __hgr_wrd_pxl2( ((a & 0x00000380) >>  7), (a & 0x00000001) ) << 4 ) |
		               ( __hgr_pxl( ((a & 0x00000060) >>  5), (a & 0x00000001) )      ) ;
		*dest++ =      ( __hgr_pxl( ((a & 0x00000018) >>  3), (a & 0x00000001) ) << 4 ) |
		               ( __hgr_pxl( ((a & 0x00000006) >>  1), (a & 0x00000001) )      ) ;
	}
}

void a2_decode_dhr_buffer(unsigned char * dbuf, unsigned char * sbuf)
{
	int i;
	unsigned long a;
	unsigned long b;
	unsigned char t;
	unsigned char * src = sbuf;
	unsigned char * dest = dbuf;
	for (i=0; i<1920; i++) {
		a =             (__bit_reverse(*src++) >> 1) ;
		a = ((a << 7) | (__bit_reverse(*src++) >> 1));
		a = ((a << 7) | (__bit_reverse(*src++) >> 1));
		a = ((a << 7) | (__bit_reverse(*src++) >> 1));
		b =             (__bit_reverse(*src++) >> 1) ;
		b = ((b << 7) | (__bit_reverse(*src++) >> 1));
		b = ((b << 7) | (__bit_reverse(*src++) >> 1));
		b = ((b << 7) | (__bit_reverse(*src++) >> 1));
		      t =   ((a & 0x0FF00000) >> 20);
		*dest++ = ((t & 0x55) | ((t & 0x88)>>2) | ((t & 0x22)<<2));
		      t =   ((a & 0x000FF000) >> 12);
		*dest++ = ((t & 0x55) | ((t & 0x88)>>2) | ((t & 0x22)<<2));
		      t =   ((a & 0x00000FF0) >>  4);
		*dest++ = ((t & 0x55) | ((t & 0x88)>>2) | ((t & 0x22)<<2));
		      t = ( ((a & 0x0000000F) <<  4)
		          | ((b & 0x0F000000) >> 24) );
		*dest++ = ((t & 0x55) | ((t & 0x88)>>2) | ((t & 0x22)<<2));
		      t =   ((b & 0x00FF0000) >> 16);
		*dest++ = ((t & 0x55) | ((t & 0x88)>>2) | ((t & 0x22)<<2));
		      t =   ((b & 0x0000FF00) >>  8);
		*dest++ = ((t & 0x55) | ((t & 0x88)>>2) | ((t & 0x22)<<2));
		      t =   ((b & 0x000000FF)      );
		*dest++ = ((t & 0x55) | ((t & 0x88)>>2) | ((t & 0x22)<<2));
	}
}


