/*
 * Appleblossom - Portable Open-Source Apple IIe Emulator
 * Copyright (C) 2005 Jonathan Bettencourt (jonrelay)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * For hard disk emulation, I use the peripheral card ROM and I/O addresses
 * from AppleWin 1.12.6.0.
 */

#include <stdio.h>
#include <string.h>
#include "a2vars.h"

int a2_hdisk_drive2;
int a2_hdisk_command;
int a2_hdisk_error;
int a2_hdisk_block[2];
int a2_hdisk_memloc[2];
unsigned char a2_hdisk_buffer[2][514];
unsigned char * a2_hdisk_ptr[2];

void a2_hdisk_reset(void)
{
	A2_HDISK[0]=NULL;
	A2_HDISK[1]=NULL;
	A2_HDISK_LEN[0]=0;
	A2_HDISK_LEN[1]=0;
	A2_HDISK_PATH[0][0]=0;
	A2_HDISK_PATH[1][0]=0;
	A2_HDISK_WRPROTECT[0]=0;
	A2_HDISK_WRPROTECT[1]=0;
	A2_HDISK_DIRTY[0]=0;
	A2_HDISK_DIRTY[1]=0;
	a2_hdisk_drive2 = 0;
	a2_hdisk_command = 0;
	a2_hdisk_error = 0;
	a2_hdisk_block[0] = 0;
	a2_hdisk_block[1] = 0;
	a2_hdisk_memloc[0] = 0;
	a2_hdisk_memloc[1] = 0;
	a2_hdisk_ptr[0] = 0;
	a2_hdisk_ptr[1] = 0;
}

void a2_read_hdisk_rom(char * fn)
{
	FILE * f;
	f = fopen(fn, "rb");
	if (f) {
		fread(&A2_PERIPH_ROM[0x700], 1, 0x100, f);
		fclose(f);
	}
}

void a2_load_hdisk(int drive2, char * fn)
{
	FILE * f;
	f = fopen(fn, "rb");
	if (f) {
		if (A2_HDISK[drive2] != NULL) free(A2_HDISK[drive2]);
		fseek(f, 0, SEEK_END);
		A2_HDISK_LEN[drive2] = ftell(f);
		fseek(f, 0, SEEK_SET);
		if ( (A2_HDISK[drive2] = (unsigned char *)malloc(A2_HDISK_LEN[drive2])) != NULL ) {
			fread(A2_HDISK[drive2], 1, A2_HDISK_LEN[drive2], f);
			strcpy(A2_HDISK_PATH[drive2], fn);
			A2_HDISK_DIRTY[drive2] = 0;
		} else {
			A2_HDISK_LEN[drive2] = 0;
		}
		fclose(f);
	}
}

void a2_write_hdisk(int drive2)
{
	FILE * f;
	if (A2_HDISK_DIRTY[drive2]) {
		f = fopen(A2_HDISK_PATH[drive2], "wb");
		if (f) {
			fwrite(A2_HDISK[drive2], 1, A2_HDISK_LEN[drive2], f);
			fclose(f);
		}
		A2_HDISK_DIRTY[drive2] = 0;
	}
}

void a2_eject_hdisk(int drive2)
{
	int i;
	if (A2_HDISK[drive2] != NULL) free(A2_HDISK[drive2]);
	A2_HDISK[drive2]=NULL;
	A2_HDISK_LEN[drive2]=0;
	A2_HDISK_PATH[drive2][0]=0;
	A2_HDISK_DIRTY[drive2]=0;
}

/*
Memory map:

    C0F0	(r)   EXECUTE AND RETURN STATUS
	C0F1	(r)   STATUS (or ERROR)
	C0F2	(r/w) COMMAND
	C0F3	(r/w) UNIT NUMBER
	C0F4	(r/w) LOW BYTE OF MEMORY BUFFER
	C0F5	(r/w) HIGH BYTE OF MEMORY BUFFER
	C0F6	(r/w) LOW BYTE OF BLOCK NUMBER
	C0F7	(r/w) HIGH BYTE OF BLOCK NUMBER
	C0F8    (r)   NEXT BYTE
*/

unsigned char a2_hdisk_peek(unsigned short a)
{
	int d = a2_hdisk_drive2;
	int i;
	switch (a) {
	case 0xC0F0:
		if (!A2_HDISK[d] || !A2_HDISK_PATH[d][0]) {
			a2_hdisk_error = 1;
			return 3;
		}
		switch (a2_hdisk_command) {
		default:
		case 0: /* status */
			if (A2_HDISK_LEN[d] == 0) {
				a2_hdisk_error = 1;
				return 8;
			}
			a2_hdisk_error = 0;
			return 0;
			break;
		case 1: /* read */
			if (a2_hdisk_block[d]*512 >= A2_HDISK_LEN[d]) {
				a2_hdisk_error = 1;
				a2_hdisk_ptr[d] = NULL;
				return 8;
			}
			memcpy(a2_hdisk_buffer[d], &A2_HDISK[d][a2_hdisk_block[d]*512], 512);
			a2_hdisk_error = 0;
			a2_hdisk_ptr[d] = a2_hdisk_buffer[d];
			break;
		case 2: /* write */
			A2_HDISK_DIRTY[d] = 1;
			if (a2_hdisk_block[d]*512 >= A2_HDISK_LEN[d]) {
				/* realloc memory to make disk bigger */
				int bytesadded = (a2_hdisk_block[d]*512 - A2_HDISK_LEN[d]) + 512;
				unsigned char * t;
				if ( (t = (unsigned char *)realloc(A2_HDISK[d], A2_HDISK_LEN[d]+bytesadded)) == NULL ) {
					/* for now just error out */
					a2_hdisk_error = 1;
					a2_hdisk_ptr[d] = NULL;
					return 8;
				} else {
					A2_HDISK[d] = t;
					A2_HDISK_LEN[d] += bytesadded;
				}
			}
			for (i=0; i<512; i++) {
				a2_hdisk_buffer[d][i] = a2_mmu_peek(a2_hdisk_memloc[d]+i);
			}
			/* memcpy(a2_hdisk_buffer[d], &A2_MAIN[a2_hdisk_memloc[d]], 512); */
			memcpy(&A2_HDISK[d][a2_hdisk_block[d]*512], a2_hdisk_buffer[d], 512);
			a2_hdisk_error = 0;
			break;
		case 3: /* format */
			break;
		}
		break;
	case 0xC0F1:
		return a2_hdisk_error;
		break;
	case 0xC0F2:
		return a2_hdisk_command;
		break;
	case 0xC0F3:
		return (((!!a2_hdisk_drive2) << 7) | 0x70);
		break;
	case 0xC0F4:
		return (a2_hdisk_memloc[d] & 0xFF);
		break;
	case 0xC0F5:
		return ((a2_hdisk_memloc[d] & 0xFF00) >> 8);
		break;
	case 0xC0F6:
		return (a2_hdisk_block[d] & 0xFF);
		break;
	case 0xC0F7:
		return ((a2_hdisk_block[d] & 0xFF00) >> 8);
		break;
	case 0xC0F8:
		if (!a2_hdisk_ptr[d]) {
			return 0;
		}
		return *(a2_hdisk_ptr[d]++);
		break;
	}
	return 0;
}

void a2_hdisk_poke(unsigned short a, unsigned char ch)
{
	int d = a2_hdisk_drive2;
	switch (a) {
	case 0xC0F2:
		a2_hdisk_command = ch;
		break;
	case 0xC0F3:
		a2_hdisk_drive2 = ((ch & 0x80) >> 7);
		break;
	case 0xC0F4:
		a2_hdisk_memloc[d] = ((a2_hdisk_memloc[d] & 0xFF00) | ch);
		break;
	case 0xC0F5:
		a2_hdisk_memloc[d] = ((a2_hdisk_memloc[d] & 0x00FF) | (ch << 8));
		break;
	case 0xC0F6:
		a2_hdisk_block[d] = ((a2_hdisk_block[d] & 0xFF00) | ch);
		break;
	case 0xC0F7:
		a2_hdisk_block[d] = ((a2_hdisk_block[d] & 0x00FF) | (ch << 8));
		break;
	}
}

