/*
 * Appleblossom - Portable Open-Source Apple IIe Emulator
 * Copyright (C) 2005 Jonathan Bettencourt (jonrelay)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Much of a2disk.c was built off Rich Skrenta's 'a2' Apple II emulator's 'disk.c'.
 * I copied the tables and the algorithm, but the code itself is my own rewrite.
 * His code is barely readable.
 */

#include <stdio.h>
#include <string.h>
#include "a2vars.h"

#define __nib1(a) (((a)>>1)|0xAA)
#define __nib2(a) ((a)|0xAA)
#define __denib(a,b) (((((a)&0x55)<<1)&0xFF)|((b)&0x55))

const unsigned char a2_disk_gap = 0xFF;
const unsigned char a2_disk_nibbletable1[] = {
	0x00, 0x08, 0x04, 0x0C, 0x20, 0x28, 0x24, 0x2C,
	0x10, 0x18, 0x14, 0x1C, 0x30, 0x38, 0x34, 0x3C,
	0x80, 0x88, 0x84, 0x8C, 0xA0, 0xA8, 0xA4, 0xAC,
	0x90, 0x98, 0x94, 0x9C, 0xB0, 0xB8, 0xB4, 0xBC,
	0x40, 0x48, 0x44, 0x4C, 0x60, 0x68, 0x64, 0x6C,
	0x50, 0x58, 0x54, 0x5C, 0x70, 0x78, 0x74, 0x7C,
	0xC0, 0xC8, 0xC4, 0xCC, 0xE0, 0xE8, 0xE4, 0xEC,
	0xD0, 0xD8, 0xD4, 0xDC, 0xF0, 0xF8, 0xF4, 0xFC
};
const unsigned char a2_disk_nibbletable2[] = {
	0x96, 0x97, 0x9A, 0x9B, 0x9D, 0x9E, 0x9F, 0xA6, 
	0xA7, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB2, 0xB3, 
	0xB4, 0xB5, 0xB6, 0xB7, 0xB9, 0xBA, 0xBB, 0xBC, 
	0xBD, 0xBE, 0xBF, 0xCB, 0xCD, 0xCE, 0xCF, 0xD3, 
	0xD6, 0xD7, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 
	0xDF, 0xE5, 0xE6, 0xE7, 0xE9, 0xEA, 0xEB, 0xEC, 
	0xED, 0xEE, 0xEF, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 
	0xF7, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
};
const unsigned char a2_disk_sector_map[] = {
	0x00, 0x0D, 0x0B, 0x09, 0x07, 0x05, 0x03, 0x01,
	0x0E, 0x0C, 0x0A, 0x08, 0x06, 0x04, 0x02, 0x0F
};
const unsigned char a2_disk_sector_pmap[] = {
	0x00, 0x0E, 0x0D, 0x0C, 0x0B, 0x0A, 0x09, 0x08,
	0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x0F
};

int a2_disk_drive2;
int a2_disk_track[2];
int a2_disk_sector[2];
int a2_disk_sector_dirty[2];
unsigned char a2_disk_buffer[2][1024];
unsigned char * a2_disk_sptr;
unsigned char * a2_disk_spt[2];
unsigned char a2_disk_wr_reg;
int a2_disk_write;

unsigned char a2_disk_nibbletable2r(unsigned char ch)
{
	unsigned char i;
	ch |= 0x80;
	for (i=0; i<64; i++) if (a2_disk_nibbletable2[i]>=ch) return i;
	return 0;
}
unsigned char a2_disk_nibbletable1r(unsigned char ch)
{
	unsigned char i;
	ch &= 0xFC;
	for (i=0; i<64; i++) if (a2_disk_nibbletable1[i]==ch) return i;
	return 0;
}


void a2_disk_reset(void)
{
	A2_DISK[0]=NULL;
	A2_DISK[1]=NULL;
	A2_DISK_PATH[0][0]=0;
	A2_DISK_PATH[1][0]=0;
	A2_DISK_WRPROTECT[0]=0;
	A2_DISK_WRPROTECT[1]=0;
	A2_DISK_PRODOS_ORDER[0]=0;
	A2_DISK_PRODOS_ORDER[1]=0;
	A2_DISK_DIRTY[0]=0;
	A2_DISK_DIRTY[1]=0;
	a2_disk_drive2 = 0;
	a2_disk_track[0] = 0;
	a2_disk_track[1] = 0;
	a2_disk_sector[0] = 0;
	a2_disk_sector[1] = 0;
	a2_disk_sptr = NULL;
	a2_disk_spt[0] = NULL;
	a2_disk_spt[1] = NULL;
	a2_disk_wr_reg = 0;
	a2_disk_write = 0;
}

void a2_read_disk_rom(char * fn)
{
	FILE * f;
	f = fopen(fn, "rb");
	if (f) {
		fread(&A2_PERIPH_ROM[0x600], 1, 0x100, f);
		fclose(f);
	}
}

void a2_load_disk(int drive2, int pdorder, char * fn)
{
	FILE * f;
	if (pdorder < 0) {
		int fnl = strlen(fn);
		if ((fn[fnl-4] == '.') && ((fn[fnl-3] == 'n') || (fn[fnl-3] == 'N')) && ((fn[fnl-2] == 'i') || (fn[fnl-2] == 'I')) && ((fn[fnl-1] == 'b') || (fn[fnl-1] == 'B'))) {
			pdorder = 2;
		} else if ((fn[fnl-3] == '.') && ((fn[fnl-2] == 'p') || (fn[fnl-2] == 'P')) && ((fn[fnl-1] == 'o') || (fn[fnl-1] == 'O'))) {
			pdorder = 1;
		} else {
			pdorder = 0;
		}
	}
	f = fopen(fn, "rb");
	if (f) {
		if (A2_DISK[drive2] != NULL) free(A2_DISK[drive2]);
		if ( (A2_DISK[drive2] = (unsigned char *)malloc(0x38E00)) != NULL ) {
			fread(A2_DISK[drive2], 1, 0x38E00, f); /* was 0x23000, which was fine for a 140K image, but .NIBs need 0x38E00 */
			strcpy(A2_DISK_PATH[drive2], fn);
			A2_DISK_PRODOS_ORDER[drive2] = pdorder;
			A2_DISK_DIRTY[drive2] = 0;
			fclose(f);
		}
	}
}

void a2_write_disk(int drive2)
{
	FILE * f;
	if (A2_DISK_DIRTY[drive2]) {
		f = fopen(A2_DISK_PATH[drive2], "wb");
		if (f) {
			fwrite(A2_DISK[drive2], 1, (A2_DISK_PRODOS_ORDER[drive2] == 2)?0x38E00:0x23000, f);
			fclose(f);
		}
		A2_DISK_DIRTY[drive2] = 0;
	}
}

void a2_eject_disk(int drive2)
{
	int i;
	if (A2_DISK[drive2] != NULL) free(A2_DISK[drive2]);
	A2_DISK[drive2]=NULL;
	A2_DISK_PATH[drive2][0]=0;
	A2_DISK_PRODOS_ORDER[drive2]=0;
	A2_DISK_DIRTY[drive2]=0;
}

void a2_disk_decode(int tr, int sec)
{
	unsigned char buf[344];
	unsigned char * a;
	unsigned char * b;
	unsigned char * c;
	unsigned char * dest;
	int i;
	int d = a2_disk_drive2;
	for (i=3; i<72; i++) {
		if (a2_disk_buffer[d][i-3]==0xD5 && a2_disk_buffer[d][i-2]==0xAA && a2_disk_buffer[d][i-1]==0xAD) {
			break;
		}
	}
	if (i==72) return;
	memcpy(buf, &a2_disk_buffer[d][i], 343);
	for (i=0; i<=342; i++) buf[i]=(a2_disk_nibbletable2r(buf[i]))<<2;
	for (i=1; i<=342; i++) buf[i]=buf[i-1]^buf[i]; /* those crazy Apple guys and their rippling XORs... */
	c=&buf[344];
	b=&buf[258];
	a=&buf[172];
	dest=&buf[86];
	while (dest != buf) {
		i = a2_disk_nibbletable1r(*(--dest));
		*(--a) |= (i & 0x03);
		*(--b) |= ((i >> 2) & 0x03);
		*(--c) |= ((i >> 4) & 0x03);
	}
	memcpy(&A2_DISK[d][256*(tr*16 + sec)], &buf[86], 256);
}

void a2_disk_encode(int tr, int sec)
{
	unsigned char buf[344];
	unsigned char * a;
	unsigned char * b;
	unsigned char * c;
	unsigned char * dest;
	int i;
	int d = a2_disk_drive2;
	memcpy(&buf[86], &A2_DISK[d][256*(tr*16 + sec)], 256);
	buf[342]=0;
	buf[343]=0;
	dest=buf;
	a=&buf[86];
	b=&buf[172];
	c=&buf[258];
	do {
		i = (*a++ & 0x03) | ((*b++ & 0x03)<<2) | ((*c++ & 0x03) << 4);
		*dest++ = a2_disk_nibbletable1[i];
	} while (dest != &buf[86]);
	a2_disk_sptr[0]=buf[0];
	for (i=1; i<=342; i++) a2_disk_sptr[i]=buf[i-1]^buf[i];
	for (i=0; i<=342; i++) a2_disk_sptr[i]=a2_disk_nibbletable2[a2_disk_sptr[i]>>2];
	a2_disk_sptr = &a2_disk_sptr[343];
}

void a2_disk_unsetup_sector(int tr, int sec)
{
	int phys_sec = a2_disk_sector_map[sec];
	int virt_sec;
	int i;
	int d = a2_disk_drive2;
	if (A2_DISK_PRODOS_ORDER[d] == 2) {        /* nibble */
		virt_sec = a2_disk_sector_map[sec];
		
		memcpy(&A2_DISK[d][416*(tr*16+virt_sec)], a2_disk_buffer[d], 416);
		return;
	} else if (A2_DISK_PRODOS_ORDER[d] == 1) { /* prodos order */
		virt_sec = a2_disk_sector_pmap[sec];
	} else {                                   /* dos order */
		virt_sec = sec;
	}
	a2_disk_decode(tr, virt_sec);
}

void a2_disk_setup_sector(int tr, int sec)
{
	int chk;
	int phys_sec = a2_disk_sector_map[sec];
	int virt_sec;
	int i;
	int d = a2_disk_drive2;
	if (A2_DISK_PRODOS_ORDER[d] == 2) {        /* nibble */
		virt_sec = a2_disk_sector_map[sec];
		
		memcpy(a2_disk_buffer[d], &A2_DISK[d][416*(tr*16+virt_sec)], 416);
		a2_disk_buffer[d][416] = 0;
		a2_disk_sptr = a2_disk_buffer[d];
		return;
	} else if (A2_DISK_PRODOS_ORDER[d] == 1) { /* prodos order */
		virt_sec = a2_disk_sector_pmap[sec];
	} else {                                   /* dos order */
		virt_sec = sec;
	}
	/*
	 * Rich Skrenta's original nibblization routine doesn't work. It makes
	 * DOS 3.3 crash at $9D86 and ProDOS fail with a Reloc/Configuration Error.
	 * Here is my nibblization routine, based on observations of a .NIB file.
	 * I can now boot both DOS 3.3 and ProDOS.
	 */
	a2_disk_sptr = a2_disk_buffer[d];
	for (i=0; i<21; i++) { /* 16 in Skrenta's routine; it actually doesn't matter how big these gaps are */
		*a2_disk_sptr++ = a2_disk_gap;
	}
	*a2_disk_sptr++ = 0xD5;
	*a2_disk_sptr++ = 0xAA;
	*a2_disk_sptr++ = 0x96;
	*a2_disk_sptr++ = 0xFF;
	*a2_disk_sptr++ = 0xFE;
	*a2_disk_sptr++ = __nib1(tr);
	*a2_disk_sptr++ = __nib2(tr);
	*a2_disk_sptr++ = __nib1(phys_sec);
	*a2_disk_sptr++ = __nib2(phys_sec);
	chk = 254^tr^phys_sec;
	*a2_disk_sptr++ = __nib1(chk);
	*a2_disk_sptr++ = __nib2(chk);
	*a2_disk_sptr++ = 0xDE;
	*a2_disk_sptr++ = 0xAA;
	*a2_disk_sptr++ = 0xEB; /* not in Skrenta's routine; this is the critical part that was missing */
	for (i=0; i<17; i++) { /* 8 in Skrenta's routine; it actually doesn't matter how big these gaps are */
		*a2_disk_sptr++ = a2_disk_gap;
	}
	*a2_disk_sptr++ = 0xD5;
	*a2_disk_sptr++ = 0xAA;
	*a2_disk_sptr++ = 0xAD;
	a2_disk_encode(tr, virt_sec);
	*a2_disk_sptr++ = 0xDE;
	*a2_disk_sptr++ = 0xAA;
	*a2_disk_sptr++ = 0xEB;
	for (i=0; i<15; i++) { /* not in Skrenta's routine; it actually doesn't matter how big these gaps are */
		*a2_disk_sptr++ = a2_disk_gap;
	}
	a2_disk_buffer[d][416] = 0;
	a2_disk_sptr = a2_disk_buffer[d];
}

void a2_disk_motor(unsigned char a)
{
	static int mag[2][4] = {{0,0,0,0},{0,0,0,0}};
	static int pmag[2][4] = {{0,0,0,0},{0,0,0,0}};
	static int ppmag[2][4] = {{0,0,0,0},{0,0,0,0}};
	static int pnum[2] = {0,0};
	static int ppnum[2] = {0,0};
	static int trackpos[2] = {0,0};
	static int ptrack[2] = {0,0};
	int magnum, tr;
	int d = a2_disk_drive2;
	magnum = a >> 1;
	ppmag[d][ppnum[d]] = pmag[d][ppnum[d]];
	ppnum[d] = pnum[d];
	pmag[d][pnum[d]] = mag[d][pnum[d]];
	pnum[d] = magnum;
	if (!(a&1)) {
		mag[d][magnum] = 0;
	} else {
		if (ppmag[d][(magnum+1)&3]) {
			trackpos[d]--;
			if (trackpos[d]<0) {
				trackpos[d]=0;
			}
		}
		if (ppmag[d][(magnum-1)&3]) {
			trackpos[d]++;
			if (trackpos[d]>140) {
				trackpos[d]=140;
			}
		}
		mag[d][magnum] = 1;
	}
	tr = (trackpos[d]+1)/2;
	if (a2_disk_track[d] != tr) {
		if (a2_disk_sector_dirty[d] && A2_DISK_PRODOS_ORDER[d] != 2) {
			a2_disk_unsetup_sector(a2_disk_track[d], a2_disk_sector[d]);
			a2_disk_sector_dirty[d]=0;
		}
		a2_disk_track[d] = tr;
	}
	if (a2_disk_track[d] != ptrack[d]) {
		a2_disk_spt[d] = NULL;
		a2_disk_sector[d] = 0;
		ptrack[d] = a2_disk_track[d];
	}
}

unsigned char a2_disk_memacc(unsigned short addr, unsigned char ch)
{
	int d = a2_disk_drive2;
	switch (addr) {
	case 0xC0E0: /* phase 0 off */
	case 0xC0E1: /* phase 0 on */
	case 0xC0E2: /* phase 1 off */
	case 0xC0E3: /* phase 1 on */
	case 0xC0E4: /* phase 2 off */
	case 0xC0E5: /* phase 2 on */
	case 0xC0E6: /* phase 3 off */
	case 0xC0E7: /* phase 3 on */
		a2_disk_motor(addr & 7);
		break;
	case 0xC0E8: /* drive off */
		break;
	case 0xC0E9: /* drive on */
		break;
	case 0xC0EA: /* drive 1 */
		a2_disk_spt[d] = a2_disk_sptr;
		a2_disk_drive2 = 0;
		a2_disk_sptr = a2_disk_spt[0];
		break;
	case 0xC0EB: /* drive 2 */
		a2_disk_spt[d] = a2_disk_sptr;
		a2_disk_drive2 = 1;
		a2_disk_sptr = a2_disk_spt[1];
		break;
	case 0xC0EC: /* shift data reg */
		if ((!A2_DISK_PATH[d][0])||(!A2_DISK[d])) {
			return 255;
		}
		if (  (!a2_disk_sptr) || (!(*a2_disk_sptr))  ) {
			if (a2_disk_sector_dirty[d] && A2_DISK_PRODOS_ORDER[d] != 2) {
				a2_disk_unsetup_sector(a2_disk_track[d], a2_disk_sector[d]);
				a2_disk_sector_dirty[d]=0;
			}
			a2_disk_sector[d]--;
			if (a2_disk_sector[d] < 0) { a2_disk_sector[d]=15; }
			a2_disk_setup_sector(a2_disk_track[d], a2_disk_sector[d]);
		}
		if (a2_disk_write && a2_disk_wr_reg) {
			A2_DISK_DIRTY[d] = 1;
			(*a2_disk_sptr++) = a2_disk_wr_reg;
			if (A2_DISK_PRODOS_ORDER[d] == 2) {
				int tr = a2_disk_track[d];
				int sec = a2_disk_sector[d];
				int phys_sec = a2_disk_sector_map[sec];
				int byte_num = a2_disk_sptr-a2_disk_buffer[d]+1;
				A2_DISK[d][416*(tr*16+phys_sec)+byte_num] = a2_disk_wr_reg;
			} else {
				a2_disk_sector_dirty[d]=1;
			}
			return a2_disk_wr_reg;
		} else {
			return (*a2_disk_sptr++);
		}
	case 0xC0ED: /* load data reg */
		a2_disk_wr_reg = ch;
		break;
	case 0xC0EE: /* read */
		a2_disk_write = 0;
		return (A2_DISK_WRPROTECT[d]?0xFF:0);
		break;
	case 0xC0EF: /* write */
		a2_disk_write = 1;
		a2_disk_wr_reg = ch;
		break;
	}
	return 0;
}

unsigned char a2_disk_peek(unsigned short addr)
{
	return a2_disk_memacc(addr, 0);
}

void a2_disk_poke(unsigned short addr, unsigned char ch)
{
	a2_disk_memacc(addr, ch);
}

void a2_disk_test(void)
{
	int i;
	A2_DISK[1] = (unsigned char *)malloc(0x38E00);
	strcpy(A2_DISK_PATH[1], "/dev/null");
	A2_DISK_PRODOS_ORDER[1] = 0;
	A2_DISK_WRPROTECT[1] = 0;
	a2_disk_drive2 = 1;
	a2_disk_track[1]=0;
	a2_disk_sector[1]=0;
	
	printf("Nib1:\n");
	for (i=0; i<64; i++) {
		printf("%02X ",a2_disk_nibbletable1[i]);
	}
	printf("\nNib1r:\n");
	for (i=0; i<64; i++) {
		printf("%02X ",a2_disk_nibbletable1r(a2_disk_nibbletable1[i]));
	}
	printf("\n");
	
	printf("Nib2:\n");
	for (i=0; i<64; i++) {
		printf("%02X ",a2_disk_nibbletable2[i]);
	}
	printf("\nNib2r:\n");
	for (i=0; i<64; i++) {
		printf("%02X ",a2_disk_nibbletable2r(a2_disk_nibbletable2[i]));
	}
	printf("\n\n");
	
	for (i=0; i<256; i++) {
		A2_DISK[1][i]=i;
	}
	a2_disk_setup_sector(0,0);
	printf("HexCrap Nibbed:\n");
	for (i=0; i<416; i++) {
		printf("%02X ",a2_disk_buffer[1][i]);
		if ((i%32)==31) { printf("\n"); }
	}
	a2_disk_unsetup_sector(0,0);
	printf("\nHexCrap Denibbed:\n");
	for (i=0; i<256; i++) {
		printf("%02X ",A2_DISK[1][i]);
		if ((i%16)==15) { printf("\n"); }
	}
	printf("\n");
	
	for (i=0; i<6; i++) {
		strncpy(&A2_DISK[1][i*48], "Margaret Chastain is a Child Diva!!!! <3 xox <3 ", 48);
	}
	a2_disk_setup_sector(0,0);
	printf("Margaret Nibbed:\n");
	for (i=0; i<416; i++) {
		printf("%02X ",a2_disk_buffer[1][i]);
		if ((i%32)==31) { printf("\n"); }
	}
	a2_disk_unsetup_sector(0,0);
	printf("\nMargaret Denibbed:\n");
	for (i=0; i<256; i++) {
		printf("%c ",A2_DISK[1][i]);
		if ((i%16)==15) { printf("\n"); }
	}
	printf("\n");
}

