	Page	58,132
	Title	COMMAND.ASM	Apple Emulator Debugger Commands
;******************************************************************************
;
;   Name:	COMMAND.ASM	Apple Emulator Debugger Commands
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the Apple
;	emulator resident debugger commands.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Execute_Command 	; Debugger execute command routine
	Public	Go_Command		; Debugger go command routine
	Public	Eval_Command		; Debugger eval command routine
	Public	Dump_Command		; Debugger dump command routine
	Public	Dbyte_Command		; Debugger dump byte command routine
	Public	Dword_Command		; Debugger dump word command routine
	Public	Code_Command		; Debugger unassemble command routine
	Public	Reg_Command		; Debugger register command routine
	Public	Radix_Command		; Debugger radix command routine
	Public	Help_Command		; Debugger help command routine
	Public	Trace_Command		; Debugger trace command routine
	Public	Enter_Command		; Debugger enter command routine
	Public	Ebyte_Command		; Debugger enter byte command routine
	Public	Eword_Command		; Debugger enter word command routine
	Public	Fill_Command		; Debugger fill command routine
	Public	Move_Command		; Debugger move command routine
	Public	Compare_Command 	; Debugger compare command routine
	Public	Search_Command		; Debugger search command routine
	Public	Bpoint_Command		; Debugger set breakpoint routine
	Public	Bdsable_Command 	; Debugger disable breakpoint routine
	Public	Benable_Command 	; Debugger enable breakpoint routine
	Public	Bclear_Command		; Debugger clear breakpoint routine
	Public	Blist_Command		; Debugger list breakpoint routine
	Public	Proceed_Command 	; Debugger process command routine
	Public	Flip_Command		; Debugger fill command routine
;
;  External Declarations
;
	Extrn	TTY_Init:Near		; Initialize TTY mode routine	  (TTY)
	Extrn	TTY_Reset:Near		; Reset TTY mode routine	  (TTY)
	Extrn	Write_TTY:Near		; Write TTY routine		  (TTY)
	Extrn	Set_Position:Near	; Set current position routine	  (TTY)
	Extrn	Set_Row:Near		; Set current row routine	  (TTY)
	Extrn	Set_Column:Near 	; Set current column routine	  (TTY)
	Extrn	Set_Attribute:Near	; Set current attribute routine   (TTY)
	Extrn	Set_Foreground:Near	; Set current foreground routine  (TTY)
	Extrn	Set_Background:Near	; Set current background routine  (TTY)
	Extrn	Set_Cursor:Near 	; Set cursor position routine	  (TTY)
	Extrn	Set_Type:Near		; Set cursor type routine	  (TTY)
	Extrn	Get_Position:Near	; Get current position routine	  (TTY)
	Extrn	Get_Row:Near		; Get current row routine	  (TTY)
	Extrn	Get_Column:Near 	; Get current column routine	  (TTY)
	Extrn	Get_Attribute:Near	; Get current attribute routine   (TTY)
	Extrn	Get_Foreground:Near	; Get current foreground routine  (TTY)
	Extrn	Get_Background:Near	; Get current background routine  (TTY)
	Extrn	Get_Cursor:Near 	; Get cursor position routine	  (TTY)
	Extrn	Get_Type:Near		; Get cursor type routine	  (TTY)
	Extrn	Read_Memory:Near	; Read 65C02 memory routine	(APPLE)
	Extrn	Write_Memory:Near	; Write 65C02 memory routine	(APPLE)
	Extrn	Break_Set:Near		; Breakpoint set routine	(BREAK)
	Extrn	Break_Find:Near 	; Breakpoint find routine	(BREAK)
	Extrn	Break_Clear:Near	; Breakpoint clear routine	(BREAK)
	Extrn	Break_Enable:Near	; Breakpoint enable routine	(BREAK)
	Extrn	Break_Disable:Near	; Breakpoint disable routine	(BREAK)
	Extrn	Break_List:Near 	; Breakpoint list routine	(BREAK)
	Extrn	Get_Window:Near 	; Get current window routine   (WINDOW)
	Extrn	Set_Window:Near 	; Set current window routine   (WINDOW)
	Extrn	Scroll_Up:Near		; Scroll window up routine     (WINDOW)
	Extrn	Scroll_Down:Near	; Scroll window down routine   (WINDOW)
	Extrn	Clear_Window:Near	; Clear window routine	       (WINDOW)
	Extrn	Save_Window:Near	; Save window routine	       (WINDOW)
	Extrn	Restore_Window:Near	; Restore window routine       (WINDOW)
	Extrn	Get_Parameter:Near	; Get parameter routine       (SUPPORT)
	Extrn	Upper_Case:Near 	; Convert to upper case       (SUPPORT)
	Extrn	Match_Parameter:Near	; Match parameter routine     (SUPPORT)
	Extrn	ASCII_Binary:Near	; ASCII to binary conversion  (SUPPORT)
	Extrn	Get_Key:Near		; Get keyboard code routine  (KEYBOARD)
	Extrn	Printf:Near		; Generic Printf routine       (PRINTF)
	Extrn	Dump_Code:Near		; Dump code listing routine	(DEBUG)
	Extrn	Dump_Data:Near		; Dump RAM data routine 	(DEBUG)
	Extrn	Dump_Stack:Near 	; Dump stack data routine	(DEBUG)
	Extrn	Print_Title:Near	; Print debugger title routine	(DEBUG)
	Extrn	Print_Status:Near	; Print status routine		(DEBUG)
	Extrn	Print_Register:Near	; Print registers routine	(DEBUG)
	Extrn	Get_Expression:Near	; Get expression value routine(EXPRESS)
	Extrn	Key_Status:Byte 	; Keyboard status byte	     (KEYBOARD)
	Extrn	RAM_Space:Word		; RAM space segment value	 (DATA)
	Extrn	ALT_Space:Word		; Alternate RAM space segment	 (DATA)
	Extrn	Video_Segment:Word	; Video segment value		 (DATA)
	Extrn	Input_Buffer:Byte	; Input buffer storage area	 (DATA)
	Extrn	Parm_Buffer:Byte	; Parameter buffer storage area  (DATA)
	Extrn	Data_Buffer:Byte	; Data buffer storage area	 (DATA)
	Extrn	Delimit_Input:Byte	; Input delimiter table 	 (DATA)
	Extrn	Delimit_Debug:Byte	; Debug delimiter table 	 (DATA)
	Extrn	Delimit_Express:Byte	; Expression delimiter table	 (DATA)
	Extrn	Flag_Encode:Byte	; CPU flag encoding table	 (DATA)
	Extrn	Flag_Decode:Byte	; CPU flag decoding table	 (DATA)
	Extrn	Emulate_Flag:Byte	; Emulator system flag byte	 (DATA)
	Extrn	System_Flag:Byte	; Apple emulator system flag byte(DATA)
	Extrn	Video_Flag:Byte 	; Video system flag byte	 (DATA)
	Extrn	Radix:Word		; Current numeric radix value	 (DATA)
	Extrn	Screen_Window:Word	; Entire video screen window	 (DATA)
	Extrn	Title_Window:Word	; Debugger title window 	(DEBUG)
	Extrn	Data_Window:Word	; Debugger data window		(DEBUG)
	Extrn	Status_Window:Word	; Debugger status window	(DEBUG)
	Extrn	Code_Window:Word	; Debugger code window		(DEBUG)
	Extrn	Register_Window:Word	; Debugger register window	(DEBUG)
	Extrn	Stack_Window:Word	; Debugger stack window 	(DEBUG)
	Extrn	Command_Window:Word	; Debugger command window	(DEBUG)
	Extrn	Code_Address:Word	; Current code dump address	(DEBUG)
	Extrn	Data_Address:Word	; Current data dump address	(DEBUG)
	Extrn	Current_Address:Word	; Current address value 	(DEBUG)
	Extrn	Compare_Address:Word	; Compare address value 	(DEBUG)
	Extrn	Last_Code:Word		; Last code dump address	(DEBUG)
	Extrn	Last_Data:Word		; Last data dump address	(DEBUG)
	Extrn	Dump_Format:Word	; Current data dump format	(DEBUG)
	Extrn	Enter_Format:Word	; Current data entry format	(DEBUG)
	Extrn	Color_Table:Byte	; Debugger color table		(DEBUG)
	Extrn	Debug_Table:Byte	; Debugger command table	(DEBUG)
	Extrn	Value:Word		; Current value storage area	(DEBUG)
	Extrn	Compare:Word		; Comparison value storage area (DEBUG)
	Extrn	Trace_Count:Word	; Trace count value storage area(DEBUG)
	Extrn	Go_Usage:Byte		; Go command usage format     (MESSAGE)
	Extrn	Eval_Usage:Byte 	; Eval command usage format   (MESSAGE)
	Extrn	Eval_Arg:Word		; Evaluation argument list    (MESSAGE)
	Extrn	Eval_Fmt:Byte		; Evaluation format string    (MESSAGE)
	Extrn	Dump_Usage:Byte 	; Dump command usage format   (MESSAGE)
	Extrn	Code_Usage:Byte 	; Code command usage format   (MESSAGE)
	Extrn	Reg_Usage:Byte		; Reg. command usage format   (MESSAGE)
	Extrn	Radix_Usage:Byte	; Radix command usage format  (MESSAGE)
	Extrn	Radix_Arg:Word		; Radix dump argument list    (MESSAGE)
	Extrn	Radix_Fmt:Byte		; Radix dump format string    (MESSAGE)
	Extrn	Help_Usage:Byte 	; Help command usage format   (MESSAGE)
	Extrn	Trace_Usage:Byte	; Trace command usage format  (MESSAGE)
	Extrn	Enter_Usage:Byte	; Enter command usage format  (MESSAGE)
	Extrn	Fill_Usage:Byte 	; Fill command usage format   (MESSAGE)
	Extrn	Move_Usage:Byte 	; Move command usage format   (MESSAGE)
	Extrn	Compare_Usage:Byte	; Compare command usage format(MESSAGE)
	Extrn	Search_Usage:Byte	; Search command usage format (MESSAGE)
	Extrn	Bpoint_Usage:Byte	; Software breakpoint usage   (MESSAGE)
	Extrn	Bdsable_Usage:Byte	; Disable breakpoint usage    (MESSAGE)
	Extrn	Benable_Usage:Byte	; Enable breakpoint usage     (MESSAGE)
	Extrn	Bclear_Usage:Byte	; Clear breakpoint usage      (MESSAGE)
	Extrn	Blist_Usage:Byte	; List breakpoint usage       (MESSAGE)
	Extrn	Proceed_Usage:Byte	; Process command usage       (MESSAGE)
	Extrn	Flip_Usage:Byte 	; Flip command usage format   (MESSAGE)
	Extrn	Continue_Fmt:Byte	; Continue message format     (MESSAGE)
	Extrn	Diff_Arg:Word		; Difference argument list    (MESSAGE)
	Extrn	Diff_Fmt:Byte		; Difference format string    (MESSAGE)
	Extrn	Find_Arg:Word		; Find message argument list  (MESSAGE)
	Extrn	Find_Fmt:Byte		; Find message format string  (MESSAGE)
	Extrn	Help_Table:Word 	; Debug commands help table   (MESSAGE)
	Extrn	Usage_Table:Word	; Debug commands usage table  (MESSAGE)
;
;  LOCAL Equates
;
HEX		Equ	16		; Base 16 - Hexadecimal
DECIMAL 	Equ	10		; Base 10 - Decimal
OCTAL		Equ	8		; Base 8  - Octal
BINARY		Equ	2		; Base 2  - Binary
LINE_COUNT	Equ	08h		; Number of bytes per data line
FORMAT_BYTE	Equ	00h		; Data byte format value
FORMAT_WORD	Equ	02h		; Data word format value
INTEL_ATTR	Equ	00h		; Intel character attribute base
APPLE_ATTR	Equ	80h		; Apple character attribute base
INTEL_STRING	Equ	DOUBLE_QUOTE	; Intel character string delimiter
APPLE_STRING	Equ	SINGLE_QUOTE	; Apple character string delimiter
RADIX_COUNT	Equ	04h		; Number of valid radix values (4)
JSR_OPCODE	Equ	20h		; Jump to subroutine opcode value
JSR_SIZE	Equ	03h		; Jump to subroutine instruction size
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include 	Strucs.inc	; Include the structure definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Execute_Command Apple Emulator Debug Execute Command
	Page	+
;******************************************************************************
;
;	Execute_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Set carry indicating no more commands (Run emulator)
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Execute_Command Proc	Near		; Debug execute command procedure
	xor	ax,ax			; Clear zero to indicate
	dec	ax			;			 debug mode exit
	stc				; Set carry indicating no more input
Execute_Exit:
	ret				; Return to the caller
Execute_Command Endp			; End of the Execute_Command procedure
	Subttl	Go_Command	Apple Emulator Debug Go Command
	Page	+
;******************************************************************************
;
;	Go_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		If assignment type delimiter
;			Call routine to get go address
;			If valid go address
;				Setup the new execution address
;			Else
;				Call routine to print go usage message
;				Clear carry indicating more input
;			Endif
;		Endif
;		Call routine to get expression value
;		If valid expression value given (Breakpoint)
;			Call routine to setup the local breakpoint
;		Else
;			If assignment type expression
;				Call routine to get go address
;				If valid go address
;					Setup the new execution address
;				Else
;					Call routine to print go usage message
;					Clear carry indicating more input
;				Endif
;			Endif
;		Endif
;		Set carry indicating no more commands (Run emulator)
;		Clear zero flag indicating to exit debug
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Go_Command	Proc	Near		; Debug go command procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Go_Assign		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Go_Assign:
	test	ah,ASSIGN_TYPE		; Check for go at location command
	jz	Go_Check		; Jump if not go at location command
Go_Location:
	call	Get_Expression		; Call routine to get expression value
	jbe	Go_Error		; Jump if no/illegal expression value
	Save	si,ds			; Save the debug command pointer
	mov	si,ax			; Get location value (From expression)
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	Setup				; Setup to
	mov	ss:[bp.SI_Register],si	;	   execute from
	mov	ss:[bp.DI_Register],di	;			desired location
	Restore si,ds			; Restore pointer to debug command
Go_Check:
	call	Get_Expression		; Call routine to get expression value
	ja	Go_Break		; Jump if breakpoint given
	jc	Go_Error		; Jump if expression error
	or	bh,bh			; Check for valid delimiter
	jz	Go_Execute		; Jump if end of input buffer
	and	bh,SPC_MASK		; Mask off all but special field
	cmp	bh,SPC_ASSIGN		; Check for assignment delimiter
	jne	Go_Error		; Jump if unknown delimiter
	inc	dx			; Increment past assignment delimiter
	jmp	Short Go_Location	; Go try to get the location value
Go_Error:
	lea	si,ds:[Go_Usage]	; Get pointer to go usage message
	call	Printf			; Call routine to print error message
	clc				; Clear carry indicating more input
	jmp	Short Go_Exit		; Go return control to the caller
Go_Break:
	mov	bx,ax			; Save the breakpoint address
	mov	ax,BREAKPOINTS		; Get the local breakpoint number
	mov	cl,TYPE_SOFT		; Get the software type value
	call	Break_Set		; Call routine to set breakpoint
	jc	Go_Error		; Jump if error setting breakpoint
Go_Execute:
	xor	ax,ax			; Clear zero to indicate
	dec	ax			;			 debug mode exit
	stc				; Set carry indicating no more input
Go_Exit:
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Go_Command	Endp			; End of the Go_Command procedure
	Subttl	Eval_Command	Apple Emulator Debug Eval Command
	Page	+
;******************************************************************************
;
;	Eval_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		While there are more expressions
;			Call routine to get the expression value
;			If a valid expression was given
;				Call routine to print the expression value
;			Else
;				Call routine to print eval command usage message
;				Exit the while loop
;			Endif
;		EndWhile
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Eval_Command	Proc	Near		; Debug eval command procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Eval_Loop		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Eval_Loop:
	call	Get_Expression		; Call routine to get expression value
	ja	Print_Value		; Jump if valid expression value
	jc	Eval_Error		; Jump if expression error
	or	bh,bh			; Check for valid delimiter
	jz	Eval_Exit		; Jump if end of input buffer
	and	bh,SPC_MASK		; Mask off all but special delimiters
	jz	Short Eval_Loop 	; Jump if not a special delimiter
Eval_Error:
	lea	si,ds:[Eval_Usage]	; Get pointer to eval usage message
	call	Printf			; Call routine to print error message
	jmp	Short Eval_Exit 	; Go return control to the caller
Print_Value:
	push	si			; Save the debug command offset
	mov	ds:[Value],ax		; Save the actual expression value
	lea	bx,ds:[Eval_Arg]	; Get pointer to eval argument list
	lea	si,ds:[Eval_Fmt]	; Get pointer to eval format string
	call	Printf			; Call routine to print the values
	pop	si			; Restore the debug command offset
	jmp	Short Eval_Loop 	; Loop till all expressions printed
Eval_Exit:
	Restore si,di			; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Eval_Command	Endp			; End of the Eval_Command procedure
	Subttl	Dump_Command	Apple Emulator Debug Dump Command
	Page	+
;******************************************************************************
;
;	Dump_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get the expression value
;		If a valid expression was given
;			Set dump address to the expression value
;		Else
;			If no expression was given
;				Calculate the new dump adress from last address
;				Set dump address to the calculated value
;			Else
;				Call routine to print dump command usage message
;			Endif
;		Endif
;		Call routine to dump the data
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Dump_Command	Proc	Near		; Debug dump command procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Dump_Address		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Dump_Address:
	call	Get_Expression		; Call routine to get expression value
	jae	Check_Value		; Jump if valid expression value
Dump_Error:
	lea	si,ds:[Dump_Usage]	; Get pointer to dump usage message
	call	Printf			; Call routine to print error message
	jmp	Short Dump_Exit 	; Go return control to the caller
Check_Value:
	jz	Calculate_Data		; Jump if no expression was given
	mov	ds:[Data_Address],ax	; Save the actual expression value
	jmp	Short Perform_Dump	; Go dump the requested data
Calculate_Data:
	call	Get_Window		; Get the current window structure
	mov	ax,ds:[si.Lower_Right_Row]
	sub	ax,ds:[si.Upper_Left_Row]
	mov	ah,LINE_COUNT		; Get the number of bytes per line
	mul	ah			; Compute the next data address offset
	add	ds:[Data_Address],ax	; Update the data dump address
Perform_Dump:
	call	Dump_Data		; Call routine to dump requested data
Dump_Exit:
	Restore si,di			; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Dump_Command	Endp			; End of the Dump_Command procedure
	Subttl	Dbyte_Command	Apple Emulator Debug Dump Byte Command
	Page	+
;******************************************************************************
;
;	Dbyte_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Set current data format to bytes
;		Call routine to handle dump command
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Dbyte_Command	Proc	Near		; Debug dump byte command procedure
	mov	ds:[Dump_Format],FORMAT_BYTE
	call	Dump_Command		; Call routine to handle dump command
Dbyte_Exit:
	ret				; Return to the caller
Dbyte_Command	Endp			; End of the Dbyte_Command procedure
	Subttl	Dword_Command	Apple Emulator Debug Dump Word Command
	Page	+
;******************************************************************************
;
;	Dword_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Set current data format to words
;		Call routine to handle dump command
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Dword_Command	Proc	Near		; Debug dump word command procedure
	mov	ds:[Dump_Format],FORMAT_WORD
	call	Dump_Command		; Call routine to handle dump command
Dword_Exit:
	ret				; Return to the caller
Dword_Command	Endp			; End of the Dword_Command procedure
	Subttl	Code_Command	Apple Emulator Debug Unassemble Command
	Page	+
;******************************************************************************
;
;	Code_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get the expression value
;		If a valid expression was given
;			Set code address to the expression value
;		Else
;			If no expression was given
;				Use current code address value
;			Else
;				Call routine to print code command usage message
;			Endif
;		Endif
;		Call routine to dump the code
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Code_Command	Proc	Near		; Debug unassemble command procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Get_Address		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Get_Address:
	call	Get_Expression		; Call routine to get expression value
	jae	Value_Check		; Jump if valid expression value
Code_Error:
	lea	si,ds:[Code_Usage]	; Get pointer to code usage message
	call	Printf			; Call routine to print error message
	jmp	Short Code_Exit 	; Go return control to the caller
Value_Check:
	jz	Perform_Unassemble	; Jump if no expression was given
	mov	ds:[Code_Address],ax	; Save the actual expression value
Perform_Unassemble:
	call	Dump_Code		; Call routine to dump requested code
Code_Exit:
	Restore si,di			; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Code_Command	Endp			; End of the Code_Command procedure
	Subttl	Reg_Command	Apple Emulator Debug Register Command
	Page	+
;******************************************************************************
;
;	Reg_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Reg_Command	Proc	Near		; Debug register command procedure
	Save	si,di,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jnz	Reg_Error		; Jump if there was a numeric delimiter
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,CR			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Debug]	; Get pointer to debug delimiter table
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	lea	si,ds:[Input_Buffer]	; Get pointer to the input buffer
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	call	Get_Parameter		; Call routine to get next parameter
	jbe	Reg_Error		; Jump if no/null parameter
	mov	bx,ax			; Save parameter length/delimiter type
	test	bh,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Reg_Convert		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Reg_Convert:
	call	Upper_Case		; Call routine to convert to uppercase
	push	si			; Save the input parameter pointer
	lea	si,ds:[Register_Table]	; Get pointer to start of register table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the match table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	pop	si			; Restore the input parameter pointer
	jnc	Get_Register		; Jump if register match is found
Reg_Error:
	lea	si,ds:[Reg_Usage]	; Get pointer to register usage message
	call	Printf			; Call routine to print error message
	jmp	Short Reg_Exit		; Go return control to the caller
Get_Register:
	cbw				; Convert matching index to full word
	mov	cx,ax			; Save the matching register index
	call	Get_Expression		; Call routine to get expression value
	jbe	Reg_Error		; Jump if no/invalid expression value
	mov	bx,cx			; Get the matching register index
	shl	bx,1			; Convert match number to table index
	call	ds:[Set_Table + bx]	; Call routine to set register value
Reg_Exit:
	Restore si,di,es		; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Reg_Command	Endp			; End of the Reg_Command procedure
	Subttl	Radix_Command	Apple Emulator Debug Radix Command
	Page	+
;******************************************************************************
;
;	Radix_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get the expression value
;		If a valid expression was given
;			Check for a valid radix value (Binary,Octal,Decimal,Hex)
;			If valid radix value given
;				Set current radix to given value
;			Else
;				Call routine to print radix command usage
;			Endif
;		Else
;			Call routine to print radix command usage message
;		Endif
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Radix_Command	Proc	Near		; Debug radix command procedure
	Save	si,di,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Get_Radix		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Get_Radix:
	call	Get_Expression		; Call routine to get expression value
	ja	Check_Radix		; Jump if valid expression value
	jc	Radix_Error		; Jump if invalid expression value
	or	bh,bh			; Check for valid delimiter
	jnz	Radix_Error		; Jump if not end of input buffer
Radix_Dump:
	lea	bx,ds:[Radix_Arg]	; Get pointer to radix argument list
	lea	si,ds:[Radix_Fmt]	; Get pointer to radix format string
	call	Printf			; Call routine to print current radix
	jmp	Short Radix_Exit	; Go return control to the caller
Radix_Error:
	lea	si,ds:[Radix_Usage]	; Get pointer to radix usage message
	call	Printf			; Call routine to print error message
	jmp	Short Radix_Exit	; Go return control to the caller
Check_Radix:
	mov	bx,ds			; Setup access to
	mov	es,bx			;		  the data segment
	mov	cx,RADIX_COUNT		; Get the number of valid radix values
	lea	di,ds:[Radix_Table]	; Get pointer to the radix table
	repne	scasw			; Check for a valid radix value
	jne	Radix_Error		; Jump if an invalid radix given
	mov	ds:[Radix],ax		; Setup the new radix value
Radix_Exit:
	Restore si,di,es		; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Radix_Command	Endp			; End of the Radix_Command procedure
	Subttl	Set_PC		Apple Emulator Set Program Counter
	Page	+
;******************************************************************************
;
;	Set_PC(Value, Stack_Frame)
;
;		Setup to execute from given value (Set PC)
;		Call routine to print status line
;		If current address within last code dumped
;			Use last code address for dump
;		Else
;			Use current address for dump
;		Endif
;		Call routine to dump requested code (Refresh)
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Value
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Set_PC		Proc	Near		; Set 65C02 program counter procedure
	Save	si,ds			; Save the debug command pointer
	mov	si,ax			; Get the new program counter value
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	Setup				; Setup to
	mov	ss:[bp.SI_Register],si	;	   execute from
	mov	ss:[bp.DI_Register],di	;			desired location
	mov	ax,si			; Compute the actual
	dec	ax			;		     program counter
	Restore si,ds			; Restore pointer to debug command
	cmp	ax,ds:[Last_Code]	; Check against last code address
	jb	Code_Update		; Jump if outside last code dump
	cmp	ax,ds:[Code_Address]	; Check against last code address
	jae	Code_Update		; Jump if outside last code dump
	mov	ax,ds:[Last_Code]	; Use last code dump address value
Code_Update:
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Print_Status		; Call routine to print the status line
	call	Dump_Code		; Call routine to dump requested code
	ret				; Return to the caller
Set_PC		Endp			; End of the Set_PC procedure
	Subttl	Set_SP		Apple Emulator Set Stack Pointer
	Page	+
;******************************************************************************
;
;	Set_SP(Value, Stack_Frame)
;
;		Set 65C02 stack pointer to given value
;		Call routine to print status line
;		Call routine to dump the stack
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Value
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Set_SP		Proc	Near		; Set 65C02 stack pointer procedure
	mov	ss:[bp.BX_Register],ax	; Set stack pointer to given value
	call	Print_Status		; Call routine to print the status line
	call	Dump_Stack		; Call routine to dump the stack
	ret				; Return to the caller
Set_SP		Endp			; End of the Set_SP procedure
	Subttl	Set_ACC 	Apple Emulator Set Accumulator
	Page	+
;******************************************************************************
;
;	Set_ACC(Value, Stack_Frame)
;
;		Set 65C02 accumulator to given value (Byte)
;		Call routine to print register line
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Value (Byte)
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Set_ACC 	Proc	Near		; Set 65C02 accumulator procedure
	mov	ss:[bp.DL_Register],al	; Set accumulator to given value (Byte)
	call	Print_Register		; Call routine to print register line
	ret				; Return to the caller
Set_ACC 	Endp			; End of the Set_ACC procedure
	Subttl	Set_X		Apple Emulator Set X Index
	Page	+
;******************************************************************************
;
;	Set_X(Value, Stack_Frame)
;
;		Set 65C02 X index to given value (Byte)
;		Call routine to print register line
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Value (Byte)
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Set_X		Proc	Near		; Set 65C02 X index procedure
	mov	ss:[bp.CH_Register],al	; Set X index to given value (Byte)
	call	Print_Register		; Call routine to print register line
	ret				; Return to the caller
Set_X		Endp			; End of the Set_X procedure
	Subttl	Set_Y		Apple Emulator Set Y Index
	Page	+
;******************************************************************************
;
;	Set_Y(Value, Stack_Frame)
;
;		Set 65C02 Y index to given value (Byte)
;		Call routine to print register line
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Value (Byte)
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Y		Proc	Near		; Set 65C02 Y index procedure
	mov	ss:[bp.CL_Register],al	; Set Y index to given value (Byte)
	call	Print_Register		; Call routine to print register line
	ret				; Return to the caller
Set_Y		Endp			; End of the Set_Y procedure
	Subttl	Set_Flags	Apple Emulator Set Flags
	Page	+
;******************************************************************************
;
;	Set_Flags(Value, Stack_Frame)
;
;		Decode flag bits from 65C02 to 80x86
;		Clear the r (Reserved) flag bit
;		Set 65C02 flags to given value
;		Call routine to print register line
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Value (Byte)
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Flags	Proc	Near		; Set 65C02 flags procedure
	mov	bx,ax			; Get the new flags register value
	xor	bh,bh			; Convert flags value to full word
	mov	al,cs:[bx + Flag_Decode]; Get the decoded flag value (80x86)
	and	al,Not CPU_R		; Make sure reserved (r) bit is clear
	mov	ss:[bp.DH_Register],al	; Set the 65C02 flags register
	call	Print_Register		; Call routine to print register line
	ret				; Return to the caller
Set_Flags	Endp			; End of the Set_Flags procedure
	Subttl	Help_Command	Apple Emulator Debug Help Command
	Page	+
;******************************************************************************
;
;	Help_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get next parameter (Specific command help)
;		If there was no parameter given
;			Call routine to clear current window (Command)
;			Call routine to get current window
;			Calculate the maximum lines at a time
;			While there is more to print
;				Print the next help line(s)
;				Print continue message on screen
;				Wait for user key input
;			EndWhile
;			Print the help command usage message
;		Else a parameter was given
;			Call routine to convert to uppercase
;			Call routine to check for command match
;			If this is a valid command
;				Print the command usage message
;			Endif
;			Print help command usage message
;		Endif
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Help_Command	Proc	Near		; Debug help command procedure
	Save	si,di,bp,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jnz	Help_Error		; Jump if there was a numeric delimiter
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,CR			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	lea	si,ds:[Input_Buffer]	; Get pointer to the input buffer
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	call	Get_Parameter		; Call routine to get next parameter
	ja	Check_Command		; Jump if valid parameter given
Help_Index:
	call	Clear_Window		; Call routine to clear current window
	call	Get_Window		; Get the current window structure
	mov	bp,ds:[si.Lower_Right_Row]
	sub	bp,ds:[si.Upper_Left_Row]
	lea	si,ds:[Help_Table]	; Get pointer to the help table
	mov	cx,bp			; Get the number of lines to print
Help_Loop:
	lodsw				; Get the next message to print
	or	ax,ax			; Check for end of messages
	jz	Help_Done		; Jump if all messages have been printed
	xchg	ax,si			; Setup to print the next help message
	call	Printf			; Call routine to print the message
	xchg	ax,si			; Restore the help table pointer
	loop	Help_Loop		; Loop till all lines are printed
	push	si			; Save the help table pointer
	call	Get_Attribute		; Get the current attribute values
	xchg	ax,bx			; Swap foreground/background colors
	call	Set_Attribute		; Set the new attribute value (Invert)
	lea	si,ds:[Continue_Fmt]	; Get pointer to the continue message
	call	Printf			; Call routine to print continue message
	call	Get_Key 		; Call routine to wait for a key
	push	ax			; Save the actual keystroke
	call	Get_Attribute		; Get the current attribute values
	xchg	ax,bx			; Swap foreground/background colors
	call	Set_Attribute		; Set the new attribute value (Restore)
	mov	cx,bp			; Get the number of lines to print
	pop	ax			; Restore the keystroke value
	pop	si			; Restore the help table pointer
	cmp	al,ESCAPE		; Check for an escape character
	je	Help_Done		; Jump if user presses escape character
	jmp	Short Help_Loop 	; Keep looping till all printed
Help_Error:
	lea	si,ds:[Help_Usage]	; Get pointer to help usage message
	call	Printf			; Call routine to print error message
	jmp	Short Help_Exit 	; Go return control to the caller
Help_Done:
	cmp	cx,bp			; Check for just continued
	je	Help_Error		; Jump if we just did a continue
	call	Get_Attribute		; Get the current attribute values
	xchg	ax,bx			; Swap foreground/background colors
	call	Set_Attribute		; Set the new attribute value (Invert)
	lea	si,ds:[Continue_Fmt]	; Get pointer to the continue message
	call	Printf			; Call routine to print continue message
	call	Get_Key 		; Call routine to wait for a key
	call	Get_Attribute		; Get the current attribute values
	xchg	ax,bx			; Swap foreground/background colors
	call	Set_Attribute		; Set the new attribute value (Restore)
	jmp	Short Help_Error	; Go print help system usage message
Check_Command:
	call	Upper_Case		; Call routine to convert to uppercase
	push	si			; Save the input parameter pointer
	lea	si,ds:[Debug_Table]	; Get pointer to start of debug table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the match table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	pop	si			; Restore the input parameter pointer
	jc	Help_Error		; Jump if no command match is found
Print_Usage:
	cbw				; Convert matching index to full word
	mov	bx,ax			; Get the matching register index
	shl	bx,1			; Convert match number to table index
	mov	si,ds:[Usage_Table + bx]; Get pointer to usage format string
	call	Printf			; Call routine to print usage message
Help_Exit:
	Restore si,di,bp,es		; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Help_Command	Endp			; End of the Help_Command procedure
	Subttl	Trace_Command	Apple Emulator Debug Trace Command
	Page	+
;******************************************************************************
;
;	Trace_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		If assignment type delimiter
;			Call routine to get trace address
;			If valid trace address
;				Setup the new execution address
;			Else
;				Call routine to print trace usage message
;				Clear carry indicating more input
;			Endif
;		Endif
;		Call routine to get the expression value
;		If a valid expression was given
;			Set trace count to given value
;			Set the trace interrupt flag
;			Set the emulator interrupt flag
;			Set carry to indicate no more input
;		Else
;			If assignment type expression
;				Call routine to get trace address
;				If valid trace address
;					Setup the new execution address
;				Else
;					Call routine to print usage
;					Clear carry indicating input
;				Endif
;			Endif
;			If no expression was given
;				Set trace count to one
;				Set the trace interrupt flag
;				Set the emulator interrupt flag
;				Set carry to indicate no more input
;			Else invalid expression
;				Call routine to print trace usage message
;				Clear carry to indicate more input
;			Endif
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Trace_Command	Proc	Near		; Debug trace command procedure
	Save	si,di,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Trace_Assign		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Trace_Assign:
	test	ah,ASSIGN_TYPE		; Check for trace at location command
	jz	Trace_Check		; Jump if not trace at location command
Trace_Location:
	call	Get_Expression		; Call routine to get expression value
	jbe	Trace_Error		; Jump if no/illegal expression value
	Save	si,ds			; Save the debug command pointer
	mov	si,ax			; Get location value (From expression)
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	Setup				; Setup to
	mov	ss:[bp.SI_Register],si	;	   execute from
	mov	ss:[bp.DI_Register],di	;			desired location
	Restore si,ds			; Restore pointer to debug command
Trace_Check:
	call	Get_Expression		; Call routine to get expression value
	jnc	Check_Count		; Jump if valid expression value
Trace_Error:
	lea	si,ds:[Trace_Usage]	; Get pointer to trace usage message
	call	Printf			; Call routine to print error message
	clc				; Clear carry indicating more input
	jmp	Short Trace_Exit	; Go return control to the caller
Check_Count:
	jnz	Set_Count		; Jump if a count value was given
	or	bh,bh			; Check for valid delimiter
	jz	Default_Count		; Jump if end of input buffer
	and	bh,SPC_MASK		; Mask off all but special field
	cmp	bh,SPC_ASSIGN		; Check for assignment delimiter
	jne	Trace_Error		; Jump if invalid delimiter type
	inc	dx			; Increment past assignment delimiter
	jmp	Short Trace_Location	; Go try to get the location value
Default_Count:
	mov	ax,1			; Default to trace a single instruction
Set_Count:
	mov	ds:[Trace_Count],ax	; Setup the trace count value
	or	ds:[Emulate_Flag],SYS_TRACE
	or	ss:[bp.DH_Register],CPU_R
	xor	ax,ax			; Set zero flag to stay in debug mode
	stc				; Set carry indicating no more input
Trace_Exit:
	Restore si,di,es		; Restore the required registers
	ret				; Return to the caller
Trace_Command	Endp			; End of the Trace_Command procedure
	Subttl	Enter_Command	Apple Emulator Debug Enter Command
	Page	+
;******************************************************************************
;
;	Enter_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get enter address
;		If a valid enter address was given
;			While there are more data values
;				Call routine to get next data value
;				If a valid data value was given
;					If a data value was given
;						Call routine to enter the data
;					Else (No expression)
;						If next data value is a string
;							Call routine to enter
;						Endif
;					Endif
;				Else
;					Call routine to print enter usage
;					Clear carry to indicate more input
;					Exit the while loop
;				Endif
;			EndWhile
;		Else
;			Call routine to print enter usage message
;			Clear carry to indicate more input
;		Endif
;		Call routine to dump the data (Refresh)
;		Call routine to dump the code (Refresh)
;		Call routine to dump the stack (Refresh)
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Enter_Command	Proc	Near		; Debug enter command procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Enter_Address		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Enter_Address:
	call	Get_Expression		; Call routine to get entry address
	mov	di,ax			; Save the expression value if any
	ja	Enter_Loop		; Jump if a valid expression was given
Enter_Error:
	lea	si,ds:[Enter_Usage]	; Get pointer to enter usage message
	call	Printf			; Call routine to print error message
	jmp	Short Enter_Exit	; Go return control to the caller
Enter_Loop:
	call	Get_Expression		; Call routine to get next data value
	jc	Enter_Error		; Jump if expression error
	jnz	Enter_Value		; Jump if a valid expression given
	or	bh,bh			; Check for valid delimiter
	jz	Enter_Exit		; Jump if end of input buffer
	xor	ax,ax			; Default to a zero data value
	and	bh,SPC_MASK		; Mask off all but special delimiters
	cmp	bh,SPC_STRING		; Check for a string delimiter
	jne	Enter_Loop		; Jump if this is not a string delimiter
	call	Enter_String		; Call routine to enter the string
	jmp	Short Enter_Loop	; Loop till all data values gotten
Enter_Value:
	mov	bx,ds:[Enter_Format]	; Get the current data entry format
	call	ds:[Enter_Table + bx]	; Call correct routine to enter the data
	jmp	Short Enter_Loop	; Loop till all data values gotten
Enter_Exit:
	mov	ax,ds:[Last_Code]	; Use last code dump address value
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Dump_Data		; Call routine to dump requested data
	call	Dump_Code		; Call routine to dump requested code
	call	Dump_Stack		; Call routine to dump the stack
	Restore si,di			; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Enter_Command	Endp			; End of the Enter_Command procedure
	Subttl	Fill_Command	Apple Emulator Debug Fill Command
	Page	+
;******************************************************************************
;
;	Fill_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get start address
;		If a valid start address was given
;			Call routine to get end address
;			If a valid end address was given
;				Call routine to get the data values
;				While there are bytes to fill
;					Get next fill byte value
;					Call routine to write the filler byte
;					Update the current offset value
;				EndWhile
;			Else invalid end address
;				Call routine to print fill usage message
;				Clear carry to indicate more input
;		Else invalid start address
;			Call routine to print fill usage message
;			Clear carry to indicate more input
;		Endif
;		Call routine to dump the data (Refresh)
;		Call routine to dump the code (Refresh)
;		Call routine to dump the stack (Refresh)
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Fill_Command	Proc	Near		; Debug fill command procedure
	Save	si,di,ds,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Fill_Start		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Fill_Start:
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	call	Get_Expression		; Call routine to get the start address
	mov	di,ax			; Save the expression value if any
	jbe	Fill_Error		; Jump if no/invalid expression given
Fill_End:
	call	Get_Expression		; Call routine to get the end address
	jc	Fill_Error		; Jump if an invalid expression given
	jnz	Fill_Check		; Jump if an end address was given
	and	bh,SPC_MASK		; Mask off all but special delimiters
	cmp	bh,SPC_LENGTH		; Check for a length delimiter
	jne	Fill_Error		; Jump if not a length delimiter
	inc	dx			; Increment past the length delimiter
Fill_Length:
	call	Get_Expression		; Call routine to get length value
	jbe	Fill_Error		; Jump if no/invalid expression given
	or	ax,ax			; Check for a non-zero length
	jnz	Short Fill_Data 	; Go try and get the fill data
Fill_Error:
	lea	si,ds:[Fill_Usage]	; Get pointer to fill usage message
	call	Printf			; Call routine to print error message
	jmp	Short Fill_Exit 	; Go return control to the caller
Fill_Check:
	sub	ax,di			; Compute the fill length count
	jb	Fill_Error		; Jump if negative length value
	inc	ax			; Compute the actual fill count
Fill_Data:
	call	Get_Data		; Call routine to get fill data
	jcxz	Fill_Error		; Jump if there is no fill data
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	mov	bx,cx			; Save the actual data length value
	mov	dx,ax			; Save the actual fill length value
Fill_Loop:
	xor	si,si			; Initialize the data offset value
	mov	cx,dx			; Setup to use the fill length value
	cmp	cx,bx			; Check fill length against data length
	jbe	Copy_Data		; Jump if all data is present
	mov	cx,bx			; Use data length for fill length
Copy_Data:
	sub	dx,cx			; Update the actual fill length value
Write_Loop:
	mov	al,es:[Data_Buffer + si]; Get next data byte from the buffer
	Save	di,bp			; Save the required registers
	call	Write_Memory		; Call routine to write the data value
	Restore di,bp			; Restore the required registers
	inc	si			; Increment the
	inc	di			;		data pointers
	loop	Write_Loop		; Loop till all data is written
	or	dx,dx			; Check for more data to fill
	jnz	Fill_Loop		; Loop till entire area is filled
Fill_Exit:
	mov	ax,es			; Restore the
	mov	ds,ax			;	      data segment value
	mov	ax,ds:[Last_Code]	; Use last code dump address value
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Dump_Data		; Call routine to dump requested data
	call	Dump_Code		; Call routine to dump requested code
	call	Dump_Stack		; Call routine to dump the stack
	Restore si,di,ds,es		; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Fill_Command	Endp			; End of the Fill_Command procedure
	Subttl	Move_Command	Apple Emulator Debug Move Command
	Page	+
;******************************************************************************
;
;	Move_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get start address
;		If a valid start address was given
;			Call routine to get end address
;			If a valid end address was given
;				Call routine to get move address
;				If a valid move address was given
;					While more data bytes to move
;						Call routine to read data byte
;						Call routine to write data byte
;						Increment the data pointers
;					EndWhile
;			Else invalid end address
;				Call routine to print move usage message
;				Clear carry to indicate more input
;		Else invalid start address
;			Call routine to print move usage message
;			Clear carry to indicate more input
;		Endif
;		Call routine to dump the data (Refresh)
;		Call routine to dump the code (Refresh)
;		Call routine to dump the stack (Refresh)
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Move_Command	Proc	Near		; Debug move command procedure
	Save	si,di,ds,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Move_Start		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Move_Start:
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	call	Get_Expression		; Call routine to get the start address
	mov	di,ax			; Save the expression value if any
	jbe	Move_Error		; Jump if no/invalid expression given
Move_End:
	call	Get_Expression		; Call routine to get the end address
	jc	Move_Error		; Jump if an invalid expression given
	jnz	Move_Check		; Jump if an end address was given
	and	bh,SPC_MASK		; Mask off all but special delimiters
	cmp	bh,SPC_LENGTH		; Check for a length delimiter
	jne	Move_Error		; Jump if not a length delimiter
	inc	dx			; Increment past the length delimiter
Move_Length:
	call	Get_Expression		; Call routine to get length value
	jbe	Move_Error		; Jump if no/invalid expression given
	or	ax,ax			; Check for a non-zero length
	jnz	Short Move_Address	; Go try and get the move address
Move_Error:
	lea	si,ds:[Move_Usage]	; Get pointer to move usage message
	call	Printf			; Call routine to print error message
	jmp	Short Move_Exit 	; Go return control to the caller
Move_Check:
	sub	ax,di			; Compute the move length count
	jb	Move_Error		; Jump if negative length value
	inc	ax			; Compute the actual move count
Move_Address:
	mov	cx,ax			; Save the actual move count value
	call	Get_Expression		; Call routine to get move address
	jbe	Move_Error		; Jump if no/invalid move address
	mov	si,ax			; Save the move address value
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
Move_Loop:
	Save	si,di,bp		; Save the current data pointers
	call	Read_Memory		; Call routine to read the memory value
	xchg	si,di			; Setup to write the memory value
	call	Write_Memory		; Call routine to write the memory value
	Restore si,di,bp		; Restore the current data pointers
	inc	si			; Increment the
	inc	di			;		data pointers
	loop	Move_Loop		; Loop till all data values are moved
Move_Exit:
	mov	ax,es			; Restore the
	mov	ds,ax			;	      data segment value
	mov	ax,ds:[Last_Code]	; Use last code dump address value
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Dump_Data		; Call routine to dump requested data
	call	Dump_Code		; Call routine to dump requested code
	call	Dump_Stack		; Call routine to dump the stack
	Restore si,di,ds,es		; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Move_Command	Endp			; End of the Move_Command procedure
	Subttl	Compare_Command Apple Emulator Debug Compare Command
	Page	+
;******************************************************************************
;
;	Compare_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get start address
;		If a valid start address was given
;			Call routine to get end address
;			If a valid end address was given
;				Call routine to get compare address
;				If a valid compare address was given
;					While more data bytes to compare
;						Call routine to read first byte
;						Call routine to read second byte
;						If the data bytes are different
;							Call routine to print
;						Endif
;						Increment the data pointers
;					EndWhile
;			Else invalid end address
;				Call routine to print compare usage message
;				Clear carry to indicate more input
;		Else invalid start address
;			Call routine to print compare usage message
;			Clear carry to indicate more input
;		Endif
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Compare_Command Proc	Near		; Debug compare command procedure
	Save	si,di,bp,ds,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Compare_Start		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Compare_Start:
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	call	Get_Expression		; Call routine to get the start address
	mov	di,ax			; Save the expression value if any
	jbe	Compare_Error		; Jump if no/invalid expression given
Compare_End:
	call	Get_Expression		; Call routine to get the end address
	jc	Compare_Error		; Jump if an invalid expression given
	jnz	Compare_Check		; Jump if an end address was given
	and	bh,SPC_MASK		; Mask off all but special delimiters
	cmp	bh,SPC_LENGTH		; Check for a length delimiter
	jne	Compare_Error		; Jump if not a length delimiter
	inc	dx			; Increment past the length delimiter
Compare_Length:
	call	Get_Expression		; Call routine to get length value
	jbe	Compare_Error		; Jump if no/invalid expression given
	or	ax,ax			; Check for a non-zero length
	jnz	Short Comp_Address	; Go try and get the compare address
Compare_Error:
	lea	si,ds:[Compare_Usage]	; Get pointer to compare usage message
	call	Printf			; Call routine to print error message
	jmp	Short Compare_Exit	; Go return control to the caller
Compare_Check:
	sub	ax,di			; Compute the compare length count
	jb	Compare_Error		; Jump if negative length value
	inc	ax			; Compute the actual compare count
Comp_Address:
	mov	cx,ax			; Save the actual compare count value
	call	Get_Expression		; Call routine to get compare address
	jbe	Compare_Error		; Jump if no/invalid compare address
	mov	si,ax			; Save the compare address value
Compare_Loop:
	Save	si,di,ds		; Save the current data pointers
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	call	Read_Memory		; Call routine to read the first value
	mov	bl,al			; Save the first data byte value
	xchg	si,di			; Setup to read the second memory value
	call	Read_Memory		; Call routine to read the second value
	Restore si,di,ds		; Restore the current data pointers
	cmp	al,bl			; Compare the two data byte values
	je	Compare_Next		; Jump if the data bytes compare
	push	si			; Save the required registers
	mov	ds:[Current_Address],di ; Save the current address value
	mov	ds:[Compare_Address],si ; Save the compare address value
	mov	ds:[Value],bx		; Save the current data value (Byte)
	mov	ds:[Compare],ax 	; Save the compare data value (Byte)
	lea	bx,ds:[Diff_Arg]	; Get pointer to difference arguments
	lea	si,ds:[Diff_Fmt]	; Get pointer to difference format
	call	Printf			; Call routine to report the difference
	pop	si			; Restore the required registers
Compare_Next:
	inc	si			; Increment the
	inc	di			;		data pointers
	loop	Compare_Loop		; Loop till all data values are compared
Compare_Exit:
	Restore si,di,bp,ds,es		; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Compare_Command Endp			; End of the Compare_Command procedure
	Subttl	Search_Command	Apple Emulator Debug Search Command
	Page	+
;******************************************************************************
;
;	Search_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get start address
;		If a valid start address was given
;			Call routine to get end address
;			If a valid end address was given
;				Call routine to get the data values
;				Compute the actual search count
;				While more area to search
;					If data at search address matches
;						Call routine to print address
;					Endif
;					Increment the search address
;					Decrement the search count
;				EndWhile
;			Else invalid end address
;				Call routine to print search usage message
;				Clear carry to indicate more input
;		Else invalid start address
;			Call routine to print search usage message
;			Clear carry to indicate more input
;		Endif
;		Restore the required registers
;		Clear carry to indicate more input
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Search_Command	Proc	Near		; Debug search command procedure
	Save	si,di,bp,ds,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Search_Start		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Search_Start:
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	call	Get_Expression		; Call routine to get the start address
	mov	di,ax			; Save the expression value if any
	jbe	Search_Error		; Jump if no/invalid expression given
Search_End:
	call	Get_Expression		; Call routine to get the end address
	jc	Search_Error		; Jump if an invalid expression given
	jnz	Search_Check		; Jump if an end address was given
	and	bh,SPC_MASK		; Mask off all but special delimiters
	cmp	bh,SPC_LENGTH		; Check for a length delimiter
	jne	Search_Error		; Jump if not a length delimiter
	inc	dx			; Increment past the length delimiter
Search_Length:
	call	Get_Expression		; Call routine to get length value
	jbe	Search_Error		; Jump if no/invalid expression given
	or	ax,ax			; Check for a non-zero length
	jnz	Short Search_Data	; Go try and get the search data
Search_Error:
	lea	si,ds:[Search_Usage]	; Get pointer to search usage message
	call	Printf			; Call routine to print error message
	jmp	Short Search_Exit	; Go return control to the caller
Search_Check:
	sub	ax,di			; Compute the compare length count
	jb	Search_Error		; Jump if negative length value
	inc	ax			; Compute the actual compare count
Search_Data:
	mov	bx,ax			; Save the search length value
	call	Get_Data		; Call routine to get search data
	jcxz	Search_Error		; Jump if there is no search data
	sub	bx,cx			; Update the search length value
	jbe	Search_Error		; Jump if search length too short
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
Search_Loop:
	Save	cx,di			; Save search address and count
	xor	si,si			; Initialize the data offset value
	mov	es:[Compare_Address],di ; Save the current search address
Comp_Loop:
	call	Read_Memory		; Call routine to read memory value
	cmp	al,es:[Data_Buffer + si]; Compare data value to search pattern
	jne	Next_Address		; Jump if mismatch in the pattern
	inc	si			; Increment the
	inc	di			;		data pointers
	loop	Comp_Loop		; Loop until all of pattern is checked
	Save	bx,ds			; Save the required registers
	mov	ax,es			; Setup access to
	mov	ds,ax			;		  the data segment
	lea	bx,ds:[Find_Arg]	; Get pointer to find argument list
	lea	si,ds:[Find_Fmt]	; Get pointer to find message format
	call	Printf			; Call routine to print find message
	Restore bx,ds			; Restore the required registers
Next_Address:
	Restore cx,di			; Restore search address and count
	inc	di			; Increment the search address value
	dec	bx			; Decrement the search length value
	jnz	Search_Loop		; Jump if more data to search
Search_Exit:
	Restore si,di,bp,ds,es		; Restore the required registers
	clc				; Clear carry indicating more input
	ret				; Return to the caller
Search_Command	Endp			; End of the Search_Command procedure
	Subttl	Bpoint_Command	Apple Emulator Debug Set Breakpoint Command
	Page	+
;******************************************************************************
;
;	Bpoint_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		If a breakpoint number was given
;			Call routine to get the breakpoint number
;			If the breakpoint number is invalid
;				Call routine to print bpoint usage message
;				Return control to the caller
;			Endif
;			Call routine to clear this breakpoint
;		Else
;			Call routine to find a vacant breakpoint
;			If no breakpoints available
;				Call routine to print bpoint usage message
;				Return control to the caller
;			Endif
;		Endif
;		Call routine to get the breakpoint address
;		If a valid breakpoint address value was given
;			Call routine to get the breakpoint type
;			If no type value was given
;				Default to software breakpoint type
;			Endif
;			If valid breakpoint type given
;				Get the actual breakpoint type value
;				Get the breakpoint check value (Hardware)
;				Call routine to set this breakpoint
;				Call routine to dump requested code (Refresh)
;				If error setting the breakpoint
;					Call routine to print bpoint usage
;				Endif
;			Else invalid breakpoint type
;				Call routine to print bpoint usage message
;			Endif
;		Else no breakpoint address value
;			Call routine to print bpoint usage message
;		Endif
;		Clear carry indicating more input
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Bpoint_Command	Proc	Near		; Debug set breakpoint procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Find_Breakpoint 	; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
	call	Get_Expression		; Call routine to get breakpoint
	cmp	ax,BREAKPOINTS		; Check for a valid breakpoint
	jae	Bpoint_Error		; Jump if invalid breakpoint number
	call	Break_Clear		; Call routine to clear this breakpoint
	jmp	Short Break_Continue	; Go continue the set breakpoint
Find_Breakpoint:
	call	Break_Find		; Call routine to find a breakpoint
	jnc	Break_Continue		; Jump if breakpoint is available
Bpoint_Error:
	lea	si,ds:[Bpoint_Usage]	; Get pointer to bpoint usage message
	call	Printf			; Call routine to print error message
	jmp	Short Bpoint_Exit	; Go return control to the caller
Break_Continue:
	mov	ds:[Value],ax		; Save the breakpoint number
	call	Get_Expression		; Call routine to get breakpoint address
	jbe	Bpoint_Error		; Jump if no breakpoint address
	mov	ds:[Compare],ax 	; Save the breakpoint address value
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,CR			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Debug]	; Get pointer to debug delimiter table
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	lea	si,ds:[Input_Buffer]	; Get pointer to the input buffer
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	call	Get_Parameter		; Call routine to get next parameter
	mov	bl,0			; Default to software type breakpoint
	jbe	Set_Breakpoint		; Jump if no/null parameter (Software)
	call	Upper_Case		; Call routine to convert to uppercase
	push	si			; Save the input parameter pointer
	lea	si,ds:[Match_Table]	; Get pointer to start of match table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the match table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	pop	si			; Restore the input parameter pointer
	jc	Bpoint_Error		; Jump if no breakpoint type match
	mov	bl,al			; Save the matching type value
Set_Breakpoint:
	xor	bh,bh			; Convert type value to a full word
	mov	cl,ds:[Type_Table + bx] ; Get the breakpoint type value
	mov	ch,ds:[Check_Table + bx]; Get the breakpoint check value
	mov	ax,ds:[Value]		; Restore the breakpoint number
	mov	bx,ds:[Compare] 	; Get the breakpoint address value
	call	Break_Set		; Call routine to set this breakpoint
	jc	Bpoint_Error		; Jump if error setting the breakpoint
	mov	ax,ds:[Last_Code]	; Use last code dump address value
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Dump_Code		; Call routine to dump requested code
Bpoint_Exit:
	clc				; Clear carry indicating more input
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Bpoint_Command	Endp			; End of the Bpoint_Command procedure
	Subttl	Bdsable_Command Apple Emulator Debug Disable Breakpoint Command
	Page	+
;******************************************************************************
;
;	Bdsable_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		If there are breakpoints given
;			While there are more breakpoints
;				Call routine to disable this breakpoint
;				Call routine to get next breakpoint
;			EndWhile
;		Else
;			Setup to loop through all the breakpoints
;			While there are more breakpoints
;				Call routine to disable this breakpoint
;				Increment to the next breakpoint
;			EndWhile
;		Endif
;		Call routine to dump requested code (Refresh)
;		Clear carry indicating more input
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Bdsable_Command Proc	Near		; Debug disable breakpoint procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Disable_Start		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Disable_Start:
	call	Get_Expression		; Call routine to get first breakpoint
	jc	Bdsable_Error		; Jump if expression error
	jz	Disable_All		; Jump if no expression (Disable all)
Disable_Loop:
	call	Break_Disable		; Call routine to disable breakpoint
	call	Get_Expression		; Call routine to get next breakpoint
	jc	Bdsable_Error		; Jump if expression error
	jnz	Disable_Loop		; Loop till all breakpoints disabled
	jmp	Short Bdsable_Done	; Go update the debug screen
Bdsable_Error:
	lea	si,ds:[Bdsable_Usage]	; Get pointer to bdsable usage message
	call	Printf			; Call routine to print error message
	jmp	Short Bdsable_Exit	; Go return control to the caller
Disable_All:
	xor	ax,ax			; Initialize the breakpoint number
	mov	cx,BREAKPOINTS		; Initialize the breakpoint counter
Loop_Disable:
	call	Break_Disable		; Call routine to disable breakpoint
	inc	ax			; Increment to the next breakpoint
	loop	Loop_Disable		; Loop till all breakpoints disabled
Bdsable_Done:
	mov	ax,ds:[Last_Code]	; Use last code dump address value
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Dump_Code		; Call routine to dump requested code
Bdsable_Exit:
	clc				; Clear carry indicating more input
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Bdsable_Command Endp			; End of the Bdsable_Command procedure
	Subttl	Benable_Command Apple Emulator Debug Enable Breakpoint Command
	Page	+
;******************************************************************************
;
;	Benable_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		If there are breakpoints given
;			While there are more breakpoints
;				Call routine to enable this breakpoint
;				Call routine to get next breakpoint
;			EndWhile
;		Else
;			Setup to loop through all the breakpoints
;			While there are more breakpoints
;				Call routine to enable this breakpoint
;				Increment to the next breakpoint
;			EndWhile
;		Endif
;		Call routine to dump requested code (Refresh)
;		Clear carry indicating more input
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Benable_Command Proc	Near		; Debug enable breakpoint procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Enable_Start		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Enable_Start:
	call	Get_Expression		; Call routine to get first breakpoint
	jc	Benable_Error		; Jump if expression error
	jz	Enable_All		; Jump if no expression (Enable all)
Enable_Loop:
	call	Break_Enable		; Call routine to enable this breakpoint
	call	Get_Expression		; Call routine to get next breakpoint
	jc	Benable_Error		; Jump if expression error
	jnz	Enable_Loop		; Loop till all breakpoints enabled
	jmp	Short Benable_Done	; Go update the debug screen
Benable_Error:
	lea	si,ds:[Benable_Usage]	; Get pointer to benable usage message
	call	Printf			; Call routine to print error message
	jmp	Short Benable_Exit	; Go return control to the caller
Enable_All:
	xor	ax,ax			; Initialize the breakpoint number
	mov	cx,BREAKPOINTS		; Initialize the breakpoint counter
Loop_Enable:
	call	Break_Enable		; Call routine to enable this breakpoint
	inc	ax			; Increment to the next breakpoint
	loop	Loop_Enable		; Loop till all breakpoints enabled
Benable_Done:
	mov	ax,ds:[Last_Code]	; Use last code dump address value
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Dump_Code		; Call routine to dump requested code
Benable_Exit:
	clc				; Clear carry indicating more input
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Benable_Command Endp			; End of the Benable_Command procedure
	Subttl	Bclear_Command	Apple Emulator Debug Clear Breakpoint Command
	Page	+
;******************************************************************************
;
;	Bclear_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		If there are breakpoints given
;			While there are more breakpoints
;				Call routine to clear this breakpoint
;				Call routine to get next breakpoint
;			EndWhile
;		Else
;			Setup to loop through all the breakpoints
;			While there are more breakpoints
;				Call routine to clear this breakpoint
;				Increment to the next breakpoint
;			EndWhile
;		Endif
;		Call routine to dump requested code (Refresh)
;		Clear carry indicating more input
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Bclear_Command	Proc	Near		; Debug clear breakpoint procedure
	Save	si,di			; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Clear_Start		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Clear_Start:
	call	Get_Expression		; Call routine to get first breakpoint
	jc	Bclear_Error		; Jump if expression error
	jz	Clear_All		; Jump if no expression (Clear all)
Clear_Loop:
	call	Break_Clear		; Call routine to clear this breakpoint
	call	Get_Expression		; Call routine to get next breakpoint
	jc	Bclear_Error		; Jump if expression error
	jnz	Clear_Loop		; Loop till all breakpoints cleared
	jmp	Short Break_Update	; Go update the code breakpoints
Bclear_Error:
	lea	si,ds:[Bclear_Usage]	; Get pointer to bclear usage message
	call	Printf			; Call routine to print error message
	jmp	Short Bclear_Exit	; Go return control to the caller
Clear_All:
	xor	ax,ax			; Initialize the breakpoint number
	mov	cx,BREAKPOINTS		; Initialize the breakpoint counter
Loop_Clear:
	call	Break_Clear		; Call routine to clear this breakpoint
	inc	ax			; Increment to the next breakpoint
	loop	Loop_Clear		; Loop till all breakpoints cleared
Break_Update:
	mov	ax,ds:[Last_Code]	; Use last code dump address value
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Dump_Code		; Call routine to dump requested code
Bclear_Exit:
	clc				; Clear carry indicating more input
	Restore si,di			; Restore the required registers
	ret				; Return to the caller
Bclear_Command	Endp			; End of the Bclear_Command procedure
	Subttl	Blist_Command	Apple Emulator Debug List Breakpoint Command
	Page	+
;******************************************************************************
;
;	Blist_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		If there are breakpoints given
;			While there are more breakpoints
;				Call routine to list this breakpoint
;				Call routine to get next breakpoint
;			EndWhile
;		Else
;			Setup to loop through all the breakpoints
;			While there are more breakpoints
;				Call routine to list this breakpoint
;				Increment to the next breakpoint
;			EndWhile
;		Endif
;		Clear carry indicating more input
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Blist_Command	Proc	Near		; Debug list breakpoint procedure
	Save	si,di,bp		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	List_Start		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
List_Start:
	call	Get_Expression		; Call routine to get first breakpoint
	jc	Blist_Error		; Jump if expression error
	jz	List_All		; Jump if no expression given (List all)
List_Loop:
	call	Break_List		; Call routine to list this breakpoint
	call	Get_Expression		; Call routine to get next breakpoint
	jc	Blist_Error		; Jump if expression error
	jnz	List_Loop		; Loop till all breakpoints listed
	jmp	Short Blist_Exit	; Go return control to the caller
Blist_Error:
	lea	si,ds:[Blist_Usage]	; Get pointer to blist usage message
	call	Printf			; Call routine to print error message
	jmp	Short Blist_Exit	; Go return control to the caller
List_All:
	xor	ax,ax			; Initialize the breakpoint number
	mov	dx,BREAKPOINTS		; Initialize the breakpoint counter
	call	Get_Window		; Get the current window structure
	mov	bp,ds:[si.Lower_Right_Row]
	sub	bp,ds:[si.Upper_Left_Row]
List_Setup:
	mov	cx,bp			; Setup number of breakpoints to list
	cmp	cx,dx			; Check against the number left
	jbe	Loop_List		; Jump if enough breakpoints left
	mov	cx,dx			; Setup the breakpoint count value
Loop_List:
	call	Break_List		; Call routine to list this breakpoint
	jnc	Break_Next		; Jump if we actually printed something
	inc	ax			; Increment the breakpoint number
	dec	dx			; Decrement the breakpoint counter
	jnz	Loop_List		; Loop if more breakpoints to list
	jmp	Blist_Exit		; Jump if all breakpoints listed
Break_Next:
	inc	ax			; Increment to the next breakpoint
	dec	dx			; Decrement the breakpoint counter
	loop	Loop_List		; Loop till enough lines printed
	or	dx,dx			; Check for all breakpoints listed
	jz	Blist_Exit		; Jump if all breakpoints listed
	push	ax			; Save the current breakpoint number
	call	Get_Attribute		; Get the current attribute values
	xchg	ax,bx			; Swap foreground/background colors
	call	Set_Attribute		; Set the new attribute value (Invert)
	lea	si,ds:[Continue_Fmt]	; Get pointer to the continue message
	call	Printf			; Call routine to print continue message
	call	Get_Key 		; Call routine to wait for a key
	push	ax			; Save the actual keystroke
	call	Get_Attribute		; Get the current attribute values
	xchg	ax,bx			; Swap foreground/background colors
	call	Set_Attribute		; Set the new attribute value (Restore)
	mov	cx,bp			; Get the number of lines to print
	pop	bx			; Restore the keystroke value
	pop	ax			; Restore the current breakpoint number
	cmp	bl,ESCAPE		; Check for an escape character
	jne	List_Setup		; Jump if not the escape character
Blist_Exit:
	clc				; Clear carry indicating more input
	Restore si,di,bp		; Restore the required registers
	ret				; Return to the caller
Blist_Command	Endp			; End of the Blist_Command procedure
	Subttl	Proceed_Command Apple Emulator Debug Proceed Command
	Page	+
;******************************************************************************
;
;	Proceed_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		If assignment type delimiter
;			Call routine to get proceed address
;			If valid proceed address
;				Setup the new execution address
;			Else
;				Call routine to print proceed usage message
;				Clear carry indicating more input
;			Endif
;		Endif
;		Call routine to get the expression value
;		If a valid expression was given
;			Set trace count to given value
;			Set the trace interrupt flag
;			Set the emulator interrupt flag
;			Set carry to indicate no more input
;		Else
;			If assignment type expression
;				Call routine to get proceed address
;				If valid proceed address
;					Setup the new execution address
;				Else
;					Call routine to print usage
;					Clear carry indicating input
;				Endif
;			Endif
;			If no expression was given
;				Get the current opcode value
;				If the opcode is a subroutine jump
;					Set breakpoint to after subroutine jump
;					Set carry to indicate no more input
;				Else
;					Set trace count to one
;					Set the trace interrupt flag
;					Set the emulator interrupt flag
;					Set carry to indicate no more input
;				Endif
;			Else invalid expression
;				Call routine to print proceed usage message
;				Clear carry to indicate more input
;			Endif
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Proceed_Command Proc	Near		; Debug proceed command procedure
	Save	si,di,es		; Save the required registers
	test	ah,NUMERIC_TYPE 	; Check for numeric delimiter (None)
	jz	Proceed_Assign		; Jump if not numeric delimiter
	dec	dx			; Back up to actual number start
Proceed_Assign:
	test	ah,ASSIGN_TYPE		; Check for proceed at location command
	jz	Proceed_Check		; Jump if not proceed at location
Proceed_Location:
	call	Get_Expression		; Call routine to get expression value
	jbe	Proceed_Error		; Jump if no/illegal expression value
	Save	si,ds			; Save the debug command pointer
	mov	si,ax			; Get location value (From expression)
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	Setup				; Setup to
	mov	ss:[bp.SI_Register],si	;	   execute from
	mov	ss:[bp.DI_Register],di	;			desired location
	Restore si,ds			; Restore pointer to debug command
Proceed_Check:
	call	Get_Expression		; Call routine to get expression value
	jnc	Count_Check		; Jump if valid expression value
Proceed_Error:
	lea	si,ds:[Proceed_Usage]	; Get pointer to proceed usage message
	call	Printf			; Call routine to print error message
	clc				; Clear carry indicating more input
	jmp	Short Proceed_Exit	; Go return control to the caller
Count_Check:
	jnz	Count_Set		; Jump if a count value was given
	or	bh,bh			; Check for valid delimiter
	jz	Call_Check		; Jump if end of input buffer
	and	bh,SPC_MASK		; Mask off all but special field
	cmp	bh,SPC_ASSIGN		; Check for assignment delimiter
	jne	Proceed_Error		; Jump if invalid delimiter type
	inc	dx			; Increment past assignment delimiter
	jmp	Short Proceed_Location	; Go try to get the location value
Call_Check:
	push	ds			; Save the data segment value
	mov	ds,cs:[RAM_Space]	; Setup to access 65C02 RAM space
	mov	bx,ss:[bp.SI_Register]	; Get the current
	dec	bx			;		  program counter
	cmp	Byte Ptr ds:[bx],JSR_OPCODE
	pop	ds			; Restore the data segment value
	jne	Count_Default		; Jump if this is not a JSR opcode
	mov	ax,BREAKPOINTS		; Get the local breakpoint number
	add	bx,JSR_SIZE		; Compute the actual breakpoint address
	mov	cl,TYPE_SOFT		; Get the software type value
	call	Break_Set		; Call routine to set breakpoint
	jc	Proceed_Error		; Jump if error setting the breakpoint
	xor	ax,ax			; Clear zero to indicate
	dec	ax			;			 debug mode exit
	stc				; Set carry indicating no more input
	jmp	Short Proceed_Exit	; Go return control to the caller
Count_Default:
	mov	ax,1			; Default to trace a single instruction
Count_Set:
	mov	cs:[Trace_Count],ax	; Setup the trace count value
	or	cs:[Emulate_Flag],SYS_TRACE
	or	ss:[bp.DH_Register],CPU_R
	xor	ax,ax			; Set zero flag to stay in debug mode
	stc				; Set carry indicating no more input
Proceed_Exit:
	Restore si,di,ds		; Restore the required registers
	ret				; Return to the caller
Proceed_Command Endp			; End of the Proceed_Command procedure
	Subttl	Flip_Command	Apple Emulator Debug Flip Command
	Page	+
;******************************************************************************
;
;	Flip_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Save the required registers
;		Call routine to get current window
;		Save the current window value
;		Call routine to reset the TTY (Flip to Apple ][ screen)
;		Call routine to wait for a keystroke
;		Call routine to initialize the TTY interface
;		Setup the debugger background color value
;		Call routine to clear the window (Clear screen)
;		Setup the current 65C02 code address
;		Call routine to print debugger title
;		Call routine to dump the data values
;		Call routine to print the status line
;		Call routine to dump the code at current address
;		Call routine to print the 65C02 registers
;		Call routine to dump the 65C02 stack
;		Restore the original window value
;		Call routine to setup original window
;		Clear carry indicating more commands
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Flip_Command	Proc	Near		; Debug flip command procedure
	Save	si,di,ds,es		; Save the required registers
	call	Get_Window		; Call routine to get current window
	push	si			; Save the current window pointer
	mov	ds,cs:[RAM_Space]	; Setup the emulator
	mov	es,cs:[Video_Segment]	;		     segment values
	mov	al,cs:[Key_Status]	; Get current keyboard status
	Save	ax,bp			; Save the required registers
	and	cs:[Key_Status],Not TYPE_SPECIAL
	call	TTY_Reset		; Call routine to reset TTY
	Restore ax,bp			; Restore the required registers
	mov	cs:[Key_Status],al	; Restore original keyboard status
	call	Get_Key 		; Call routine to wait for keystroke
	mov	ax,cs			; Setup access to
	mov	ds,ax			;		  the data segment
	call	TTY_Init		; Call routine to initialize TTY
	mov	ax,ds:[Color_Table.Color_Back]
	call	Set_Background		; Setup the debugger background color
	call	Clear_Window		; Call routine to clear window (Screen)
	mov	ax,ds:[Last_Code]	; Use last code dump address value
	mov	ds:[Code_Address],ax	; Setup the code dump address value
	call	Print_Title		; Call routine to print debug title
	call	Dump_Data		; Call routine to dump the data
	call	Print_Status		; Call routine to print status line
	call	Dump_Code		; Call routine to dump the code
	call	Print_Register		; Call routine to print register line
	call	Dump_Stack		; Call routine to dump the stack
	pop	si			; Restore the original window pointer
	call	Set_Window		; Call routine to set current window
Flip_Exit:
	clc				; Clear carry indicating more input
	Restore si,di,ds,es		; Restore the required registers
	ret				; Return to the caller
Flip_Command	Endp			; End of the Flip_Command procedure
	Subttl	Enter_Byte	Apple Emulator Enter Byte Data Routine
	Page	+
;******************************************************************************
;
;	Enter_Byte(Value, Offset)
;
;		Save the required registers
;		Save the current offset value
;		Call routine to write value into 65C02 space (Byte)
;		Restore the current offset value
;		Update the current offst value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Data value (Byte)
;		DI    - Offset in 65C02 space
;
;	Registers on Exit:
;
;		AX    - Destroyed
;		DI    - Offset value updated
;
;******************************************************************************
		Even			; Force procedure to even address
Enter_Byte	Proc	Near		; Debug enter byte data procedure
	Save	bp,ds			; Save the required registers
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	push	di			; Save the current offset value
	call	Write_Memory		; Call routine to write the data byte
	pop	di			; Restore the current offset value
	inc	di			; Update the current offset value
Byte_Exit:
	Restore bp,ds			; Restore the required registers
	ret				; Return to the caller
Enter_Byte	Endp			; End of the Enter_Byte procedure
	Subttl	Enter_Word	Apple Emulator Enter Word Data Routine
	Page	+
;******************************************************************************
;
;	Enter_Word(Value, Offset)
;
;		Save the required registers
;		Save the current offset value and data values
;		Call routine to write first byte into 65C02 space
;		Restore the current offset and data values
;		Update the current offset value
;		Save the current offset value
;		Call routine to write second byte into 65C02 space
;		Restore the current offset value
;		Update the current offset value
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Data value (Word)
;		DI    - Offset in 65C02 space
;
;	Registers on Exit:
;
;		AX    - Destroyed
;		DI    - Offset value updated
;
;******************************************************************************
		Even			; Force procedure to even address
Enter_Word	Proc	Near		; Debug enter word data procedure
	Save	bp,ds			; Save the required registers
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	Save	ax,di			; Save current offset and data values
	call	Write_Memory		; Call routine to write first data byte
	Restore ax,di			; Restore current offset and data values
	inc	di			; Update the current offset value
	xchg	al,ah			; Setup to write the second data byte
	push	di			; Save the current offset value
	call	Write_Memory		; Call routine to write second data byte
	pop	di			; Restore the current offset value
	inc	di			; Update the current offset value
Word_Exit:
	Restore bp,ds			; Restore the required registers
	ret				; Return to the caller
Enter_Word	Endp			; End of the Enter_Word procedure
	Subttl	Enter_String	Apple Emulator Enter String Data Routine
	Page	+
;******************************************************************************
;
;	Enter_String(Pointer, Offset, Location)
;
;		Save the required registers
;		Get the string ending character value
;		Setup character attribute value base
;		While more characters in the string
;			Get the next character value (Logically Or'ed into base)
;			Call routine to write the byte value
;			Update the current location value
;		EndWhile
;		Update the current parse offset
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		DI    - Location in 65C02 space
;
;	Registers on Exit:
;
;		AX    - Destroyed
;		DX    - Current parse offset updated
;		DI    - Location value updated
;
;******************************************************************************
		Even			; Force procedure to even address
Enter_String	Proc	Near		; Debug enter string data procedure
	Save	bx,cx,si,bp,ds,es	; Save the required registers
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	mov	ds,cs:[RAM_Space]	; Setup to access the 65C02 RAM space
	mov	bx,dx			; Setup to access the character string
	mov	ah,es:[bx + si] 	; Get the character string end character
	inc	bx			; Increment to next string character
	mov	cl,INTEL_ATTR		; Default to Intel attribute base
	cmp	ah,INTEL_STRING 	; Check for Intel string delimiter
	je	String_Loop		; Jump if this is an Intel string
	mov	cl,APPLE_ATTR		; Setup to Apple attribute base
String_Loop:
	mov	al,es:[bx + si] 	; Get the next string character
	cmp	al,ah			; Check for end of string
	je	String_Update		; Jump if at end of the string
	cmp	al,CR			; Check for end of input buffer
	je	String_Exit		; Jump if at end of input buffer
	Save	ax,di			; Save current location and data values
	or	al,cl			; Setup the character attribute base
	call	Write_Memory		; Call routine to write the data byte
	Restore ax,di			; Restore current offset and data values
	inc	di			; Update the current location value
	inc	bx			; Increment the string pointer
	jmp	Short String_Loop	; Loop till all of string is entered
String_Update:
	inc	bx			; Increment past character string
String_Exit:
	mov	dx,bx			; Update the current parse offset
	Restore bx,cx,si,bp,ds,es	; Restore the required registers
	ret				; Return to the caller
Enter_String	Endp			; End of the Enter_String procedure
	Subttl	Ebyte_Command	Apple Emulator Debug Enter Byte Command
	Page	+
;******************************************************************************
;
;	Ebyte_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Set current data format to bytes
;		Call routine to handle enter command
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Ebyte_Command	Proc	Near		; Debug enter byte command procedure
	mov	ds:[Enter_Format],FORMAT_BYTE
	call	Enter_Command		; Call routine to handle enter command
Ebyte_Exit:
	ret				; Return to the caller
Ebyte_Command	Endp			; End of the Ebyte_Command procedure
	Subttl	Eword_Command	Apple Emulator Debug Enter Word Command
	Page	+
;******************************************************************************
;
;	Eword_Command(Delimiter, Offset, Command, Stack_Frame)
;
;		Set current data format to words
;		Call routine to handle enter command
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Delimiter type/Parameter length
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		FL    - Carry flag set if no more debug commands
;			Zero flag set if not to exit debug mode
;
;******************************************************************************
		Even			; Force procedure to even address
Eword_Command	Proc	Near		; Debug enter word command procedure
	mov	ds:[Enter_Format],FORMAT_WORD
	call	Enter_Command		; Call routine to handle enter command
Eword_Exit:
	ret				; Return to the caller
Eword_Command	Endp			; End of the Eword_Command procedure
	Subttl	Get_Data	Get Fill/Search Data Routine
	Page	+
;******************************************************************************
;
;	Get_Data(Offset, Stack_Frame)
;
;		Save the required registers
;		Setup pointer to the data buffer
;		While there are more expressions
;			Call routine to get the next expression
;			Is this is a string expression
;				Call routine to get the string
;			Else numeric expression
;				Save expression value in buffer
;			Endif
;		EndWhile
;		Compute the number of data bytes
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		CX    - Number of data bytes
;		DX    - Parse address updated
;		FL    - Carry flag set if no data error
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Data	Proc	Near		; Get fill/search data procedure
	Save	ax,bx,di,es		; Save the required registers
	mov	ax,ds			; Setup access to
	mov	es,ax			;		  the data segment
	lea	di,ds:[Data_Buffer]	; Setup the data buffer pointer
Get_Loop:
	call	Get_Expression		; Call routine to get next data value
	jc	Get_Exit		; Jump if expression error
	jnz	Get_Value		; Jump if a valid expression given
	or	bh,bh			; Check for valid delimiter
	jz	Get_Done		; Jump if end of input buffer
	xor	ax,ax			; Default to a zero data value
	and	bh,SPC_MASK		; Mask off all but special delimiters
	cmp	bh,SPC_STRING		; Check for a string delimiter
	jne	Get_Loop		; Jump if this is not a string delimiter
	call	Get_String		; Call routine to get the string data
	jmp	Short Get_Loop		; Loop till all data values gotten
Get_Value:
	stosb				; Save value to the data buffer
	jmp	Short Get_Loop		; Loop till all data values gotten
Get_Done:
	lea	cx,ds:[Data_Buffer]	; Setup to calculate the data length
	sub	di,cx			; Compute the number of data bytes
	mov	cx,di			; Save the actual input data length
Get_Exit:
	Restore ax,bx,di,es		; Restore the required registers
	ret				; Return control to the caller
Get_Data	Endp			; End of the Get_Data procedure
	Subttl	Get_String	Apple Emulator Get String Data Routine
	Page	+
;******************************************************************************
;
;	Get_String(Pointer, Offset, Location)
;
;		Save the required registers
;		Get the string ending character value
;		Setup character attribute value base
;		While more characters in the string
;			Get the next character value (Logically Or'ed into base)
;			Put character value into buffer
;			Update the current location value
;		EndWhile
;		Update the current parse offset
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DX    - Current parse offset
;		DS:SI - Pointer to debug command
;		ES:DI - Pointer to data buffer
;
;	Registers on Exit:
;
;		DX    - Current parse offset updated
;		DI    - Location value updated
;
;******************************************************************************
		Even			; Force procedure to even address
Get_String	Proc	Near		; Debug enter word data procedure
	Save	ax,bx,cx		; Save the required registers
	mov	bx,dx			; Setup to access the character string
	mov	ah,es:[bx + si] 	; Get the character string end character
	inc	bx			; Increment to next string character
	mov	cl,INTEL_ATTR		; Default to Intel attribute base
	cmp	ah,INTEL_STRING 	; Check for Intel string delimiter
	je	Data_Loop		; Jump if this is an Intel string
	mov	cl,APPLE_ATTR		; Setup to Apple attribute base
Data_Loop:
	mov	al,es:[bx + si] 	; Get the next string character
	cmp	al,ah			; Check for end of string
	je	Data_Update		; Jump if at end of the string
	cmp	al,CR			; Check for end of input buffer
	je	Data_Exit		; Jump if at end of input buffer
	stosb				; Store character value into buffer
	inc	bx			; Increment the string pointer
	jmp	Short Data_Loop 	; Loop till all of string is gotten
Data_Update:
	inc	bx			; Increment past character string end
Data_Exit:
	mov	dx,bx			; Update the current parse offset
	Restore ax,bx,cx		; Restore the required registers
	ret				; Return to the caller
Get_String	Endp			; End of the Get_String procedure
;******************************************************************************
;
;	Define all the tables needed by the commands
;
;******************************************************************************
Register_Table	Equ	This Byte	; Debug register match table
		Db	3		; Debug register table entry size
		Db	2,"PC"          ; 65C02 program counter register
		Db	2,"SP"          ; 65C02 stack pointer register
		Db	2,"AC"          ; 65C02 accumulator register
		Db	1,"X "          ; 65C02 X index register
		Db	1,"Y "          ; 65C02 Y index register
		DB	1,"FL"          ; 65C02 flags register
		Db	0		; End of debug match table
Set_Table	Equ	This Word	; Debug register set jump table
		Dw	Set_PC		; Set 65C02 PC register routine
		Dw	Set_SP		; Set 65C02 SP register routine
		Dw	Set_ACC 	; Set 65C02 ACC register routine
		Dw	Set_X		; Set 65C02 X register routine
		Dw	Set_Y		; Set 65C02 Y register routine
		Dw	Set_Flags	; Set 65C02 flags register routine
Match_Table	Equ	This Byte	; Breakpoint type match table
		Db	3		; Breakpoint type table entry size
		Db	1,"SO"          ; Software breakpoint type
		Db	1,"RE"          ; Hardware breakpoint type (Read)
		Db	1,"WR"          ; Hardware breakpoint type (Write)
		Db	1,"AC"          ; Hardware breakpoint type (Access)
		Db	1,"EX"          ; Execution breakpoint type
Type_Table	Equ	This Byte	; Breakpoint type value table
		Db	TYPE_SOFT	; Software breakpoint type
		Db	TYPE_HARD	; Hardware breakpoint type (Read)
		Db	TYPE_HARD	; Hardware breakpoint type (Write)
		Db	TYPE_HARD	; Hardware breakpoint type (Access)
		Db	TYPE_EXECUTE	; Execution breakpoint type
Check_Table	Equ	This Byte	; Breakpoint check table (Hardware)
		Db	0		; ***** Illegal for software *****
		Db	READ		; Read check value
		Db	WRITE		; Write check value
		Db	ACCESS		; Access check value
		Db	0		; ***** Illegal for execution *****
Radix_Table	Equ	This Word	; Debug valid radix values table
		Dw	BINARY		; Binary base value (2)
		Dw	OCTAL		; Octal base value (8)
		Dw	DECIMAL 	; Decimal base value (10)
		Dw	HEX		; Hexadecimal base value (16)
Enter_Table	Equ	This Word	; Debug enter format jump table
		Dw	Enter_Byte	; Enter data byte routine
		Dw	Enter_Word	; Enter data word routine
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Command module
