	Page	58,132
	Title	CARTRIGE.ASM	Cartridge Routines
;******************************************************************************
;
;   Name:	CARTRIGE.ASM	Cartridge Routines
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the Apple
;	cartridge.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Cart_ID 		; Cartridge ID string
	Public	Cart_Init		; Cartridge initialization routine
	Public	Cart_Ctrl		; Cartridge control routine
	Public	Cart_Rd 		; Cartridge read routine
	Public	Cart_Wrt		; Cartridge write routine
	Public	Cart_Mem_Rd		; Cartridge memory read routine
	Public	Cart_Mem_Wrt		; Cartridge memory write routine
	Public	Cart_Exp_Rd		; Cartridge expansion read routine
	Public	Cart_Exp_Wrt		; Cartridge expansion write routine
	Public	Cart_Reset		; Cartridge reset routine
	Public	Cart_Data		; Cartridge data segment pointers
;
;  External Declarations
;
	Extrn	Get_Parameter:Near	; Get parameter routine       (SUPPORT)
	Extrn	Upper_Case:Near 	; Convert to upper case       (SUPPORT)
	Extrn	Match_Parameter:Near	; Match parameter routine     (SUPPORT)
	Extrn	Move_Block:Near 	; Move memory block routine   (SUPPORT)
	Extrn	Slot_Address:Near	; Get expansion slot address   (DEVICE)
	Extrn	Assignment:Near 	; Assignment test routine      (CONFIG)
	Extrn	Unknown:Near		; Unknown parameter routine    (CONFIG)
	Extrn	Noise:Near		; Skip noise routine	       (CONFIG)
	Extrn	Error:Near		; Apple emulator error routine	(APPLE)
	Extrn	Exit:Near		; Apple emulator exit routine	(APPLE)
	Extrn	RAM_Space:Word		; RAM space segment value	 (DATA)
	Extrn	Current_Owner:Byte	; Current owner slot number	 (DATA)
	Extrn	Delimit_Input:Byte	; Input delimiter table 	 (DATA)
	Extrn	Parm_Buffer:Byte	; Parameter buffer storage area  (DATA)
	Extrn	System_Flag:Byte	; Apple emulator system flag byte(DATA)
	Extrn	ERR_NO_MEMORY:Abs	; Not enough memory error code	 (DATA)
	Extrn	ERR_FILE_ERROR:Abs	; Missing filename error code	 (DATA)
;
;  LOCAL Equates
;
CTRL_SIZE	Equ	((Size Cart_Card) + 0Fh) Shr 4
CART_MAX	Equ	8000h		; Maximum cartridge size (32k Bytes)
AREA_MAX	Equ	0400h		; Maximum area size (2k/1k Bytes/Words)
AREA_START	Equ	0C800h		; Area starting address
AREA_SHIFT	Equ	03h		; Area shift count value
CTRL_INIT	equ	0FFh		; Cartridge control initialization
CONTROL_MASK	Equ	0Fh		; Cartridge control bits mask
ASCII_CONVERT	Equ	30h		; ASCII conversion value (Slot)
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include 	Strucs.inc	; Include the structure definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Cart_Init	Cartridge Initialization
	Page	+
;******************************************************************************
;
;	Cart_Init(Data, Offset, Slot_Number)
;
;		Save the required registers
;		Save the current parse position
;		Try to allocate memory for the cartridge area
;		If errors allocating memory
;			Set error code to not enough memory
;			Call the error routine
;			Call routine to exit the emulator
;		Endif
;		Save address of cartridge data area
;		Initialize the cartridge control byte
;		Try to open the cartridge file
;		If no errors opening cartridge file
;			Try to read the cartridge image
;		Endif for opening cartridge file
;		If configuration data is present
;			Call cartridge configuration routine
;		Endif
;		Update the current parse position
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Slot number (0 - 7)
;		DX    - Current parse offset
;		DS:SI - Pointer to configuration data (65C02 RAM segment)
;
;	Registers on Exit:
;
;		AX-CX - Destroyed
;		DX    - Updated to next parse offset
;		SI-DI - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Init	Proc	Near		; Cartridge initialization procedure
	Save	bp,ds,es		; Save the required registers
	push	si			; Save the configuration data pointer
	mov	bp,dx			; Save the current parse offset value
	mov	di,ax			; Get the language card slot number
	shl	di,1			; Convert slot number to table index
Cart_Allocate:
	mov	ah,ALLOCATE_MEMORY	; Get the allocate memory function code
	mov	bx,CTRL_SIZE		; Get number of paragraphs to allocate
	int	DOS			; Try to allocate cartridge space
	jnc	Cart_Setup		; Jump if no errors allocating space
Memory_Error:
	mov	al,ERR_NO_MEMORY	; Get not enough memory error code
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit the emulator
Cart_Setup:
	mov	ds,ax			; Setup cartridge segment address
	mov	cs:[di + Cart_Data],ax	; Save cartridge segment address
	mov	ds:[Cart_Byte],CTRL_INIT; Initialize the cartridge control byte
	mov	ax,ds			; Get the cartridge segment
	mov	es,ax			; Set ES to cartridge segment
	mov	ax,cs			; Get current CS register value
	mov	ds,ax			; Set DS to current CS register value
	mov	dx,di			; Get the cartridge slot index
	shr	dx,1			; Convert slot index to slot number
	add	dl,ASCII_CONVERT	; Convert slot to ASCII
	lea	si,ds:[Base_File]	; Get pointer to base file name
	lea	di,es:[Cart_File]	; Get pointer to cartridge file name
	mov	cx,Size Cart_Name	; Get length of the base file name
	rep	movsb			; Setup base file name for drive A
	mov	es:[Cart_File.Cart_Slot],dl
	mov	ax,es			; Get cartridge segment value
	mov	ds,ax			; Set DS to cartridge segment
	pop	si			; Restore configuration data pointer
	test	cs:[System_Flag],CONFIG_DATA
	jz	Open_Cart_File		; Jump if no configuration data present
	call	Cart_Config		; Call cartridge configuration routine
Open_Cart_File:
	mov	ah,OPEN_FILE		; Get the open file function code
	mov	al,READ_ONLY		; Get read only file access code
	mov	dx,Cart_File		; Get pointer to file name
	int	DOS			; Try to open cartridge file
	jc	Cart_Exit		; Jump if errors opening the file
Read_Cart_File:
	mov	bx,ax			; Move file handle to BX register
	mov	ah,READ_FILE		; Get read file function code
	mov	dx,Cart_Image		; Setup the buffer address
	mov	cx,CART_MAX		; Get maximum cartridge size (32k)
	int	DOS			; Try to read the cartridge image
	mov	ah,CLOSE_FILE		; Get close file function code
	int	DOS			; Close the disk ROM file
Cart_Exit:
	mov	dx,bp			; Update the current parse offset
	Restore bp,ds,es		; Restore the required registers
	ret				; Return to the caller
Cart_Init	Endp			; End of the Cart_Init procedure
	Subttl	Cart_Config	Cartridge Configuration Routine
	Page	+
;******************************************************************************
;
;	Cart_Config(Controller_Data, Pointer, Offset)
;
;		Save the required registers
;		Setup access to the configuration data
;		Save the controller card segment
;		While not at the next section
;			Call routine to get the next parameter
;			If a null parameter was given
;				If delimiter type is comment
;					Call routine to eat this comment
;				Endif
;				If delimiter type is section
;					Exit the while loop
;				Endif
;			Else valid parameter was given
;				Call routine to convert to uppercase
;				Call routine to check for parameter match
;				If no parameter match
;					Call routine for unknown parameter
;				Endif
;				Call routine to check for assignment
;				Call routine to get the next parameter
;				If there is a null parameter
;					If delimiter type is filename
;						Call routine to get next param.
;						If no valid parameter (non-null)
;							Set error to no file
;							Call routine to print
;							Call routine to exit
;						Endif
;					Else null parameter
;						Set error code to no filename
;						Call routine to print error
;						Call routine to exit emulator
;					Endif
;				Endif
;				Restore the controller card address
;				Copy the filename into position
;			Endif
;		Endwhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		SI    - Pointer to configuration data
;		BP    - Current parse offset
;		ES    - Controller data segment
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;		BP    - Current parse offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Config	Proc	Near		; Cartridge configuration procedure
	Save	ds,es			; Save the required registers
	mov	dx,bp			; Setup the current parse offset
	mov	ds,cs:[RAM_Space]	; Setup access to the 65C02 RAM space
	mov	bp,es			; Save the controller data segment
Cartridge_Loop:
	mov	ax,cs			; Setup access to
	mov	es,ax			;		  the parse tables
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	lea	di,es:[Parm_Buffer]	; Get pointer to parameter buffer
	mov	cx,PARM_SIZE		; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	jc	Config_Done		; Jump if no more parameters
	jnz	Check_Parameter 	; Jump if a non-null parameter
Check_Type:
	test	ah,SECTION_TYPE 	; Check for section type delimiter
	jnz	Config_Done		; Jump if next section reached
	test	ah,COMMENT_TYPE 	; Check for a comment type delimiter
	jz	Cartridge_Loop		; Jump if this is not a comment
	call	Noise			; Call routine to skip this noise
	jmp	Short Cartridge_Loop	; Go continue the parsing
Check_Parameter:
	call	Upper_Case		; Call routine to convert to uppercase
	Save	ax,si,di,ds,es		; Save the required register values
	mov	bx,es			; Setup to access
	mov	ds,bx			;		  parameter match table
	lea	si,ds:[Match_Table]	; Get pointer to start of match table
	xchg	al,ah			; Save the actual parameter length
	lodsb				; Get the match table entry size (Bytes)
	xchg	al,ah			; Restore the actual parameter length
	call	Match_Parameter 	; Call routine to check for a match
	mov	bl,al			; Save the matching parameter number
	Restore ax,si,di,ds,es		; Restore the required register values
	jnc	Process_Parameter	; Jump if a parameter match was found
Parameter_Error:
	call	Unknown 		; Call the unknown parameter routine
Config_Done:
	jmp	Short Config_Exit	; Go return control to the caller
Process_Parameter:
	call	Assignment		; Call routine to check for assignment
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	lea	bx,es:[Delimit_Input]	; Get pointer to input delimiter table
	mov	cx,Size File_Name	; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	jc	File_Error		; Jump if no parameter given
	jnz	Get_File		; Jump if a valid parameter given
Check_File:
	test	ah,FILENAME_TYPE	; Check for filename type delimiter
	jz	File_Error		; Jump if not a filename delimiter
	mov	al,NULL 		; Get the parameter terminator byte
	mov	ah,EOF			; Get the buffer terminator byte
	mov	cx,Size File_Name	; Get parameter buffer size [Bytes]
	call	Get_Parameter		; Call routine to get next parameter
	jbe	File_Error		; Jump if no/null parameter given
Get_File:
	Save	ax,si,di,ds,es		; Save the required register values
	xor	ax,ax			; Setup for variable length move
	mov	cx,es			; Setup to
	mov	ds,cx			;	   copy the given
	mov	si,di			;			  filename
	mov	es,bp			; Restore the controller data segment
	lea	di,es:[Cart_File]	; Get pointer to cartridge filename area
Copy_Name:
	call	Move_Block		; Call routine to move the filename
	Restore ax,si,di,ds,es		; Restore the required register values
	jmp	Short Check_Type	; Go continue the parsing
File_Error:
	mov	al,ERR_FILE_ERROR
	call	Error			; Call routine to print the error
	call	Exit			; Call routine to exit emulator
Config_Exit:
	dec	dx			; Update the current
	mov	bp,dx			;		     parse offset
	Restore ds,es			; Restore the required registers
	ret				; Return to the caller
Cart_Config	Endp			; End of the Cart_Config procedure
	Subttl	Cart_Ctrl	Cartridge Control
	Page	+
;******************************************************************************
;
;	Cart_Ctrl(RAM_Space, Slot_Number)
;
;
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Slot number (0 - 7)
;		DS    - 65C02 RAM space
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Ctrl	Proc	Near		; Cartridge card control procedure

	ret				; Return to the caller
Cart_Ctrl	Endp			; End of the Cart_Ctrl procedure
	Subttl	Cart_Rd 	Cartridge Read
	Page	+
;******************************************************************************
;
;	Cart_Rd(Effective_Address, Slot_Index)
;
;		Save the required registers
;		Setup the cartridge data segment
;		Get the cartridge control bits (From effective address)
;		Get the current control bits value
;		Update the control bits value
;		Call the cartridge update routine
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		AL    - Read value
;		AH    - Destroyed
;		BP    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Rd 	Proc	Near		; Cartridge read procedure
	Save	es			; Save the required registers
	mov	es,cs:[bp + Cart_Data]	; Setup the cartridge data segment
	mov	ax,di			; Get the effective address
	and	ax,CONTROL_MASK 	; Mask off all but the control bits
	mov	ah,es:[Cart_Byte]	; Get the current control bit values
	mov	es:[Cart_Byte],al	; Update the control bits value
	call	Cart_Update		; Call the cartridge update routine
Cart_Rd_Exit:
	Restore es			; Restore the required registers
	ret				; Return to the caller
Cart_Rd 	Endp			; End of the Cart_Rd procedure
	Subttl	Cart_Wrt	Cartridge Write
	Page	+
;******************************************************************************
;
;	Cart_Wrt(Effective_Address, Slot_Index, Write_Value)
;
;		Save the required registers
;		Setup the cartridge data segment
;		Get the cartridge control bits (From effective address)
;		Get the current control bits value
;		Update the control bits value
;		Call the cartridge update routine
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Write value
;		BP    - Slot index (Slot number * 2)
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		BP    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Wrt	Proc	Near		; Cartridge write procedure
	Save	ax,es			; Save the required registers
	mov	es,cs:[bp + Cart_Data]	; Setup the cartridge data segment
	mov	ax,di			; Get the effective address
	and	ax,CONTROL_MASK 	; Mask off all but the control bits
	mov	ah,es:[Cart_Byte]	; Get the current control bit values
	mov	es:[Cart_Byte],al	; Update the control bits value
	call	Cart_Update		; Call the cartridge update routine
Cart_Wrt_Exit:
	Restore ax,es			; Restore the required registers
	ret				; Return to the caller
Cart_Wrt	Endp			; End of the Cart_Wrt procedure
	Subttl	Cart_Mem_Rd	Cartridge Memory Read
	Page	+
;******************************************************************************
;
;	Cart_Mem_Rd(Effective_Address)
;
;		Read the memory location value (Byte)
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		AL    - Memory value
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Mem_Rd	Proc	Near		; Cartridge memory read procedure
	ret				; Return to the caller
Cart_Mem_Rd	Endp			; End of the Cart_Mem_Rd procedure
	Subttl	Cart_Mem_Wrt	Cartridge Memory Write
	Page	+
;******************************************************************************
;
;	Cart_Mem_Wrt(Effective_Address, Memory_Value)
;
;		Write the memory location value (Standard RAM)
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Memory value
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Mem_Wrt	Proc	Near		; Cartridge memory write procedure
	mov	Byte Ptr ds:[di],al	; Write the memory location
	ret				; Return to the caller
Cart_Mem_Wrt	Endp			; End of the Cart_Mem_Wrt procedure
	Subttl	Cart_Exp_Rd	Cartridge Expansion Read
	Page	+
;******************************************************************************
;
;	Cart_Exp_Rd(Effective_Address)
;
;		Read the memory location value (Byte)
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		AL    - Memory value
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Exp_Rd	Proc	Near		; Cartridge expansion read procedure
	mov	al,ds:[di]		; Read the memory location
	ret				; Return to the caller
Cart_Exp_Rd	Endp			; End of the Cart_Exp_Rd procedure
	Subttl	Cart_Exp_Wrt	Cartridge Expansion Write
	Page	+
;******************************************************************************
;
;	Cart_Exp_Wrt(Effective_Address, Memory_Value)
;
;		Return to the caller (Cartridge is NOT writable)
;
;	Registers on Entry:
;
;		AL    - Memory value
;		DS:DI - 65C02 Effective address
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Exp_Wrt	Proc	Near		; Cartridge expansion write procedure
	ret				; Return to the caller
Cart_Exp_Wrt	Endp			; End of the Cart_Exp_Wrt procedure
	Subttl	Cart_Reset	Cartridge Reset Routine
	Page	+
;******************************************************************************
;
;	Cart_Reset()
;
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Reset	Proc	Near		; Cartridge reset procedure

	ret				; Return to the caller
Cart_Reset	Endp			; End of the Cart_Reset procedure
	Subttl	Cart_Update	Cartridge Update Routine
	Page	+
;******************************************************************************
;
;	Cart_Update(Old_Control, New_Control, RAM_Space, Cart_Segment, Slot)
;
;		Save the required registers
;		If control bits have changed
;			Get the size of area to update
;			Compute the source address
;			Setup destination to area
;			Update the memory area from cartridge
;		Else
;			If current owner NOT this slot
;				Get the size of area to update
;				Compute the source address
;				Setup destination to area
;				Update the memory area from cartridge
;			Endif
;		Update the current owner to this slot
;		Endif for control bits
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AH    - Old control bits
;		AL    - New control bits
;		BP    - Slot index (Slot number * 2)
;		DS    - 65C02 RAM space
;		ES    - Cartridge segment
;
;	Registers on Exit:
;
;		AX-BX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Cart_Update	Proc	Near		; Cartridge update procedure
	Save	cx,si,di,ds,es		; Save the required registers
	cmp	al,ah			; Check for control bits change
	jne	Update_Area		; Jump if control bits have changed
	mov	bx,bp			; Get the slot index value
	shr	bx,1			; Compute the actual slot number
	cmp	bl,cs:[Current_Owner]	; Check slot number against current
	je	Update_Done		; Jump if slot number still current
Update_Area:
	mov	cx,AREA_MAX		; Get area maximum size (Words)
	mov	ah,al			; Get the new control bit settings
	xor	al,al			; Convert control bits to full word
	shl	ax,AREA_SHIFT		; Shift control bits into position
	mov	si,ax			; Setup the source index value
	add	si,Cart_Image		; Compute the actual source index
	mov	ax,ds			; Get the 65C02 RAM space segment
	mov	di,es			; Get the cartridge segment value
	mov	ds,di			; Setup DS to the cartridge segment
	mov	es,ax			; Setup ES to the 65C02 RAM space
	mov	di,AREA_START		; Setup destination to the area
	rep	movsw			; Move correct section into the area
	mov	cs:[Current_Owner],bl	; Update the current slot owner value
Update_Done:
	Restore cx,si,di,ds,es		; Restore the required registers
Update_Exit:
	ret				; Return to the caller
Cart_Update	Endp			; End of the Cart_Update procedure
;******************************************************************************
;
;	Define the cartridge data areas
;
;******************************************************************************
Cart_Data	Equ	This Word	; Define the cartridge pointers
		Slot_Data	<>	; Pointers to the cartridge data areas
Base_File	Cart_Name	<>	; Define base cartridge file name
Cart_ID 	Equ	This Byte	; Cartridge ID string
		Db	"Cartridge",0
Match_Table	Equ	This Byte	; Start of floppy parameter match table
		Db	10		; Match table entry size
		Db	4,"CARTRIDGE"   ; Specify cartridge file parameter
		Db	0		; End of the match table
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Cartrige module
