	Page	58,132
	Title	BREAK.ASM	Apple System Break Routines
;******************************************************************************
;
;   Name:	BREAK.ASM	Apple System Break Routines
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the support code for the Apple ][
;	system debugger break routines.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Break_Setup		; Breakpoint setup routine
	Public	Break_Reset		; Breakpoint reset routine
	Public	Break_Set		; Breakpoint set routine
	Public	Break_Find		; Breakpoint find routine
	Public	Break_Clear		; Breakpoint clear routine
	Public	Break_Enable		; Breakpoint enable routine
	Public	Break_Disable		; Breakpoint disable routine
	Public	Break_List		; Breakpoint list routine
	Public	Break_Check		; Breakpoint check routine
;
;  External Declarations
;
	Extrn	Printf:Near		; Generic Printf routine       (PRINTF)
	Extrn	Emulate_Flag:Byte	; Emulator system flag byte	 (DATA)
	Extrn	System_Flag:Byte	; Apple emulator system flag byte(DATA)
	Extrn	RAM_Space:Word		; RAM space segment value	 (DATA)
	Extrn	Value:Word		; Current value storage area	(DEBUG)
	Extrn	Break_Address:Word	; Breakpoint address value	(DEBUG)
	Extrn	Code_Count:Word 	; Code breakpoints count value	(DEBUG)
	Extrn	Data_Count:Word 	; Data breakpoints count value	(DEBUG)
	Extrn	Code_Table:Word 	; Code breakpoint address table (DEBUG)
	Extrn	Data_Table:Word 	; Data breakpoint address table (DEBUG)
	Extrn	Type_Table:Word 	; Debugger type table		(DEBUG)
	Extrn	Break_Table:Byte	; Debug breakpoint table	(DEBUG)
	Extrn	Local_Break:Byte	; Local breakpoint structure	(DEBUG)
	Extrn	Break_Arg:Word		; Breakpoint argument list    (MESSAGE)
	Extrn	Break_Fmt:Byte		; Breakpoint format string    (MESSAGE)
	Extrn	Break_Status:Word	; Pointer to status string    (MESSAGE)
	Extrn	Break_Type:Word 	; Pointer to type string      (MESSAGE)
	Extrn	Enable_String:Byte	; Breakpoint enabled string   (MESSAGE)
	Extrn	Disable_String:Byte	; Breakpoint disabled string  (MESSAGE)
	Extrn	Software_String:Byte	; Software type string	      (MESSAGE)
	Extrn	Illegal_String:Byte	; Illegal type string	      (MESSAGE)
	Extrn	Read_String:Byte	; Hardware read type string   (MESSAGE)
	Extrn	Write_String:Byte	; Hardware write type string  (MESSAGE)
	Extrn	Access_String:Byte	; Hardware access type string (MESSAGE)
	Extrn	Execute_String:Byte	; Execution type string       (MESSAGE)
;
;  LOCAL Equates
;
HARD_ROTATE	Equ	02h		; Hardware type rotate count
HARD_MASK	Equ	03h		; Hardware type mask value
BREAK_OPCODE	Equ	0Bh		; Debugger break opcode value
BREAK_OFFSET	Equ	02h		; Breakpoint offset value
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include 	Strucs.inc	; Include the structure definitions
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Break_Setup	Setup Debug Breakpoints Routine
	Page	+
;******************************************************************************
;
;	Break_Setup()
;
;		Save the required registers
;		Zero the code and data counters
;		Clear system break interrupt
;		Set to check all breakpoints (Including local)
;		While there are more breakpoints
;			Get the breakpoint type value
;			If this breakpoint is used
;				If this breakpoint is enabled
;					Call routine to setup breakpoint
;				Endif
;			Endif
;		EndWhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Break_Setup	Proc	Near		; Setup debug breakpoints procedure
	Save	si,di,bp,ds		; Save the required registers
	mov	ax,cs			; Setup access to
	mov	ds,ax			;		  the data segment
	and	cs:[Emulate_Flag],Not SYS_BREAK
	xor	ax,ax			; Setup to zero the count values
	mov	ds:[Code_Count],ax	; Initialize code and
	mov	ds:[Data_Count],ax	;		      data count values
	xor	bx,bx			; Initialize code and
	xor	bp,bp			;		      data table offsets
	mov	cx,BREAKPOINTS + 1	; Setup the breakpoint counter (Local)
	lea	si,ds:[Break_Table]	; Get pointer to breakpoint table
Setup_Loop:
	mov	al,ds:[si.Debug_Type]	; Get the breakpoint type value
	test	al,TYPE_USED		; Check for this breakpoint used
	jz	Next_Setup		; Jump if this breakpoint not used
	test	al,TYPE_ENABLED 	; Check for this breakpoint enabled
	jz	Next_Setup		; Jump if this breakpoint not enabled
	and	ax,BREAK_MASK		; Mask off all but the breakpoint type
	shl	ax,1			; Convert type value to table index
	mov	di,ax			; Setup to access the setup table
	call	ds:[Setup_Table + di]	; Call routine to setup breakpoint
Next_Setup:
	add	si,Size Debug_Break	; Increment to the next breakpoint entry
	loop	Setup_Loop		; Loop till all breakpoints setup
Setup_Exit:
	Restore si,di,bp,ds		; Restore the required registers
	ret				; Return control to the caller
Break_Setup	Endp			; End of the Break_Setup procedure
	Subttl	Setup_Soft	Setup Software Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Setup_Soft(Breakpoint)
;
;		Save the required registers
;			Get the breakpoint address value
;			Read the current memory value
;			Save the current opcode value
;			Put break opcode value into place
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Setup_Soft	Proc	Near		; Setup software breakpoint procedure
	Save	di,ds			; Save the required registers
	mov	ds,cs:[RAM_Space]	; Setup to access 65C02 RAM space
	mov	di,cs:[si.Soft_Address] ; Setup the breakpoint address value
	mov	al,ds:[di]		; Read the current opcode value
	mov	cs:[si.Soft_Opcode],al	; Save the current opcode value
	mov	Byte Ptr ds:[di],BREAK_OPCODE
	Restore di,ds			; Restore the required registers
	ret				; Return control to the caller
Setup_Soft	Endp			; End of the Setup_Soft procedure
	Subttl	Setup_Hard	Setup Hardware Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Setup_Hard(Breakpoint, Offset)
;
;		Get the breakpoint address value
;		Save breakpoint address in data table
;		Increment the data count value
;		Get the breakpoint check value
;		Convert check value to a full word
;		Save check value in type table
;		Increment the data table offset
;		Set system break interrupt
;		Return to the caller
;
;	Registers on Entry:
;
;		BP    - Data table offset
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX    - Destroyed
;		BP    - Data table offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Setup_Hard	Proc	Near		; Setup hardware breakpoint procedure
	mov	ax,ds:[si.Hard_Address] ; Get the breakpoint address value
	mov	ds:[Data_Table + bp],ax ; Save address value in data table
	inc	ds:[Data_Count] 	; Increment the data count value
	mov	al,ds:[si.Hard_Check]	; Get the hardware check value
	xor	ah,ah			; Convert check value to a full word
	mov	ds:[Type_Table + bp],ax ; Save the check value in the type table
	add	bp,2			; Increment to the next table offset
	or	cs:[Emulate_Flag],SYS_BREAK
	ret				; Return control to the caller
Setup_Hard	Endp			; End of the Setup_Hard procedure
	Subttl	Setup_Exec	Setup Execution Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Setup_Exec(Breakpoint, Offset)
;
;		Get the breakpoint address value
;		Save breakpoint address in code table
;		Increment the code count value
;		Increment the code table offset
;		Set system break interrupt
;		Return to the caller
;
;	Registers on Entry:
;
;		BX    - Code table offset
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX    - Destroyed
;		BX    - Code table offset updated
;
;******************************************************************************
		Even			; Force procedure to even address
Setup_Exec	Proc	Near		; Setup execution breakpoint procedure
	mov	ax,ds:[si.Exec_Address] ; Get the breakpoint address value
	mov	ds:[Code_Table + bx],ax ; Save address value in code table
	inc	ds:[Code_Count] 	; Increment the code count value
	add	bx,2			; Increment to the next table offset
	or	cs:[Emulate_Flag],SYS_BREAK
	ret				; Return control to the caller
Setup_Exec	Endp			; End of the Setup_Exec procedure
	Subttl	Break_Reset	Reset Debug Breakpoints Routine
	Page	+
;******************************************************************************
;
;	Break_Reset(Stack_Frame)
;
;		Save the required registers
;		Zero the code and data counters
;		Set to check all breakpoints (Including local)
;		While there are more breakpoints
;			Get the breakpoint type value
;			If this breakpoint is used
;				If this breakpoint is enabled
;					Call routine to reset breakpoint
;				Endif
;			Endif
;		EndWhile
;		Clear the local breakpoint
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		SS:BP - Pointer to debug stack frame (65C02 Registers)
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Break_Reset	Proc	Near		; Reset debug breakpoints procedure
	Save	si,di,ds		; Save the required registers
	mov	ax,cs			; Setup access to
	mov	ds,ax			;		  the data segment
	xor	ax,ax			; Setup to zero the count values
	mov	ds:[Code_Count],ax	; Initialize code and
	mov	ds:[Data_Count],ax	;		      data count values
	mov	cx,BREAKPOINTS + 1	; Setup the breakpoint counter (Local)
	lea	si,ds:[Break_Table]	; Get pointer to breakpoint table
Reset_Loop:
	mov	al,ds:[si.Debug_Type]	; Get the breakpoint type value
	test	al,TYPE_USED		; Check for this breakpoint used
	jz	Next_Reset		; Jump if this breakpoint not used
	test	al,TYPE_ENABLED 	; Check for this breakpoint enabled
	jz	Next_Reset		; Jump if this breakpoint not enabled
	and	ax,BREAK_MASK		; Mask off all but the breakpoint type
	shl	ax,1			; Convert type value to table index
	mov	di,ax			; Setup to access the setup table
	call	ds:[Reset_Table + di]	; Call routine to reset breakpoint
Next_Reset:
	add	si,Size Debug_Break	; Increment to the next breakpoint entry
	loop	Reset_Loop		; Loop till all breakpoints reset
Reset_Exit:
	mov	ds:[Local_Break.Debug_Type],0
	Restore si,di,ds		; Restore the required registers
	ret				; Return control to the caller
Break_Reset	Endp			; End of the Break_Reset procedure
	Subttl	Reset_Soft	Reset Software Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Reset_Soft(Breakpoint)
;
;		Save the required registers
;			Get the breakpoint address value
;			Get the saved opcode value
;			Put saved opcode value back into place
;			If program counter is on this breakpoint
;				Decrement program counter to replaced opcode
;				Setup the new execution address
;			Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Reset_Soft	Proc	Near		; Reset software breakpoint procedure
	Save	si,di,ds		; Save the required registers
	mov	ds,cs:[RAM_Space]	; Setup to access 65C02 RAM space
	mov	di,cs:[si.Soft_Address] ; Setup the breakpoint address value
	mov	al,cs:[si.Soft_Opcode]	; Get the saved opcode value
	mov	ds:[di],al		; Restore the original opcode value
	mov	si,cs:[si.Soft_Address] ; Get the breakpoint address
	add	si,BREAK_OFFSET 	; Setup to check for break execution
	cmp	si,ss:[bp.SI_Register]	; Check for breakpoint execution
	jne	Restore_Exit		; Jump if did not execute breakpoint
	sub	si,BREAK_OFFSET 	; Decrement to original instruction
	Setup				; Setup to
	mov	ss:[bp.SI_Register],si	;	   execute from
	mov	ss:[bp.DI_Register],di	;			desired location
Restore_Exit:
	Restore si,di,ds		; Restore the required registers
	ret				; Return control to the caller
Reset_Soft	Endp			; End of the Reset_Soft procedure
	Subttl	Reset_Hard	Reset Hardware Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Reset_Hard(Breakpoint)
;
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Reset_Hard	Proc	Near		; Reset hardware breakpoint procedure
	ret				; Return control to the caller
Reset_Hard	Endp			; End of the Reset_Hard procedure
	Subttl	Reset_Exec	Reset Execution Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Reset_Exec(Breakpoint)
;
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Reset_Exec	Proc	Near		; Reset execution breakpoint procedure
	ret				; Return control to the caller
Reset_Exec	Endp			; End of the Reset_Exec procedure
	Subttl	Break_Set	Set Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Break_Set(Breakpoint, Address, Type, Data)
;
;		Save the required registers
;		If this breakpoint is not used
;			Get the breakpoint type value
;			Call correct routine to set this breakpoint
;		Else this breakpoint already used
;			Set carry indicating cannot set breakpoint
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Breakpoint number
;		BX    - Breakpoint address
;		CL    - Breakpoint type
;		CH    - Breakpoint specific data (Hardware only)
;		DS    - Data segment value
;
;	Registers on Exit:
;
;		FL    - Carry set if error setting breakpoint
;
;******************************************************************************
		Even			; Force procedure to even address
Break_Set	Proc	Near		; Set breakpoint procedure
	Save	ax,bx,cx,dx,si,di	; Save the required registers
	lea	si,ds:[Break_Table]	; Get pointer to breakpoint table
	mov	ah,Size Debug_Break	; Get size of the breakpoint structure
	mul	ah			; Compute the actual table offset
	add	si,ax			; Setup to set requested breakpoint
	test	ds:[si.Debug_Type],TYPE_USED
	stc				; Set carry in case breakpoint used
	jnz	Set_Exit		; Jump if this breakpoint already used
	mov	di,cx			; Get the breakpoint type value
	and	di,BREAK_MASK		; Mask off all but the actual type
	shl	di,1			; Convert breakpoint type to table index
	call	ds:[Set_Table + di]	; Call routine to set this breakpoint
Set_Exit:
	Restore ax,bx,cx,dx,si,di	; Restore the required registers
	ret				; Return control to the caller
Break_Set	Endp			; End of the Break_Set procedure
	Subttl	Set_Soft	Set Software Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Set_Soft(Breakpoint, Address, Type, Data)
;
;		Save the breakpoint address value
;		Setup software breakpoint type (Software + Used + Enabled)
;		Clear carry indicating able to set breakpoint
;		Return to the caller
;
;	Registers on Entry:
;
;		BX    - Breakpoint address
;		CL    - Breakpoint type (Software)
;		CH    - Breakpoint specific data (Not Used)
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;		FL    - Carry flag set if error setting breakpoint
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Soft	Proc	Near		; Set software breakpoint procedure
	mov	ds:[si.Soft_Address],bx ; Set the software breakpoint address
	or	cl,TYPE_USED + TYPE_ENABLED
	mov	ds:[si.Soft_Type],cl	; Setup the breakpoint type value
Soft_Exit:
	clc				; Clear carry indicating no errors
	ret				; Return control to the caller
Set_Soft	Endp			; End of the Set_Soft procedure
	Subttl	Set_Hard	Set Hardware Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Set_Hard(Breakpoint, Address, Type, Data)
;
;		Setup the hardware breakpoint address
;		Setup the hardware check type value (Read/Write/Access)
;		Setup the hardware breakpoint type (Hardware + Used + Enabled)
;		Clear carry indicating no errors
;		Return to the caller
;
;	Registers on Entry:
;
;		BX    - Breakpoint address
;		CL    - Breakpoint type (Hardware)
;		CH    - Breakpoint specific data (Type - Read/Write/Access)
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;		FL    - Carry flag set if error setting breakpoint
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Hard	Proc	Near		; Set hardware breakpoint procedure
	mov	ds:[si.Hard_Address],bx ; Set the hardware breakpoint address
	mov	ds:[si.Hard_Check],ch	; Set the hardware breakpoint type
	or	cl,TYPE_USED + TYPE_ENABLED
	mov	ds:[si.Hard_Type],cl	; Setup the breakpoint type value
Hard_Exit:
	clc				; Clear carry indicating no errors
	ret				; Return control to the caller
Set_Hard	Endp			; End of the Set_Hard procedure
	Subttl	Set_Exec	Set Execution Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Set_Exec(Breakpoint, Address, Type, Data)
;
;		Setup the execution breakpoint address
;		Setup the execution breakpoint type (Execute + Used + Enabled)
;		Clear carry indicating no errors
;		Return to the caller
;
;	Registers on Entry:
;
;		BX    - Breakpoint address
;		CL    - Breakpoint type (Execution)
;		CH    - Breakpoint specific data (Not Used)
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX-DX - Destroyed
;		SI-DI - Destroyed
;		FL    - Carry flag set if error setting breakpoint
;
;******************************************************************************
		Even			; Force procedure to even address
Set_Exec	Proc	Near		; Set execution breakpoint procedure
	mov	ds:[si.Exec_Address],bx ; Set the execution breakpoint address
	or	cl,TYPE_USED + TYPE_ENABLED
	mov	ds:[si.Exec_Type],cl	; Setup the breakpoint type value
Exec_Exit:
	clc				; Clear carry indicating no errors
	ret				; Return control to the caller
Set_Exec	Endp			; End of the Set_Exec procedure
	Subttl	Break_Find	Find Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Break_Find()
;
;		Save the required registers
;		While there are more breakpoint entries
;			If this is a non-used breakpoint
;				Compute the actual breakpoint numer
;				Exit the while loop
;			Endif
;			Increment to the next entry
;		EndWhile
;		If no non-used breakpoints found
;			Set carry indicating no breakpoint
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		DS    - Data segment value
;
;	Registers on Exit:
;
;		AX    - Breakpoint number
;		FL    - Carry set if no available breakpoints
;
;******************************************************************************
		Even			; Force procedure to even address
Break_Find	Proc	Near		; Find breakpoint procedure
	Save	cx,si			; Save the required registers
	mov	cx,BREAKPOINTS		; Get the number of breakpoints to check
	lea	si,ds:[Break_Table]	; Initialize breakpoint pointer
Find_Loop:
	test	ds:[si.Debug_Type],TYPE_USED
	jz	Found_Breakpoint	; Jump if this breakpoint not used
	add	si,Size Debug_Break	; Increment to next breakpoint entry
	loop	Find_Loop		; Loop till all entries are checked
	stc				; Set carry indicating no breakpoint
	jmp	Short Find_Exit 	; Go return control to the caller
Found_Breakpoint:
	mov	ax,BREAKPOINTS		; Compute the actual
	sub	ax,cx			;		     breakpoint number
Find_Exit:
	Restore cx,si			; Restore the required registers
	ret				; Return control to the caller
Break_Find	Endp			; End of the Break_Find procedure
	Subttl	Break_Clear	Clear Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Break_Clear(Breakpoint)
;
;		Save the required registers
;		Clear the requested breakpoint
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Breakpoint number
;		DS    - Data segment value
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Break_Clear	Proc	Near		; Clear breakpoint procedure
	Save	ax,si			; Save the required registers
	lea	si,ds:[Break_Table]	; Get pointer to breakpoint table
	mov	ah,Size Debug_Break	; Get size of the breakpoint structure
	mul	ah			; Compute the actual table offset
	add	si,ax			; Setup to clear requested breakpoint
	and	ds:[si.Debug_Type],Not TYPE_USED
Clear_Exit:
	Restore ax,si			; Restore the required registers
	ret				; Return control to the caller
Break_Clear	Endp			; End of the Break_Clear procedure
	Subttl	Break_Enable	Enable Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Break_Enable(Breakpoint)
;
;		Save the required registers
;		If the requested breakpoint is used
;			Enable the requested breakpoint
;			Clear carry indicating no errors
;		Else
;			Set carry indicating enable error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Breakpoint number
;		DS    - Data segment value
;
;	Registers on Exit:
;
;		FL    - Carry set if error enabling breakpoint
;
;******************************************************************************
		Even			; Force procedure to even address
Break_Enable	Proc	Near		; Enable breakpoint procedure
	Save	ax,si			; Save the required registers
	lea	si,ds:[Break_Table]	; Get pointer to breakpoint table
	mov	ah,Size Debug_Break	; Get size of the breakpoint structure
	mul	ah			; Compute the actual table offset
	add	si,ax			; Setup to enable requested breakpoint
Enable_Check:
	test	ds:[si.Debug_Type],TYPE_USED
	stc				; Set carry flag in case no breakpoint
	jz	Enable_Exit		; Jump if breakpoint not used
	or	ds:[si.Debug_Type],TYPE_ENABLED
Enable_Exit:
	Restore ax,si			; Restore the required registers
	ret				; Return control to the caller
Break_Enable	Endp			; End of the Break_Enable procedure
	Subttl	Break_Disable	Disable Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Break_Disable(Breakpoint)
;
;		Save the required registers
;		If the requested breakpoint is used
;			Disable the requested breakpoint
;			Clear carry indicating no errors
;		Else
;			Set carry indicating disable error
;		Endif
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Breakpoint number
;		DS    - Data segment value
;
;	Registers on Exit:
;
;		FL    - Carry set if error disabling breakpoint
;
;******************************************************************************
		Even			; Force procedure to even address
Break_Disable	Proc	Near		; Disable breakpoint procedure
	Save	ax,si			; Save the required registers
	lea	si,ds:[Break_Table]	; Get pointer to breakpoint table
	mov	ah,Size Debug_Break	; Get size of the breakpoint structure
	mul	ah			; Compute the actual table offset
	add	si,ax			; Setup to disable requested breakpoint
Disable_Check:
	test	ds:[si.Debug_Type],TYPE_USED
	stc				; Set carry flag in case no breakpoint
	jz	Disable_Exit		; Jump if breakpoint not used
	and	ds:[si.Debug_Type],Not TYPE_ENABLED
Disable_Exit:
	Restore ax,si			; Restore the required registers
	ret				; Return control to the caller
Break_Disable	Endp			; End of the Break_Disable procedure
	Subttl	Break_List	List Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Break_List(Breakpoint)
;
;		Save the required registers
;
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Breakpoint number
;		DS    - Data segment value
;
;	Registers on Exit:
;
;		FL    - Carry set if error listing breakpoint
;
;******************************************************************************
		Even			; Force procedure to even address
Break_List	Proc	Near		; List breakpoint procedure
	Save	ax,bx,si		; Save the required registers
	lea	si,ds:[Break_Table]	; Get pointer to breakpoint table
	mov	ds:[Value],ax		; Save the actual breakpoint number
	mov	ah,Size Debug_Break	; Get size of the breakpoint structure
	mul	ah			; Compute the actual table offset
	add	si,ax			; Setup to list requested breakpoint
List_Check:
	test	ds:[si.Debug_Type],TYPE_USED
	stc				; Set carry flag in case no breakpoint
	jz	List_Exit		; Jump if breakpoint not used
	mov	bl,ds:[si.Debug_Type]	; Get the debug breakpoint type value
	and	bx,BREAK_MASK		; Mask off all but the breakpoint type
	shl	bx,1			; Convert breakpoint type to table index
	call	ds:[List_Table + bx]	; Call routine to print breakpoint
	clc				; Clear carry indicating no error
List_Exit:
	Restore ax,bx,si		; Restore the required registers
	ret				; Return control to the caller
Break_List	Endp			; End of the Break_List procedure
	Subttl	List_Soft	List Software Breakpoint Routine
	Page	+
;******************************************************************************
;
;	List_Soft(Breakpoint)
;
;		Get and save the software breakpoint address
;		Set breakpoint type string to software
;		If this breakpoint is enabled
;			Set status string to enabled
;		Else
;			Set status string to disabled
;		Endif
;		Get pointer to arguments and format string
;		Call routine to print this breakpoint
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX-BX - Destroyed
;		SI    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
List_Soft	Proc	Near		; List software breakpoint procedure
	mov	ax,ds:[si.Soft_Address] ; Get software breakpoint address
	mov	ds:[Break_Address],ax	; Save the software breakpoint address
	lea	ax,ds:[Software_String] ; Get pointer to software type string
	mov	ds:[Break_Type],ax	; Setup the breakpoint type string
	lea	ax,ds:[Enable_String]	; Default to breakpoint enabled
	test	ds:[si.Soft_Type],TYPE_ENABLED
	jnz	Soft_Status		; Jump if breakpoint is enabled
	lea	ax,ds:[Disable_String]	; Setup pointer to disable string
Soft_Status:
	mov	ds:[Break_Status],ax	; Save the breakpoint status string
	lea	bx,ds:[Break_Arg]	; Get pointer to breakpoint arguments
	lea	si,ds:[Break_Fmt]	; Get pointer to breakpoint format
	call	Printf			; Call routine to print this breakpoint
	ret				; Return control to the caller
List_Soft	Endp			; End of the List_Soft procedure
	Subttl	List_Hard	List Hardware Breakpoint Routine
	Page	+
;******************************************************************************
;
;	List_Hard(Breakpoint)
;
;		Get and save the hardware breakpoint address
;		Setup the breakpoint type string (Read, Write, Access)
;		If this breakpoint is enabled
;			Set status string to enabled
;		Else
;			Set status string to disabled
;		Endif
;		Get pointer to arguments and format string
;		Call routine to print this breakpoint
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX-BX - Destroyed
;		SI    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
List_Hard	Proc	Near		; List hardware breakpoint procedure
	mov	ax,ds:[si.Hard_Address] ; Get hardware breakpoint address
	mov	ds:[Break_Address],ax	; Save the hardware breakpoint address
	mov	bl,ds:[si.Hard_Check]	; Get the hardware check value
	rol	bl,HARD_ROTATE		; Rotate check bits into position
	and	bx,HARD_MASK		; Mask off all but the check bits
	shl	bx,1			; Convert check bits to table index
	mov	bx,ds:[Hard_Table + bx] ; Get pointer to correct type string
	mov	ds:[Break_Type],bx	; Save the breakpoint type string
	lea	ax,ds:[Enable_String]	; Default to breakpoint enabled
	test	ds:[si.Hard_Type],TYPE_ENABLED
	jnz	Hard_Status		; Jump if breakpoint is enabled
	lea	ax,ds:[Disable_String]	; Setup pointer to disable string
Hard_Status:
	mov	ds:[Break_Status],ax	; Save the breakpoint status string
	lea	bx,ds:[Break_Arg]	; Get pointer to breakpoint arguments
	lea	si,ds:[Break_Fmt]	; Get pointer to breakpoint format
	call	Printf			; Call routine to print this breakpoint
	ret				; Return control to the caller
List_Hard	Endp			; End of the List_Hard procedure
	Subttl	List_Execute	List Execution Breakpoint Routine
	Page	+
;******************************************************************************
;
;	List_Execute(Breakpoint)
;
;		Get and save the execution breakpoint address
;		If this breakpoint is enabled
;			Set status string to enabled
;		Else
;			Set status string to disabled
;		Endif
;		Get pointer to arguments and format string
;		Call routine to print this breakpoint
;		Return to the caller
;
;	Registers on Entry:
;
;		DS:SI - Pointer to breakpoint structure
;
;	Registers on Exit:
;
;		AX-BX - Destroyed
;		SI    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
List_Execute	Proc	Near		; List execution breakpoint procedure
	mov	ax,ds:[si.Exec_Address] ; Get execution breakpoint address
	mov	ds:[Break_Address],ax	; Save the execution breakpoint address
	lea	ax,ds:[Execute_String]	; Get pointer to execution type string
	mov	ds:[Break_Type],ax	; Setup the breakpoint type string
	lea	ax,ds:[Enable_String]	; Default to breakpoint enabled
	test	ds:[si.Exec_Type],TYPE_ENABLED
	jnz	Exec_Status		; Jump if breakpoint is enabled
	lea	ax,ds:[Disable_String]	; Setup pointer to disable string
Exec_Status:
	mov	ds:[Break_Status],ax	; Save the breakpoint status string
	lea	bx,ds:[Break_Arg]	; Get pointer to breakpoint arguments
	lea	si,ds:[Break_Fmt]	; Get pointer to breakpoint format
	call	Printf			; Call routine to print this breakpoint
	ret				; Return control to the caller
List_Execute	Endp			; End of the List_Execute procedure
	Subttl	Break_Check	Check Breakpoint Routine
	Page	+
;******************************************************************************
;
;	Break_Check(Address)
;
;		Save the required registers
;		Set to check all breakpoints (Excluding local)
;		While there are more breakpoints
;			Get the breakpoint type value
;			If this breakpoint is used
;				If this is not a hardware breakpoint
;					If breakpoint matches current address
;						Save the breakpoint number
;						Clear carry indicating match
;						Check for breakpoint enabled
;						Exit the while loop
;					Endif
;				Endif
;			Endif
;		EndWhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AX    - Address to check
;		DS    - Data segment value
;
;	Registers on Exit:
;
;		AX    - Matching breakpoint number (If carry clear)
;		FL    - Carry set if no breakpoint match
;			Zero set if breakpoint disabled (Breakpoint match only)
;
;******************************************************************************
		Even			; Force procedure to even address
Break_Check	Proc	Near		; Check breakpoint procedure
	Save	bx,cx,si		; Save the required registers
	mov	bx,ax			; Save the address value
	mov	cx,BREAKPOINTS		; Setup the breakpoint counter
	lea	si,ds:[Break_Table]	; Get pointer to breakpoint table
Check_Loop:
	mov	al,ds:[si.Debug_Type]	; Get the breakpoint type value
	test	al,TYPE_USED		; Check for this breakpoint used
	jz	Next_Check		; Jump if this breakpoint not used
	and	al,BREAK_MASK		; Mask off all but breakpoint type
	cmp	al,TYPE_HARD		; Check for a hardware breakpoint
	je	Next_Check		; Jump if this is a hardware breakpoint
	cmp	bx,ds:[si.Debug_Address]; Check for a breakpoint match
	jne	Next_Check		; Jump if no breakpoint match
	mov	ax,BREAKPOINTS		; Compute the matching
	sub	ax,cx			;		       breakpoint number
	test	ds:[si.Debug_Type],TYPE_ENABLED
	jmp	Short Check_Exit	; Go return control to the caller
Next_Check:
	add	si,Size Debug_Break	; Increment to the next breakpoint entry
	loop	Check_Loop		; Loop till all breakpoints setup
	stc				; Set carry indicating no match
Check_Exit:
	Restore bx,cx,si		; Restore the required registers
	ret				; Return control to the caller
Break_Check	Endp			; End of the Break_Check procedure
;******************************************************************************
;
;	Define all the required tables
;
;******************************************************************************
Setup_Table	Equ	This Word	; Start of the breakpoint setup table
	Dw	Setup_Soft		; Setup software type breakpoint
	Dw	Setup_Hard		; Setup hardware type breakpoint
	Dw	Setup_Exec		; Setup execute type breakpoint
Reset_Table	Equ	This Word	; Start of the breakpoint reset table
	Dw	Reset_Soft		; Reset software type breakpoint
	Dw	Reset_Hard		; Reset hardware type breakpoint
	Dw	Reset_Exec		; Reset execute type breakpoint
Set_Table	Equ	This Word	; Start of the set breakpoint table
	Dw	Set_Soft		; Set software type breakpoint
	Dw	Set_Hard		; Set hardware type breakpoint
	Dw	Set_Exec		; Set execute type breakpoint
List_Table	Equ	This Word	; Start of the list breakpoint table
	Dw	List_Soft		; List software type breakpoint
	Dw	List_Hard		; List hardware type breakpoint
	Dw	List_Execute		; List execute type breakpoint
Hard_Table	Equ	This Word	; Start of hardware type string table
	Dw	Illegal_String		; Illegal string type pointer
	Dw	Write_String		; Write string type pointer
	Dw	Read_String		; Read string type pointer
	Dw	Access_String		; Access string type pointer
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Break module
