/*
 *  Apple II emulator by Alexander Jean-Claude Bottema (C) 1994
 *
 *  $Id: keys.c,v 1.17 1998/08/25 03:00:01 chernabog Exp $
 *
 *  MODIFICATION HISTORY
 *   v0.3 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jan 1997.
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *   v0.5 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Feb 1998.
 *   v0.6 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Aug 1998.
 *     This code has nothing to do with my employer, GTE Internetworking,
 *     BBN Technologies.  It was written completely on my own time and on
 *     my own machine.
 *
 */


#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <linux/keyboard.h>

#include "keys.h"
#include "apple2.h"
#include "misc.h"
#include "video.h"
#include "interface.h"

/* from misc.c */
extern uid_t user, privileged;

#ifdef DEBUGGER
/* from debugger.c */
extern void c_do_debugging();
#endif


/* parameters for generic and keyboard-simulated joysticks */
short		joy_x = 127;
short		joy_y = 127;
unsigned char	joy_button0 = 0;
unsigned char	joy_button1 = 0;
unsigned char	joy_button2 = 0;

/* from joystick.c */
#ifdef PC_JOYSTICK
#include "linux/joystick.h"
extern int js_fd;
extern struct JS_DATA_TYPE js;
extern int js_center_x, js_center_y, js_offset_x, js_offset_y;
extern float js_adjustlow_x, js_adjustlow_y, js_adjusthigh_x, js_adjusthigh_y;
int x_val, y_val;
#endif

static int		menu_key = -1;
static int		flash_cnt = 0;
static char		caps_lock = 1;	/* is enabled */

/* ----------------------------------------------------
    Keymap. Mapping scancodes to Apple II+ US Keyboard
   ---------------------------------------------------- */
static int apple_ii_keymap_plain[128] =
    { -1 , 27 , '1', '2', '3', '4', '5', '6',	/* 00-07   */
      '7', '8', '9', '0', ':', '-',  8 , 27 ,   /* 08-15   */
      'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I',   /* 16-23   */
      'O', 'P', -1 , 8  , 13 , -1 , 'A', 'S',   /* 24-31   */
      'D', 'F', 'G', 'H', 'J', 'K', 'L', ';',   /* 32-39   */
      8  , -1 , -1 , 21 , 'Z', 'X', 'C', 'V',   /* 40-47   */
      'B', 'N', 'M', ',', '.', '/', -1 , -1 ,   /* 48-55   */
      -1 , ' ', -1 , kF1 , kF2 , kF3 , kF4 , kF5 ,   /* 56-63   */
      kF6 , kF7 , kF8 , kF9 , kF10, -1 , -1 , JUL,   /* 64-71   */
      J_U, JUR, S_D, J_L, J_C, J_R, S_I, JDL,   /* 72-79   */
      J_D, JDR, -1 , -1 , -1 , kF11, kF12, -1 ,   /* 80-87   */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 ,   /* 88-95   */
      -1 , -1 , -1 , RST, -1 , -1 , -1 , -1 ,   /* 96-103  */
      -1 , 8  , 21 , JB1, -1 , JB2, -1 , JB0,   /* 104-111 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , BOT,   /* 112-119 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 };  /* 120-127 */

static int apple_ii_keymap_ctrl[128] =
    { -1 , 027, '1', '2', '3', '4', '5', '6',	/* 00-07   */
      '7', '8', '9', '0', ':', '-',  8 , 27 ,   /* 08-15   */
      17 , 23 , 5  , 18 , 20 , 25 , 21 , 9  ,   /* 16-23   */
      15 , 16 , -1 , 8  , 13 , -1 , 1  , 19 ,   /* 24-31   */
      4  , 6  , 7  , 8  , 10 , 11 , 12 , ';',   /* 32-39   */
      8  , -1 , -1 , 21 , 26 , 24 , 3  , 22 ,   /* 40-47   */
      2  , 14 , 13 , ',', '.', '/', -1 , -1 ,   /* 48-55   */
      -1 , ' ', -1 , kF1 , kF2 , kF3 , kF4 , kF5 ,   /* 56-63   */
      kF6 , kF7 , kF8 , kF9 , kF10, -1 , -1 , JUL,   /* 64-71   */
      J_U, JUR, S_D, J_L, J_C, J_R, S_I, JDL,   /* 72-79   */
      J_D, JDR, -1 , -1 , -1 , kF11, kF12, -1 ,   /* 80-87   */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 ,   /* 88-95   */
      -1 , -1 , -1 , RST, -1 , -1 , -1 , -1 ,   /* 96-103  */
      -1 , 8  , 21 , JB1, -1 , JB2, -1 , JB0,   /* 104-111 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , BOT,   /* 112-119 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 };  /* 120-127 */

static int apple_ii_keymap_shifted[128] =
    { -1 , 27 , '!', '"', '#', '$', '%', '&',	/* 00-07   */
      39 , '(', ')', '0', '*', '=',  8 , 27 ,   /* 08-15   */
      'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I',   /* 16-23   */
      'O', '@', -1 , 8  , 13 , -1 , 'A', 'S',   /* 24-31   */
      'D', 'F', 'G', 'H', 'J', 'K', 'L', '+',   /* 32-39   */
      8  , -1 , -1 , 21 , 'Z', 'X', 'C', 'V',   /* 40-47   */
      'B', '^', 'M', '<', '>', '?', -1 , -1 ,   /* 48-55   */
      -1 , ' ', -1 , kF1 , kF2 , kF3 , kF4 , kF5 ,   /* 56-63   */
      kF6 , kF7 , kF8 , kF9 , kF10, -1 , -1 , JUL,   /* 64-71   */
      J_U, JUR, S_D, J_L, J_C, J_R, S_I, JDL,   /* 72-79   */
      J_D, JDR, -1 , -1 , -1 , kF11, kF12, -1 ,   /* 80-87   */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 ,   /* 88-95   */
      -1 , -1 , -1 , RST, -1 , -1 , -1 , -1 ,   /* 96-103  */
      -1 , 8  , 21 , JB1, -1 , JB2, -1 , JB0,   /* 104-111 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , BOT,   /* 112-119 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 };  /* 120-127 */

#ifdef APPLE_IIE
/* ----------------------------------------------------
    //e Keymap. Mapping scancodes to Apple //e US Keyboard
   ---------------------------------------------------- */
static int apple_iie_keymap_plain[128] =
    { -1 , 27 , '1', '2', '3', '4', '5', '6',	/* 00-07   */
      '7', '8', '9', '0', '-', '=',  8 ,  9 ,   /* 08-15   */
      'q', 'w', 'e', 'r', 't', 'y', 'u', 'i',   /* 16-23   */
      'o', 'p', '[', ']', 13 , -1 , 'a', 's',   /* 24-31   */
      'd', 'f', 'g', 'h', 'j', 'k', 'l', ';',   /* 32-39   */
     '\'', '`', -1 ,'\\', 'z', 'x', 'c', 'v',   /* 40-47   */
      'b', 'n', 'm', ',', '.', '/', -1 , -1 ,   /* 48-55   */
      -1 , ' ', -1 , kF1 , kF2 , kF3 , kF4 , kF5 ,   /* 56-63   */
      kF6 , kF7 , kF8 , kF9 , kF10, -1 , -1 , JUL,   /* 64-71   */
      J_U, JUR, S_D, J_L, J_C, J_R, S_I, JDL,   /* 72-79   */
      J_D, JDR, -1 , -1 , -1 , kF11, kF12, -1 ,   /* 80-87   */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 ,   /* 88-95   */
      -1 , -1 , -1 , RST, -1 , -1 , -1 , 11 ,   /* 96-103  */
      -1 , 8  , 21 , JB1, 10 , JB2, 127, JB0,   /* 104-111 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , BOT,   /* 112-119 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 };  /* 120-127 */

static int apple_iie_keymap_ctrl[128] =
    { -1 , 27 , '1', '2', '3', '4', '5', '6',	/* 00-07   */
      '7', '8', '9', '0', '-', '=',  8 ,  9 ,   /* 08-15   */
      17 , 23 , 5  , 18 , 20 , 25 , 21 ,  9 ,   /* 16-23   */
      15 , 16 , 27 , 29 , 13 , -1 , 1  , 19 ,   /* 24-31   */
      4  , 6  , 7  , 8  , 10 , 11 , 12 , ';',   /* 32-39   */
     '\'', '`', -1 ,'\\', 26 , 24 , 3  , 22 ,   /* 40-47   */
      2  , 14 , 13 , ',', '.', '/', -1 , -1 ,   /* 48-55   */
      -1 , ' ', -1 , kF1 , kF2 , kF3 , kF4 , kF5 ,   /* 56-63   */
      kF6 , kF7 , kF8 , kF9 , kF10, -1 , -1 , JUL,   /* 64-71   */
      J_U, JUR, S_D, J_L, J_C, J_R, S_I, JDL,   /* 72-79   */
      J_D, JDR, -1 , -1 , -1 , kF11, kF12, -1 ,   /* 80-87   */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 ,   /* 88-95   */
      -1 , -1 , -1 , RST, -1 , -1 , -1 , 11 ,   /* 96-103  */
      -1 , 8  , 21 , JB1, 10 , JB2, 127, JB0,   /* 104-111 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , BOT,   /* 112-119 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 };  /* 120-127 */

static int apple_iie_keymap_shifted[128] =
    { -1 , 27 , '!', '@', '#', '$', '%', '^',	/* 00-07   */
      '&', '*', '(', ')', '_', '+',  8 ,  9 ,   /* 08-15   */
      'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I',   /* 16-23   */
      'O', 'P', '{', '}', 13 , -1 , 'A', 'S',   /* 24-31   */
      'D', 'F', 'G', 'H', 'J', 'K', 'L', ':',   /* 32-39   */
      '"', '~', -1 , '|', 'Z', 'X', 'C', 'V',   /* 40-47   */
      'B', 'N', 'M', '<', '>', '?', -1 , -1 ,   /* 48-55   */
      -1 , ' ', -1 , kF1 , kF2 , kF3 , kF4 , kF5 ,   /* 56-63   */
      kF6 , kF7 , kF8 , kF9 , kF10, -1 , -1 , JUL,   /* 64-71   */
      J_U, JUR, S_D, J_L, J_C, J_R, S_I, JDL,   /* 72-79   */
      J_D, JDR, -1 , -1 , -1 , kF11, kF12, -1 ,   /* 80-87   */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 ,   /* 88-95   */
      -1 , -1 , -1 , RST, -1 , -1 , -1 , 11 ,   /* 96-103  */
      -1 , 8  , 21 , JB1, 10 , JB2, 127, JB0,   /* 104-111 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , BOT,   /* 112-119 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 };  /* 120-127 */

static int apple_iie_keymap_caps[128] =
    { -1 , 27 , '1', '2', '3', '4', '5', '6',	/* 00-07   */
      '7', '8', '9', '0', '-', '=',  8 ,  9 ,   /* 08-15   */
      'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I',   /* 16-23   */
      'O', 'P', '[', ']', 13 , -1 , 'A', 'S',   /* 24-31   */
      'D', 'F', 'G', 'H', 'J', 'K', 'L', ';',   /* 32-39   */
     '\'', '`', -1 ,'\\', 'Z', 'X', 'C', 'V',   /* 40-47   */
      'B', 'N', 'M', ',', '.', '/', -1 , -1 ,   /* 48-55   */
      -1 , ' ', -1 , kF1 , kF2 , kF3 , kF4 , kF5 ,   /* 56-63   */
      kF6 , kF7 , kF8 , kF9 , kF10, -1 , -1 , JUL,   /* 64-71   */
      J_U, JUR, S_D, J_L, J_C, J_R, S_I, JDL,   /* 72-79   */
      J_D, JDR, -1 , -1 , -1 , kF11, kF12, -1 ,   /* 80-87   */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 ,   /* 88-95   */
      -1 , -1 , -1 , RST, -1 , -1 , -1 , 11 ,   /* 96-103  */
      -1 , 8  , 21 , JB1, 10 , JB2, 127, JB0,   /* 104-111 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , BOT,   /* 112-119 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 };  /* 120-127 */

static int apple_iie_keymap_shift_ctrl[128] =
    { -1 , 27 , '1',  0 , '3', '4', '5', 30 ,	/* 00-07   */
      '7', '8', '9', '0', 31 , '=',  8 ,  9 ,   /* 08-15   */
      17 , 23 , 5  , 18 , 20 , 25 , 21 ,  9 ,   /* 16-23   */
      15 , 16 , 27 , 29 , 13 , -1 , 1  , 19 ,   /* 24-31   */
      4  , 6  , 7  , 8  , 10 , 11 , 12 , ';',   /* 32-39   */
     '\'', '`', 28 , -1 , 26 , 24 , 3  , 22 ,   /* 40-47   */
      2  , 14 , 13 , ',', '.', '/', -1 , -1 ,   /* 48-55   */
      -1 , ' ', -1 , kF1 , kF2 , kF3 , kF4 , kF5 ,   /* 56-63   */
      kF6 , kF7 , kF8 , kF9 , kF10, -1 , -1 , JUL,   /* 64-71   */
      J_U, JUR, S_D, J_L, J_C, J_R, S_I, JDL,   /* 72-79   */
      J_D, JDR, -1 , -1 , -1 , kF11, kF12, -1 ,   /* 80-87   */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 ,   /* 88-95   */
      -1 , -1 , -1 , RST, -1 , -1 , -1 , 11 ,   /* 96-103  */
      -1 , 8  , 21 , JB1, 10 , JB2, 127, JB0,   /* 104-111 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , BOT,   /* 112-119 */
      -1 , -1 , -1 , -1 , -1 , -1 , -1 , -1 };  /* 120-127 */
#endif

static unsigned short		max_speed = 0;
static char			key_pressed[ 256 ];


/* ----------------------------------------------------
    functions
   ---------------------------------------------------- */

#ifdef APPLE_IIE
/* HACK unused.  HMMM, what to do about BS versus DEL?... */
void c_set_rubout_key(int ascii) {
    apple_iie_keymap_plain[14] = ascii&0xff;
    apple_iie_keymap_ctrl[14] = ascii&0xff;
    apple_iie_keymap_shifted[14] = ascii&0xff;
    apple_iie_keymap_caps[14] = ascii&0xff;
    apple_iie_keymap_shift_ctrl[14] = ascii&0xff;
}
#endif


/* -------------------------------------------------------------------------
 * This routine is called periodically to update the state of the emulator.
 * 	-handles switching to menus
 * 	-polls PC Joystick
 * 	-update palette for flashing text
 * ------------------------------------------------------------------------- */
void c_periodic_update() {

#ifdef DEBUGGER
    /* debugger hooks - proper exception_flag is set if we've already
       got a kF7 */
    if ((menu_key >= 0) || (exception_flag & EnterDebugSig))
    {
	int key = (exception_flag & EnterDebugSig) ? kF7 : menu_key;
#else
    if (menu_key >= 0)
    {
	int key = menu_key;
#endif
	menu_key = -1;/* got it */
	switch (key) {
	case kF1:
	    c_keyboard_off();
	    c_interface_select_diskette( 0 );
	    c_keyboard_on();
	    break;
	case kF2:
	    c_keyboard_off();
	    c_interface_select_diskette( 1 );
	    c_keyboard_on();
	    break;
	case kF4:
	    c_keyboard_off();
	    while (c_mygetch(1) == -1) { }/*busy loop*/
	    c_keyboard_on();
	    break;
	case kF5:
	    c_keyboard_off();
	    c_interface_keyboard_layout();
	    c_keyboard_on();
	    break;
#ifdef DEBUGGER
	case kF7:
	    /* set exception_flag and return to complete execution of
               current 6502 instruction */
	    if (!(exception_flag & EnterDebugSig)) {
		exception_flag = EnterDebugSig;
		break;
	    }
	    exception_flag &= ~EnterDebugSig;
	    c_keyboard_off();
	    c_do_debugging();
	    c_keyboard_on();
	  break;
#endif
#if 0
	case kF8:
	    c_keyboard_off();
	    c_interface_words();
	    c_keyboard_on();
	    break;
#endif
	case kF9:
	    if (max_speed != 0)
	        apple_speed = max_speed, max_speed = 0;
	    else
	        max_speed = apple_speed, apple_speed = 1;
	    break;
	case kF10:
	    if (max_speed != 0)
		apple_speed = max_speed, max_speed = 0;
	    c_keyboard_off();
	    c_interface_parameters();
	    c_keyboard_on();
	    break;
	}
    }

    /* cursor */
    flash_cnt += (int)TIMER_DELAY;
    if (flash_cnt > 333333) {
	flash_cnt = 0;
	c_flash_cursor(flash_cnt);
    } else if (flash_cnt == (200000 / TIMER_DELAY) * TIMER_DELAY) {
	c_flash_cursor(flash_cnt);
    }

    /* simulated joystick */
    if (joy_mode == JOY_KYBD) {
	if (key_pressed[ SCODE_J_U ])
	{
		if (joy_y > joy_step)
		    joy_y -= joy_step;
		else
		    joy_y = 0;
	}

	if (key_pressed[ SCODE_J_D ])
	{
		if (joy_y < joy_range - joy_step)
		    joy_y += joy_step;
		else
		    joy_y = joy_range-1;
	}

	if (key_pressed[ SCODE_J_L ])
	{
		if (joy_x > joy_step)
		    joy_x -= joy_step;
		else
		    joy_x = 0;
	}

	if (key_pressed[ SCODE_J_R ])
	{
		if (joy_x < joy_range - joy_step)
		    joy_x += joy_step;
		else
		    joy_x = joy_range-1;
	}
    }
#ifdef PC_JOYSTICK
    else if ((joy_mode == JOY_PCJOY) && !(js_fd < 0)) {
	read(js_fd, &js, JS_RETURN);

	x_val = (js.x < js_center_x)
	    ? (js.x - js_offset_x) * js_adjustlow_x
	    : (js.x - js_center_x) * js_adjusthigh_x + half_joy_range;

	y_val = (js.y < js_center_y)
	    ? (js.y - js_offset_y) * js_adjustlow_y
	    : (js.y - js_center_y) * js_adjusthigh_y + half_joy_range;

	joy_y = (y_val > 0xff) ? 0xff : (y_val < 0) ? 0 : y_val;
	joy_x = (x_val > 0xff) ? 0xff : (x_val < 0) ? 0 : x_val;

/* 	almost_x = (x_val > 0xff) ? 0xff : (x_val < 0) ? 0 : x_val; */
/* 	adj_x = (3-(joy_y/0x40)) + 10; */
/* 	turnpt_x = joy_y + adj_x; */
/* 	almost_x = (almost_x < turnpt_x) */
/* 	    ? almost_x */
/* 	    : (almost_x - turnpt_x) * adjusthigh_x; */

/* 	joy_x = (almost_x > 0xff) ? 0xff : (almost_x < 0) ? 0 : almost_x; */

	/* sample buttons only if apple keys aren't pressed. keys get set to
	 * 0xff, and js buttons are set to 0x80. */
	if (!(joy_button0 & 0x7f))
	    joy_button0 = (js.buttons & 0x01) ? 0x80 : 0x0;
	if (!(joy_button1 & 0x7f))
	    joy_button1 = (js.buttons & 0x02) ? 0x80 : 0x0;
	if (!(joy_button2 & 0x7f))
	    joy_button2 = (js.buttons & 0x03) ? 0x80 : 0x0;
    }
#endif
    else if (joy_mode == JOY_OFF)
        joy_x = joy_y = 256;
}

/* -------------------------------------------------------------------------
    void c_read_raw_key() : handle a scancode
   ------------------------------------------------------------------------- */
void c_read_raw_key(int scancode, int pressed) {
    int *keymap = NULL;

    /* determine which key mapping to use */
#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	/* set/reset caps lock */
	if (key_pressed[ SCODE_CAPS ])
	    caps_lock = !caps_lock;

	if ((key_pressed[ SCODE_L_SHIFT ] ||		/* shift-ctrl */
	     key_pressed[ SCODE_R_SHIFT ]) &&
	    (key_pressed[ SCODE_L_CTRL ] ||
	     key_pressed[ SCODE_R_CTRL ]))
	    keymap = apple_iie_keymap_shift_ctrl;
	else if (key_pressed[ SCODE_L_CTRL ] ||		/* ctrl */
		 key_pressed[ SCODE_R_CTRL ])
	    keymap = apple_iie_keymap_ctrl;
	else if (key_pressed[ SCODE_L_SHIFT ] ||	/* shift */
		 key_pressed[ SCODE_R_SHIFT ])
	    keymap = apple_iie_keymap_shifted;
	else if (caps_lock)				/* caps lock */
	    keymap = apple_iie_keymap_caps;
	else						/* plain */
	    keymap = apple_iie_keymap_plain;
    } else
#endif
    {
	if (key_pressed[ SCODE_L_CTRL ] ||
	    key_pressed[ SCODE_R_CTRL ])
	    keymap = apple_ii_keymap_ctrl;
	else if (key_pressed[ SCODE_L_SHIFT ] ||
		 key_pressed[ SCODE_R_SHIFT ])
	    keymap = apple_ii_keymap_shifted;
	else
	    keymap = apple_ii_keymap_plain;
    }

    /* key is pressed */
    if (pressed) {

	key_pressed[ scancode ] = 1;
	if ((unsigned int)(keymap[ scancode ]) < 128) {
	    apple_ii_64k[0][0xC000] = keymap[scancode] | 0x80;
	    apple_ii_64k[1][0xC000] = keymap[scancode] | 0x80;
	    return;
	}

	switch (keymap[ scancode ]) {
	case RST:
	    exception_flag = ResetSig;
	    break;
	case BOT:
	    exception_flag = RebootSig;
	    break;
	case J_C:
	    joy_x = joy_center_x;
	    joy_y = joy_center_y;
	    break;
	case JB0:
	    joy_button0 = 0xff;/* open apple */
	    break;
	case JB1:
	    joy_button1 = 0xff;/* closed apple */
	    break;
	case JB2:
	    joy_button2 = 0xff;/* unused? */
	    break;
	default:
	    break;
	}

	/* menu keys */
	if ((keymap[ scancode ] >= kF1) &&
	    (keymap[ scancode ] <= kF10))
	{
	    menu_key = keymap[ scancode ];
	}
    }

    /* key is released */
    else {
        key_pressed[ scancode ] = 0;
	switch (keymap[ scancode ]) {
	case JB0:
	    joy_button0 = 0x00;
	    break;
	case JB1:
	    joy_button1 = 0x00;
	    break;
	case JB2:
	    joy_button2 = 0x00;
	    break;
	}
    }
}
