/*
 *  Apple II emulator originally by Alexander Jean-Claude Bottema (C) 1994
 *
 *  $Id: defs.h,v 1.3 1998/08/23 17:12:52 chernabog Exp $
 *
 *  This file was renamed from defs.H to appease the emacs font/indention
 *  gods.
 *
 * MODIFICATION HISTORY
 *   v0.3 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jan 1997.
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *   v0.5 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Feb 1998.
 *   v0.6 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Aug 1998.
 *     This code has nothing to do with my employer, GTE Internetworking,
 *     BBN Technologies.  It was written completely on my own time and on
 *     my own machine.
 *
 *  Global defines/macros for assembly code.
 *  	- generic macros
 *  	- instruction help macros
 *  	- graphics macros
 */

/* -------------------------------------------------------------------------
    CPU (6502) Helper Routines
   ------------------------------------------------------------------------- */

#define GetFromPC_B	xorl	%eax, %eax;			\
			movw	PC_Reg, EffectiveAddr;		\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			incw	PC_Reg;

#define GetFromPC_W	xorl	%eax, %eax;			\
			movw	PC_Reg, EffectiveAddr;		\
			incw	EffectiveAddr;			\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			xchgb	%al, %ah;			\
			decw	EffectiveAddr;			\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			addw	$2, PC_Reg;

#define GetFromEA_B	call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);

#define GetFromMem_B(x)						\
			movl	x, EffectiveAddr_E;		\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);

#define GetFromMem_W(x)						\
			movl	x, EffectiveAddr_E;		\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			xchgb	%al, %ah;			\
			incw	EffectiveAddr;			\
			call	*SYMBOL_NAME(table_read_memory)	\
				(,EffectiveAddr_E,4);		\
			xchgb	%al, %ah;


#define Wait		movw	SYMBOL_NAME(apple_speed), %ax;	\
		   0:						\
			decw	%ax;				\
			jnz	0b;

#ifdef DEBUGGER
#define SaveState	movl	PC_Reg, SYMBOL_NAME(debug_PC);		\
			movl	EffectiveAddr, SYMBOL_NAME(debug_EA);	\
			movl	SP_Reg, SYMBOL_NAME(debug_SP);		\
			movb	X_Reg, SYMBOL_NAME(debug_X);		\
			movb	Y_Reg, SYMBOL_NAME(debug_Y);		\
			movb	A_Reg, SYMBOL_NAME(debug_A);		\
			movb	F_Reg, SYMBOL_NAME(debug_F);

#define ReplaceState	movl	SYMBOL_NAME(debug_PC), PC_Reg;		\
			movl	SYMBOL_NAME(debug_EA), EffectiveAddr;	\
			movl	SYMBOL_NAME(debug_SP), SP_Reg;		\
			movb	SYMBOL_NAME(debug_X), X_Reg;		\
			movb	SYMBOL_NAME(debug_Y), Y_Reg;		\
			movb	SYMBOL_NAME(debug_A), A_Reg;		\
			movb	SYMBOL_NAME(debug_F), F_Reg;

/* DEBUGGER version of Continue.  (we have other exception_flag values) */
#define Continue	Wait						   \
			testb   $0xFF, SYMBOL_NAME(exception_flag);	   \
			jnz     1f;					   \
			GetFromPC_B					   \
			jmp	*table_opcodes(,%eax,4);		   \
									   \
/* exception */	1:	cmpb	RebootSig, SYMBOL_NAME(exception_flag);	   \
			jz	4f;/*reboot*/				   \
			cmpb	ResetSig, SYMBOL_NAME(exception_flag);	   \
			jz	5f;/*reset*/				   \
			cmpb	DebugStepSig, SYMBOL_NAME(exception_flag); \
			jz	2f;/*debug step*/			   \
			jmp	3f;/*enter debugger*/			   \
									   \
/* step */	2:	SaveState;					   \
			popal;						   \
			ret;						   \
									   \
/* enter */	3:	SaveState;					   \
			pushal;						   \
/* do debugging */	call	SYMBOL_NAME(c_periodic_update);		   \
			popal;						   \
			ReplaceState					   \
			GetFromPC_B					   \
			jmp	*table_opcodes(,%eax,4);		   \
									   \
/* reboot */	4:	movb	$0, SYMBOL_NAME(exception_flag);	   \
			ret;						   \
/* reset */	5:	movb	$0, SYMBOL_NAME(exception_flag);	   \
			movw	$0xfffc, EffectiveAddr;			   \
			call	read_memory_word;			   \
			movw    %ax, PC_Reg;				   \
			GetFromPC_B					   \
			jmp	*table_opcodes(,%eax,4);
#else /* ! DEBUGGER */
#define Continue	Wait						\
			testb   $0xFF, SYMBOL_NAME(exception_flag);	\
			jnz     1f;					\
			GetFromPC_B					\
			jmp	*table_opcodes(,%eax,4);		\
/* exception */	1:	cmpb	RebootSig, SYMBOL_NAME(exception_flag);	\
			jz	2f;/*reboot*/				\
			movb	$0, SYMBOL_NAME(exception_flag);	\
			movw	$0xfffc, EffectiveAddr;			\
			call	read_memory_word;			\
			movw    %ax, PC_Reg;				\
			GetFromPC_B					\
			jmp	*table_opcodes(,%eax,4);		\
/* reboot */	2:	movb	$0, SYMBOL_NAME(exception_flag);	\
			ret;
#endif /* DEBUGGER */


#define FlagC		lahf;				\
  			andb	C_Flag, %ah;		\
  			andb	C_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagZ		lahf;				\
  			andb	Z_Flag, %ah;		\
  			andb	Z_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagN		lahf;				\
  			andb	N_Flag, %ah;		\
  			andb	N_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagNV		lahf;				\
  			andb	NV_Flag, %ah;		\
  			andb	NV_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagNZ		lahf;				\
  			andb	NZ_Flag, %ah;		\
  			andb	NZ_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagNZC		lahf;				\
			andb	NZC_Flag, %ah;		\
			andb	NZC_Flag_Not, F_Reg;	\
			orb	%ah, F_Reg;

#define FlagNVZC	lahf;				\
			seto	%al;			\
			shlb    $3, %al;		\
			andb	NZC_Flag, %ah;		\
			andb	NVZC_Flag_Not, F_Reg;	\
			orb	%al, %ah;		\
			orb	%ah, F_Reg;

#define Push(x)		movb	x, SYMBOL_NAME(apple_ii_64k)(,SP_Reg,1); \
			decb	SP_Reg_L;

#define Pop(x)		incb	SP_Reg_L;				 \
			movb	SYMBOL_NAME(apple_ii_64k)(,SP_Reg,1), x;

/* Immediate Addressing - the operand is contained in the second byte of the
   instruction. */
#define GetImm		movw	PC_Reg, EffectiveAddr;	\
			incw	PC_Reg;

/* Absolute Addressing - the second byte of the instruction is the low
   order address, and the third byte is the high order byte. */
#define GetAbs		GetFromPC_W;			\
			movw	%ax, EffectiveAddr;

/* Zero Page Addressing - the second byte of the instruction is an
   address on the zero page */
#define GetZPage	GetFromPC_B;			\
			movw	%ax, EffectiveAddr;

/* Zero Page Indexed Addressing - The effective address is calculated by
   adding the second byte to the contents of the index register.  Due
   to the zero page addressing nature of this mode, no carry is added
   to the high address byte, and the crossing of page boundaries does
   not occur. */
#define GetZPage_X	GetFromPC_B;			\
			addb	X_Reg, %al;		\
			movw	%ax, EffectiveAddr;

#define GetZPage_Y	GetFromPC_B;			\
			addb	Y_Reg, %al;		\
			movw	%ax, EffectiveAddr;

/* Absolute Indexed Addressing - The effective address is formed by
   adding the contents of X or Y to the address contained in the
   second and third bytes of the instruction. */
#define GetAbs_X	GetFromPC_W;			\
			addb	X_Reg, %al;		\
			adcb	$0, %ah;		\
			movw	%ax, EffectiveAddr;

#define GetAbs_Y	GetFromPC_W;			\
			addb	Y_Reg, %al;		\
			adcb	$0, %ah;		\
			movw	%ax, EffectiveAddr;

/* Absolute Indirect Addressing - The second and third bytes of the
   instruction are the low and high bytes of an address, respectively.
   The contents of the fully specified memory location is the
   low-order byte of the effective address.  The next memory location
   contains the high order byte of the effective address. */
/*HACK - used?*/
#define GetInd		GetFromPC_W;		\
			GetFromMem_W(%eax)

/* Zero Page Indirect Addressing (65c02) - The second byte of the
   instruction points to a memory location on page zero containing the
   low order byte of the effective address.  The next location on page
   zero contains the high order byte of the address. */
#define GetIndZPage	GetFromPC_B;			\
			movw	%ax, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			incw	EffectiveAddr;		\
			andw	$0xFF, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			movw	%ax, EffectiveAddr;

/* Zero Page Indexed Indirect Addressing - The second byte is added to
   the contents of the X index register; the carry is discarded.  The
   result of this addition points to a memory location on page zero
   whose contents is the low order byte of the effective address.  The
   next memory location in page zero contains the high-order byte of
   the effective address.  Both memory locations specifying the high
   and low-order bytes must be in page zero. */
#define GetIndZPage_X	GetFromPC_B;			\
			addb	X_Reg, %al;		\
			movw	%ax, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			incw	EffectiveAddr;		\
			andw	$0xFF, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			movw	%ax, EffectiveAddr;

/* Indirect Indexed Addressing - The second byte of the instruction
   points to a memory location in page zero.  The contents of this
   memory location are added to the contents of the Y index register,
   the result being the low order byte of the effective address.  The
   carry from this addition is added to the contents of the next page
   zero memory location, the result being the high order byte of the
   effective address. */
#define GetIndZPage_Y   GetFromPC_B;			\
			movw	%ax, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			incw	EffectiveAddr;		\
			andw	$0xFF, EffectiveAddr;	\
			GetFromEA_B;			\
			xchgb	%al, %ah;		\
			addb	Y_Reg, %al;		\
			adcb	$0, %ah;		\
			movw	%ax, EffectiveAddr;

#define DoADC_b		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			adcb	%al, A_Reg;		\
			FlagNVZC

#define	DoADC_d		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			adcb	A_Reg, %al;		\
			daa;				\
			movb	%al, A_Reg;		\
			FlagNVZC

#define DoAND		call	read_memory;	\
			andb	%al, A_Reg;	\
			FlagNZ

#define DoASL		call	read_memory;	\
			shlb	$1, %al;	\
			FlagNZC			\
			call	write_memory;

#define DoBIT		andb	NVZ_Flag_Not, F_Reg;	\
			call	read_memory;		\
			testb	%al, A_Reg;		\
			FlagZ				\
			movb    %al, %ah;		\
			andb	$0x40, %al;		\
			shrb	$3, %al;		\
			orb	%al, F_Reg;		\
			andb    $0x80, %ah;		\
			orb	%ah, F_Reg;

#define DoCMP		call	read_memory;	\
			cmpb	%al, A_Reg;	\
			cmc;			\
			FlagNZC

#define DoCPX		call	read_memory;	\
			cmpb	%al, X_Reg;	\
			cmc;			\
			FlagNZC

#define DoCPY		call	read_memory;	\
			cmpb	%al, Y_Reg;	\
			cmc;			\
			FlagNZC

#define DoDEC		call	read_memory;	\
			decb	%al;		\
			FlagNZ			\
			call	write_memory;

#define DoEOR		call	read_memory;	\
			xorb	%al, A_Reg;	\
			FlagNZ

#define DoINC		call	read_memory;	\
			incb	%al;		\
			FlagNZ			\
			call	write_memory;

#define DoJMP		movw	EffectiveAddr, PC_Reg;

#define DoJSR		movw	PC_Reg, %ax;		\
			decw	%ax;			\
			Push(%ah)			\
			Push(%al)			\
			movw	EffectiveAddr, PC_Reg;

#define DoLDA		call	read_memory;	\
			movb	%al, A_Reg;	\
			orb	%al, %al;	\
			FlagNZ

#define DoLDX		call	read_memory;	\
			movb	%al, X_Reg;	\
			orb	%al, %al;	\
			FlagNZ

#define DoLDY		call	read_memory;	\
			movb	%al, Y_Reg;	\
			orb	%al, %al;	\
			FlagNZ

#define DoLSR		call	read_memory;	\
			shrb	$1, %al;	\
			FlagNZC			\
			call	write_memory;

#define DoORA		call	read_memory;	\
			orb	%al, A_Reg;	\
			FlagNZ

#define DoROL(L,CONT)	call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rclb	$1, %al;		\
			jc	L;			\
			orb	%al, %al;		\
			clc;				\
			FlagNZC				\
			call	write_memory;		\
			CONT				\
	  L##:		orb	%al, %al;		\
			stc;				\
			FlagNZC				\
			call	write_memory;		\
			CONT

#define DoROR(L,CONT)	call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rcrb	$1, %al;		\
			jc	L;			\
			orb	%al, %al;		\
			clc;				\
			FlagNZC				\
			call	write_memory;		\
			CONT				\
	  L##:		orb	%al, %al;		\
			stc;				\
			FlagNZC				\
			call	write_memory;		\
			CONT

#define DoSBC_b		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			sbbb	%al, A_Reg;		\
			cmc;				\
			FlagNVZC

#define DoSBC_d		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			xchgb	A_Reg, %al;		\
			sbbb	A_Reg, %al;		\
			das;				\
			movb	%al, A_Reg;		\
			cmc;				\
			FlagNVZC

#define DoSTA		movb	A_Reg, %al;	\
			call	write_memory;

#define DoSTX		movb	X_Reg, %al;	\
			call	write_memory;

#define DoSTY		movb	Y_Reg, %al;	\
			call	write_memory;

/* -------------------------------------------------------------------------
    65c02 instructions
   ------------------------------------------------------------------------- */

#define DoSTZ		movb	$0x0, %al;	\
			call	write_memory;

#define DoTRB		call	read_memory;	\
			andb	$0x3f, %al;	\
			FlagNV			\
			orb	%ah, %al;	\
			notb	A_Reg;		\
			andb	A_Reg, %al;	\
			FlagZ			\
			call	write_memory;	\
			notb	A_Reg;

#define DoTSB		call	read_memory;	\
			andb	$0x3f, %al;	\
			FlagNV			\
			orb	%ah, %al;	\
			orb	A_Reg, %al;	\
			FlagZ			\
			call	write_memory;

/* -------------------------------------------------------------------------
    Undocumented 6502 (Illegal instructions)
   ------------------------------------------------------------------------- */

        /* AAX = A AND X -> M */
#define DoAAX		movb	A_Reg, %al;	\
			andb	X_Reg, %al;	\
			FlagNZ			\
			call	write_memory;

	/* AMA = ORA 238, AND M, TAX */
#define DoAMA		orb	$238, A_Reg;	\
			call	read_memory;	\
			andb	%al, A_Reg;	\
			movb	A_Reg, X_Reg;	\
			FlagNZ

	/* ANA = AND M, Carry = BIT 7 */
#define DoANA(CONT)	call	read_memory;	\
			andb	%al, A_Reg;	\
			js	1f;		\
			clc;			\
			FlagNZC			\
			CONT			\
		1:	stc;			\
			FlagNZC			\
			CONT

	/* ANB = same as ANA */
#define DoANB(CONT)	call	read_memory;	\
			andb	%al, A_Reg;	\
			js	1f;		\
			clc;			\
			FlagNZC			\
			CONT			\
		1:	stc;			\
			FlagNZC			\
			CONT

	/* AXM = (A AND X) - M -> X */
#define DoAXM		call	read_memory;		\
			andb	A_Reg, X_Reg;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			sbbb	%al, X_Reg;		\
			cmc;				\
			FlagNVZC

	/* AXS = (A AND X) -> S, A AND X AND 17 -> M */
			/* HACK!!!!!!!!!!!!!!! */
#define DoAXS		movb	A_Reg, SP_Reg_L;	\
			andb	X_Reg, SP_Reg_L;	\
			movb	SP_Reg_L, %al;		\
			andb	$17, %al;		\
			FlagNZ	/* \ wasn't here */	\
			call	write_memory; 

	/* DCP = DEC M, CMP M */
#define	DoDCP		call	read_memory;	\
			decb	%al;		\
			cmpb	%al, A_Reg;	\
			cmc;			\
			FlagNZC			\
			call	write_memory;

	/* ISB = INC M, SBC M */
#define DoISB_b		call	read_memory;		\
			incb	%al;			\
			call	write_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			sbbb	%al, A_Reg;		\
			cmc;				\
			FlagNVZC

#define DoISB_d		call	read_memory;		\
			incb	%al;			\
			call	write_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			cmc;				\
			xchgb	A_Reg, %al;		\
			sbbb	A_Reg, %al;		\
			das;				\
			movb	%al, A_Reg;		\
			cmc;				\
			FlagNVZC

	/* LAN = ROL M, AND M */
#define DoLAN(CONT)	call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rclb	$1, %al;		\
			jc	1f;			\
			call	write_memory;		\
			andb	%al, A_Reg;		\
			clc;				\
			FlagNZC				\
			CONT				\
		   1:	call	write_memory;		\
			andb	%al, A_Reg;		\
			stc;				\
			FlagNZC				\
			CONT

	/* LAS = LDA M, TAX, TXS */
#define DoLAS		call	read_memory;	\
			movb	%al, A_Reg;	\
			movb	%al, X_Reg;	\
			movb	%al, SP_Reg_L;	\
			orb	%al, %al;	\
			FlagNZ

	/* LAX = LDA M, TAX */
#define DoLAX		call	read_memory;	\
			movb	%al, A_Reg;	\
			movb	%al, X_Reg;	\
			orb	%al, %al;	\
			FlagNZ

	/* LOR = ASL M, ORA M */
#define DoLOR		call	read_memory;	\
			shlb	$1, %al;	\
			FlagC			\
			call	write_memory;	\
			orb	%al, A_Reg;	\
			FlagNZ

	/* RAD = ROR M, ADC M */
#define DoRAD_b		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rcrb	$1, %al;		\
			adcb	%al, A_Reg;		\
			pushl	%eax;			\
			FlagNVZC			\
			popl	%eax;			\
			call	write_memory;

#define DoRAD_d		call	read_memory;		\
			bt	C_Flag_Bit, FF_Reg;	\
			rcrb	$1, %al;		\
			pushfl;				\
			call	write_memory;		\
			popfl;				\
			adcb	A_Reg, %al;		\
			daa;				\
			movb	%al, A_Reg;		\
			FlagNVZC

	/* RAM = AND M, LSR A */
#define DoRAM		call	read_memory;	\
			andb	%al, A_Reg;	\
			shrb	$1, A_Reg;	\
			FlagNZC

	/* RBM = same as RAM */
#define DoRBM		DoRAM

	/* REO = LSR M, EOR M */
#define DoREO		call	read_memory;	\
			shrb	$1, %al;	\
			xorb	%al, A_Reg;	\
			FlagNZC			\
			call	write_memory;

	/* DoZBC = same as SBC */
#define DoZBC_b		DoSBC_b
#define DoZBC_d		DoSBC_d

	/* TEA = (A AND X AND (OP+2)+1) -> M */
#define	DoTEA		pushl	EffectiveAddr_E;	\
			movw	PC_Reg, EffectiveAddr;	\
			decw	EffectiveAddr;		\
			call	read_memory;		\
			popl	EffectiveAddr_E;	\
			incb	%al;			\
			andb	A_Reg, %al;		\
			andb	X_Reg, %al;		\
			FlagNZ				\
			call	write_memory;

	/* TEX = (X AND (OP+2)+1) -> M */
#define DoTEX		pushl	EffectiveAddr_E;	\
			movw	PC_Reg, EffectiveAddr;	\
			decw	EffectiveAddr;		\
			call	read_memory;		\
			popl	EffectiveAddr_E;	\
			incb	%al;			\
			andb	X_Reg, %al;		\
			FlagNZ				\
			call	write_memory;

	/* TEY = (Y AND 1) -> M */
#define DoTEY		movb	Y_Reg, %al;	\
			andb	$1, %al;	\
			FlagNZ			\
			call	write_memory;

	/* XMA = (X AND M) AND (A OR 238) -> A */
			/* HACK!!!!!!!!!!!!!!! */
#define DoXMA	/* the \ wasn't here before */	\
			call	read_memory;	\
			andb	X_Reg, %al;	\
			orb	$238, A_Reg;	\
			andb	%al, A_Reg;	\
			FlagNZ


/* -------------------------------------------------------------------------
    Graphics routines.
    Care has been taken to isolate the dimension-dependent (320x200 vs 640x400)
    routines.
   ------------------------------------------------------------------------- */


/* -------------------------------------------------------------------------
 * Plot exatly 7 pixels from FROM to TO.
 * ecx: scratch
 * ------------------------------------------------------------------------- */
#define Plot7Pixels(FROM,TO)\
		movl	(FROM), %ecx;			/* long -> GM */      \
		movl	%ecx, (TO);				      \
		addl	$4, FROM;			/* inc pointers */    \
		addl	$4, TO;					      \
		movw	(FROM), %cx;			/* word -> GM */      \
		movw	%cx, (TO);				      \
		addl	$2, FROM;			/* inc pointers */    \
		addl	$2, TO;					      \
		movb	(FROM), %cl;			/* byte -> GM */      \
		movb	%cl, (TO);


#ifdef _640x400

#define LoadHiresTableRef(TABLE)\
		leal	SYMBOL_NAME(expanded_col_hires_##TABLE), %ebx;\
		shll	$4, %eax;/* *16 */\
		addl	%eax, %ebx;

/* -------------------------------------------------------------------------
 * Plot a normal swath of pixels.
 * For 640x400 this is 2 rows of 14 pixels.
 * ebx: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotPixels\
		PlotPixels640\
		subl	$12, %ebx;\
		addl	$SCANWIDTH-12, %eax;\
		PlotPixels640\
		subl	$SCANWIDTH, %eax;\
		addl	$1, %eax;
#define PlotPixels640\
		movl	(%ebx), %ecx;			/* long -> GM */      \
		movl	%ecx, (%eax);				      \
		addl	$4, %eax;			/* inc pointers */    \
		addl	$4, %ebx;					      \
		movl	(%ebx), %ecx;			/* long -> GM */      \
		movl	%ecx, (%eax);				      \
		addl	$4, %eax;			/* inc pointers */    \
		addl	$4, %ebx;					      \
		movl	(%ebx), %ecx;			/* long -> GM */      \
		movl	%ecx, (%eax);				      \
		addl	$4, %eax;			/* inc pointers */    \
		addl	$4, %ebx;					      \
		movw	(%ebx), %cx;			/* word -> GM */      \
		movw	%cx, (%eax);

/* -------------------------------------------------------------------------
 * Plot a dynamically interpolated swath of pixels.
 * For 640x400 this is 2 rows of 18 pixels.
 * ebx: from table
 * eax: to graphics memory
 * ecx: scratch
 * ------------------------------------------------------------------------- */
#define PlotPixelsExtra\
		subl	$2, %eax;\
		pushl	%edx;\
		xorl	%edx, %edx;\
		PlotPixelsExtraRow;\
		addl	$SCANWIDTH-18, %eax;\
		subl	$9, %ebx;\
		PlotPixelsExtraRow;\
		popl	%edx;
#define PlotPixelsExtraRow\
		movb	$9, %dl;\
	1:	movb	(%ebx), %cl;\
		movb	%cl, %ch;\
		movw	%cx, (%eax);\
		incl	%ebx;\
		addl	$2, %eax;\
		decb	%dl;\
		jnz	1b;


/* -------------------------------------------------------------------------
 * Plot an 80 column character row.  We can do this only in 640x400 resolution.
 * For 640x400 this is 2 rows of 7 pixels.
 * esi: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotCharacter80Row\
		Plot7Pixels(%esi,%eax);\
		addl	$ SCANWIDTH-6, %eax;/* Go to next row */\
		subl	$6, %esi;\
		Plot7Pixels(%esi,%eax);\

/* -------------------------------------------------------------------------
 * Plot a 40 column character row.
 * For 640x400 this is 2 rows of 14 pixels.
 * esi: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotCharacter40Row\
		PlotCharacter40Row640\
		subl	$12, %esi;\
		addl	$ SCANWIDTH-12, %eax;\
		PlotCharacter40Row640\
		addl	$2, %esi;
#define PlotCharacter40Row640\
		movl	(%esi), %ecx;	\
		movl	%ecx, (%eax);	\
		addl	$4, %esi;	\
		addl	$4, %eax;	\
		movl	(%esi), %ecx;	\
		movl	%ecx, (%eax);	\
		addl	$4, %esi;	\
		addl	$4, %eax;	\
		movl	(%esi), %ecx;	\
		movl	%ecx, (%eax);	\
		addl	$4, %esi;	\
		addl	$4, %eax;	\
		movw	(%esi), %cx;	\
		movw	%cx, (%eax);

/* -------------------------------------------------------------------------
 * Plot a 40 column row of lores graphics.
 * For 640x400 this is 2 rows of 14 pixels.
 * esi: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotBlockRow				\
		PlotBlockRow640			\
		addl	$ SCANWIDTH-12, %eax;	\
		PlotBlockRow640
#define PlotBlockRow640\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movw	%dx, (%eax);

/* -------------------------------------------------------------------------
 * Get the adjancent color bytes in memory.
 * For 640x400 mode, we need to remember to move around by a factor of 2.
 * ebx: graphics memory index
 * eax: temp buffer for comparison
 * ------------------------------------------------------------------------- */
#define GrabAdjGMBytes\
		subl	$3, %ebx;\
		movw	(%ebx), %cx;\
		movw	%cx, (%eax);			/* GM -> temp */\
		addl	$9, %eax;\
		addl	$18, %ebx;\
		movw	(%ebx), %cx;\
		movw	%cx, (%eax);			/* GM -> temp */\
		decl	%eax;

/* -------------------------------------------------------------------------
 * Plots a normalized byte of dhires color directly into graphics memory.
 * eax: graphics memory index
 * ebx: dhires_colors index
 * edx: scratch
 * ------------------------------------------------------------------------- */
#define PlotDHiresByte\
		movb	SYMBOL_NAME(dhires_colors2)(,%ebx,1), %dl;\
		movb	%dl, %dh;\
		shll	$16, %edx;\
		movb	SYMBOL_NAME(dhires_colors1)(,%ebx,1), %dl;\
		movb	%dl, %dh;\
		movl	%edx, (%eax);\
		movl	%edx, SCANWIDTH(%eax);\
		addl	$4, %eax;

#define PlotDHiresFirstByte\
		subl	$4, %eax;\
		PlotDHiresByte

#else /* if ! _640x400 */

#define	expanded_col8_hires_even expanded_col_hires_even
#define expanded_col8_hires_odd  expanded_col_hires_odd

#define LoadHiresTableRef(TABLE)\
		leal	SYMBOL_NAME(expanded_col_hires_##TABLE)(,%eax,8), %ebx;


/* -------------------------------------------------------------------------
 * Plot a normal swath of pixels.
 * For 320x200 this is exactly 7 pixels.
 * ebx: from table
 * eax: to graphics memory
 * ------------------------------------------------------------------------- */
#define PlotPixels\
		Plot7Pixels(%ebx,%eax)
#define PlotCharacter40Row		\
		Plot7Pixels(%esi,%eax)
#define PlotBlockRow				\
		movl	%edx, (%eax);		\
		addl	$4, %eax;		\
		movw	%dx, (%eax);		\
		addl	$2, %eax;		\
		movb	%dl, (%eax);

/* -------------------------------------------------------------------------
 * Plot a dynamically interpolated swath of pixels
 * For 320x200 this is exactly 9 pixels.
 * ebx: from table
 * eax: to graphics memory
 * ecx: scratch
 * ------------------------------------------------------------------------- */
#define PlotPixelsExtra\
		decl	%eax;\
		movl	(%ebx), %ecx;\
		movl	%ecx, (%eax);\
		addl	$4, %eax;\
		addl	$4, %ebx;\
		movl	(%ebx), %ecx;\
		movl	%ecx, (%eax);\
		addl	$4, %eax;\
		addl	$4, %ebx;\
		movb	(%ebx), %cl;\
		movb	%cl, (%eax);

/* -------------------------------------------------------------------------
 * Get the adjancent color bytes in memory.
 * ebx: graphics memory index
 * eax: temp buffer for comparison
 * ------------------------------------------------------------------------- */
#define GrabAdjGMBytes\
		subl	$2, %ebx;\
		movw	(%ebx), %cx;\
		movw	%cx, (%eax);			/* GM -> temp */\
		addl	$9, %eax;\
		addl	$9, %ebx;\
		movw	(%ebx), %cx;\
		movw	%cx, (%eax);			/* GM -> temp */\
		decl	%eax;

/* -------------------------------------------------------------------------
 * Plots a normalized byte of dhires color directly into graphics memory.
 * eax: graphics memory index
 * ebx: dhires_colors index
 * edx: scratch
 * ------------------------------------------------------------------------- */
#define PlotDHiresByte							\
		movb	SYMBOL_NAME(dhires_colors1)(,%ebx,1), %dl;	\
		movb	SYMBOL_NAME(dhires_colors2)(,%ebx,1), %dh;	\
		movw	%dx, (%eax);					\
		addl	$2, %eax;

#define PlotDHiresFirstByte\
		subl	$2, %eax;\
		PlotDHiresByte

#endif/*_640x400*/


/* -------------------------------------------------------------------------
 * Calculate the graphics memory offset based on EffectiveAddr.
 * BASE 0x2000, 0x4000
 * PTR register to store the offset
 * ------------------------------------------------------------------------- */
#define CalcHiresGM(BASE,PTR)\
		movl	EffectiveAddr_E, %ecx;		/* ecx = mem addrs */ \
		subw	BASE, EffectiveAddr;		/* - graphics base */ \
		movl	SYMBOL_NAME(hires_row_offset)			      \
			(,EffectiveAddr_E,4), PTR;	/* PTR = GM offset */ \
		movl	%ecx, EffectiveAddr_E;		/* + graphics base */ \
		addl	SYMBOL_NAME(GM), PTR;		/* PTR += GM base */


/* -------------------------------------------------------------------------
 * PlotByte - macro to plot a hires byte into graphics memory.
 * BASE = 0x2000,0x4000.
 * TABLE = expanded_col_hires_even, expanded_col_hires_odd.
 * OPP_TABLE = opposite table
 * INTERP_COLOR = even_colors, odd_colors
 * ALT_INTERP_COLOR = opposite colors
 * ------------------------------------------------------------------------- */
#define PlotByte(BASE,X,TABLE,OPP_TABLE,INTERP_COLOR,ALT_INTERP_COLOR)	      \
		pushl	%eax;				/* save regs */	      \
		pushl	%ebx;						      \
		pushl	%ecx;						      \
		\
		xorb	%ah, %ah;			/* clear noise */     \
		testb	$0xFF, SYMBOL_NAME(strict_color);		      \
		jnz	PB_dynamic##X;			/* dynamic color mode */\
		LoadHiresTableRef(TABLE);\
		CalcHiresGM(BASE,%eax);			/* eax = GM */\
		PlotPixels;				/* temp -> GM */\
		jmp	PB_exit##X;\
		\
PB_dynamic##X:\
		leal	SYMBOL_NAME(expanded_col8_hires_##TABLE)(,%eax,8), %ebx;\
		leal	SYMBOL_NAME(temp), %eax;	/* eax = temp */\
		addl	$2, %eax;\
		Plot7Pixels(%ebx,%eax);			/* 7bytes -> temp+2 */\
		\
		subl	$8, %eax;\
		CalcHiresGM(BASE,%ebx);			/* ebx = GM */\
		/* copy adjacent color bytes into temp array */\
		GrabAdjGMBytes;\
		\
		/* calculate dynamic colors in temp array */\
		DynamicCalculateColor(X,OPP_TABLE,INTERP_COLOR,ALT_INTERP_COLOR);\
PB_plot_dynamic##X:\
		leal	SYMBOL_NAME(temp), %ebx;	/* ebx = temp */\
		incl	%ebx;\
		CalcHiresGM(BASE,%eax);			/* eax = GM */\
		PlotPixelsExtra				/* temp -> GM: 1 + 7 + 1 */\
PB_exit##X:\
		popl	%ecx;				/* restore regs */    \
		popl	%ebx;						      \
		popl	%eax;


/* -------------------------------------------------------------------------
 * Dynamic calculation of color at the edges of bytes.
 * ------------------------------------------------------------------------- */
#define DynamicCalculateColor(X,OPP_TABLE,INTERP_COLOR,ALT_INTERP_COLOR);\
		movw	(%eax), %cx;					      \
		testb	$0xFF, %ch;	/* check right color */		      \
		jz	PB_next0##X;	/* right black, do other end */	      \
	movw	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %cx;\
		andb	$1, %ch;					      \
		jz	PB_black0##X;	/* right black */		      \
		andb	$0x40, %cl;					      \
		jz	PB_black0##X;	/* inside black, right colored */     \
		movw	$0x3737, (%eax); /* edge is white (#55) */	      \
		jmp	PB_next0##X;					      \
PB_black0##X:								      \
	movzwl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		movb	%ch, %cl;					      \
		xorb	%ch, %ch;					      \
		leal	SYMBOL_NAME(expanded_col8_hires_##OPP_TABLE)	      \
			(,%ecx,8), %ebx;				      \
		incl	%eax;						      \
		movb	(%ebx), %cl;					      \
		movb	%cl, (%eax);					      \
		decl	%eax;						      \
PB_next0##X:								      \
		decw	EffectiveAddr;		/* previous byte */	      \
		subl	$7, %eax;	/* left edge of byte */		      \
		movb	(%eax), %cl;					      \
		testb	$0xFF, %cl;	/* check left color */		      \
		jz	PB_next1##X;	/* left black, done */		      \
	movw	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %cx;\
		andb	$0x40, %cl;					      \
		jz	PB_black1##X;	/* left black */		      \
		andb	$0x1, %ch;					      \
		jz	PB_black1##X;	/* left colored, inside black */      \
		movw	$0x3737, (%eax); /* edge is white (#55) */	      \
		jmp	PB_next1##X;					      \
PB_black1##X:								      \
	movzbl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		leal	SYMBOL_NAME(expanded_col8_hires_##OPP_TABLE)	      \
			(,%ecx,8), %ebx;				      \
		addb	$6, %ebx;					      \
		movb	(%ebx), %cl;					      \
		movb	%cl, (%eax);					      \
PB_next1##X:								      \
		incw	EffectiveAddr;                                        \
		/* do extra calculation for interpolated colors */	      \
		cmpb	$2, SYMBOL_NAME(strict_color);			      \
		jne	PB_plot_dynamic##X;				      \
									      \
		decw	EffectiveAddr;					      \
		CalculateInterpColor(X,2,ALT_INTERP_COLOR);		      \
PB_next2##X:								      \
		incw	EffectiveAddr;					      \
		incl	%eax;						      \
		CalculateInterpColor(X,3,INTERP_COLOR);			      \
PB_next3##X:								      \
		addl	$6, %eax;					      \
		CalculateInterpColor(X,4,INTERP_COLOR);			      \
PB_next4##X:								      \
		incw	EffectiveAddr;					      \
		incl	%eax;						      \
		CalculateInterpColor(X,5,ALT_INTERP_COLOR);		      \
PB_next5##X:								      \
	decw	EffectiveAddr;


/* -------------------------------------------------------------------------
 * Calculates the color at the edge of interpolated bytes.
 * Done 4 times in little endian order (...7 0...7 0...)
 * ------------------------------------------------------------------------- */
#define CalculateInterpColor(X,Y,INTERP_COLOR)				\
		testb	$0xFF, (%eax);					\
		jnz	PB_next##Y##X;	/* not black, next */		\
		movw	(%eax), %cx;	/* off+1 in %ch */		\
		testb	$0xFF, %ch;					\
		jz	PB_next##Y##X;	/* off+1 is black, next */	\
		movb	-1(%eax), %cl;	/* off-1 in %cl */		\
		testb	$0xFF, %cl;					\
		jz	PB_next##Y##X;	/* off-1 is black, next */	\
		cmpb	$55, %cl;	/* off-1 is white? */		\
		je	PB_white0##X##Y;				\
		movb	%cl, (%eax);	/* store non-white */		\
		jmp	PB_next##Y##X;	/* next */			\
PB_white0##X##Y:							\
		cmpb	$55, %ch;	/* off+1 is white? */		\
		je	PB_white##X##Y;					\
		movb	%ch, (%eax);	/* store non-white */		\
		jmp	PB_next##Y##X;	/* next */			\
PB_white##X##Y:				/* both sides are white */	\
	movzbl	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %ecx;\
		shrb	$7, %cl;					\
 		movb	SYMBOL_NAME(INTERP_COLOR)(,%ecx,1), %bl;	\
		movb	%bl, (%eax);


/* -------------------------------------------------------------------------
 * compeletely update all the hires/dhires page rows.
 * X=0,1
 * OFFSET=0x2027,0x4027 (page 1, 2)
 * ------------------------------------------------------------------------- */
#define UpdateHiresRows(PRE,X,OFFSET)					 \
update_hires_rows_##X:							 \
		movl	$20, %ecx;	/* ECX: 40 column counter */	 \
		movl	%ebx, %edx;	/* EBX: pixel row counter */	 \
		shrb	$3, %dl;	/* normalize to 0 - 23 */	 \
					/* EDI: row offset */		 \
		movw	SYMBOL_NAME(video_line_offset)(,%edx,2),	 \
			EffectiveAddr;					 \
		movl	%ebx, %edx;					 \
		andb	$0x7, %dl;					 \
		shlw	$10, %dx;	/* EDX: offset range 0 - 1C00 */ \
		addw	OFFSET, %dx;	/* add base end-row offset */	 \
		addw	%dx, EffectiveAddr;	/* EDI: mem address */	 \
update_hires_columns_##X:						 \
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		cmpw	$159, %bx;	/* mixed mode boundary */	 \
		jg	update_hires_mixed_##X;				 \
		call	SYMBOL_NAME(PRE##ram_hires_page##X##_odd);	 \
		decw	EffectiveAddr;	/* previous address */		 \
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		call	SYMBOL_NAME(PRE##ram_hires_page##X##_even);	 \
		jmp	update_hires_cont_##X;				 \
update_hires_mixed_##X:							 \
		call	SYMBOL_NAME(PRE##ram_hires_mixed##X##_odd);	 \
		decw	EffectiveAddr;	/* previous address */		 \
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		call	SYMBOL_NAME(PRE##ram_hires_mixed##X##_even);	 \
update_hires_cont_##X:							 \
		decw	EffectiveAddr;	/* previous address */		 \
		decb	%cl;		/* dec column counter */	 \
		jnz	update_hires_columns_##X;			 \
		decw	%bx;		/* dec row */			 \
		jns	update_hires_rows_##X;


/* -------------------------------------------------------------------------
 * compeletely update all the text page rows.
 * X=0,1
 * OFF=0x427,0x827 (page 1, 2)
 * ------------------------------------------------------------------------- */
#define UpdateRows(PRE,X,OFFSET)				\
update_rows_##X:							\
		movl	$39, %ecx;					\
		movw	SYMBOL_NAME(video_line_offset)(,%ebx,2),	\
			EffectiveAddr;					\
		addw	OFFSET, EffectiveAddr;				\
update_columns_##X:							\
	movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al;\
		cmpb	$19, %bl;					\
		jg	update_mixed_##X;				\
		call	SYMBOL_NAME(PRE##ram_text_page##X);		\
		jmp	update_cont_##X;				\
update_mixed_##X:							\
		call	SYMBOL_NAME(PRE##ram_text_mixed##X);		\
update_cont_##X:							\
		decw	%di;						\
		decb	%cl;						\
		jns	update_columns_##X;				\
		decb	%bl;						\
		jns	update_rows_##X;


/* -------------------------------------------------------------------------
 * Plot a full double hires color byte into GM
 * OFF 0x2000, 0x4000
 * PROBLEMS:
 * 	graphics artifiacts are not implemented correctly.
 * ------------------------------------------------------------------------- */
#define PlotDHires(OFF,X)						      \
		pushl	%eax;				/* save regs */	      \
		pushl	%ebx;						      \
		pushl	%ecx;						      \
		pushl	%edx;						      \
		pushl	EffectiveAddr_E;				      \
									      \
		andw	$0xFFFF, EffectiveAddr;		/* erase offset */    \
		btr	$0, EffectiveAddr_E;		/* normalize */	      \
		movl	EffectiveAddr_E, %ecx;		/* ecx = mem addrs */ \
		subw	OFF, EffectiveAddr;		/* - graphics base */ \
		movl	SYMBOL_NAME(hires_row_offset)			      \
			(,EffectiveAddr_E,4), %eax;	/* eax = GM offset */ \
		movb	SYMBOL_NAME(hires_col_offset)			      \
			(,EffectiveAddr_E,1), %bl;			      \
		addl	SYMBOL_NAME(GM), %eax;		/* eax += GM base */  \
									      \
	leal	SYMBOL_NAME(apple_ii_64k), EffectiveAddr_E;\
		addl	%ecx, EffectiveAddr_E;				      \
		movl	EffectiveAddr_E, %ecx;				      \
		addl	BANK2, %ecx;					      \
									      \
		testb	$0xFF, %bl;					      \
		jz	plot_dhires##X##_cont;				      \
		movzbl	-1(EffectiveAddr_E), %ebx;			      \
		movb	0(%ecx), %bh;					      \
		btr	$7, %ebx;					      \
		shrb	$3, %bl;					      \
		shlb	$4, %bh;					      \
		orb	%bh, %bl;					      \
		xorb	%bh, %bh;					      \
		PlotDHiresFirstByte					      \
									      \
plot_dhires##X##_cont:							      \
		movl	%ecx, %edx;					      \
		movb	2(%edx), %cl;					      \
		shll	$28, %ecx;					      \
									      \
		movzbl	1(EffectiveAddr_E), %ebx;			      \
		btr	$7, %ebx;		/* erase msb */		      \
		shll	$21, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	1(%edx), %ebx;					      \
		btr	$7, %ebx;		/* erase msb */		      \
		shll	$14, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	0(EffectiveAddr_E), %ebx;			      \
		btr	$7, %ebx;		/* erase msb */		      \
		shll	$7, %ebx;					      \
		orl	%ebx, %ecx;					      \
									      \
		movzbl	0(%edx), %ebx;					      \
		btr	$7, %ebx;		/* erase msb */		      \
		orl	%ebx, %ecx;					      \
		/* 00000001 11111122 22222333 3333xxxx */		      \
									      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		shrl	$4, %ecx;					      \
		movb	%cl, %bl;					      \
		PlotDHiresByte						      \
		popl	EffectiveAddr_E;				      \
		andl	$0xFFFF, EffectiveAddr_E;/* for safety */	      \
		popl	%edx;						      \
		popl	%ecx;			 /* restore regs */	      \
		popl	%ebx;						      \
		popl	%eax;						      \
		ret;

#define offset_RAMRD_RAMWRT						\
		movl	SYMBOL_NAME(ramrd_offset), %eax;		\
		movl	%eax, SYMBOL_NAME(read_text_page_offset);	\
		movl	%eax, SYMBOL_NAME(read_hires_page_offset);	\
		movl	SYMBOL_NAME(ramwrt_offset), %eax;		\
		movl	%eax, SYMBOL_NAME(write_text_page_offset);	\
		movl	%eax, SYMBOL_NAME(write_hires_page_offset);


/* -------------------------------------------------------------------------
 * setup to plot the text/lores stuff.
 * eax: graphics memory pointer
 * ------------------------------------------------------------------------- */
#define PlotTextPagePre(OFF,OP,PAGE)\
		pushal;					/*Save everything -MUST BE MATCHED!*/\
		xorb	%ah, %ah;\
		movl	%eax, %esi;			/*ESI=EAX=Chr code*/\
		subw	OFF, EffectiveAddr;		/*Normalize scrn addr*/\
		testb	$0xFF, SYMBOL_NAME(vmode_active);/*Current SVGA page*/\
		OP	10f;\
		movl	PAGE, %eax;			/*Choose page PAGE*/\
		call	set_page;			/*Set page PAGE*/\
	10:\
							/*Compute row*/\
		movl	SYMBOL_NAME(text_row_offset)(,EffectiveAddr_E,4), %eax;\
		addl	SYMBOL_NAME(GM), %eax;		/*Graphic addr*/


/* -------------------------------------------------------------------------
 * Common code for plotting an 80 column character.
 * Only 640x400 resolution can do this.
 * eax: graphics memory pointer
 * esi: precalculated font pointer
 * OFF 0x400, 0x800
 * PAGE 0, 1
 * ------------------------------------------------------------------------- */
#define Plot80Character(TAG,OFF,OP,PAGE)\
		PlotTextPagePre(OFF,OP,PAGE)/* does a pushal */\
plot_80character_correct_page##TAG:\
    		addw	%bx, %ax;				/*screen offset*/\
		shll	$6, %esi;				/* * 64 = 8cols * 8rows*/\
		addl	$ SYMBOL_NAME(intermediate_font), %esi;	/*Font addr*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		addl	$2, %esi;\
		addl	$ SCANWIDTH-6, %eax;			/*Go to next row*/\
		\
		PlotCharacter80Row;\
		\
		popal;	/* MATCHES pushal from PlotTextPagePre */
