/* ********************************************************************
 *  Apple II emulator by Alexander Jean-Claude Bottema (C) 1994
 *
 *  $Id: apple2.S,v 1.19 1998/08/25 03:00:00 chernabog Exp $
 *
 *  MODIFICATION HISTORY
 *   v0.3 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jan 1997.
 *   v0.4 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Jun 1997.
 *   v0.5 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Feb 1998.
 *   v0.6 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Aug 1998.
 *     This code has nothing to do with my employer, GTE Internetworking,
 *     BBN Technologies.  It was written completely on my own time and on
 *     my own machine.
 *
 * ***************************************************************** */

#define __ASSEMBLY__
#include <linux/linkage.h>
#include "apple2.h"
#include "defs.h"


	/* --------------------------------------------------------------------
	   ....................................................................
	    Apple ][+ Memory documentation (//e differs mostly in softswitches)
	  
	    I use two jump tables; one for get_memory and one for set_memory
	    respectively. The jump tables contain exactly 64k entries to
            avoid unnecessary range checking. Furthermore, special access
	    or to serve memory mapped I/O services are easily obtained by
	    patching the jump table with various routines.
	  
	    Apple 64k address space:
	  
	    (Two byte addresses are represented with least significant
	     byte first, e.g. address FA59 is represented as 59 FA)
	  
	    Address		Description
	  
	    0000 - 00FF   	Zero page RAM
	    0100 - 01FF		Stack
	    0200 - 03EF		RAM
	    03F0 - 03F1		Address for BRK instruction
	                        (normally 59 FA = FA59)
	    03F2 - 03F3         Soft entry vector (normally BF 9D = 9DBF)
	    03F4                Power-up byte
	    03F5 - 03F7         Jump instruction to the subroutine which
	                        handles Applesoft II &-commands
	   			(normally 4C 58 FF = JMP FF58)
	    03F8 - 03FA		Jump instruction to the subroutine which
	                        handles User CTRL-Y commands
	                        (normally 4C 65 FF = JMP FF65)
	    03FB - 03FD		Jump instruction to the subroutine which
	                        handles Non-Maskable Interrupts (NMI)
	                        (normally 4C 65 FF = JMP FF65)
	    03FE - 03FF         Address of the subroutine which handles
	                        Interrupt ReQuests (IRQ)
	                        (normally 65 FF = FF65)
	    0400 - 07FF         Basically primary video text page
	    0478 - 047F		I/O Scratchpad RAM Addresses for Slot 0 - 7
	    04F8 - 04FF		               - " " -
	    0578 - 057F		               - " " -
	    05F8 - 05FF		               - " " -
	    0678 - 067F		               - " " -
	    06F8 - 06FF		               - " " -
	    0778 - 077F		               - " " -
	    07F8 - 07FF		               - " " -
	    ----- These two addresses are pretty strange; the values
	          provided were inconsistent at least on my Apple II
	    05F8		Holds the slot number of the disk
	  			controller card from which DOS was
	                        booted.
	    07F8		Holds the slot number (CX, X = Slot #)
	                        of the slot that is currently active.
	  
	    0800 - 0BFF		Basically secondary video text page
	  
	    0C00 - 1FFF		Plain RAM
	    2000 - 3FFF		Primary video hires page (RAM)
	    4000 - 5FFF         Secondary video Hi-Res page (RAM)
	    6000 - BFFF		Plain RAM
	                        (Normally the operating system is
	                         loaded into ~= 9C00 - BFFF)
	    C000 - C00F		Keyboard data
	                        (C00X contains the character ASCII
	                         code of the pressed key. The
	                         value is available at any C00X
	                         address)
	    C010 - C01F         Clear Keyboard strobe
	    C020 - C02F         Cassette output toggle
	    C030 - C03F         Speaker toggle
	    C040 - C04F         Utility strobe
	    C050		Set graphics mode
	    C051		Set text mode
	    C052		Set all text or graphics
	    C053		Mix text and graphics
	    C054		Display primary page
	    C055		Display secondary page
	    C056		Display low-res graphics
	    C057		Display hi-res graphics
	    C058 - C05F		Annunciator outputs
	    C060		Cassette input
	    C061 - C063		Pushbutton inputs (button 1, 2 and 3)
	    C064 - C067		Game controller inputs
	    C068 - C06F		Same as C060 - C067
	    C070 - C07F		Game controller strobe
	    C080 - C08F		Slot 0 I/O space (usually a language card)
	                        --- If language card
	    C080		    Reset language card
	                            Read mode enabled
	  			    Write mode disabled
	                            Read from language card
	  			    Write to ROM (impossible of course)
	    C081		--- First access
	  			    Read mode disabled
	                            Read from ROM
	  			--- On second access
	                            Write mode enabled
	  			    Write to language card
	    C082		--- (Disable language card)
	                            Read mode disabled
	                            Write mode disabled
	                            Read from ROM
	                            Write to ROM
	    C083		--- First access
	                            Read mode enabled
	  			    Read from language card
	  		 	--- On second access
	  			    Write mode enabled
	  			    Write to language card
	    C088 - C08B		Same as C080 - C083, but
	  			switch to second bank, i.e.
	  			map addresses D000-DFFF to
	  			other 4k area.
	    C100 - C1FF		Slot 1 PROM
	    C200 - C2FF		Slot 2 PROM
	    C300 - C3FF		Slot 3 PROM
	    C400 - C4FF		Slot 4 PROM
	    C500 - C5FF		Slot 5 PROM
	    C600 - C6FF		Slot 6 PROM
	    C700 - C7FF		Slot 7 PROM
	    C800 - CFFF		Expansion ROM (for peripheral cards)
	    CFFF		Disable access to expansion ROM for
	  			ALL peripheral cards.
	    D000 - DFFF		ROM or 4k RAM if language card is
	  			enabled. However, there are TWO 4k
	  			banks that can be mapped onto addresses
	  			D000 - DFFF. See C088 - C08B.
	    E000 - FFFF		ROM or 8k RAM if language card is
	  			enabled.
	   ----------------------------------------------------------------- */


   /* ----------------------------------------------------------------------
       6502 instructions
      ---------------------------------------------------------------------- */

#include "6502.h"
	
   /* ----------------------------------------------------------------------
       Undocumented 6502 (Illegal instructions)
      ---------------------------------------------------------------------- */

#include "undocumented.h"

   /* ----------------------------------------------------------------------
       65c02 instructions
      ---------------------------------------------------------------------- */

#ifdef APPLE_IIE
#include "65c02.h"
#endif

	/* -----------------------------------------------------------------
	 * Begin emulation.
	 * ----------------------------------------------------------------- */
ENTRY(cpu65x02)
		xorl	%eax, %eax
		call	set_page

		xorl	%eax, %eax		# Clear registers
		xorl	%ebx, %ebx
		xorl	%ecx, %ecx
		xorl	%edx, %edx
		xorl	%esi, %esi
		xorl	%edi, %edi

		movl	$0xFA62, %esi		# For now
		# movl	$0xFF59, %esi
		xorb	X_Reg, X_Reg		# Clear X register
		xorb	Y_Reg, Y_Reg		# Clear Y register
		xorb	A_Reg, A_Reg		# Clear A register
		xorb	F_Reg, F_Reg		# Clear flags
		movl	$0x1FF, SP_Reg		# Stack pointer
		cld				# Clear direction flag

		Continue


	/* -----------------------------------------------------------------
	 * Scan through video memory (text & graphics) and call the updating
	 * routines.  Depending on softswitch settings, either text or
	 * graphics, page 1 or 2 will be rendered.
	 * This is called on exit from menu screens, etc.
	 * ebx: number of rows (counting down)
	 * ----------------------------------------------------------------- */
ENTRY(update_video_screen)
		pushal

		xorl	%eax, %eax
		xorl	%edi, %edi

		/* 24 rows text/lores page 0 */
		movl	$23, %ebx
#ifdef APPLE_IIE
		UpdateRows(iie_soft_write_,0,$0x427)
#else
		UpdateRows(write_,0,$0x427)
#endif

		/* 24 rows text/lores page 1 */
		movl	$23, %ebx
#ifdef APPLE_IIE
		UpdateRows(iie_soft_write_,1,$0x827)
#else
		UpdateRows(write_,1,$0x827)
#endif

		/* 192 rows hires page 0 */
		movl	$191, %ebx
#ifdef APPLE_IIE
		UpdateHiresRows(iie_soft_write_,0,$0x2027)
#else
		UpdateHiresRows(write_,0,$0x2027)
#endif

		/* 192 rows hires page 1 */
		movl	$191, %ebx
#ifdef APPLE_IIE
		UpdateHiresRows(iie_soft_write_,1,$0x4027)
#else
		UpdateHiresRows(write_,1,$0x4027)
#endif

		popal
		ret
	

	/* --------------------------------------------------------------------
	 * This is a convenience wrapper around c_setpage() (video.h).
	 * Call set_page with appropriate content in eax.
	 * eax: page number (0, 1)
	 * ----------------------------------------------------------------- */
		.align 4
set_page:
		movb	%al, SYMBOL_NAME(vmode_active)# set apple2 active page
		orb	%al, %al		# Check content of AL
		jnz	set_page_1		# AL = 1 ?
set_page_0:
		pushal
		pushl	$0
set_page_c:	call	SYMBOL_NAME(c_setpage)
		addl	$4, %esp
		popal
		ret
set_page_1:
		pushal
		pushl	$1
		jmp	set_page_c

	/* --------------------------------------------------------------------
	    Routines	: read_memory, read_memory_word
	    Input	: EffectiveAddr (Currently EDI)
	    Output	: AL = Byte value at 6502 address EffectiveAddr or,
			  AX = Word value at 6502 address EffectiveAddr.
	  		  All other registers are unaffected
	    Usage	: call read_memory (with appropriate content in ESI)
	    Comment	: The read_memory routine uses a jump table (thus
	                  avoiding range checking of particular address areas)
	                  to jump to the appropriate routine given the address.
	  		  Hence, the jump table is consisted of 64k addresses
			  (this consumes 256k of memory).
	   ----------------------------------------------------------------- */
		.align 4
read_memory:
	/* watch reading of data */
#ifdef DEBUGGER
		movb	$0, SYMBOL_NAME(watch_write)
		movl	EffectiveAddr_E, SYMBOL_NAME(watch_addrs)
#endif
		jmp	* SYMBOL_NAME(table_read_memory)(,EffectiveAddr_E,4)


		.align 4
read_memory_word:
	/* watch reading of data */
#ifdef DEBUGGER
		movb	$0, SYMBOL_NAME(watch_write)
		movl	EffectiveAddr_E, SYMBOL_NAME(watch_addrs)
#endif
		incw	EffectiveAddr
		call	* SYMBOL_NAME(table_read_memory)(,EffectiveAddr_E,4)
		shlw	$8, %ax
		decw	EffectiveAddr
		call	* SYMBOL_NAME(table_read_memory)(,EffectiveAddr_E,4)
		ret


	/* --------------------------------------------------------------------
	    Routine	: write_memory
	    Input	: EffectiveAddr (Currently EDI)
	    Output	: AL = Byte value
	  		  All other registers are unaffected
	    Usage	: call write_memory (with appropriate content in EDI
			  and AL)
	    Comment	: The write_memory routine uses a jump table (thus
	                  avoiding range checking of particular address areas)
	                  to jump to the appropriate routine given the address.
	  		  Hence, the jump table is consisted of 64k addresses
			  (this consumes 256k of memory).
	   ----------------------------------------------------------------- */
		.align 4
write_memory:
	/* watch writing of data */
#ifdef DEBUGGER
		movb	$1, SYMBOL_NAME(watch_write)
		movl	EffectiveAddr_E, SYMBOL_NAME(watch_addrs)
		movb	%al, SYMBOL_NAME(watch_data)
#endif
		jmp	* SYMBOL_NAME(table_write_memory)(,EffectiveAddr_E,4)

#ifdef DEBUGGER
/* -------------------------------------------------------------------------
    do_write_memory () - called from debugger to write values to memory.
   ------------------------------------------------------------------------- */
ENTRY (do_write_memory)
		movl	SYMBOL_NAME(debug_scratch), EffectiveAddr_E
		movb	SYMBOL_NAME(watch_data), %al
		jmp	write_memory

/* -------------------------------------------------------------------------
    do_write_lc () - called from debugger to write unconditionally to
    language card.
   ------------------------------------------------------------------------- */
ENTRY (do_write_lc)
		movl	SYMBOL_NAME(debug_scratch), EffectiveAddr_E
		movb	SYMBOL_NAME(watch_data), %al
		cmpl	$0, SYMBOL_NAME(watch_addrs)
		jz	do_bank0
		jns	do_bank1
		subw	$0xE000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_card)(,EffectiveAddr_E,1)
		ret
do_bank0:	subw	$0xD000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		ret
do_bank1:	subw	$0xC000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		ret
#endif

		/******************************************/

/* ram_nop, read_ram_default, write_ram_default: default ram/rom
   read/write functions. */
ENTRY(ram_nop)
		ret

ENTRY(read_ram_default)
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		ret

ENTRY(write_ram_default)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		ret

ENTRY(read_unmapped_softswitch)
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
    		ret

ENTRY(write_unmapped_softswitch)
    		ret

#ifdef APPLE_IIE
/* iie_read_ram_default - read from 128k ram */
ENTRY(iie_read_ram_default)
		orl	SYMBOL_NAME(ramrd_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)\
			(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_write_ram_default - read from 128k ram */
ENTRY(iie_write_ram_default)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
		ret
#endif

		/******************************************/

/* write to LC RAM 0xe000 -> 0xffff.  we check to make sure that we
   can indeed write to the language card. */
ENTRY(write_ram_lc)
		cmpb	$0, SYMBOL_NAME(language_card_write)
		je	SYMBOL_NAME(ram_nop)
		pushl	EffectiveAddr_E
		subw	$0xE000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_card)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret

#ifdef APPLE_IIE
/* IIE write to LC RAM 0xe000 -> 0xffff.  we check to make sure that
   we can indeed write to the language card. */
ENTRY(iie_write_ram_lc)
		cmpb	$0, SYMBOL_NAME(language_card_write)
		je	SYMBOL_NAME(ram_nop)
		pushl	EffectiveAddr_E
		subw	$0xE000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(language_card)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret
#endif

/* write to LC RAM 0xd000 -> 0xdfff banks. we check to make sure that
   we can indeed write to the language card. */
ENTRY(write_ram_bank)
		cmpb	$0, SYMBOL_NAME(language_card_write)
		je	SYMBOL_NAME(ram_nop)
		pushl	EffectiveAddr_E
		cmpb	$0, SYMBOL_NAME(language_current_bank)
		jne	write_bank1
		subw	$0xD000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret
write_bank1:	subw	$0xC000, EffectiveAddr
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret

#ifdef APPLE_IIE
/* write to LC RAM 0xd000 -> 0xdfff banks. we check to make sure that
   we can indeed write to the language card. */
ENTRY(iie_write_ram_bank)
		cmpb	$0, SYMBOL_NAME(language_card_write)
		je	SYMBOL_NAME(ram_nop)
		pushl	EffectiveAddr_E
		cmpb	$0, SYMBOL_NAME(language_current_bank)
		jne	iie_write_bank1
		subw	$0xD000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret
iie_write_bank1:
		subw	$0xC000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1)
		popl	EffectiveAddr_E
		ret
#endif

		/******************************************/

/* read LC RAM or ROM 0xe000 -> 0xffff. */
ENTRY(read_ram_lc)
		cmpb	$0, SYMBOL_NAME(language_card_read)
		je	SYMBOL_NAME(read_ram_default)
		pushl	EffectiveAddr_E
		subw	$0xE000, EffectiveAddr
		movb	SYMBOL_NAME(language_card)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret

#ifdef APPLE_IIE
/* read LC RAM or ROM 0xe000 -> 0xffff. */
ENTRY(iie_read_ram_lc)
		cmpb	$0, SYMBOL_NAME(language_card_read)
		je	iie_read_ram_lc_rom
		pushl	EffectiveAddr_E
		subw	$0xE000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(language_card)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret
iie_read_ram_lc_rom:	/* don't need offsets in ROM! */
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		ret
#endif

/* read to LC RAM or ROM 0xd000 -> 0xdfff banks */
ENTRY(read_ram_bank)
		cmpb	$0, SYMBOL_NAME(language_card_read)
		je	SYMBOL_NAME(read_ram_default)
		pushl	EffectiveAddr_E
		cmpb	$0, SYMBOL_NAME(language_current_bank)
		jne	read_bank1
		subw	$0xD000, EffectiveAddr
		movb	SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret
read_bank1:	subw	$0xC000, EffectiveAddr
		movb	SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret

#ifdef APPLE_IIE
/* read to LC RAM or ROM 0xd000 -> 0xdfff banks */
ENTRY(iie_read_ram_bank)
		cmpb	$0, SYMBOL_NAME(language_card_read)
		je	iie_read_ram_bank_rom
		pushl	EffectiveAddr_E
		cmpb	$0, SYMBOL_NAME(language_current_bank)
		jne	iie_read_bank1
		subw	$0xD000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret
iie_read_bank1:	subw	$0xC000, EffectiveAddr
		orl	SYMBOL_NAME(lc_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(language_banks)(,EffectiveAddr_E,1), %al
		popl	EffectiveAddr_E
		ret
iie_read_ram_bank_rom:	/* don't need offsets in ROM! */
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		ret

/* read/write //e zpage and stack */
ENTRY(iie_read_ram_zpage_and_stack)
		orl	SYMBOL_NAME(zp_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

ENTRY(iie_write_ram_zpage_and_stack)
		orl	SYMBOL_NAME(zp_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
		ret
#endif

		/******************************************/

ENTRY(write_ram_text_page0)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	plot_character0
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # lores mode?
		jz	plot_block0
		ret

ENTRY(write_ram_text_mixed0)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	plot_character0
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	plot_character0
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # Not hires mode?
		jz	plot_block0
		ret

ENTRY(write_ram_text_page1)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	plot_character1
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # lores mode?
		jz	plot_block1
		ret

ENTRY(write_ram_text_mixed1)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	plot_character1
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	plot_character1
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # Not hires mode?
		jz	plot_block1
		ret

#ifdef APPLE_IIE
/* iie_write_ram_text_page0 - handle text page //e specific */
ENTRY(iie_write_ram_text_page0)
		orl	SYMBOL_NAME(write_text_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_text_page0:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jz	iie_write_lores0		# graphics
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jnz	plot_80character0		# 80 col text

		/* check for screen update */
		testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	plot_character0			# 40 col text
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_text_page_offset)
		jnz	ram_nop				# NOP (in auxram)
		jmp	plot_character0			# 40 col text

iie_write_lores0:
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # lores mode?
		jz	plot_block0			# lores
		ret

/* iie_read_ram_text_page0 - read value from text page 0 */
ENTRY(iie_read_ram_text_page0)
		orl	SYMBOL_NAME(read_text_page_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_read_screen_hole_text_page0 - read value out of screen hole */
ENTRY(iie_read_screen_hole_text_page0)
		orl	SYMBOL_NAME(read_text_page_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_write_screen_hole_text_page0 - write value into screen hole */
ENTRY(iie_write_screen_hole_text_page0)
		orl	SYMBOL_NAME(write_text_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_write_ram_text_mixed0 - handle mixed text page //e specific */
ENTRY(iie_write_ram_text_mixed0)
		orl	SYMBOL_NAME(write_text_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_text_mixed0:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	iie_mixed_text0			# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jz	iie_mixed_lores0

iie_mixed_text0:
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
		jnz	plot_80character0
		/* check for screen update */
		testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	plot_character0			# 40 col text
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_text_page_offset)
		jnz	ram_nop				# NOP (in auxram)
		jmp	plot_character0			# 40 col text

iie_mixed_lores0:
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # Not hires mode?
		jz	plot_block0			# lores
		ret

/* iie_write_ram_text_page1 - handle text page1 //e specific */
ENTRY(iie_write_ram_text_page1)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_text_page1:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jz	iie_write_lores1		# graphics
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jnz	plot_80character1		# 80 col text

		/* check for screen update */
		testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	plot_character1			# 40 col text
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_text_page_offset)
		jnz	ram_nop				# NOP (in auxram)
		jmp	plot_character1			# 40 col text

iie_write_lores1:
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # lores mode?
		jz	plot_block1			# lores
		ret


/* iie_write_ram_text_mixed1 - handle mixed page 1 //e specific */
ENTRY(iie_write_ram_text_mixed1)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_text_mixed1:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	iie_mixed_text1			# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jz	iie_mixed_lores1

iie_mixed_text1:
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
		jnz	plot_80character1
		/* check for screen update */
		testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	plot_character1			# 40 col text
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_text_page_offset)
		jnz	ram_nop				# NOP (in auxram)
		jmp	plot_character1			# 40 col text

iie_mixed_lores1:
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # Not hires mode?
		jz	plot_block1			# lores
		ret


#endif /* APPLE_IIE */

ENTRY(write_ram_hires_page0_even)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_even_byte0
		ret

ENTRY(write_ram_hires_mixed0_even)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_even_byte0
		ret

ENTRY(write_ram_hires_page0_odd)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_odd_byte0
		ret

ENTRY(write_ram_hires_mixed0_odd)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_odd_byte0
		ret

#ifdef APPLE_IIE
/* iie_write_ram_hires_page0_even - handle hires page //e specific */
ENTRY(iie_write_ram_hires_page0_even)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_page0_even:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires0		# dhires

		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page0_even
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
_iie_plot_hires_page0_even:
		jmp	plot_even_byte0			# plot hires


/* iie_write_ram_hires_mixed0_even - handle mixed hires page //e specific */
ENTRY(iie_write_ram_hires_mixed0_even)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_mixed0_even:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires0		# dhires

		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page0_even
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
		jmp	plot_even_byte0			# plot hires

/* iie_write_ram_hires_page0_odd - handle hires page //e specific */
ENTRY(iie_write_ram_hires_page0_odd)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_page0_odd:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires0		# dhires

		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page0_odd
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
_iie_plot_hires_page0_odd:
		jmp	plot_odd_byte0			# plot hires

/* iie_write_ram_hires_mixed0_odd - handle mixed hires page //e specific */
ENTRY(iie_write_ram_hires_mixed0_odd)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_mixed0_odd:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires0		# dhires

		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page0_odd
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
		jmp	plot_odd_byte0			# plot hires

/* iie_read_ram_hires_page0 - read value from hires page */
ENTRY(iie_read_ram_hires_page0)
		orl	SYMBOL_NAME(read_hires_page_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_read_screen_hole_hires_page0 - read a value from hires screen hole */
ENTRY(iie_read_screen_hole_hires_page0)
		orl	SYMBOL_NAME(read_hires_page_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* iie_write_screen_hole_hires_page0 - write a value into hires screen hole */
ENTRY(iie_write_screen_hole_hires_page0)
		orl	SYMBOL_NAME(write_hires_page_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
		ret
#endif

ENTRY(write_ram_hires_page1_even)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_even_byte1
		ret

ENTRY(write_ram_hires_mixed1_even)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_even_byte1
		ret

ENTRY(write_ram_hires_page1_odd)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_odd_byte1
		ret

ENTRY(write_ram_hires_mixed1_odd)
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jnz	plot_odd_byte1
		ret

#ifdef APPLE_IIE
/* iie_write_ram_hires_page1_even - write hires page1 //e specific */
ENTRY(iie_write_ram_hires_page1_even)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_page1_even:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires1		# dhires
		/* hires page is in main memory, draw it if we're
                   repainting the screen. */
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page1_even
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
_iie_plot_hires_page1_even:
		jmp	plot_even_byte1			# plot hires

/* iie_write_ram_hires_mixed1_even - write hires page1 //e specific */
ENTRY(iie_write_ram_hires_mixed1_even)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_mixed1_even:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires1		# dhires
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page1_even
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
		jmp	plot_even_byte1			# plot hires

/* iie_write_ram_hires_page1_odd - write hires page1 //e specific */
ENTRY(iie_write_ram_hires_page1_odd)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_page1_odd:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires1		# dhires
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page1_odd
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
_iie_plot_hires_page1_odd:
		jmp	plot_odd_byte1			# plot hires

/* iie_write_ram_hires_mixed1_odd - write hires page1 //e specific */
ENTRY(iie_write_ram_hires_mixed1_odd)
		orl	SYMBOL_NAME(ramwrt_offset), EffectiveAddr_E
		movb	%al, SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1)
		andl	$0xFFFF, EffectiveAddr_E
iie_soft_write_ram_hires_mixed1_odd:
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	ram_nop				# text
		testb	$0xFF, SYMBOL_NAME(vmode_hires) # hires mode?
		jz	ram_nop				# lores
		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	1f				# not dhires
		testb	$0xFF, SYMBOL_NAME(vmode_dhires)# dhires mode?
		jnz	iie_plot_dhires1		# dhires
	1:	testb	$0xFF, SYMBOL_NAME(updating_screen)
		jnz	_iie_plot_hires_page1_odd
		testl	$0xFFFFFFFF, SYMBOL_NAME(write_hires_page_offset)
		jnz	ram_nop				# in auxram
		jmp	plot_odd_byte1			# plot hires

		.align	4
iie_plot_dhires0:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jz	iie_plot_dhires0_correct_page
		pushal
		xorl	%eax, %eax			# Choose page 0
		call	set_page			# Set page 0
		popal
iie_plot_dhires0_correct_page:
		PlotDHires($0x2000,0)
		ret

iie_plot_dhires1:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jnz	iie_plot_dhires1_correct_page
		pushal
		xorl	%eax, %eax
		movb	$1, %eax			# Choose page 1
		call	set_page			# Set page 1
		popal
iie_plot_dhires1_correct_page:
		PlotDHires($0x4000,1)
		ret


#ifdef _640x400
		.align 4
plot_80character0:
		pushl	%ebx
		pushl	EffectiveAddr_E

		orl	$0x10000, EffectiveAddr_E	# aux ram
		movl	$0, %ebx			# +0 screen offset
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		Plot80Character(0a,$0x400,jz,$0)

		popl	EffectiveAddr_E			# main ram
		movl	$7, %ebx			# +7 screen offset
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		Plot80Character(0b,$0x400,jz,$0)

		popl	%ebx
		ret

		.align 4
plot_80character1:
		pushl	%ebx
		pushl	EffectiveAddr_E

		orl	$0x10000, EffectiveAddr_E	# aux ram
		movl	$0, %ebx			# +0 screen offset
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		Plot80Character(1a,$0x800,jnz,$1)

		popl	EffectiveAddr_E			# main ram
		movl	$7, %ebx			# +7 screen offset
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		Plot80Character(1b,$0x800,jnz,$1)

		popl	%ebx
		ret

#else /* !640x400 resolution is not sufficient for 80 column */
		.align 4
plot_80character0:
		andl	$0xFFFF, EffectiveAddr_E/* for safety */
		ret

		.align 4
plot_80character1:
		andl	$0xFFFF, EffectiveAddr_E/* for safety */
		ret
#endif /* _640x400 */

#endif /* APPLE_IIE */


/* plot character on first text page */
		.align 4
plot_character0:
		PlotTextPagePre($0x400,jz,$0)
plot_character_correct_page:
#ifdef _640x400
		shll	$7, %esi				# * 128
#else
		shll	$6, %esi				# * 64
#endif
		addl	$ SYMBOL_NAME(expanded_font), %esi	# Font addr

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row
		addl	$2, %esi;
		addl	$ SCANSTEP, %eax			# Go to next row

		PlotCharacter40Row

		popal
		ret


/* plot character on second text page */
		.align 4
plot_character1:
		PlotTextPagePre($0x800,jnz,$1)
		jmp	plot_character_correct_page	# same as page 0


/* plot lores block first page */
	
plot_block0:
		PlotTextPagePre($0x400,jz,$0)
plot_block_correct_page:
		movw	%si, %dx			# Compute color
		andb	$0x0F, %dl
		shlb	$4, %dl
		movb	%dl, %dh
		shll	$16, %edx
		movw	%si, %dx
		andb	$0x0F, %dl
		shlb	$4, %dl
		movb	%dl, %dh

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		movw	%si, %dx			# Compute color
		andb	$0xF0, %dl
		movb	%dl, %dh
		shll	$16, %edx
		movw	%si, %dx
		andb	$0xF0, %dl
		movb	%dl, %dh

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow
		addl	$ SCANSTEP, %eax		# Go to next row

		PlotBlockRow

		popal
		ret

		.align 4
plot_block1:
		PlotTextPagePre($0x800,jnz,$1)
    		jmp plot_block_correct_page
	
	

/* plot even column hires byte on page 0 */
		.align 4
plot_even_byte0:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jz	plot_even_byte0_correct_page
		pushal
		xorl	%eax, %eax			# Choose page 0
		call	set_page			# Set page 0
		popal
plot_even_byte0_correct_page:
		PlotByte($0x2000,0,even,odd,even_colors,odd_colors)
		ret
	
/* plot odd column hires byte on page 0 */
		.align 4
plot_odd_byte0:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jz	plot_odd_byte0_correct_page
		pushal
		xorl	%eax, %eax			# Choose page 0
		call	set_page			# Set page 0
		popal
plot_odd_byte0_correct_page:
		PlotByte($0x2000,1,odd,even,odd_colors,even_colors)
		ret

/* plot even column hires byte on page 1 */	
		.align 4
plot_even_byte1:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jnz	plot_even_byte1_correct_page
		pushal
		movl	$1, %eax			# Choose page 1
		call	set_page			# Set page 1
		popal
plot_even_byte1_correct_page:
		PlotByte($0x4000,2,even,odd,even_colors,odd_colors)
		ret

/* plot odd column hires byte on page 1 */
		.align 4
plot_odd_byte1:
		testb	$0xFF, SYMBOL_NAME(vmode_active)# Current SVGA page
		jnz	plot_odd_byte1_correct_page
		pushal
		movl	$1, %eax			# Choose page 1
		call	set_page			# Set page 1
		popal
plot_odd_byte1_correct_page:
		PlotByte($0x4000,3,odd,even,odd_colors,even_colors)
		ret
	
	

ENTRY(read_keyboard)
		movb	SYMBOL_NAME(apple_ii_64k)+0xC000, %al
        	ret

ENTRY(read_keyboard_strobe)
		andb	$0x7f, SYMBOL_NAME(apple_ii_64k)+0xC000
		andb	$0x7f, SYMBOL_NAME(apple_ii_64k)+0x1C000
		movb	SYMBOL_NAME(apple_ii_64k)+0xC000, %al /* HACK: necessary? */
		ret

ENTRY(read_random)
		pushal
		call	SYMBOL_NAME(c_read_random)
		popal
		movb	SYMBOL_NAME(random_value), %al
		ret

ENTRY(read_speaker_toggle_pc)
		inb	$0x61, %al
		xorb	$0x2, %al
		outb	%al, $0x61
		ret

ENTRY(read_switch_primary_page)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)
		jnz	_read_switch_primary
		ret
_read_switch_primary:
		movb	$0, SYMBOL_NAME(vmode_page2)
		movb	$0, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$0
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
		ret

ENTRY(read_switch_secondary_page)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)
		jz	_read_switch_secondary
		ret
_read_switch_secondary:
		movb	$1, SYMBOL_NAME(vmode_page2)
		movb	$1, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$1
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
		ret

#ifdef APPLE_IIE
/* PAGE2 off. if 80STORE on then we use main text page1, and if HIRES
	on we also use main hires page1, regardless of RAMRD and
	RAMWRT. */
ENTRY(iie_page2_off)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)	# already off?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_page2)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)
		jz	_iie_page2_off_hires_off	# no 80STORE
		movl	$0, SYMBOL_NAME(read_text_page_offset)
		movl	$0, SYMBOL_NAME(write_text_page_offset)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	_iie_page2_off_hires_off	# no HIRES
		movl	$0, SYMBOL_NAME(read_hires_page_offset)
		movl	$0, SYMBOL_NAME(write_hires_page_offset)
_iie_page2_off_hires_off:
		movb	$0, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$0				# page 1 main
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)# redraw */
		ret


/* PAGE2 on.  if 80STORE on then we use aux text page 1, and if HIRES
	on we also use aux hires page 1, regardless of RAMRD and
	RAMWRT. */
ENTRY(iie_page2_on)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)	# already on?
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_page2)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)
		jz	_iie_page2_on_80store_off	# no 80STORE
		movl	BANK2, SYMBOL_NAME(read_text_page_offset)
		movl	BANK2, SYMBOL_NAME(write_text_page_offset)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	_iie_page2_on_cont
		movl	BANK2, SYMBOL_NAME(read_hires_page_offset)
		movl	BANK2, SYMBOL_NAME(write_hires_page_offset)
_iie_page2_on_cont:
		movb	$0, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$0				# page 1 aux
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)# redraw */
		ret

_iie_page2_on_80store_off:				# no 80STORE
		movb	$1, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$1				# page 2 main
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)# redraw */
		ret
#endif

ENTRY(read_switch_graphics)
		testb	$0xFF, SYMBOL_NAME(vmode_text)
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_text)
		call	SYMBOL_NAME(update_video_screen)
		ret		

ENTRY(read_switch_text)
		testb	$0xFF, SYMBOL_NAME(vmode_text)
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_text)
		call	SYMBOL_NAME(update_video_screen)
		ret

ENTRY(read_switch_no_mixed)
		testb	$0xFF, SYMBOL_NAME(vmode_mixed)
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_mixed)
		call	SYMBOL_NAME(update_video_screen)
		ret		

ENTRY(read_switch_mixed)
		testb	$0xFF, SYMBOL_NAME(vmode_mixed)
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_mixed)
		call	SYMBOL_NAME(update_video_screen)
		ret

ENTRY(read_switch_lores)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_hires)
		call	SYMBOL_NAME(update_video_screen)
		ret

ENTRY(read_switch_hires)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_hires)
		call	SYMBOL_NAME(update_video_screen)
		ret

#ifdef APPLE_IIE
/* HIRES off. use RAMRD/RAMWRT offsets for hires page 1. */
ENTRY(iie_hires_off)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)		# already off?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_hires)		# no hires
		movl	SYMBOL_NAME(ramrd_offset), %eax
		movl	%eax, SYMBOL_NAME(read_hires_page_offset)
		movl	SYMBOL_NAME(ramwrt_offset), %eax
		movl	%eax, SYMBOL_NAME(write_hires_page_offset)
		call	SYMBOL_NAME(update_video_screen)	# update screen
		ret

/* HIRES on. if 80STORE on, use PAGE2 offset for hires page 1. */
ENTRY(iie_hires_on)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)		# already on?
		jnz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_hires)		# hires on
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)
		jz	iie_hires_on_80store_off		# no 80STORE

		xorl	%eax, %eax
		movb	SYMBOL_NAME(vmode_page2), %ah
		shll	$8, %eax				# 0 or 0x10000
		movl	%eax, SYMBOL_NAME(read_hires_page_offset)
		movl	%eax, SYMBOL_NAME(write_hires_page_offset)
iie_hires_on_80store_off:
		call	SYMBOL_NAME(update_video_screen)	# update screen
		ret
#endif

/*****************************************************************************/

.comm		joy_trigger0, 2
.comm		joy_trigger1, 2
.comm		joy_trigger2, 2
.comm		joy_trigger3, 2

ENTRY(read_button0)
		movb	SYMBOL_NAME(joy_button0), %al
		ret

ENTRY(read_button1)
		movb	SYMBOL_NAME(joy_button1), %al
		ret

ENTRY(read_button2)
		movb	SYMBOL_NAME(joy_button2), %al
		ret

ENTRY(read_gc0)
		cmpw	$0xFF, joy_trigger0
		je	read_gc0_cont
		incw	joy_trigger0
read_gc0_cont:
		movw	joy_trigger0, %ax
		cmpw	%ax, SYMBOL_NAME(joy_x)
		jge	read_gc0_ff	/* XXXX is this correct? */
		movb	$0, %al
		ret
read_gc0_ff:
		movb	$0xFF, %al
		ret

ENTRY(read_gc1)
		cmpw	$0xFF, joy_trigger1
		je	read_gc1_cont
		incw	joy_trigger1
read_gc1_cont:
		movw	joy_trigger1, %ax
		cmpw	%ax, SYMBOL_NAME(joy_y)
		jge	read_gc1_ff
		movb	$0, %al
		ret
read_gc1_ff:
		movb	$0xFF, %al
		ret

/* HACK not doing anything... */
ENTRY(iie_read_gc2)
		ret

ENTRY(iie_read_gc3)
		ret

ENTRY(read_gc_strobe)
		movb	$0, joy_trigger0
		movb	$0, joy_trigger1
		movb	$0, joy_trigger2
		movb	$0, joy_trigger3
		ret

/* c080: read RAM; no write; use $D000 bank 2. */
ENTRY(lc_c080)
		movb	$0, SYMBOL_NAME(language_card_second)
		movb	$0, SYMBOL_NAME(language_card_write)
		movb	$1, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_current_bank)
		ret

/* c081: read ROM; write RAM; use $D000 bank 2. */
ENTRY(lc_c081)
		movb	$0, SYMBOL_NAME(language_card_read)
		cmpb	$0, SYMBOL_NAME(language_card_second)
		je	lc_c081_exit
		movb	$0, SYMBOL_NAME(language_current_bank)
		movb	$1, SYMBOL_NAME(language_card_write)
lc_c081_exit:
		movb	$1, SYMBOL_NAME(language_card_second)
		ret

/* c082: read ROM; no write; use $D000 bank 2. */
ENTRY(lc_c082)
		movb	$0, SYMBOL_NAME(language_card_second)
		movb	$0, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_card_write)
		ret

/* c083: read and write RAM; no write; use $D000 bank 2. */
ENTRY(lc_c083)
		movb	$1, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_current_bank)
		cmpb	$0, SYMBOL_NAME(language_card_second)
		je	lc_c083_exit
		movb	$1, SYMBOL_NAME(language_card_write)
lc_c083_exit:
		movb	$1, SYMBOL_NAME(language_card_second)
		ret

/* c088: read RAM; no write; use $D000 bank 1. */
ENTRY(lc_c088)
		movb	$0, SYMBOL_NAME(language_card_second)
		movb	$1, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_card_write)
		movb	$1, SYMBOL_NAME(language_current_bank)
		ret

/* c089: read ROM; write RAM; use $D000 bank 1. */
ENTRY(lc_c089)
		movb	$0, SYMBOL_NAME(language_card_read)
		cmpb	$0, SYMBOL_NAME(language_card_second)
		jz	lc_c089_exit
		movb	$1, SYMBOL_NAME(language_card_write)
		movb	$1, SYMBOL_NAME(language_current_bank)
lc_c089_exit:
		movb	$1, SYMBOL_NAME(language_card_second)
		ret

/* c08a: read ROM; no write; use $D000 bank 1. */
ENTRY(lc_c08a)
		movb	$0, SYMBOL_NAME(language_card_second)
		movb	$0, SYMBOL_NAME(language_card_read)
		movb	$0, SYMBOL_NAME(language_card_write)
		ret

/* c08b: read and write RAM; use $D000 bank 1. */
ENTRY(lc_c08b)
		movb	$1, SYMBOL_NAME(language_card_read)
		movb	$1, SYMBOL_NAME(language_current_bank)
		cmpb	$0, SYMBOL_NAME(language_card_second)
		jz	lc_c08b_exit
		movb	$1, SYMBOL_NAME(language_card_write)
lc_c08b_exit:
		movb	$1, SYMBOL_NAME(language_card_second)
		ret
/* -------------------------------------------------------------------------
 * misc //e functions
 * ------------------------------------------------------------------------- */

#ifdef APPLE_IIE

/* 80STORE off. use RAMRD/RAMWRT for text/hires display page 1 */
ENTRY(iie_80store_off)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# already off?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(eighty_store_flag)
		offset_RAMRD_RAMWRT	# RAMRD/RAMWRT are text/hires offset
		movzbl	SYMBOL_NAME(vmode_page2), %eax
		movb	%al, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	%eax
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

/* 80STORE on. we use PAGE2 offset to access text page 1.  if HIRES on
	we also use PAGE2 offset to access hires page 1 */
ENTRY(iie_80store_on)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# already on?
		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(eighty_store_flag)
		testb	$0xFF, SYMBOL_NAME(vmode_page2)
		jnz	iie_80store_on_page2_on

		movl	$0, SYMBOL_NAME(read_text_page_offset)
		movl	$0, SYMBOL_NAME(write_text_page_offset)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	iie_80store_on_cont
		movl	$0, SYMBOL_NAME(read_hires_page_offset)
		movl	$0, SYMBOL_NAME(write_hires_page_offset)
		jmp	iie_80store_on_cont

iie_80store_on_page2_on:
		movl	BANK2, SYMBOL_NAME(read_text_page_offset)
		movl	BANK2, SYMBOL_NAME(write_text_page_offset)
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jz	iie_80store_on_cont
		movl	BANK2, SYMBOL_NAME(read_hires_page_offset)
		movl	BANK2, SYMBOL_NAME(write_hires_page_offset)
iie_80store_on_cont:
		movb	$0, SYMBOL_NAME(vmode_screen)
		pushal
		pushl	$0					# page 1
		call	SYMBOL_NAME(c_setscreen)
		addl	$4, %esp
		popal
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

ENTRY(iie_check_80store)
		movb	SYMBOL_NAME(eighty_store_flag), %al
		ret

/* RAMRD main. if 80STORE off then text/hires display pages are in
	main memory. */
ENTRY(iie_ramrd_main)
		testb	$0xFF, SYMBOL_NAME(ramrd_flag)		# already on?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(ramrd_flag)
		movl	$0, SYMBOL_NAME(ramrd_offset)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# 80store?
		jnz	iie_ramrd_main_80store_on
		movl	$0, SYMBOL_NAME(read_text_page_offset)
		movl	$0, SYMBOL_NAME(read_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret
iie_ramrd_main_80store_on:
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop					# already set
		movl	$0, SYMBOL_NAME(read_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

/* RAMRD aux. if 80STORE off then text/hires display pages are in aux
	memory. */
ENTRY(iie_ramrd_aux)
		testb	$0xFF, SYMBOL_NAME(ramrd_flag)		# already off?
		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(ramrd_flag)
		movl	BANK2, SYMBOL_NAME(ramrd_offset)
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# 80store?
		jnz	iie_ramrd_aux_80store_on
		movl	BANK2, SYMBOL_NAME(read_text_page_offset)
		movl	BANK2, SYMBOL_NAME(read_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret
iie_ramrd_aux_80store_on:
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop					# already set
		movl	BANK2, SYMBOL_NAME(read_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

ENTRY(iie_check_ramrd)
		movb	SYMBOL_NAME(ramrd_flag), %al
		ret

/* RAMWRT main. if 80STORE off then text/hires display pages are in
	main memory. */
ENTRY(iie_ramwrt_main)
		testb	$0xFF, SYMBOL_NAME(ramwrt_flag)		# already on?
		jz	ram_nop
		movb	$0, SYMBOL_NAME(ramwrt_flag)
		movl	$0, SYMBOL_NAME(ramwrt_offset)	# main RAM
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# 80store?
		jnz	iie_ramwrt_main_80store_on
		movl	$0, SYMBOL_NAME(write_text_page_offset)
		movl	$0, SYMBOL_NAME(write_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret
iie_ramwrt_main_80store_on:
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop					# already set
		movl	$0, SYMBOL_NAME(write_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

/* RAMWRT aux. if 80STORE off then write to text/hires display pages
	are in aux memory. */
ENTRY(iie_ramwrt_aux)
		testb	$0xFF, SYMBOL_NAME(ramwrt_flag)		# already off?
		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(ramwrt_flag)
		movl	BANK2, SYMBOL_NAME(ramwrt_offset)	# aux RAM
		testb	$0xFF, SYMBOL_NAME(eighty_store_flag)	# 80store?
		jnz	iie_ramwrt_aux_80store_on
		movl	BANK2, SYMBOL_NAME(write_text_page_offset)
		movl	BANK2, SYMBOL_NAME(write_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret
iie_ramwrt_aux_80store_on:
		testb	$0xFF, SYMBOL_NAME(vmode_hires)
		jnz	ram_nop					# already set
		movl	BANK2, SYMBOL_NAME(write_hires_page_offset)
/* 		call	SYMBOL_NAME(update_video_screen)	# redraw */
		ret

ENTRY(iie_check_ramwrt)
		movb	SYMBOL_NAME(ramwrt_flag), %al
		ret
	
ENTRY(iie_altzp_main)
		movb	$0, SYMBOL_NAME(altzp_flag)
		movl	$0, SYMBOL_NAME(lc_offset)
		movl	$0, SYMBOL_NAME(zp_offset)
		andl	$0xFFFF, SP_Reg
		ret

ENTRY(iie_altzp_aux)
		movb	$0x80, SYMBOL_NAME(altzp_flag)
		movl	$0x2000, SYMBOL_NAME(lc_offset)
		movl	BANK2, SYMBOL_NAME(zp_offset)
		orl	BANK2, SP_Reg
		ret

ENTRY(iie_check_altzp)
		movb	SYMBOL_NAME(altzp_flag), %al
		ret

ENTRY(iie_80col_off)
    		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jz	ram_nop
		movb	$0, SYMBOL_NAME(eighty_col_flag)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	SYMBOL_NAME(update_video_screen)# redraw text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	SYMBOL_NAME(update_video_screen)# redraw text
		ret

ENTRY(iie_80col_on)
    		testb	$0xFF, SYMBOL_NAME(eighty_col_flag)
    		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(eighty_col_flag)
		testb	$0xFF, SYMBOL_NAME(vmode_text)	# Text mode?
		jnz	SYMBOL_NAME(update_video_screen)# redraw text
		testb	$0xFF, SYMBOL_NAME(vmode_mixed) # Mixed mode?
		jnz	SYMBOL_NAME(update_video_screen)# redraw text
		ret

ENTRY(iie_check_80col)
		movb	SYMBOL_NAME(eighty_col_flag), %al
		ret

ENTRY(iie_altchar_off)
		testb	$0xff, SYMBOL_NAME(altchar_flag)
    		jz	ram_nop
		movb	$0, SYMBOL_NAME(altchar_flag)
		pushal
		call	SYMBOL_NAME(c_set_primary_char)
		popal
		ret

ENTRY(iie_altchar_on)
		testb	$0xff, SYMBOL_NAME(altchar_flag)
    		jnz	ram_nop
		movb	$0x80, SYMBOL_NAME(altchar_flag)
		pushal
		call	SYMBOL_NAME(c_set_altchar)
		popal
		ret

ENTRY(iie_check_altchar)
		movb	SYMBOL_NAME(altchar_flag), %al
		ret

/* unset ioudis flag and read_gc_strobe */
ENTRY(iie_ioudis_off)
		movb	$0, SYMBOL_NAME(ioudis_flag)
		jmp	SYMBOL_NAME(read_gc_strobe)

/* set ioudis flag and read_gc_strobe */
ENTRY(iie_ioudis_on)
		movb	$0x80, SYMBOL_NAME(ioudis_flag)
		jmp	SYMBOL_NAME(read_gc_strobe)

/* check ioudis flag and read_gc_strobe */
ENTRY(iie_check_ioudis)
		call	SYMBOL_NAME(read_gc_strobe)
		movb	SYMBOL_NAME(ioudis_flag), %al
		ret

ENTRY(iie_dhires_on)
		testb	$0xFF, SYMBOL_NAME(ioudis_flag)
		jz	ram_nop
		movb	$1, SYMBOL_NAME(vmode_dhires)
		movb	$0x80, SYMBOL_NAME(dhires_flag)
		call	SYMBOL_NAME(update_video_screen)
		ret

ENTRY(iie_dhires_off)
		testb	$0xFF, SYMBOL_NAME(ioudis_flag)
		jz	ram_nop
		movb	$0, SYMBOL_NAME(vmode_dhires)
		movb	$0, SYMBOL_NAME(dhires_flag)
		call	SYMBOL_NAME(update_video_screen)
		ret

/* check dhires flag and read_gc_strobe */
ENTRY(iie_check_dhires)
		call	SYMBOL_NAME(read_gc_strobe)
		movb	SYMBOL_NAME(dhires_flag), %al
		ret

ENTRY(iie_check_text)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(vmode_text), %ah
		shrw	%ax
		ret

ENTRY(iie_check_mixed)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(vmode_mixed), %ah
		shrw	%ax
		ret

ENTRY(iie_check_page2)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(vmode_page2), %ah
		shrw	%ax
		ret

ENTRY(iie_check_hires)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(vmode_hires), %ah
		shrw	%ax
		ret

ENTRY(iie_check_bank)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(language_current_bank), %ah
		shrw	%ax
		xorb	$0x80, %al	/* I'm backwards */
		ret

ENTRY(iie_check_lcram)
		xorw	%ax, %ax
		movb	SYMBOL_NAME(language_card_read), %ah
		shrw	%ax
		ret

/* HACK not doing anything... */
ENTRY(iie_check_vbl)
		ret

	/* ---------------------------------- */

ENTRY(iie_c3rom_peripheral)
		movb	$0x80, SYMBOL_NAME(c3rom_flag)
		movl	$0, SYMBOL_NAME(c3rom_offset)
		ret

ENTRY(iie_c3rom_internal)
		movb	$0, SYMBOL_NAME(c3rom_flag)
		movl	BANK2, SYMBOL_NAME(c3rom_offset)
		ret

ENTRY(iie_check_c3rom)
		movb	SYMBOL_NAME(c3rom_flag), %al
		ret

ENTRY(iie_cxrom_peripheral)
		movb	$0, SYMBOL_NAME(cxrom_flag)
		movl	$0, SYMBOL_NAME(cxrom_offset)
		ret

ENTRY(iie_cxrom_internal)
		movb	$0x80, SYMBOL_NAME(cxrom_flag)
		movl	BANK2, SYMBOL_NAME(cxrom_offset)
	/* HACK - do we reset c3rom_offset when we go to peripheral? */
		movl	BANK2, SYMBOL_NAME(c3rom_offset)
		ret

ENTRY(iie_check_cxrom)
		movb	SYMBOL_NAME(cxrom_flag), %al
		ret

ENTRY(iie_read_slot3)
		orl	SYMBOL_NAME(c3rom_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

ENTRY(iie_read_slot6)
		orl	SYMBOL_NAME(cxrom_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

ENTRY(iie_read_slotx)
		orl	SYMBOL_NAME(cxrom_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* HACK!!!!! - how does the expansion slot get referenced? need to handle other roms requesting to use this area!!!! */
ENTRY(iie_read_slot_expansion)
		orl	SYMBOL_NAME(c8rom_offset), EffectiveAddr_E
		movb	SYMBOL_NAME(apple_ii_64k)(,EffectiveAddr_E,1), %al
		andl	$0xFFFF, EffectiveAddr_E
		ret

/* HACK!!!!! - how does the expansion slot get referenced? */
ENTRY(iie_disable_slot_expansion)
		ret

#endif /* APPLE_IIE */


#ifdef DEBUGGER
/* -------------------------------------------------------------------------
    do_step() 
	steps the simulation while remaining in the debugger's control
   ------------------------------------------------------------------------- */
ENTRY(do_step)
    		pushal
		ReplaceState
		GetFromPC_B
		jmp	*table_opcodes(,%eax,4)
#endif
