/*
 *  Apple II debugger routines
 *  by Aaron Culliney - chernabog@baldmountain.bbn.com - (C) 1997
 *  This code has nothing to do with my employer, BBN.  It was written
 *  completely on my own time and on my own machine.
 *
 *  debugger.c - Main debugger command routines.
 *
 *  $Id: debugger.c,v 1.4 1997/03/01 19:01:48 chernabog Exp $
 *
 **/

#ifdef DEBUGGER
#include "defs.H"
#include "colors.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>


/* debugger defines */
#define SCREEN_X	41
#define SCREEN_Y	24
#define PROMPT_X	2
#define PROMPT_Y	BUF_Y - 1
#define PROMPT_END_X	BUF_X - 2
#define command_line	command_buf[PROMPT_Y]

/* debugger globals */
static char command_buf[BUF_Y][BUF_X];	/* command line prompt */
char second_buf[BUF_Y][BUF_X];		/* scratch buffer for output */
char lexbuf[BUF_X+2];			/* comman line to be flex'ed */
int num_buffer_lines;			/* num lines of output */
int arg1, arg2;				/* command arguments */

int debug_PC;				/* debugger's program counter */
int debug_EA;				/* debugger's effective addrs */
int debug_SP;				/* debugger's stack pointer */
unsigned char debug_X;			/* debugger's X register */
unsigned char debug_Y;			/* debugger's Y register */
unsigned char debug_A;			/* debugger's accumulator */
unsigned char debug_F;			/* debugger's flag register */
unsigned char debug_no_graphics;	/* enable/disable graphics drawing */
unsigned char debug_no_sound;		/* enable/disable sound */
unsigned char debug_step;		/* we're stepping the simulation */
int debug_scratch;			/* scratch variable */
int step_next;				/* stepping over instructions */

int breakpoints[MAX_BRKPTS];		/* memory breakpoints */
int watchpoints[MAX_BRKPTS];		/* memory watchpoints */

unsigned char watch_write;              /* watchpoint read/write access? */
unsigned char watch_olddata;            /* old byte */
unsigned char watch_data;               /* new byte */
int watch_addrs;                        /* address of tripped watchpoint */

/* in apple2.S */
extern void read_speaker_toggle_pc();
extern void ram_nop();
extern void do_step();
extern void do_write_memory();
extern void do_write_lc();

/* in tokens.l */
extern int yylex();
extern void init_lex(char *buf, int size);

/* in misc.c */
extern short sound_mode;
extern Function table_read_memory[64 K];
extern Function table_write_memory[64 K];
extern unsigned char apple_ii_64k[];
extern unsigned char language_card[], language_bank0[], language_bank1[];
extern unsigned char language_current_bank, language_card_second,
                     language_card_read, language_card_write;

extern void c_setpage(int);
extern void c_setscreen(int);

/* in disk.c */
extern int drive_6, motor_6, ddrw_6, disk_byte_6, volume_6, checksum_6;
extern char file_name_6[2][1024];	/* HACK!!!!!! */
extern int nibblized[], protected_6[], phase_6[], phase_change_6[], sector_6[];
extern long file_size_6[];

/* in interface.c */
extern void c_interface_translate_screen(unsigned char screen[24][41]);
extern void c_interface_exit();
extern int c_mygetch();
extern void c_interface_normal_keyboard_on();
extern void c_interface_textcolor(short , short);
extern void c_interface_print(int, int, unsigned char *);
extern void c_interface_print_char(int, int, unsigned char);

/* opcode struct used for disassembling memory */
struct opcode_struct {
    char code[4];
    char fmt[32];
    unsigned char val;
    int numargs;
};

/* 256 Apple II opcodes.  My convention: undocumented instructions
   have >3char mnemonics */
struct opcode_struct opcodes[256] = {
        { "BRK", "BRK", 0x00, 0 },
	{ "ORA", "ORA ($%02X,X)", 0x01, 1 },
	{ "???", "HANG", 0x02, 0 },		/* undocumented */
	{ "???", "LORX $%02X", 0x03, 1 },	/* undocumented */
	{ "???", "NOP2", 0x04, 0 },		/* undocumented */
	{ "ORA", "ORA $%02X", 0x05, 1 },
	{ "ASL", "ASL $%02X", 0x06, 1 },
	{ "???", "LORZ $%02X", 0x07, 1 },	/* undocumented */
	{ "PHP", "PHP", 0x08, 0 },
	{ "ORA", "ORA #$%02X", 0x09, 1 },
	{ "ASL", "ASL", 0x0A, 0 },
	{ "???", "ANA_ #$%02X", 0x0B, 1 },	/* undocumented */
	{ "???", "NOP3", 0x0C, 0 },		/* undocumented */
	{ "ORA", "ORA $%02X%02X", 0x0D, 2 },
	{ "ASL", "ASL $%02X%02X", 0x0E, 2 },
	{ "???", "LORa $%02X%02X", 0x0F, 2 },	/* undocumented */
	{ "BPL", "BPL $%04X (_$%02X)", 0x10, 3 },
	{ "ORA", "ORA ($%02X),Y", 0x11, 1 },
	{ "???", "HANG", 0x12, 0 },		/* undocumented */
	{ "???", "LORY $%02X", 0x13, 1 },	/* undocumented */
	{ "???", "NOP2", 0x14, 0 },		/* undocumented */
	{ "ORA", "ORA $%02X,X", 0x15, 1 },
	{ "ASL", "ASL $%02X,X", 0x16, 1 },
	{ "???", "LORZX $%02X", 0x17, 1 },	/* undocumented */
	{ "CLC", "CLC", 0x18, 0 },
	{ "ORA", "ORA $%02X%02X,Y", 0x19, 2 },
	{ "???", "NOP_", 0x1A, 0 },		/* undocumented */
	{ "???", "LORy $%02X%02X", 0x1B, 2 },	/* undocumented */
	{ "???", "NOP3", 0x1C, 0 },		/* undocumented */
	{ "ORA", "ORA $%02X%02X,X", 0x1D, 2 },
	{ "ASL", "ASL $%02X%02X,X", 0x1E, 2 },
	{ "???", "LORx $%02X%02X", 0x1F, 2 },	/* undocumented */
	{ "JSR", "JSR %02X%02X", 0x20, 2 },
	{ "AND", "AND ($%02X,X)", 0x21, 1 },
	{ "???", "HANG", 0x22, 0 },		/* undocumented */
	{ "???", "LANX $%02X", 0x23, 1 },	/* undocumented */
	{ "BIT", "BIT $%02X", 0x24, 1 },
	{ "AND", "AND $%02X", 0x25, 1 },
	{ "ROL", "ROL $%02X", 0x26, 1 },
	{ "???", "LANZ $%02X", 0x27, 1 },	/* undocumented */
	{ "PLP", "PLP", 0x28, 0 },
	{ "AND", "AND #$%02X", 0x29, 1 },
	{ "ROL", "ROL", 0x2A, 0 },
	{ "???", "ANB_ #$%02X", 0x2B, 1 },	/* undocumented */
	{ "BIT", "BIT $%02X%02X", 0x2C, 2 },
	{ "AND", "AND $%02X%02X", 0x2D, 2 },
	{ "ROL", "ROL $%02X%02X", 0x2E, 2 },
	{ "???", "LANa $%02X%02X", 0x2F, 2 },	/* undocumented */
	{ "BMI", "BMI $%04X (_$%02X)", 0x30, 3 },
	{ "AND", "AND ($%02X),Y", 0x31, 1 },
	{ "???", "HANG", 0x32, 0 },		/* undocumented */
	{ "???", "LANY $%02X", 0x33, 1 },	/* undocumented */
	{ "???", "NOP2", 0x34, 0 },		/* undocumented */
	{ "AND", "AND $%02X,X", 0x35, 1 },
	{ "ROL", "ROL $%02X,X", 0x36, 1 },
	{ "???", "LANZX $%02X", 0x37, 1 },	/* undocumented */
	{ "SEC", "SEC", 0x38, 0 },
	{ "AND", "AND $%02X%02X,Y", 0x39, 2 },
	{ "???", "NOP_", 0x3A, 0 },		/* undocumented */
	{ "???", "LANy $%02X%02X", 0x3B, 2 },	/* undocumented */
	{ "???", "NOP3", 0x3C, 0 },
	{ "AND", "AND $%02X%02X,X", 0x3D, 2 },
	{ "ROL", "ROL $%02X%02X,X", 0x3E, 2 },
	{ "???", "LANx $%02X%02X", 0x3F, 2 },	/* undocumented */
	{ "RTI", "RTI", 0x40, 0 },
	{ "EOR", "EOR ($%02X,X)", 0x41, 1 },
	{ "???", "HANG", 0x42, 0 },		/* undocumented */
	{ "???", "REOX $%02X", 0x43, 1 },	/* undocumented */
	{ "???", "NOP2", 0x44, 0 },		/* undocumented */
	{ "EOR", "EOR $%02X", 0x45, 1 },
	{ "LSR", "LSR $%02X", 0x46, 1 },
	{ "???", "REOZ $%02X", 0x47, 1 },	/* undocumented */
	{ "PHA", "PHA", 0x48, 0 },
	{ "EOR", "EOR #$%02X", 0x49, 1 },
	{ "LSR", "LSR", 0x4A, 0 },
	{ "???", "RAM_ #$%02X", 0x4B, 1 },	/* undocumented */
	{ "JMP", "JMP %02X%02X", 0x4C, 2 },
	{ "EOR", "EOR %02X%02X", 0x4D, 2 },
	{ "LSR", "LSR %02X%02X", 0x4E, 2 },
	{ "???", "REOa %02X%02X", 0x4F, 2 },	/* undocumented */
	{ "BVC", "BVC $%04X (_$%02X)", 0x50, 3 },
	{ "EOR", "EOR ($%02X),Y", 0x51, 1 },
	{ "???", "HANG", 0x52, 0 },		/* undocumented */
	{ "???", "REOY $%02X", 0x53, 1 },	/* undocumented */
	{ "???", "NOP2", 0x54, 0 },		/* undocumented */
	{ "EOR", "EOR $%02X,X", 0x55, 1 },
	{ "LSR", "LSR $%02X,X", 0x56, 1 },
	{ "???", "REOZX $%02X", 0x57, 1 },	/* undocumented */
	{ "CLI", "CLI", 0x58, 0 },
	{ "EOR", "EOR $%02X%02X,Y", 0x59, 2 },
	{ "???", "NOP_", 0x5A, 0 },		/* undocumented */
	{ "???", "REOy $%02X%02X", 0x5B, 2 },	/* undocumented */
	{ "???", "NOP3", 0x5C, 0 },		/* undocumented */
	{ "EOR", "EOR $%02X%02X,X", 0x5D, 2 },
	{ "LSR", "LSR $%02X%02X,X", 0x5E, 2 },
	{ "???", "REOx $%02X%02X", 0x5F, 2 },	/* undocumented */
	{ "RTS", "RTS", 0x60, 0 },
	{ "ADC", "ADC ($%02X,X)", 0x61, 1 },
	{ "???", "HANG", 0x62, 0 },		/* undocumented */
	{ "???", "RADX $%02X", 0x63, 1 },	/* undocumented */
	{ "???", "NOP2", 0x64, 0 },		/* undocumented */
	{ "ADC", "ADC $%02X", 0x65, 1 },
	{ "ROR", "ROR $%02X", 0x66, 1 },
	{ "???", "RADZ $%02X", 0x67, 1 },	/* undocumented */
	{ "PLA", "PLA", 0x68, 0 },
	{ "ADC", "ADC #$%02X", 0x69, 1 },
	{ "ROR", "ROR", 0x6A, 0 },
	{ "???", "RBM_ #$%02X", 0x6B, 1 },	/* undocumented */
	{ "JMP", "JMP ($%02X%02X)", 0x6C, 2 },
	{ "ADC", "ADC $%02X%02X", 0x6D, 2 },
	{ "ROR", "ROR $%02X%02X", 0x6E, 2 },
	{ "???", "RADa $%02X%02X", 0x6F, 2 },	/* undocumented */
	{ "BVS", "BVS $%04X (_$%02X)", 0x70, 3 },
	{ "ADC", "ADC ($%02X),Y", 0x71, 1 },
	{ "???", "HANG", 0x72, 0 },		/* undocumented */
	{ "???", "RADY $%02X", 0x73, 1 },	/* undocumented */
	{ "???", "NOP2", 0x74, 0 },		/* undocumented */
	{ "ADC", "ADC $%02X,X", 0x75, 1 },
	{ "ROR", "ROR $%02X,X", 0x76, 1 },
	{ "???", "RADZX $%02X", 0x77, 1 },	/* undocumented */
	{ "SEI", "SEI", 0x78, 0 },
	{ "ADC", "ADC $%02X%02X,Y", 0x79, 2 },
	{ "???", "NOP_", 0x7A, 0 },		/* undocumented */
	{ "???", "RADy $%02X%02X", 0x7B, 2 },	/* undocumented */
	{ "???", "NOP3", 0x7C, 0 },		/* undocumented */
	{ "ADC", "ADC $%02X%02X,X", 0x7D, 2 },
	{ "ROR", "ROR $%02X%02X,X", 0x7E, 2 },
	{ "???", "RADx $%02X%02X", 0x7F, 2 },	/* undocumented */
	{ "???", "NOP2", 0x80, 0 },		/* undocumented */
	{ "STA", "STA ($%02X,X)", 0x81, 1 },
	{ "???", "NOP2", 0x82, 0 },		/* undocumented */
	{ "???", "AAXX $%02X", 0x83, 1 },	/* undocumented */
	{ "STY", "STY $%02X", 0x84, 1 },
	{ "STA", "STA $%02X", 0x85, 1 },
	{ "STX", "STX $%02X", 0x86, 1 },
	{ "???", "AAXZ $%02X", 0x87, 1 },	/* undocumented */
	{ "DEY", "DEY", 0x88, 0 },
	{ "???", "NOP2", 0x89, 0 },		/* undocumented */
	{ "TXA", "TXA", 0x8A, 0 },
	{ "???", "XMA_ $%02X", 0x8B, 1 },	/* undocumented */
	{ "STY", "STY $%02X%02X", 0x8C, 2 },
	{ "STA", "STA $%02X%02X", 0x8D, 2 },
	{ "STX", "STX $%02X%02X", 0x8E, 2 },
	{ "???", "AAXa $%02X%02X", 0x8F, 2 },	/* undocumented */
	{ "BCC", "BCC $%04X (_$%02X)", 0x90, 3 },
	{ "STA", "STA ($%02X),Y", 0x91, 1 },
	{ "???", "HANG", 0x92, 0 },		/* undocumented */
	{ "???", "AAXY $%02X", 0x93, 1 },	/* undocumented */
	{ "STY", "STY $%02X,X", 0x94, 1 },
	{ "STA", "STA $%02X,X", 0x95, 1 },
	{ "STX", "STX $%02X,Y", 0x96, 1 },
	{ "???", "AAXZY $%02X", 0x97, 1 },	/* undocumented */
	{ "TYA", "TYA", 0x98, 0 },
	{ "STA", "STA $%02X%02X,Y", 0x99, 2 },
	{ "TXS", "TXS", 0x9A, 0 },
	{ "???", "AXSa $%02X%02X", 0x9B, 2 },	/* undocumented */
	{ "???", "TEYx $%02X%02X", 0x9C, 2 },	/* undocumented */
	{ "STA", "STA $%02X%02X,X", 0x9D, 2 },
	{ "???", "TEXy $%02X%02X", 0x9E, 2 },	/* undocumented */
	{ "???", "TEAy $%02X%02X", 0x9F, 2 },	/* undocumented */
	{ "LDY", "LDY #$%02X", 0xA0, 1 },
	{ "LDA", "LDA ($%02X,X)", 0xA1, 1 },
	{ "LDX", "LDX #$%02X", 0xA2, 1 },
	{ "???", "LAXX $%02X", 0xA3, 1 },	/* undocumented */
	{ "LDY", "LDY $%02X", 0xA4, 1 },
	{ "LDA", "LDA $%02X", 0xA5, 1 },
	{ "LDX", "LDX $%02X", 0xA6, 1 },
	{ "???", "LAXZ $%02X", 0xA7, 1 },	/* undocumented */
	{ "TAY", "TAY", 0xA8, 0 },
	{ "LDA", "LDA #$%02X", 0xA9, 1 },
	{ "TAX", "TAX", 0xAA, 0 },
	{ "???", "AMA_ #$%02X", 0xAB, 1 },	/* undocumented */
	{ "LDY", "LDY $%02X%02X", 0xAC, 2 },
	{ "LDA", "LDA $%02X%02X", 0xAD, 2 },
	{ "LDX", "LDX $%02X%02X", 0xAE, 2 },
	{ "???", "LAXa $%02X%02X", 0xAF, 2 },	/* undocumented */
	{ "BCS", "BCS $%04X (_$%02X)", 0xB0, 3 },
	{ "LDA", "LDA ($%02X),Y", 0xB1, 1 },
	{ "???", "HANG", 0xB2, 0 },		/* undocumented */
	{ "???", "LAXY $%02X", 0xB3, 1 },	/* undocumented */
	{ "LDY", "LDY $%02X,X", 0xB4, 1 },
	{ "LDA", "LDA $%02X,X", 0xB5, 1 },
	{ "LDX", "LDX $%02X,Y", 0xB6, 1 },
	{ "???", "LAXZY $%02X", 0xB7, 1 },	/* undocumented */
	{ "CLV", "CLV", 0xB8, 0 },
	{ "LDA", "LDA $%02X%02X,Y", 0xB9, 2 },
	{ "TSX", "TSX", 0xBA, 0 },
	{ "???", "LASy $%02X%02X", 0xBB, 2 },	/* undocumented */
	{ "LDY", "LDY $%02X%02X,X", 0xBC, 2 },
	{ "LDA", "LDA $%02X%02X,X", 0xBD, 2 },
	{ "LDX", "LDX $%02X%02X,Y", 0xBE, 2 },
	{ "???", "LAXy $%02X%02X", 0xBF, 2 },	/* undocumented */
	{ "CPY", "CPY #$%02X", 0xC0, 1 },
	{ "CMP", "CMP ($%02X,X)", 0xC1, 1 },
	{ "???", "NOP2", 0xC2, 0 },		/* undocumented */
	{ "???", "DCPX $%02X", 0xC3, 1 },	/* undocumented */
	{ "CPY", "CPY $%02X", 0xC4, 1 },
	{ "CMP", "CMP $%02X", 0xC5, 1 },
	{ "DEC", "DEC $%02X", 0xC6, 1 },
	{ "???", "DCPZ $%02X", 0xC7, 1 },	/* undocumented */
	{ "INY", "INY", 0xC8, 0 },
	{ "CMP", "CMP #$%02X", 0xC9, 1 },
	{ "DEX", "DEX", 0xCA, 0 },
	{ "???", "AXM_ #$%02X", 0xCB, 1 },	/* undocumented */
	{ "CPY", "CPY $%02X%02X", 0xCC, 2 },
	{ "CMP", "CMP $%02X%02X", 0xCD, 2 },
	{ "DEC", "DEC $%02X%02X", 0xCE, 2 },
	{ "???", "DCPa $%02X%02X", 0xCF, 2 },	/* undocumented */
	{ "BNE", "BNE $%04X (_$%02X)", 0xD0, 3 },
	{ "CMP", "CMP ($%02X),Y", 0xD1, 1 },
	{ "???", "HANG", 0xD2, 0 },		/* undocumented */
	{ "???", "DCPY $%02X", 0xD3, 1 },	/* undocumented */
	{ "???", "NOP2", 0xD4, 0 },		/* undocumented */
	{ "CMP", "CMP $%02X,X", 0xD5, 1 },
	{ "DEC", "DEC $%02X,X", 0xD6, 1 },
	{ "???", "DCPZX $%02X", 0xD7, 1 },	/* undocumented */
	{ "CLD", "CLD", 0xD8, 0 },
	{ "CMP", "CMP $%02X%02X,Y", 0xD9, 2 },
	{ "???", "NOP_", 0xDA, 0 },		/* undocumented */
	{ "???", "DCPy $%02X%02X", 0xDB, 2 },	/* undocumented */
	{ "???", "NOP3", 0xDC, 0 },		/* undocumented */
	{ "CMP", "CMP $%02X%02X,X", 0xDD, 2 },
	{ "DEC", "DEC $%02X%02X,X", 0xDE, 2 },
	{ "???", "DCPx $%02X%02X", 0xDF, 2 },	/* undocumented */
	{ "CPX", "CPX #$%02X", 0xE0, 1 },
	{ "SBC", "SBC ($%02X,X)", 0xE1, 0 },
	{ "???", "NOP2", 0xE2, 0 },		/* undocumented */
	{ "???", "ISBx $%02X", 0xE3, 1 },	/* undocumented */
	{ "CPX", "CPX $%02X", 0xE4, 1 },
	{ "SBC", "SBC $%02X", 0xE5, 1 },
	{ "INC", "INC $%02X", 0xE6, 1 },
	{ "???", "ISBZ $%02X", 0xE7, 1 },	/* undocumented */
	{ "INX", "INX", 0xE8, 0 },
	{ "SBC", "SBC #$%02X", 0xE9, 1 },
	{ "NOP", "NOP", 0xEA, 0 },
	{ "???", "ZBC_ #$%02X", 0xEB, 1 },	/* undocumented */
	{ "CPX", "CPX $%02X%02X", 0xEC, 2 },
	{ "SBC", "SBC $%02X%02X", 0xED, 2 },
	{ "INC", "INC $%02X%02X", 0xEE, 2 },
	{ "???", "ISBa $%02X%02X", 0xEF, 2 },	/* undocumented */
	{ "BEQ", "BEQ $%04X (_$%02X)", 0xF0, 3 },
	{ "SBC", "SBC ($%02X),Y", 0xF1, 1 },
	{ "???", "HANG", 0xF2, 0 },		/* undocumented */
	{ "???", "ISBY $%02X", 0xF3, 1 },	/* undocumented */
	{ "???", "NOP2", 0xF4, 0 },		/* undocumented */
	{ "SBC", "SBC $%02X,X", 0xF5, 1 },
	{ "INC", "INC $%02X,X", 0xF6, 1 },
	{ "???", "ISBZX $%02X", 0xF7, 1 },	/* undocumented */
	{ "SED", "SED", 0xF8, 0 },
	{ "SBC", "SBC $%02X%02X,Y", 0xF9, 2 },
	{ "???", "NOP_", 0xFA, 0 },		/* undocumented */
	{ "???", "ISBy $%02X%02X", 0xFB, 2 },	/* undocumented */
	{ "???", "NOP3", 0xFC, 0 },		/* undocumented */
	{ "SBC", "SBC $%02X%02X,X", 0xFD, 2 },
	{ "INC", "INC $%02X%02X,X", 0xFE, 2 },
	{ "???", "ISBx $%02X%02X", 0xFF, 2 }	/* undocumented */
};

/* -------------------------------------------------------------------------
    dump_mem () - hexdump of memory to debug console
	we DO NOT wrap the display : 0xffff -> 0x0 (programs can't wrap?)
   ------------------------------------------------------------------------- */

void dump_mem(int addrs, int len, int lc, int do_ascii) {
    int i, j, mod, op, end;

    if (lc) {
	if ((addrs >= 0xd000) && (addrs <= 0xffff)) addrs -= 0xd000;
	if ((addrs < 0) || (addrs > 0x2fff)) addrs = 0;
    }
    else if ((addrs < 0) || (addrs > 0xffff)) addrs = debug_PC;

    if ((len < 1) || (len > 256)) len = 256;
    if (do_ascii && (len > 128)) len = 128;


    /* save hexdump in second_buf */
    end = (lc) ? 0x3000 : 0x10000;
    for (i = num_buffer_lines-1, j = 0; ((j < len) && (addrs+j < end)); j++) {

	mod = j % (16 >> do_ascii);

	if (lc) {
	    op = (addrs+j >= 0x1000) ? language_card[(addrs+j)-0x1000]
		: (lc == 1) ? language_bank0[addrs+j]
		: language_bank1[addrs+j] ;
	}
	else op = apple_ii_64k[addrs+j];

	if (!mod) {
	    if (++i) {
		for (mod=0; mod<BUF_X; mod++)
		    if (second_buf[i-1][mod] == '\0')
			second_buf[i-1][mod] = ' ';
	    }
	    memset(second_buf[i], ' ', BUF_X); mod = 0;
	    sprintf(second_buf[i], "%04X:%02X", addrs+j, op);
	    if (do_ascii) sprintf(second_buf[i]+23, "%c",
				  (op > 31 && op < 127) ? op : '.');
	    continue;
	}
	sprintf(second_buf[i]+5+mod*2, "%02X", op);
	if (do_ascii) sprintf(second_buf[i]+23+mod, "%c",
			      (op > 31 && op < 127) ? op : '.');
    }
    for (mod=0; mod<BUF_X; mod++)
	if (second_buf[i][mod] == '\0') second_buf[i][mod] = ' ';
    num_buffer_lines = i + 1;
}


/* -------------------------------------------------------------------------
    search_mem () - search memory for bytes
   ------------------------------------------------------------------------- */
void search_mem(char *hexstr, int lc) {
    int i = 0, j = 0, end, op;
    static char scratch[3];
    unsigned char byte;

    end = (lc) ? 0x3000 : 0x10000;

    /* iterate over memory */
    for (i = 0; i < end; i++) {
	strncpy(scratch, hexstr+j, 2);		/* extract a byte */
	byte = (unsigned char) strtol(scratch, (char**)NULL, 16);

	if (lc) {
	    op = (i >= 0x1000) ? language_card[i-0x1000]
		: (lc == 1) ? language_bank0[i] : language_bank1[i] ;
	}
	else op = apple_ii_64k[i];

	if (byte == op) {		/* matched byte? */
	    ++j;				/* increment */
	    if (!isxdigit(*(hexstr+j))) {	/* end of bytes? */
		/* then we found a match */
		sprintf(second_buf[num_buffer_lines], "%04X: %s",
			i-(j>>1), hexstr);
		num_buffer_lines = (num_buffer_lines + 1) % (BUF_Y-2);
		j = 0; continue;
	    }
	    ++j;
	    if (!isxdigit(*(hexstr+j))) {	/* end of bytes? */
		/* then we found a match */
		sprintf(second_buf[num_buffer_lines], "%04X: %s",
			i-(j>>1)+1, hexstr);
		num_buffer_lines = (num_buffer_lines + 1) % (BUF_Y-2);
		j = 0; continue;
	    }
	    continue;
	}
	j = 0;
    }
}


/* -------------------------------------------------------------------------
    set_mem () - write to apple_ii_64k main memory.
   ------------------------------------------------------------------------- */
void set_mem(int addrs, char *hexstr) {
    static char scratch[3];

    if ((addrs < 0) || (addrs > 0xffff)) {
	sprintf(second_buf[num_buffer_lines++], "invalid address");
	return;
    }

    while (*hexstr) {
	strncpy(scratch, hexstr, 2);
	watch_data = (unsigned char) strtol(scratch, (char**)NULL, 16);
	debug_scratch = addrs;

	/* call the set_memory routine, which knows how to route the
           request */
	do_write_memory();

	++hexstr;
	if (!*hexstr) break;
	++hexstr;
	if (++addrs > 0xffff) return;  /* don't overwrite memory */
    }
}


/* -------------------------------------------------------------------------
    set_lc_mem () - specifically write to apple II language card RAM memory
   ------------------------------------------------------------------------- */
void set_lc_mem(int addrs, int bank, char *hexstr) {
    static char scratch[3];

    if ((addrs >= 0xd000) && (addrs <= 0xffff)) addrs -= 0xd000;
    if ((addrs < 0) || (addrs > 0x2fff)) {
	sprintf(second_buf[num_buffer_lines++], "invalid LC address");
	return;
    }

    while (*hexstr) {
	strncpy(scratch, hexstr, 2);
	watch_data = (unsigned char) strtol(scratch, (char**)NULL, 16);
	watch_addrs = (addrs < 0x1000) ? bank-1 : -1; /* -1, 0, 1 */
	debug_scratch = addrs; debug_scratch += 0xd000;

	/* call the set_memory routine */
	do_write_lc();

	++hexstr;
	if (!*hexstr) break;
	++hexstr;
	if (++addrs > 0x2fff) return;
    }
}


/* -------------------------------------------------------------------------
    disasm () - disassemble 6502 instructions
	we DO NOT wrap the display : 0xffff -> 0x0 (programs can't wrap?)
   ------------------------------------------------------------------------- */

void disasm(int addrs, int len, int lc) {
    static char fmt[64];
    unsigned char op;
    char arg1, arg2;
    int i, j, end;

    if (lc) {
	if ((addrs >= 0xd000) && (addrs <= 0xffff)) addrs -= 0xd000;
	if ((addrs < 0) || (addrs > 0x2fff)) addrs = 0;
    }
    else if ((addrs < 0) || (addrs > 0xffff)) addrs = debug_PC;

    if (len > BUF_Y - 2) len = BUF_Y - 2 - num_buffer_lines;

    /* save hexdump in second_buf */
    end = (lc) ? 0x3000 : 0x10000;
    for (i = num_buffer_lines, j = addrs; ((i<len) && (j<end)); i++, j++) {

	if (lc)
	    op = (j >= 0x1000) ? language_card[j-0x1000]
	        : (lc == 1) ? language_bank0[j] : language_bank1[j];
	else
	    op = apple_ii_64k[j];

	sprintf(fmt, "%04X: %s", j, opcodes[op].fmt);

	switch ((int)opcodes[op].numargs) {
	case 0:					/* no args */
	    sprintf(second_buf[i], fmt);
	    break;

	case 1:					/* 1 arg */
	    if (j == 0xffff) {
		num_buffer_lines = i;
		return;
	    }

	    if (lc)
		arg1 = (j >= 0x1000) ? language_card[++j-0x1000]
	            : (lc == 1) ? language_bank0[++j] : language_bank1[++j];
	    else
		arg1 = apple_ii_64k[++j];

	    sprintf(second_buf[i], fmt, (unsigned char)arg1);
	    break;

	case 2:					/* 2 args */
	    if (j >= 0xfffe) {
		num_buffer_lines = i;
		return;
	    }

	    if (lc) {
		arg1 = (j >= 0x1000) ? language_card[++j-0x1000]
	            : (lc == 1) ? language_bank0[++j] : language_bank1[++j];
		arg2 = (j >= 0x1000) ? language_card[++j-0x1000]
	            : (lc == 1) ? language_bank0[++j] : language_bank1[++j];
	    }
	    else {
		arg1 = apple_ii_64k[++j];
		arg2 = apple_ii_64k[++j];
	    }

	    sprintf(second_buf[i], fmt,
		    (unsigned char)arg2, (unsigned char)arg1);
	    break;

	case 3:					/* special branch */
	    if (j == 0xffff) {
		num_buffer_lines = i;
		return;
	    }

	    if (lc)
		arg1 = (j >= 0x1000) ? language_card[++j-0x1000]
	            : (lc == 1) ? language_bank0[++j] : language_bank1[++j];
	    else
		arg1 = apple_ii_64k[++j];

	    if (arg1 < 0) {
	      sprintf(second_buf[i], fmt,
		      j + arg1 + 1, (unsigned char)((~arg1) + 1));
	      second_buf[i][17] = '-';		/* branch back */
	    }
	    else {
	      sprintf(second_buf[i], fmt,
		      j + arg1 + 1, (unsigned char)arg1+1);
	      second_buf[i][17] = '+';		/* branch ahead */
	    }
	    break;

	default:				/* shouldn't happen */
	    sprintf(second_buf[i], "args to opcode incorrect!");
	    break;
	}
    }
    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    show_regs () - shows 6502 registers
   ------------------------------------------------------------------------- */

void show_regs() {
    sprintf(second_buf[num_buffer_lines++], "PC = %04X EA = %04X SP = %04X",
	    debug_PC, debug_EA, debug_SP);
    sprintf(second_buf[num_buffer_lines++],
	    "X = %02X Y = %02X A = %02X F = %02X",
	    debug_X, debug_Y, debug_A, debug_F);

    memset(second_buf[num_buffer_lines], ' ', BUF_X);
    if (debug_F & C_Flag) second_buf[num_buffer_lines][0]='C';
    if (debug_F & X_Flag) second_buf[num_buffer_lines][1]='X';
    if (debug_F & I_Flag) second_buf[num_buffer_lines][2]='I';
    if (debug_F & V_Flag) second_buf[num_buffer_lines][3]='V';
    if (debug_F & B_Flag) second_buf[num_buffer_lines][4]='B';
    if (debug_F & D_Flag) second_buf[num_buffer_lines][5]='D';
    if (debug_F & Z_Flag) second_buf[num_buffer_lines][6]='Z';
    if (debug_F & N_Flag) second_buf[num_buffer_lines][7]='N';
    
    ++num_buffer_lines;
}

/* -------------------------------------------------------------------------
    will_branch () = will instruction branch?
		-1 - n/a
		0  - no it won't
		>0  - yes it will
   ------------------------------------------------------------------------- */
int will_branch() {
    unsigned char op = apple_ii_64k[debug_PC];
    
    switch (op) {
    case 0x10:				/* BPL */
      return (int) !(debug_F & N_Flag);
    case 0x30:				/* BMI */
      return (int) (debug_F & N_Flag);
    case 0x50:				/* BVC */
      return (int) !(debug_F & V_Flag);
    case 0x70:				/* BVS */
      return (int) (debug_F & V_Flag);
    case 0x90:				/* BCC */
      return (int) !(debug_F & C_Flag);
    case 0xb0:				/* BCS */
      return (int) (debug_F & C_Flag);
    case 0xd0:				/* BNE */
      return (int) !(debug_F & Z_Flag);
    case 0xf0:				/* BEQ */
      return (int) (debug_F & Z_Flag);
    }

    return -1;
}


/* -------------------------------------------------------------------------
    set_halt () = set a breakpoint or watchpoint in memory
	type = points to "watchpoints" or "breakpoints" array
   ------------------------------------------------------------------------- */
void set_halt(int *type, int addrs) {
    int i;

    for (i = 0; i < MAX_BRKPTS; i++) {
	if (type[i] == -1) {
	    type[i] = addrs;
	    sprintf(second_buf[num_buffer_lines++], "set at %04X", addrs);
	    return;
	}
    }
    sprintf(second_buf[num_buffer_lines++], "too many!");
}

/* -------------------------------------------------------------------------
    clear_halt () = unset a critical breakpoint or watchpoint in memory
	type = points to "watchpoints" or "breakpoints" array
	pt = (pt - 1) into type.  0 indicates clear all.
   ------------------------------------------------------------------------- */
void clear_halt(int *type, int pt) {
    int i;

    if (!pt) {		/* unset all */
	for (i = 0; i < MAX_BRKPTS; i++)
	    type[i] = -1;
	return;
    }
    type[pt-1] = -1;	/* unset single */
}

/* -------------------------------------------------------------------------
    at_haltpt () - tests if at haltpt
	returns 0 = no breaks or watches
		1 = one break or watchpoint fired
		n = two or more breaks and/or watches fired
   ------------------------------------------------------------------------- */
int at_haltpt() {
    int i;

    for (i = 0; i < MAX_BRKPTS; i++) {

	if (debug_PC == breakpoints[i]) {
	    sprintf(second_buf[num_buffer_lines++], "stopped at %04X",
		    breakpoints[i]);
	}
    }
    for (i = 0; i < MAX_BRKPTS; i++) {
	if (watch_addrs == watchpoints[i]) {
	    if (watch_write)
		sprintf(second_buf[num_buffer_lines++],
			"read: %04X", watchpoints[i]);
	    else sprintf(second_buf[num_buffer_lines++],
			 "wrote: %04X: %02X",
			 watchpoints[i], watch_data);
	}
    }

    return num_buffer_lines;	/* 0 indicates nothing happened */
}

/* -------------------------------------------------------------------------
    show_breakpts () - show breakpoints and watchpoints
   ------------------------------------------------------------------------- */
void show_breakpts() {
    int i=num_buffer_lines, k;

    for (k = 0; k < MAX_BRKPTS; k++) {
	if ((breakpoints[k] >= 0) && (watchpoints[k] >= 0)) {
	    sprintf(second_buf[i++], "break %02d at %04X  watch %02d at %04X",
		    k+1, breakpoints[k], k+1, watchpoints[k]);
	}
	else if (breakpoints[k] >= 0) {
	    sprintf(second_buf[i++], "break %02d at %04X",
		    k+1, breakpoints[k]);
	}
	else if (watchpoints[k] >= 0) {
	    memset(second_buf[i], ' ', BUF_X);
	    sprintf(second_buf[i++]+16, "  watch %02d at %04X",
		    k+1, watchpoints[k]);
	}
    }
    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    show_lc_info () - show language card info
   ------------------------------------------------------------------------- */
void show_lc_info() {
    int i = num_buffer_lines;
    sprintf(second_buf[i++], "bank = %d", language_current_bank+1);
    (language_card_write) ? sprintf(second_buf[i++], "write LC")
	                  : sprintf(second_buf[i++], "LC write protected");
    (language_card_read)  ? sprintf(second_buf[i++], "read LC")
	                  : sprintf(second_buf[i++], "read ROM");
    sprintf(second_buf[i++], "second = %d", language_card_second);
    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    show_disk_info () - disk II info
   ------------------------------------------------------------------------- */
void show_disk_info() {
    static char tmp[32];
    int i = num_buffer_lines, len = 0, off = 19;

    /* generic information */
    sprintf(second_buf[i++], "drive %s", (drive_6) ? "B" : "A");
    sprintf(second_buf[i++], "motor %s", (motor_6) ? "off" : "on");
    sprintf(second_buf[i++], "%s", (ddrw_6) ? "write" : "read");
    sprintf(second_buf[i++], "byte = %02X", disk_byte_6);
    sprintf(second_buf[i++], "volume = %d", volume_6);
    sprintf(second_buf[i++], "checksum = %d", checksum_6);

    sprintf(second_buf[i++], "-------------------------------------");

    /* drive / image specific information */
    len = strlen(file_name_6[0]);
    if (len) while ((--len) && (file_name_6[0][len] != '/'));
    strncpy(tmp, file_name_6[0] + len + 1, 31);
    memset(second_buf[i], ' ', BUF_X);
    *(second_buf[i] + sprintf(second_buf[i], "%s", tmp)) = ' ';

    len = strlen(file_name_6[1]);
    if (len) while ((--len) && (file_name_6[1][len] != '/'));
    strncpy(tmp, file_name_6[1] + len + 1, 31);
    sprintf(second_buf[i++]+off, "%s", tmp);

    memset(second_buf[i], ' ', BUF_X);
    *(second_buf[i] + sprintf(second_buf[i], "%s %d bytes",
	(nibblized[0]) ? ".nib" : ".dsk", (int)file_size_6[0])) = ' ';
    sprintf(second_buf[i++]+off, "%s %d bytes",
	    (nibblized[1]) ? ".nib" : ".dsk", (int)file_size_6[1]);

    memset(second_buf[i], ' ', BUF_X);
    *(second_buf[i] + sprintf(second_buf[i], "write %s",
	    (protected_6[0]) ? "protected" : "enabled")) = ' ';
    sprintf(second_buf[i++]+off, "write %s",
	    (protected_6[1]) ? "protected" : "enabled");

    memset(second_buf[i], ' ', BUF_X);
    *(second_buf[i] + sprintf(second_buf[i], "phase %d %s", phase_6[0],
	    (phase_change_6[0]) ? "(new)" : "")) = ' ';
    sprintf(second_buf[i++]+off, "phase %d %s", phase_6[1],
	    (phase_change_6[1]) ? "(new)" : "");

    memset(second_buf[i], ' ', BUF_X);
    if (!nibblized[0]) {
	*(second_buf[i] + sprintf(second_buf[i], "sector %d",
				  sector_6[0])) = ' ';
	if (nibblized[1]) ++i;
    }
    if (!nibblized[1])
	sprintf(second_buf[i++]+off, "sector %d", sector_6[1]);

    num_buffer_lines = i;
}

/* -------------------------------------------------------------------------
    end_step () - finish a stepping command
	display the next instruction, and tell whether it will branch
   ------------------------------------------------------------------------- */
void end_step() {
    int branch;

    disasm(debug_PC, 1, 0);		/* show next instruction */
    branch = will_branch();		/* test if it will branch */
    if (branch == -1) return;		/* n/a */
    sprintf(second_buf[num_buffer_lines++], "%s",
	    (branch) ? "will branch" : "will not branch");
}


/* -------------------------------------------------------------------------
    c_do_step () - step into or step over commands
   ------------------------------------------------------------------------- */
void c_do_step(int step_count) {
    char ch;
    int step_frame = 0;

    /* do step while step_count AND no breaks AND no keypress */
    do {
	if (step_next && apple_ii_64k[debug_PC] == 0x20) {
	    do {
		if (apple_ii_64k[debug_PC] == 0x20) ++step_frame;/* JSR */
		if (apple_ii_64k[debug_PC] == 0x60) --step_frame;/* RTS */
		do_step();
	    } while (((ch = c_mygetch()) == -1) && !at_haltpt() && step_frame);
	}
	else do_step();			/* step one instruction */
    } while (--step_count && !at_haltpt() && (c_mygetch() == -1));

    end_step();				/* print location */
}

/* -------------------------------------------------------------------------
    display_help ()
	show quick reference command usage
   ------------------------------------------------------------------------- */
void display_help() {
    /*                     "|||||||||||||||||||||||||||||||||||||" */
    int i = num_buffer_lines;
    sprintf(second_buf[i++], "d{is} {lc1|lc2} {addr} {+}{len}  ");
    sprintf(second_buf[i++], "m{em} {lc1|lc2} {addr} {+}{len}  ");
    sprintf(second_buf[i++], "a{scii} {lc1|lc2} {addr} {+}{len}");
    sprintf(second_buf[i++], "r{egs}                           ");
    sprintf(second_buf[i++], "<addr> {lc1|lc2} : <byteseq>     ");
    sprintf(second_buf[i++], "(s{tep} | n{ext}) {len}          ");
    sprintf(second_buf[i++], "f{inish}                         ");
    sprintf(second_buf[i++], "u{ntil}                          ");
    sprintf(second_buf[i++], "g{o} {addr}                      ");
    sprintf(second_buf[i++], "sea{rch} {lc1|lc2} <bytes>       ");
    sprintf(second_buf[i++], "(b{reak} | w{atch}) {addr}       ");
    sprintf(second_buf[i++], "(c{lear} | i{gnore}) {num}       ");
    sprintf(second_buf[i++], "sta{tus}                         ");
    sprintf(second_buf[i++], "l{ang}                           ");
    sprintf(second_buf[i++], "dr{ive}                          ");
    sprintf(second_buf[i++], "gr{aphics}                       ");
    sprintf(second_buf[i++], "so{und}                          ");
    sprintf(second_buf[i++], "(? | h{elp})                     ");
    num_buffer_lines = i;
}


/* -------------------------------------------------------------------------
    c_do_debug_command ()
	perform a debugger command
   ------------------------------------------------------------------------- */

void c_do_debug_command() {
    int i = 0, j = 0, k = 0;
    
    /* reset key local vars */
    debug_step = 0;
    step_next = 0;
    num_buffer_lines = 0;

    /* call lex to perform the command.*/
    strncpy(lexbuf, command_line + PROMPT_X, BUF_X);
    init_lex(lexbuf, BUF_X+2);
    yylex();

    /* set up to copy results into main buffer */
    if (num_buffer_lines >= PROMPT_Y) {
	k = BUF_Y - PROMPT_Y;
    } else {
	/* scroll buffer */
	for (i = 0, j = 0; i < PROMPT_Y - num_buffer_lines; i++, j = 0) {
	    memcpy(command_buf[i], command_buf[num_buffer_lines+1+i], BUF_X);
	    while ((j < BUF_X) && (command_buf[i][j] != '\0')) j++;
	    memset (command_buf[i] + j, ' ', BUF_X - j);
	    command_buf[i][BUF_X - 1] = '\0';
	}
    }
    
    /* copy the debug results into debug console window. change '\0's
       to ' 's and cap with a single '\0' */
    while (i < PROMPT_Y) {
	j = 0;
	memcpy(command_buf[i], second_buf[k++], BUF_X);
	while ((j < BUF_X) && (command_buf[i][j] != '\0')) ++j;
	memset(command_buf[i] + j, ' ', BUF_X - j);
	command_buf[i++][BUF_X - 1] = '\0';
    }

    /* new prompt */
    memset(command_line, ' ', BUF_X);
    command_line[0] = '>';
    command_line[BUF_X - 1] = '\0';

    /* display the new information */
    c_interface_textcolor(COLOR_LIGHT_GREEN, 0);
    for (i=0; i<BUF_Y; i++) {
	c_interface_print(1, 1+i, command_buf[i]);
    }

    /* reset before we return */
    num_buffer_lines = 0;
    debug_step = 0;
}

/* -------------------------------------------------------------------------
    c_do_debugging()
	main debugging console
   ------------------------------------------------------------------------- */

void c_do_debugging() {
    static unsigned char screen[SCREEN_Y][SCREEN_X] =
    { "||||||||||||||||||||||||||||||||||||||||",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "|                                      |",
      "||||||||||||||||||||||||||||||||||||||||" };

    static char lex_initted = 0;

    int	i;
    char ch;
    int command_pos = PROMPT_X;

    debug_step = 0;

    /* initialize the buffers */
    for (i=0; i<BUF_Y; i++) {
	memset(command_buf[i], '\0', BUF_X);
	memset(second_buf [i], '\0', BUF_X);
    }
    memset(command_line, ' ', BUF_X);
    command_line[0] = '>';
    command_line[BUF_X - 1] = '\0';

    /* initialize the lexical scanner */
    if (!lex_initted) {
	memset(lexbuf, '\0', BUF_X+2);
	/*init_lex(lexbuf, BUF_X+2);*/
	lex_initted = 1;
    }

    c_setpage( 0 );
    c_setscreen( 0 );
    c_interface_translate_screen( screen );
    c_interface_textcolor( COLOR_LIGHT_RED, 0 );
    for (i = 0; i < 24; i++)
	c_interface_print(0, i, screen[ i ] );
    c_interface_normal_keyboard_on();

    for (;;) {
	/* print command line */
	c_interface_textcolor(COLOR_LIGHT_GREEN, 0);
	c_interface_print(1, 1+PROMPT_Y, command_line);

	/* highlight cursor */
	c_interface_textcolor(COLOR_LIGHT_GREEN, COLOR_MEDIUM_BLUE);
	(command_line[command_pos])
	    ? c_interface_print_char(1+command_pos, 1+PROMPT_Y,
				     command_line[command_pos])
	    : c_interface_print_char(1+command_pos, 1+PROMPT_Y, 
				     command_line[command_pos]);
	c_interface_textcolor(COLOR_LIGHT_GREEN, 0);

	while ((ch = c_mygetch()) == -1) ;

	if (ch == 27) {
	    char ch = c_mygetch();
	    if (ch == '[') {
		char ch = c_mygetch();
		switch(ch) {
		case 'C':	/* right arrow */
		case 'D':	/* left arrow */
		default:
		    continue;
		}
	    }
	    else if (ch == -1) {
		debug_step = 0;
		debug_no_graphics = 0;
		debug_no_sound = 0;
		for (i = 0xC030; i < 0xC040; i++)
		    table_read_memory[i] = table_write_memory[i] =
			sound_mode ? read_speaker_toggle_pc : ram_nop;
		c_interface_exit();
		return;
	    }
	}
	else {
	    /* backspace */
	    if ((ch == 127 || ch == 8) &&
		(command_pos > PROMPT_X)) {
		command_line[--command_pos] = ' ';
	    }
	    /* return */
	    else if (ch == 13) {
		command_line[command_pos] = '\0';
		c_do_debug_command();
		command_pos = PROMPT_X;
	    }
	    /* normal character */
	    else if ((ch >= ' ') && (ch < 127) &&
		     (command_pos < PROMPT_END_X)) {
		command_line[command_pos++] = ch;
	    }
	}
    }
}

#endif /* DEBUGGER */
