/* 
 * Apple // emulator for Linux: Miscellaneous support
 *
 * Copyright 1994 Alexander Jean-Claude Bottema
 * Copyright 1995 Stephen Lee
 * Copyright 1997, 1998 Aaron Culliney
 * Copyright 1998, 1999 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/ioctl.h>

#include "misc.h"
#include "apple2.h"
#include "video.h"
#include "disk.h"
#include "interface.h"
#include "keys.h"
#ifdef DEBUGGER
#include "debug.h"
#endif

/* in joystick.c */
#ifdef PC_JOYSTICK
extern int js_center_x, js_center_y, js_max_x, js_max_y, js_min_x, js_min_y;
extern long js_timelimit;
#endif


/* different than in defs.h! */
#define C_Flag_6502	0x1
#define X_Flag_6502	0x20
#define I_Flag_6502	0x4
#define V_Flag_6502	0x40
#define B_Flag_6502	0x10
#define D_Flag_6502	0x8
#define Z_Flag_6502	0x2
#define N_Flag_6502	0x80

/* ----------------------------------
    internal apple2 variables
   ---------------------------------- */

static unsigned char	apple_ii_rom[12288];
static unsigned char	slot6_rom[256];
#ifdef APPLE_IIE
static unsigned char	apple_iie_rom[32768];		/* //e */
#endif

#ifdef _640x400
unsigned char	intermediate_font[0x800*8];/* NAMEHACK used when in 80 col mode */
#else
#define		intermediate_font expanded_font
#endif

/* in prefs.l */
extern void lowercase_string(char *s);
extern FILE *prefsin;			/* yyin */
extern int prefslex();			/* yylex() */
extern int line_num;

#ifdef DEBUGGER
/* in debugger.c */
extern int breakpoints[];
extern int watchpoints[];
extern int op_breakpoints[256];
extern unsigned char watch_data;	/* new byte */
extern int watch_addrs;			/* address of tripped watchpoint */
#endif

/* 40col/80col/lores/hires/dhires line offsets */
unsigned short video_line_offset[24] =
		   { 0x000, 0x080, 0x100, 0x180, 0x200, 0x280, 0x300, 0x380,
		     0x028, 0x0A8, 0x128, 0x1A8, 0x228, 0x2A8, 0x328, 0x3A8,
		     0x050, 0x0D0, 0x150, 0x1D0, 0x250, 0x2D0, 0x350, 0x3D0 };

unsigned char dhires_colors1[256] = {
    0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9,0xa,0xb,0xc,0xd,0xe,0xf,
    0x0,0x1,0x3,0x3,0x5,0x5,0x7,0x7,0x9,0x9,0xb,0xb,0xd,0xd,0xf,0xf,
    0x0,0x1,0x2,0x3,0x6,0x5,0x6,0x7,0xa,0x9,0xa,0xb,0xe,0xd,0xe,0xf,
    0x0,0x1,0x3,0x3,0x7,0x5,0x7,0x7,0xb,0x9,0xb,0xb,0xf,0xd,0xf,0xf,
    0x0,0x1,0x2,0x3,0x4,0x4,0x6,0x7,0x8,0x9,0xa,0xb,0xc,0xd,0xe,0xf,
    0x0,0x1,0x3,0x3,0x5,0x5,0x7,0x7,0xd,0x9,0xb,0xb,0xd,0xd,0xf,0xf,
    0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0xe,0x9,0xa,0xb,0xe,0xd,0xe,0xf,
    0x0,0x1,0x7,0x3,0x7,0x5,0x7,0x7,0xf,0x9,0xb,0xb,0xf,0xd,0xf,0xf,
};

unsigned char dhires_colors2[256] = {
    0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x8,0x8,0x0,0xb,0x8,0xd,0x0,0x0,
    0x1,0x1,0x1,0x1,0x0,0x5,0x1,0x1,0x0,0x9,0xb,0xb,0x0,0xd,0xf,0xf,
    0x0,0x1,0x2,0x2,0x2,0x5,0x2,0x2,0x0,0xa,0xa,0xa,0xe,0xd,0x2,0x2,
    0x3,0x3,0x3,0x3,0x7,0x5,0x7,0x7,0x0,0xb,0xb,0xb,0xf,0xd,0xf,0xf,
    0x0,0x0,0x4,0x0,0x4,0x4,0x4,0x4,0xc,0x8,0x4,0x8,0xc,0xd,0x4,0x4,
    0x5,0x5,0x5,0x5,0x5,0x5,0x5,0x5,0xd,0x4,0x4,0x4,0xd,0xd,0x4,0x4,
    0x6,0x6,0x6,0x2,0xe,0x6,0x6,0x6,0xe,0xe,0xa,0xa,0xe,0x6,0xe,0x6,
    0x7,0x7,0x7,0x7,0x7,0x7,0x7,0x7,0xf,0xf,0xb,0xb,0xf,0xf,0xf,0xf,
};

#ifdef APPLE_IIE

/* -------------------------------------------------------------------------
    c_initialize_dhires_values()
   ------------------------------------------------------------------------- */
void c_initialize_dhires_values() {
    int i;
/*     int value, v; */
/*     unsigned char locolor, hicolor; */

    /* precalculate the colors for all the 256*8 bit combinations. */
/*     for (value = 0x00, v = 0; value <= 0xFF; value++) { */
/* 	locolor = (value & 0x0F) | 0x10; */
/* 	hicolor = (value << 4) | 0x10; */

/* 	dhires_colors[v++] = locolor; */
/* 	dhires_colors[v++] = locolor; */
/* 	dhires_colors[v++] = locolor; */
/* 	dhires_colors[v++] = locolor; */

/* 	dhires_colors[v++] = hicolor; */
/* 	dhires_colors[v++] = hicolor; */
/* 	dhires_colors[v++] = hicolor; */
/* 	dhires_colors[v++] = hicolor; */
/*     } */

    for (i = 0; i < 0x80; i++) {
	dhires_colors1[i+0x80] = dhires_colors1[i];
	dhires_colors2[i+0x80] = dhires_colors2[i];
    }
}
#endif

/* -------------------------------------------------------------------------
    c_initialize_hires_values()
   ------------------------------------------------------------------------- */

void c_initialize_hires_values()
{
    int     value, b, v, e, /*color_toggle,*/ last_not_black;

    /* precalculate the colors for all the 256*8 bit combinations. */
    for (value = 0x00; value <= 0xFF; value++) {
	for (e = value * 8, last_not_black = 0, v = value, b = 0;
	     b < 7; b++, v >>= 1, e++)
	{
	    if (v & 1)
	    {
		expanded_col_hires_even[ e ] = last_not_black ?
					    COLOR_LIGHT_WHITE :
					((b & 1) ?
					    ((value & 0x80) ?
					         COLOR_LIGHT_RED :
						 COLOR_LIGHT_GREEN) :
					    ((value & 0x80) ?
						 COLOR_LIGHT_BLUE :
						 COLOR_LIGHT_PURPLE));

		expanded_col_hires_odd[ e ] = last_not_black ?
					    COLOR_LIGHT_WHITE :
					((b & 1) ?
					    ((value & 0x80) ?
					         COLOR_LIGHT_BLUE :
						 COLOR_LIGHT_PURPLE) :
					    ((value & 0x80) ?
						 COLOR_LIGHT_RED :
						 COLOR_LIGHT_GREEN));

		if (last_not_black && b > 0)
		    expanded_col_hires_even[ e - 1 ] = COLOR_LIGHT_WHITE,
		    expanded_col_hires_odd[ e - 1 ] = COLOR_LIGHT_WHITE;

		last_not_black = 1;
	    }
	    else
		expanded_col_hires_even[ e ] = COLOR_BLACK,
		expanded_col_hires_odd[ e ] = COLOR_BLACK,
	        last_not_black = 0;
	}
    }
    if (color_mode == 0) {	/* Black and White */
        for (value = 0x00; value <= 0xFF; value++) {
	    for (b = 0, e = value * 8; b < 7; b++, e++)	{
		if (expanded_col_hires_even[ e ] != COLOR_BLACK)
		    expanded_col_hires_even[ e ] = COLOR_LIGHT_WHITE;
		if (expanded_col_hires_odd[ e ] != COLOR_BLACK)
		    expanded_col_hires_odd[ e ] = COLOR_LIGHT_WHITE;
	    }
	}
    }
    else if (color_mode == LAZY_INTERP) /* Lazy Interpolated color */
    {
        for (value = 0x00; value <= 0xFF; value++)
        {
            for (b = 1, e = value * 8 + 1; b <= 5; b += 2, e += 2)
            {
                if (expanded_col_hires_even[ e ] == COLOR_BLACK &&
                    expanded_col_hires_even[ e - 1 ] != COLOR_BLACK &&
                    expanded_col_hires_even[ e + 1 ] != COLOR_BLACK)
                      expanded_col_hires_even[ e ] =
                      expanded_col_hires_even[ e - 1 ];

                if (expanded_col_hires_odd[ e ] == COLOR_BLACK &&
                    expanded_col_hires_odd[ e - 1 ] != COLOR_BLACK &&
                    expanded_col_hires_odd[ e + 1 ] != COLOR_BLACK)
                      expanded_col_hires_odd[ e ] =
                      expanded_col_hires_odd[ e - 1 ];
            }

            for (b = 0, e = value * 8; b <= 6; b += 2, e += 2) {
                if (expanded_col_hires_odd[ e ] == COLOR_BLACK) {
                    if (b > 0 && b < 6) {
                    if (expanded_col_hires_even[e+1] != COLOR_BLACK &&
                        expanded_col_hires_even[e-1] != COLOR_BLACK &&
                        expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE &&
                        expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_even[e] =
                            expanded_col_hires_even[e-1];
                    } else if (b == 0) {
                    if (expanded_col_hires_even[e+1] != COLOR_BLACK &&
                        expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_even[e] =
                            expanded_col_hires_even[e+1];
                    } else {
                    if (expanded_col_hires_even[e-1] != COLOR_BLACK &&
                        expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_even[ e ] =
                            expanded_col_hires_even[ e - 1 ];
                    }
                }

                if (expanded_col_hires_odd[ e ] == COLOR_BLACK) {
                    if (b > 0 && b < 6) {
                    if (expanded_col_hires_odd[e+1] != COLOR_BLACK &&
                        expanded_col_hires_odd[e-1] != COLOR_BLACK &&
                        expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE &&
                        expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_odd[e] =
                            expanded_col_hires_odd[e-1];
                    }
                    else if (b == 0) {
                    if (expanded_col_hires_odd[e+1] != COLOR_BLACK &&
                        expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_odd[e] =
                            expanded_col_hires_odd[e+1];
                    }
                    else if
                        (expanded_col_hires_odd[e-1] != COLOR_BLACK &&
                         expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE)
                        expanded_col_hires_odd[e] =
                            expanded_col_hires_odd[e-1];
                }
            }
        }
    }
    else if (color_mode == INTERP) {	/* Color and strict interpolation */
        for (value = 0x00; value <= 0xFF; value++) {
	    for (b = 1, e = value * 8 + 1; b <= 5; b += 2, e += 2) {
		if (expanded_col_hires_even[e] == COLOR_BLACK) {
		    if (expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
		      expanded_col_hires_even[e] =
			  expanded_col_hires_even[e-1];

		    else if (
			expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] == COLOR_LIGHT_WHITE)
		      expanded_col_hires_even[e] =
			  expanded_col_hires_even[e-1];

		    else if (
			expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
		      expanded_col_hires_even[e] =
			  expanded_col_hires_even[e+1];

		    else if (
			expanded_col_hires_even[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] == COLOR_LIGHT_WHITE)
		      expanded_col_hires_even[e] = (value & 0x80)
			  ? COLOR_LIGHT_BLUE : COLOR_LIGHT_PURPLE;

		}
		if (expanded_col_hires_odd[e] == COLOR_BLACK) {
		    if (expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
		      expanded_col_hires_odd[e] =
			  expanded_col_hires_odd[e-1];

		    else if (
		        expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] == COLOR_LIGHT_WHITE)
		      expanded_col_hires_odd[e] =
			  expanded_col_hires_odd[e-1];

		    else if (
		        expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
		      expanded_col_hires_odd[e] =
			  expanded_col_hires_odd[e+1];

		    else if (
			expanded_col_hires_odd[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] = (value & 0x80)
			    ? COLOR_LIGHT_RED : COLOR_LIGHT_GREEN;
		}
	    }

	    for (b = 0, e = value * 8; b <= 6; b += 2, e += 2) {
		if (expanded_col_hires_even[ e ] == COLOR_BLACK) {
		    if (b > 0 && b < 6) {
		    if (expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
			expanded_col_hires_even[e] =
			    expanded_col_hires_even[e-1];

		    else if (
			expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_even[e] =
			    expanded_col_hires_even[e-1];

		    else if (
			expanded_col_hires_even[e-1] != COLOR_BLACK &&
			expanded_col_hires_even[e+1] != COLOR_BLACK &&
			expanded_col_hires_even[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] != COLOR_LIGHT_WHITE)
			expanded_col_hires_even[e] =
			    expanded_col_hires_even[e+1];

		    else if (
			expanded_col_hires_even[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_even[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_even[e] = (value & 0x80)
			    ? COLOR_LIGHT_RED : COLOR_LIGHT_GREEN;
		    }
		}

		if (expanded_col_hires_odd[e] == COLOR_BLACK) {
		    if (b > 0 && b < 6) {
		    if (expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] =
			    expanded_col_hires_odd[e-1];

		    else if (
			expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] != COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] =
			    expanded_col_hires_odd[e-1];

		    else if (
			expanded_col_hires_odd[e-1] != COLOR_BLACK &&
			expanded_col_hires_odd[e+1] != COLOR_BLACK &&
			expanded_col_hires_odd[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] != COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] =
			    expanded_col_hires_odd[e+1];

		    else if (
			expanded_col_hires_odd[e-1] == COLOR_LIGHT_WHITE &&
			expanded_col_hires_odd[e+1] == COLOR_LIGHT_WHITE)
			expanded_col_hires_odd[e] = (value & 0x80)
			    ? COLOR_LIGHT_BLUE : COLOR_LIGHT_PURPLE;
		    }
		}
	    }
	}
    }
#ifdef _640x400
    for (b=0; b<2048; b++) {
	expanded_col8_hires_even[b] = expanded_col_hires_even[b];
    }
    /* *2 for 640x400 */
    for (b=0, e=0; b<4096; b++, e++) {
	expanded_col_hires_even[b] = expanded_col8_hires_even[e];
	expanded_col_hires_even[++b] = expanded_col8_hires_even[e];
    }
    for (b=0; b<2048; b++) {
	expanded_col8_hires_odd[b] = expanded_col_hires_odd[b];
    }
    /* *2 for 640x400 */
    for (b=0, e=0; b<4096; b++, e++) {
	expanded_col_hires_odd[b] = expanded_col8_hires_odd[e];
	expanded_col_hires_odd[++b] = expanded_col8_hires_odd[e];
    }
#endif
}

#ifdef APPLE_IIE

/* -------------------------------------------------------------------------
    c_set_altchar() - set alternate character set
   ------------------------------------------------------------------------- */
void c_set_altchar()
{
    int i,j,v;

    i = 0x400; 			/* Start with character 0x3f */

    while (i-- != 0x200)	/* Stop at character 0x20 */
    {
        j = 8;
        v = char_rom[i];
        while (j--)
        { 
#ifdef _640x400
	    expanded_font[(i << 4) + (j << 1)] =
	    expanded_font[(i << 4) + (j << 1) + 1] =
#endif /* _640x400 */ 
            intermediate_font[(i << 3) + j] = (v & 128) 
                                          ? COLOR_LIGHT_WHITE : 
                                            COLOR_BLACK;
            v <<= 1;
        }
    }
    update_video_screen();
}
#endif/* APPLE_IIE */

/* -------------------------------------------------------------------------
    c_set_primary_char() - set primary character set
   ------------------------------------------------------------------------- */
void c_set_primary_char() 
{
    int i = 0x2000; /* Start with character 0x80 */

    while (i-- != 0x1000) /* Stop at 0x40 */
    {
#ifdef _640x400
        expanded_font[i << 1] =
        expanded_font[(i << 1) + 1] =
#endif /* _640x400 */ 
        intermediate_font[i] = (intermediate_font[i - 0x1000] 
                               == COLOR_LIGHT_WHITE) 
                             ? COLOR_FLASHING_WHITE : COLOR_BLACK;
    }
    update_video_screen();
}

/* -------------------------------------------------------------------------
    c_initialize_font():     Initialize ROM character table to primary char set
   ------------------------------------------------------------------------- */
void c_initialize_font()
{ 
    int i,j,v;

    i = 0x800; 			/* Start with character 0xff */

    while (i--)			/* Stop at character 0x00 */
    {
        j = 8;
        v = char_rom[i];
        while (j--)
        { 
#ifdef _640x400
	    expanded_font[(i << 4) + (j << 1)] =
	    expanded_font[(i << 4) + (j << 1) + 1] =
#endif /* _640x400 */ 
            intermediate_font[(i << 3) + j] = (v & 128) 
                                          ? COLOR_LIGHT_WHITE : 
                                            COLOR_BLACK;
            v <<= 1;
        }
    }

#ifdef APPLE_IIE
    if (apple_mode != IIE_MODE) 
#endif /* APPLE_IIE */
    {
	i = 0x4000; /* Character 0xff */
        while (i-- != 0x3fc0)
        {
#ifdef _640x400
	     expanded_font[i << 1] =
	     expanded_font[(i << 1) + 1] =
#endif /* _640x400 */
             intermediate_font[i] = 0;
        }
    }

    i = 0x2000; /* Start with character 0x80 */

    while (i-- != 0x1000) /* Stop at 0x40 */
    {
#ifdef _640x400
        expanded_font[i << 1] =
        expanded_font[(i << 1) + 1] =
#endif /* _640x400 */ 
        intermediate_font[i] = (intermediate_font[i - 0x1000] 
                               == COLOR_LIGHT_WHITE) 
                             ? COLOR_FLASHING_WHITE : COLOR_BLACK;
    }
}


/* -------------------------------------------------------------------------
    c_initialize_row_col_tables()
   ------------------------------------------------------------------------- */
void c_initialize_row_col_tables()
{
    int x, y, off, i, yoff;

    /* text page offsets */
    for (y = 0; y < 24; y++) {
#ifdef _640x400
	yoff = (y*16*SCANWIDTH) + 4;
#else
	yoff = (y*8*320)+/*normalize*/(4*320+20);
#endif
	for (x = 0; x < 40; x++) {
#ifdef _640x400
	    text_row_offset[ video_line_offset[ y ] + x ] = yoff + (x*14);
/*	    text_page_cols[ video_line_offset[ y ] + x ] =*/
/*		(unsigned short)(x*7);*/
#else
	    text_row_offset[ video_line_offset[ y ] + x ] = yoff + (x*7);
/*	    text_page_cols[ video_line_offset[ y ] + x ] =*/
/*		(unsigned short)(x*14);*/
#endif
	}
    }

    /* hires page offsets. initialize to invalid values. */
    for (i = 0; i < 8192; i++) {
	(long)(hires_row_offset[i]) = -1;
    }

    for (y = 0; y < 24; y++) {
	for (off = 0; off < 8; off++) {
	    for (x = 0; x < 40; x++) {
#ifdef _640x400
		hires_row_offset[video_line_offset[y] + 0x400*off + x ] =
		    (y*16 + 2*off/* + 8*/) * SCANWIDTH + x*14 + 4;
#else
		hires_row_offset[video_line_offset[y] + 0x400*off + x ] =
		    (y*8 + off + 4) * 320 + x*7 + 20;
#endif
		hires_col_offset[video_line_offset[y] + 0x400*off + x] =
		    (unsigned char)x;
	    }
	}
    }

#ifdef APPLE_IIE
    /* dhires page offsets */
    for (y = 0; y < 24; y++)
	for (off = 0; off < 8; off++)
	    for (x = 0; x < 40; x++)
		dhires_aux_page_offset[video_line_offset[y] + 0x400*off + x]
		    = (y*8 + off + 4) * 640 + x*7*2 + 40;
#endif
}

/* -------------------------------------------------------------------------
    c_initialize_code_tables()
   ------------------------------------------------------------------------- */

void c_initialize_code_tables()
{
    int		i;

    for (i = 0; i < 256; i++)
    {
	unsigned char	val = 0;

	if (i & C_Flag)
	    val |= C_Flag_6502;
	if (i & X_Flag)
	    val |= X_Flag_6502;
	if (i & I_Flag)
	    val |= I_Flag_6502;
	if (i & V_Flag)
	    val |= V_Flag_6502;
	if (i & B_Flag)
	    val |= B_Flag_6502;
	if (i & D_Flag)
	    val |= D_Flag_6502;
	if (i & Z_Flag)
	    val |= Z_Flag_6502;
	if (i & N_Flag)
	    val |= N_Flag_6502;

	table_encode_flags[ i ] = val | 0x20;
	table_decode_flags[ val ] = i;
    }
}


/* -------------------------------------------------------------------------
    c_initialize_tables()
   ------------------------------------------------------------------------- */

void c_initialize_tables() {

    int i, x, y;

    /* reset everything */
    for (i = 0; i < 0x10000; i++) {
#ifdef APPLE_IIE
	if (apple_mode == IIE_MODE) {
	    table_read_memory[i] = iie_read_ram_default;
	    table_write_memory[i] = iie_write_ram_default;
	} else
#endif
        {
	    table_read_memory[i] = read_ram_default;
	    table_write_memory[i] = write_ram_default;
	}
    }

    /* language card read/write area */
    for (i = 0xD000; i < 0xE000; i++) {
#ifdef APPLE_IIE
	if (apple_mode == 2) {
	    table_write_memory[i] =
		iie_write_ram_bank;
	    table_read_memory[i] =
		iie_read_ram_bank;
	} else
#endif
	{
	    table_write_memory[i] =
		write_ram_bank;
	    table_read_memory[i] =
		read_ram_bank;
	}
    }
    for (i = 0xE000; i < 0x10000; i++) {
#ifdef APPLE_IIE
	if (apple_mode ==2) {
	    table_write_memory[i] =
		iie_write_ram_lc;
	    table_read_memory[i] =
		iie_read_ram_lc;
	} else
#endif
	{
	    table_write_memory[i] =
		write_ram_lc;
	    table_read_memory[i] =
		read_ram_lc;
	}
    }
    /* done common initialization */

#ifdef APPLE_IIE
    /* initialize zero-page, //e specific */
    if (apple_mode == IIE_MODE) {
	for (i = 0; i < 0x200; i++) {
	    table_read_memory[i] =
		iie_read_ram_zpage_and_stack;
	    table_write_memory[i] =
		iie_write_ram_zpage_and_stack;
	}
    }
#endif

#ifdef APPLE_IIE
    /* initialize screen holes (page 1 only). page1 screen holes are
       switched with 80STORE.  page 2 screen holes are switched with
       regular RAMRD/RAMWRT. */
    for (i = 0x400; i < 0x800; i++) {
	table_read_memory[i] =
	    iie_read_screen_hole_text_page0;
	table_write_memory[i] =
	    iie_write_screen_hole_text_page0;
    }
    for (i = 0x2000; i < 0x4000; i++) {
	table_read_memory[i] =
	    iie_read_screen_hole_hires_page0;
	table_write_memory[i] =
	    iie_write_screen_hole_hires_page0;
    }
#endif

    /* initialize text/lores & hires graphics */
    for (y = 0; y < 24; y++) {		/* 24 rows */
	for (x = 0; x < 40; x++)	/* 40 cols */
	{
#ifdef APPLE_IIE
	    if (apple_mode == IIE_MODE) {
		/* //e mode: text/lores page 0 */
		table_write_memory[ video_line_offset[ y ] + x + 0x400] =
			(y < 20) ? iie_write_ram_text_page0 :
				   iie_write_ram_text_mixed0;
		table_read_memory[ video_line_offset[ y ] + x + 0x400] =
		    iie_read_ram_text_page0;
	    }
	    else
#endif
	    {
		/* ][+ modes: text/lores page 0 */
		table_write_memory[ video_line_offset[ y ] + x + 0x400] =
			(y < 20) ? write_ram_text_page0 :
				   write_ram_text_mixed0;
	    }

#ifdef APPLE_IIE
	    if (apple_mode == IIE_MODE) {
		table_write_memory[ video_line_offset[ y ] + x + 0x800] =
		    (y < 20) ? iie_write_ram_text_page1 :
		               iie_write_ram_text_mixed1;
	    }
	    else
#endif
	    {
	    /* ][+ modes: text/lores page 1 in main memory */
	    table_write_memory[ video_line_offset[ y ] + x + 0x800] =
			(y < 20) ? write_ram_text_page1 :
				   write_ram_text_mixed1;
	    }

	    for (i = 0; i < 8; i++)
	    {
		/* //e mode: hires/double hires page 0 */
#ifdef APPLE_IIE
		if (apple_mode == IIE_MODE) {
		    table_write_memory[ 0x2000 + video_line_offset[ y ]
					 + 0x400 * i + x ] =
		        (y < 20) ? ((x & 1) ? iie_write_ram_hires_page0_odd :
					      iie_write_ram_hires_page0_even)
				 : ((x & 1) ? iie_write_ram_hires_mixed0_odd :
					      iie_write_ram_hires_mixed0_even);
		    table_read_memory[ 0x2000 + video_line_offset[ y ]
					 + 0x400 * i + x ] =
			iie_read_ram_hires_page0;
		}

		/* ][+ modes: hires page 0 */
		else
#endif
		{
		    table_write_memory[ 0x2000 + video_line_offset[ y ]
					 + 0x400 * i + x ] =
		        (y < 20) ? ((x & 1) ? write_ram_hires_page0_odd :
					      write_ram_hires_page0_even)
				 : ((x & 1) ? write_ram_hires_mixed0_odd :
					      write_ram_hires_mixed0_even);
		}

#ifdef APPLE_IIE
		if (apple_mode == IIE_MODE) {
		    table_write_memory[ 0x4000 + video_line_offset[ y ]
					 + 0x400 * i + x ] =
		        (y < 20) ? ((x & 1) ? iie_write_ram_hires_page1_odd :
					      iie_write_ram_hires_page1_even)
				 : ((x & 1) ? iie_write_ram_hires_mixed1_odd :
					      iie_write_ram_hires_mixed1_even);
		}

		/* ][+ modes: hires page 1 */
		else
#endif
		{
		table_write_memory[ 0x4000 + video_line_offset[ y ]
				        + 0x400 * i + x ] =
		        (y < 20) ? ((x & 1) ? write_ram_hires_page1_odd :
					      write_ram_hires_page1_even)
				 : ((x & 1) ? write_ram_hires_mixed1_odd :
					      write_ram_hires_mixed1_even);
		}
	    }
	}
    }

    /* softswich rom */
    for (i = 0xC000; i < 0xC100; i++) {
	table_read_memory[i] =
	    read_unmapped_softswitch;
	table_write_memory[i] =
	    write_unmapped_softswitch;
    }

    /* slot rom */
    for (i = 0xC100; i < 0xD000; i++) {
	table_read_memory[i] =
#ifdef APPLE_IIE
	    iie_read_ram_default;
#else
	    read_ram_default;
#endif
	table_write_memory[i] =
	    ram_nop;
    }

    /* keyboard data and strobe (READ) */
    for (i = 0xC000; i < 0xC010; i++) {
	table_read_memory[i] =
	    (Function) read_keyboard;
    }
    for (i = 0xC010; i < 0xC020; i++) {
	table_read_memory[i] =
	table_write_memory[i] =
	    read_keyboard_strobe;
    }

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {

	/* RDBNK2 switch */
	table_read_memory[0xC011] =
	    iie_check_bank;

	/* RDLCRAM switch */
	table_read_memory[0xC012] =
	    iie_check_lcram;

	/* 80STORE switch */
	table_write_memory[0xC000] = iie_80store_off;
	table_write_memory[0xC001] = iie_80store_on;
	table_read_memory[0xC018] = iie_check_80store;

	/* RAMRD switch */
	table_write_memory[0xC002] = iie_ramrd_main;
	table_write_memory[0xC003] = iie_ramrd_aux;
	table_read_memory[0xC013] = iie_check_ramrd;

	/* RAMWRT switch */
	table_write_memory[0xC004] = iie_ramwrt_main;
	table_write_memory[0xC005] = iie_ramwrt_aux;
	table_read_memory[0xC014] = iie_check_ramwrt;

	/* ALTZP switch */
	table_write_memory[0xC008] = iie_altzp_main;
	table_write_memory[0xC009] = iie_altzp_aux;
	table_read_memory[0xC016] = iie_check_altzp;

	/* 80COL switch */
	table_write_memory[0xC00C] = iie_80col_off;
	table_write_memory[0xC00D] = iie_80col_on;
	table_read_memory[0xC01F] = iie_check_80col;

	/* ALTCHAR switch */
	table_write_memory[0xC00E] = iie_altchar_off;
	table_write_memory[0xC00F] = iie_altchar_on;
	table_read_memory[0xC01E] = iie_check_altchar;

	/* SLOTC3ROM switch */
	table_write_memory[0xC00A] = iie_c3rom_internal;
	table_write_memory[0xC00B] = iie_c3rom_peripheral;
	table_read_memory[0xC017] = iie_check_c3rom;

	/* SLOTCXROM switch */
	table_write_memory[0xC006] = iie_cxrom_peripheral;
	table_write_memory[0xC007] = iie_cxrom_internal;
	table_read_memory[0xC015] = iie_check_cxrom;

	/* RDVBLBAR switch */
	table_read_memory[0xC019] =
	    iie_check_vbl;
    }
#endif
    
    /* random number generator */
    for (i = 0xC020; i < 0xC030; i++)
	table_read_memory[i] =
	table_write_memory[i] =
	    read_random;

    /* TEXT switch */
    table_read_memory[0xC050] =
    table_write_memory[0xC050] =
	read_switch_graphics;
    table_read_memory[0xC051] =
    table_write_memory[0xC051] =
	read_switch_text;

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC01A] =
	    iie_check_text;
    }
#endif

    /* MIXED switch */
    table_read_memory[0xC052] =
    table_write_memory[0xC052] =
	read_switch_no_mixed;
    table_read_memory[0xC053] =
    table_write_memory[0xC053] =
	read_switch_mixed;

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC01B] =
	    iie_check_mixed;
    }
    /* PAGE2 switch */
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC054] =
	table_write_memory[0xC054] =
	    iie_page2_off;
    } else
#endif
    {
	table_read_memory[0xC054] =
	table_write_memory[0xC054] =
	    read_switch_primary_page;
    }

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC01C] =
	    iie_check_page2;
    }
    /* PAGE2 switch */
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC055] =
	table_write_memory[0xC055] =
	    iie_page2_on;
    } else
#endif
    {
	table_read_memory[0xC055] =
	table_write_memory[0xC055] =
	    read_switch_secondary_page;
    }

    /* HIRES switch */
#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC01D] =
	    iie_check_hires;
	table_read_memory[0xC056] =
	table_write_memory[0xC056] =
	    iie_hires_off;
	table_read_memory[0xC057] =
	table_write_memory[0xC057] =
	    iie_hires_on;
    }
    else
#endif
    {
	table_read_memory[0xC056] =
	table_write_memory[0xC056] =
	    read_switch_lores;
	table_read_memory[0xC057] =
	table_write_memory[0xC057] =
	    read_switch_hires;
    }

    /* game I/O switches */
    table_read_memory[0xC061] =
    table_read_memory[0xC069] =
        read_button0;
    table_read_memory[0xC062] =
    table_read_memory[0xC06A] =
        read_button1;
    table_read_memory[0xC063] =
    table_read_memory[0xC06B] =
        read_button2;
    table_read_memory[0xC064] =
    table_read_memory[0xC06C] =
        read_gc0;
    table_read_memory[0xC065] =
    table_read_memory[0xC06D] =
        read_gc1;
#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	table_read_memory[0xC066] =
	    iie_read_gc2;
	table_read_memory[0xC067] =
	    iie_read_gc3;
    }
#endif
    for (i = 0xC070; i < 0xC080; i++)
	table_read_memory[i] =
	table_write_memory[i] =
	    read_gc_strobe;

#ifdef APPLE_IIE
    if (apple_mode == IIE_MODE) {
	/* IOUDIS switch & read_gc_strobe */
	table_write_memory[0xC07E] =
	    iie_ioudis_on;
	table_write_memory[0xC07F] =
	    iie_ioudis_off;
	table_read_memory[0xC07E] =
	    iie_check_ioudis;
	table_read_memory[0xC07F] =
	    iie_check_dhires;

	/* DHIRES/Annunciator switches */
	table_write_memory[0xC05E] =
	table_read_memory[0xC05E] =
	    iie_dhires_on;
	table_write_memory[0xC05F] =
	table_read_memory[0xC05F] =
	    iie_dhires_off;
    }
#endif

    /* language card softswitches */
    table_read_memory[0xC080] = table_write_memory[0xC080] =
    table_read_memory[0xC084] = table_write_memory[0xC084] =
	lc_c080;
    table_read_memory[0xC081] = table_write_memory[0xC081] =
    table_read_memory[0xC085] = table_write_memory[0xC085] =
	lc_c081;
    table_read_memory[0xC082] = table_write_memory[0xC082] =
    table_read_memory[0xC086] = table_write_memory[0xC086] =
	lc_c082;
    table_read_memory[0xC083] = table_write_memory[0xC083] =
    table_read_memory[0xC087] = table_write_memory[0xC087] =
	lc_c083;

    table_read_memory[0xC088] = table_write_memory[0xC088] =
    table_read_memory[0xC08C] = table_write_memory[0xC08C] =
	lc_c088;
    table_read_memory[0xC089 ] = table_write_memory[0xC089] =
    table_read_memory[0xC08D ] = table_write_memory[0xC08D] =
	lc_c089;
    table_read_memory[0xC08A] = table_write_memory[0xC08A] =
    table_read_memory[0xC08E] = table_write_memory[0xC08E] =
	lc_c08a;
    table_read_memory[0xC08B] = table_write_memory[0xC08B] =
    table_read_memory[0xC08F] = table_write_memory[0xC08F] =
	lc_c08b;

    /* disk softswitches */
    table_read_memory[0xC0E0] = table_read_memory[0xC0E2] =
    table_read_memory[0xC0E4] = table_read_memory[0xC0E6] =
        disk_read_nop;

    table_read_memory[0xC0E1] = table_read_memory[0xC0E3] =
    table_read_memory[0xC0E5] = table_read_memory[0xC0E7] =
        disk_read_phase;

    table_read_memory[0xC0E8] =
	disk_read_motor_off;
    table_read_memory[0xC0E9] =
	disk_read_motor_on;
    table_read_memory[0xC0EA] =
	disk_read_select_a;
    table_read_memory[0xC0EB] =
	disk_read_select_b;
    table_read_memory[0xC0EC] =
	disk_read_byte;
    table_read_memory[0xC0ED] =
	disk_read_latch;	/* read latch */
    table_read_memory[0xC0EE] =
	disk_read_prepare_in;
    table_read_memory[0xC0EF] =
	disk_read_prepare_out;

    for (i = 0xC0E0; i < 0xC0F0; i++) {
        table_write_memory[i] =
	table_read_memory[i];
    }

    table_write_memory[0xC0ED] =
	disk_write_latch;	/* write latch */

#ifdef APPLE_IIE
    /* slot i/o area */
    for (i = 0xC100; i < 0xC300; i++) {
	table_read_memory[i] =
	    iie_read_slotx;		/* slots 1 & 2 (x) */
    }
    for (i = 0xC300; i < 0xC400; i++) {
	table_read_memory[i] =
	    iie_read_slot3;		/* slot 3 (80col) */
    }
    for (i = 0xC400; i < 0xC600; i++) {
	table_read_memory[i] =
	    iie_read_slotx;		/* slots 4 & 5 (x) */
    }
    for (i = 0xC600; i < 0xC700; i++) {
	table_read_memory[i] =
	    iie_read_slot6;		/* slot 6 (disk) */
    }
    for (i = 0xC700; i < 0xC800; i++) {
	table_read_memory[i] =
	    iie_read_slotx;		/* slot 7 (x) */
    }
    for (i = 0xC800; i < 0xD000; i++) {
	table_read_memory[i] =
	    iie_read_slot_expansion;	/* expansion rom */
    }
    table_read_memory[0xCFFF] =
    table_write_memory[0xCFFF] =
	iie_disable_slot_expansion;
#endif
}

/* -------------------------------------------------------------------------
    c_initialize_apple_ii_memory()
   ------------------------------------------------------------------------- */

void c_initialize_apple_ii_memory()
{
     FILE	*f;
     int	i;
     static int ii_rom_loaded = 0;
     static int slot6_rom_loaded = 0;
#ifdef APPLE_IIE
     static int iie_rom_loaded = 0;
#endif

     for (i = 0; i < 0x10000; i++) {
         apple_ii_64k[0][i] = 0;
         apple_ii_64k[1][i] = 0;
     }
     for (i = 0; i < 8192; i++)
         language_card[0][i] = language_card[1][i] = 0;
     for (i = 0; i < 8192; i++)
         language_banks[0][i] = language_banks[1][i] = 0;

     if (!ii_rom_loaded)
     {
         sprintf(temp, "%s/apple_II.rom", system_path);
         if ((f = fopen(temp, "r")) == NULL) {
	     printf("OOPS!\n");
             printf("Cannot find file '%s'.\n",temp);
	     exit(0);
         }
         fread(apple_ii_rom, 0x3000, 1, f);
         fclose(f);
	 ii_rom_loaded = 1;
     }

#ifdef APPLE_IIE
     if (!iie_rom_loaded) {
	 sprintf(temp, "%s/apple_IIe.rom", system_path);
	 if ((f = fopen(temp, "r")) == NULL) {
	     printf("Cannot find file '%s'.\n",temp);
	     exit(0);
	 }
	 fread(apple_iie_rom, 32768, 1, f);
	 fclose(f);
	 iie_rom_loaded = 1;
     }
#endif

     for (i = 0xD000; i < 0x10000; i++)
	 apple_ii_64k[0][i] = apple_ii_rom[i - 0xD000];
     for (i = 0; i < 0x1000; i++)
	 language_banks[0][i] = apple_ii_rom[i];
     for (i = 0; i < 0x2000; i++)
	 language_card[0][i] = apple_ii_rom[i + 0x1000];

#ifdef APPLE_IIE
     if (apple_mode == IIE_MODE) {
	 /* load the rom from 0xC000, slot rom main, internal rom aux */
	 for (i = 0xC000; i < 0x10000; i++) {
	     apple_ii_64k[0][i] = apple_iie_rom[i - 0xC000];
	     apple_ii_64k[1][i] = apple_iie_rom[i - 0x8000];
	 }
	 for (i = 0; i < 0x1000; i++) {
	     language_banks[0][i] = apple_iie_rom[i + 0x1000];
	     language_banks[1][i] = apple_iie_rom[i + 0x5000];
	 }
	 for (i = 0; i < 0x2000; i++) {
	     language_card[0][i] = apple_iie_rom[i + 0x2000];
	     language_card[1][i] = apple_iie_rom[i + 0x6000];
	 }
     }
     else
#endif
     /* softswitch memory HACK - why this? */
     {
	 for (i = 0xC100; i < 0xD000; i++) {
	     apple_ii_64k[0][i] = i & 0xFF;
	     apple_ii_64k[1][i] = i & 0xFF;
	 }
     }

     apple_ii_64k[0][0xC000] = 0x00;
     apple_ii_64k[1][0xC000] = 0x00;

#ifdef APPLE_IIE
     if (apple_mode == IIE_MODE) {
     } else
#endif
     {
     /* load Disk II rom */
     if (!slot6_rom_loaded) {
	 sprintf(temp, "%s/slot6.rom", system_path);
	 if ((f = fopen( temp, "r" )) == NULL) {
	     printf("Cannot find file '%s'.\n",temp);
	     exit( 0 );
	 }
	 fread(slot6_rom, 0x100, 1, f);
	 fclose(f);
	 slot6_rom_loaded = 1;
     }
     memcpy(apple_ii_64k[0] + 0xC600, slot6_rom, 0x100);
     memcpy(apple_ii_64k[1] + 0xC600, slot6_rom, 0x100);	/* slot rom */
     }
}

/* -------------------------------------------------------------------------
    void c_initialize_sound()
   ------------------------------------------------------------------------- */

void c_initialize_sound()
{
    int i;

    for (i = 0xC030; i < 0xC040; i++)
        table_read_memory[i] = table_write_memory[i] =
	    (sound_mode && soundAllowed) ? read_speaker_toggle_pc : ram_nop;
}

#ifdef APPLE_IIE
/* -------------------------------------------------------------------------
    c_initialize_iie_switches
   ------------------------------------------------------------------------- */
void c_initialize_iie_switches() {

    lc_offset = 0;
    zp_offset = 0;
    ramrd_offset = 0;
    ramwrt_offset = 0;
    write_text_page_offset = 0;
    read_text_page_offset = 0;
    write_hires_page_offset = 0;
    read_hires_page_offset = 0;

    vmode_dhires = 0;
    dhires_flag = 0;
    eighty_store_flag = 0;
    ramrd_flag = 0;			/* MAIN > 127 */
    ramwrt_flag = 0;			/* MAIN > 127 */
    altzp_flag = 0;
    eighty_col_flag = 0;
    altchar_flag = 0;
    ioudis_flag = 0x80;			/* ioudis on */

    c3rom_flag = 0x80;/*HACK0or1?*/	/* c3rom internal */
    c8rom_flag = 0x80;			/* c8rom internal */
    cxrom_flag = 0;			/* cxrom peripheral */

    c3rom_offset = 0x10000;		/* c3rom internal */
    c8rom_offset = 0x10000;		/* c8rom internal */
    cxrom_offset = 0;			/* cxrom peripheral */
}
#endif

/* -------------------------------------------------------------------------
    c_initialize_lc_settings()
   ------------------------------------------------------------------------- */
void c_initialize_lc_settings() {
    language_card_read = 0;		/* READ ROM */
    language_card_write = 1;		/* WRITE TO LC RAM */
    language_card_second = 1;		/* because we're writing */
    language_current_bank = 0;
}

/* -------------------------------------------------------------------------
    void c_initialize_vm()
   ------------------------------------------------------------------------- */
void c_initialize_vm() {
    c_initialize_font();		/* font already read in */
    c_initialize_hires_values();	/* precalculate hires values */
    c_initialize_row_col_tables();	/* precalculate hires offsets */
    c_initialize_code_tables();		/* processor flags */
    c_initialize_apple_ii_memory();	/* read in rom memory */
    c_initialize_tables();		/* read/write memory jump tables */
    c_initialize_sound();		/* sound system */
    c_init_6();				/* drive ][, slot 6 */

    c_initialize_lc_settings();		/* language card settings */
#ifdef APPLE_IIE
    c_initialize_iie_switches();	/* set the //e softswitches */
    c_initialize_dhires_values();	/* set up dhires colors */
#endif

#ifdef MOUSE_EMULATION
    c_initialize_mouse();
#endif
}

/* -------------------------------------------------------------------------
    void c_initialize_firsttime()
   ------------------------------------------------------------------------- */

static void c_initialize_firsttime()
{
    /* get IO permission for speaker port. */
    if (/*ioperm(0x42, 1, 1) ||*/ ioperm(0x61, 1, 1)) {
	perror("ioperm");
	printf("cannot get port access to PC speaker.\n");
	printf("sound will not be used.\n");
	soundAllowed=0;
    } else
	soundAllowed=1;

    /* read in system files and calculate system defaults */
    c_load_interface_font();

    /* set the state of the virtual machine */
    c_initialize_vm();

    /* initialize the video system */
    c_initialize_video();
    c_clear_video();

    vmode_text =/* vmode_mixed =*/ 1;
    vmode_mixed = 0;
    vmode_screen = vmode_page2 = vmode_hires = 0;
    vmode_active = 0;
}

void c_read_random() {
    random_value = (unsigned char)(rand() >> 8);
}

void c_system_defaults() {

    FILE *f = NULL;

    /* set system defaults before user defaults. */
    strcpy(disk_path, "./disks");
    strcpy(system_path, "./rom");

    sprintf(temp, "%s/%s", getenv("HOME"), ".apple2");
    f = fopen(temp, "r");
    if (f == NULL) {
	printf(
	    "Warning. Cannot open the .apple2 system defaults file.\n"
	    "Make sure it's readable in your home directory.");
	printf("Press RETURN to continue...");
	getchar();
	return;
    }

    line_num = 1;
    if (f) {
	prefsin = f;
	prefslex();		/* call the lexical scanner */
    }
    fclose( f );
}

/* save user prefs */
void c_save_settings() {
    FILE *f = NULL;

    sprintf(temp, "%s/%s", getenv("HOME"), ".apple2");
    f = fopen(temp, "w");
    if (f == NULL) {
	printf(
	    "Cannot open the .apple2 system defaults file for writing.\n"
	    "Make sure it has rw permission in your home directory.");
	return;
    }
    
    fprintf(f, "speed = %d\n", MAX_APPLE_DELAY + 1 - apple_speed);
    fprintf(f, "mode = %s\n",  (apple_mode == 0) ? "][+" :
			       (apple_mode == 1) ? "][+ undocumented" :
						   "//e");
    fprintf(f, "disk path = %s\n", disk_path);
    fprintf(f, "color = %s\n", (color_mode == 0) ? "black/white" :
			       (color_mode == 1) ? "lazy color" :
			       (color_mode == 2) ? "color" :
			       (color_mode == 3) ? "lazy interpolated" :
						   "interpolated");
    fprintf(f, "sound = %s\n", (sound_mode == 0) ? "off" :
						   "pc speaker");
    fprintf(f, "joystick = %s\n", (joy_mode == JOY_KYBD)    ? "keyboard" :
				  (joy_mode == JOY_DIGITAL) ? "digital" :
				  (joy_mode == JOY_PCJOY)   ? "pc joystick" :
							      "off");
    fprintf(f, "joystick range = %d\n", joy_range);
    fprintf(f, "origin_x = %d\n", joy_center_x);
    fprintf(f, "origin_y = %d\n", joy_center_y);
#ifdef PC_JOYSTICK
    fprintf(f, "pc joystick parms = %d %d %d %d %d %d %ld\n",
	    js_center_x, js_center_y, js_max_x, js_min_x,
	    js_max_y, js_min_y, js_timelimit);
#endif
    fprintf(f, "sensitivity = %d%%\n", joy_step);
    fprintf(f, "system path = %s\n", system_path);
    fclose(f);
}


int main(int sargc, char *sargv[])
{
    int		i;

    argc = sargc;
    argv = sargv;

#ifdef DEBUGGER
    watch_addrs = -1;
    watch_data = -1;
    /*  the watchpoints and breakpoints */
    for (i=0; i<MAX_BRKPTS; i++) {
	breakpoints[i] = -1;
	watchpoints[i] = -1;
    }
    for (i=0; i<0x100; i++) {
	op_breakpoints[(unsigned char)i] = 0;
    }
#endif

    for (i = 1; i < argc; i++)
    {
	lowercase_string(argv[i]);
/*
	if (strcmp(argv[i], "-vga") == 0)
	    force_vga_mode = 1;
*/
    }

    c_system_defaults();		/* user prefs */
    c_initialize_firsttime();		/* init svga graphics and vm */

    exception_flag = ResetSig;		/* flag reset state */

    for (;;) {
#ifdef APPLE_IIE
	if (apple_mode == IIE_MODE)
	    memcpy(table_opcodes,table_65c02_opcodes,1024);
	else
#endif
	if (apple_mode == IIU_MODE)
	    memcpy(table_opcodes,table_6502_undoc_opcodes,1024);
	else
	    memcpy(table_opcodes,table_6502_opcodes,1024);
	

	/* initialize pages */

        vmode_active = 0;
        c_setpage( 0 );
        c_setscreen( 0 );
   
	/* execute the emulator */
        cpu65x02();

#ifdef DEBUGGER
	/* reset the watchpoints and breakpoints */
	for (i=0; i<MAX_BRKPTS; i++) {
	    breakpoints[i] = -1;
	    watchpoints[i] = -1;
	}
	for (i=0; i<0x100; i++) {
	    op_breakpoints[(unsigned char)i] = 0;
	}
#endif

	c_initialize_vm();

	vmode_text =/* vmode_mixed =*/ 1;
	vmode_mixed = 0;
	vmode_screen = vmode_page2 = vmode_hires = 0;

	exception_flag = ResetSig;

        vmode_active = 0;
	c_setpage( 0 );
	c_setscreen( 0 );

	update_video_screen();
    }
    /* never reached */
}
