/*
 *  Apple II debugger routines
 *  by Aaron Culliney - chernabog@baldmountain.bbn.com - (C) 1997
 *  This code has nothing to do with my employer, BBN.  It was written
 *  completely on my own time and on my own machine.
 *
 *  tokens.l - lexical parser for debugger command line input.
 *
 *  $Id: tokens.l,v 1.2 1997/03/01 19:02:07 chernabog Exp $
 *
 *  ASSUMPTIONS:
 *	flex version 2.5.2
 *
 **/

%{

#include "defs.H"
#include "proto.h"
#include <strings.h>
#include <ctype.h>
#include <vga.h>


/* in debugger.c */
extern void dump_mem(int, int, int, int);
extern void search_mem(char*, int);
extern void set_mem(int, char*);
extern void set_lc_mem(int, int, char*);
extern void disasm(int, int, int);
extern void clear_halt(int*, int);
extern void set_halt(int*, int);
extern void show_breakpts();
extern void show_regs();
extern void display_help();
extern void show_lc_info();
extern void show_disk_info();
extern void c_do_step(int);
extern int at_haltpt();
extern void end_step();

extern unsigned char debug_step;
extern int step_next;
extern int debug_PC;
extern int debug_scratch;
extern unsigned char debug_no_graphics;
extern char second_buf[BUF_Y][BUF_X];
extern int num_buffer_lines;
extern int arg1, arg2;
extern int breakpoints[MAX_BRKPTS];
extern int watchpoints[MAX_BRKPTS];
extern unsigned char watch_data;		/* new byte */
extern unsigned char debug_no_sound;

/* in apple2.S */
extern void read_speaker_toggle_pc();
extern void ram_nop();
extern void do_step();

/* in misc.c */
extern unsigned char apple_ii_64k[];
extern short sound_mode;
extern Function table_read_memory[64 K];
extern Function table_write_memory[64 K];

/* in interface.c */
extern int c_mygetch();
extern void c_interface_normal_keyboard_off();

/* in keys.c */
extern void c_mouse_close();


YY_BUFFER_STATE buffer = 0;


%}


DIGIT		[0-9]
HEXDIGIT	[0-9abcdefABCDEF]
WHITESPACE	[\n\t" "]
BOS		^{WHITESPACE}*
EOS		{WHITESPACE}*\0


%%



{BOS}(me?m?|as?c?i?i?){WHITESPACE}+{HEXDIGIT}+{WHITESPACE}\+?+{HEXDIGIT}+{EOS} {
    /* mem <addrs> <len> */
    int do_ascii = 0;

    if (tolower(yytext[0]) == 'a')
	do_ascii = 1;
    while (!isspace(*yytext)) ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    arg2 = strtol(yytext, &yytext, 16);
    dump_mem(arg1, arg2, 0, do_ascii);
    return MEM;
}

{BOS}(me?m?|as?c?i?i?){WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* mem <addrs> */
    int do_ascii = 0;

    if (tolower(yytext[0]) == 'a')
	do_ascii = 1;
    while (!isspace(*yytext)) ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    dump_mem(arg1, 256, 0, do_ascii);
    return MEM;
}

{BOS}(me?m?|as?c?i?i?){WHITESPACE}+\+{HEXDIGIT}+{EOS} {
    /* mem +<len> */
    int do_ascii = 0;

    if (tolower(yytext[0]) == 'a')
	do_ascii = 1;
    while (*yytext != '+') ++yytext;
    ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    dump_mem(debug_PC, arg1, 0, do_ascii);
    return MEM;
}

{BOS}(me?m?|as?c?i?i?){EOS} {
    /* dump mem from current location */
    int do_ascii = 0;

    if (tolower(yytext[0]) == 'a')
	do_ascii = 1;
    dump_mem(debug_PC, 256, 0, do_ascii);
    return MEM;
}

{BOS}(me?m?|as?c?i?i?){WHITESPACE}*(lc?1|lc?2){WHITESPACE}+{HEXDIGIT}+{WHITESPACE}+\+?{HEXDIGIT}+{EOS} {
    /* dump mem from lc<bank> <addrs> <len> */
    int do_ascii = 0;
    int lc;

    if (tolower(yytext[0]) == 'a')
	do_ascii = 1;
    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;

    lc = strtol(yytext, &yytext, 10);
    arg1 = strtol(yytext, &yytext, 16);
    arg2 = strtol(yytext, &yytext, 16);

    dump_mem(arg1, arg2, lc, do_ascii);
    return MEM;
}

{BOS}(me?m?|as?c?i?i?){WHITESPACE}*(lc?1|lc?2){WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* dump mem from lc<bank> <addrs> */
    int do_ascii = 0;
    int lc;

    if (tolower(yytext[0]) == 'a')
	do_ascii = 1;
    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;

    lc = strtol(yytext, &yytext, 10);
    arg1 = strtol(yytext, &yytext, 16);

    dump_mem(arg1, 256, lc, do_ascii);
    return MEM;
}

{BOS}(me?m?|as?c?i?i?){WHITESPACE}*lc?{WHITESPACE}+{HEXDIGIT}+{WHITESPACE}+\+?{HEXDIGIT}+{EOS} {
    /* dump mem from lc <addrs> <len> */
    int do_ascii = 0;

    if (tolower(yytext[0]) == 'a')
	do_ascii = 1;
    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    arg2 = strtol(yytext, &yytext, 16);
    dump_mem(arg1, arg2, 1, do_ascii);
    return MEM;
}

{BOS}me?m?{WHITESPACE}*lc?{WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* dump mem from lc <addrs> */
    int do_ascii = 0;

    if (tolower(yytext[0]) == 'a')
	do_ascii = 1;
    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    dump_mem(arg1, 256, 1, do_ascii);
    return MEM;
}

{BOS}di?s?{WHITESPACE}+{HEXDIGIT}+{WHITESPACE}+\+?{HEXDIGIT}+{EOS} {
    /* disassemble at <addrs> <len> */
    while (!isspace(*yytext)) ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    arg2 = strtol(yytext, &yytext, 16);

    disasm(arg1, arg2, 0);
    return DIS;
}

{BOS}di?s?{WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* disassemble at <addrs> */
    while (!isspace(*yytext)) ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    arg2 = 256;
    if ((arg1 < 0) || (arg1 > 65535)) arg1 = debug_PC;

    disasm(arg1, arg2, 0);
    return DIS;
}

{BOS}di?s?{WHITESPACE}+\+{HEXDIGIT}+{EOS} {
    /* disassemble current location +<len> */
    while (*yytext != '+') ++yytext;
    ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    disasm(debug_PC, arg1, 0);
    return DIS;
}

{BOS}di?s?{EOS} {
    /* disassemble current location */
    disasm(debug_PC, 256, 0);
    return DIS;
}

{BOS}di?s?{WHITESPACE}*(lc?1|lc?2){WHITESPACE}+{HEXDIGIT}+{WHITESPACE}+\+?{HEXDIGIT}+{EOS} {
    /* disassemble language<bank> <addr> <len> */
    int lc;

    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;

    lc = strtol(yytext, &yytext, 10);
    arg1 = strtol(yytext, &yytext, 16);
    arg2 = strtol(yytext, &yytext, 16);
    
    disasm(arg1, arg2, lc);
    return DIS;
}

{BOS}di?s?{WHITESPACE}*(lc?1|lc?2){WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* disassemble language<bank> <addr> */
    int lc;

    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;

    lc = strtol(yytext, &yytext, 10);
    arg1 = strtol(yytext, &yytext, 16);

    disasm(arg1, 256, lc);
    return DIS;
}

{BOS}di?s?{WHITESPACE}*lc?{WHITESPACE}+{HEXDIGIT}+{WHITESPACE}+\+?{HEXDIGIT}+{EOS} {
    /* disassemble language card <addr> <len> */
    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    arg2 = strtol(yytext, &yytext, 16);
    
    disasm(arg1, arg2, 1);
    return DIS;
}

{BOS}di?s?{WHITESPACE}*lc?{WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* disassemble language card <addr> */
    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;

    arg1 = strtol(yytext, &yytext, 16);
    disasm(arg1, 256, 1);
    return DIS;
}


{BOS}re?g?s?{EOS} {
    /* show cpu state */
    show_regs();
    return REGS;
}

{BOS}{HEXDIGIT}+{WHITESPACE}*\:{WHITESPACE}*{HEXDIGIT}+{EOS} {
    /* set memory <addrs> : <hex string> */
    arg1 = strtol(yytext, &yytext, 16);

    while (*yytext != ':') ++yytext; ++yytext;
    while (isspace(*yytext)) ++yytext;

    set_mem(arg1, yytext);
    return SETMEM;
}

{BOS}{HEXDIGIT}+{WHITESPACE}*(lc?1|lc?2)\:{WHITESPACE}*{HEXDIGIT}+{EOS} {
    /* set LC memory <addrs> : <hex string> */
    int lc;

    arg1 = strtol(yytext, &yytext, 16);

    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;
    lc = strtol(yytext, &yytext, 10);
    ++yytext; while (isspace(*yytext)) ++yytext;

    set_lc_mem(arg1, lc, yytext);
    return SETMEM;
}

{BOS}(st?e?p?|ne?x?t?){EOS} {
    /* step / step next instruction */
    if (*yytext == 'n') step_next = 1;

    debug_step = 1;
    c_do_step(1);
    return STEP;
}

{BOS}(st?e?p?|ne?x?t?){WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* step / step next <n> instructions */
    if (*yytext == 'n') step_next = 1;

    while (!isspace(*yytext)) ++yytext;

    debug_step = 1;
    arg1 = strtol(yytext, (char**)NULL, 16);
    if ((arg1 < 1) || (arg1 > 255)) arg1 = 255;

    c_do_step(arg1);
    return STEP;
}

{BOS}fi?n?i?s?h?{EOS} {
    int step_frame = 1;

    /* step until finished with curent stack frame */
    debug_step = 1;
    while ((c_mygetch() == -1) && !at_haltpt()) {
	
	if (apple_ii_64k[debug_PC] == 0x20) ++step_frame;	/* JSR */
	if (apple_ii_64k[debug_PC] == 0x60) --step_frame;	/* RTS */
	
	if (!step_frame) break;				/* finished */
	do_step();
    }
    end_step();			/* print location */
    return FINISH;
}

{BOS}un?t?i?l?{EOS} {
    /* step until PC > current line.  good for backward loops, but it
       doesn't handle jsr's ahead in memory */
    debug_step = 1;
    arg1 = debug_PC;
    while ((debug_PC <= arg1) && (!at_haltpt()) && (c_mygetch() == -1))
	do_step();
    end_step();			/* print location */
    return UNTIL;
}

{BOS}go?{WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* jump to addrs and run while remaining in debugger console */
    while (!isspace(*yytext)) ++yytext;

    /* DANGEROUS! */
    debug_PC = strtol(yytext, (char**)NULL, 16);
    debug_step = 1;
    while (!at_haltpt() && (c_mygetch() == -1))
	do_step();
    end_step();			/* print location */
    return GO;
}

{BOS}go?{EOS} {
    /* run while remaining in debugger console */
    debug_step = 1;
    while (!at_haltpt() && (c_mygetch() == -1))
	do_step();
    end_step();			/* print location */
    return GO;
}

{BOS}wa?t?c?h?{EOS} {
    /* set watchpoint */
    set_halt(watchpoints, debug_PC);
    return WATCH;
}

{BOS}wa?t?c?h?{WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* set watchpoint */
    while (!isspace(*yytext)) ++yytext;

    arg1 = strtol(yytext, (char**)NULL, 16);
    if ((arg1 < 0) || (arg1 > 65535)) {
	sprintf(second_buf[num_buffer_lines++], "invalid address");
	return WATCH;
    }

    set_halt(watchpoints, arg1);
    return WATCH;
}


{BOS}br?e?a?k?{EOS} {
    /* set breakpoint */
    set_halt(breakpoints, debug_PC);
    return BREAK;
}

{BOS}br?e?a?k?{WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* set breakpoint */
    while (!isspace(*yytext)) ++yytext;

    arg1 = strtol(yytext, (char**)NULL, 16);
    if ((arg1 < 0) || (arg1 > 65535)) {
	sprintf(second_buf[num_buffer_lines++], "invalid address");
	return BREAK;
    }

    set_halt(breakpoints, arg1);
    return BREAK;
}


{BOS}ig?n?o?r?e?{EOS} {
    /* ignore everything */
    clear_halt(watchpoints, 0);
    sprintf(second_buf[num_buffer_lines++], "ignored all");
    return IGNORE;
}

{BOS}ig?n?o?r?e?({WHITESPACE}+{DIGIT}+)+{EOS} {
    /* ignore <watchpt> ... */

    while (!isspace(*yytext)) ++yytext;

    while (*yytext) {
	arg1 = strtol(yytext, &yytext, 10);
	if ((arg1 < 1) || (arg1 > MAX_BRKPTS)) {
	    sprintf(second_buf[num_buffer_lines++], "invalid watchpoint");
	    return IGNORE;
	}
	clear_halt(watchpoints, arg1);
	sprintf(second_buf[num_buffer_lines++], "ignored %d", arg1);
    }
    return IGNORE;
}


{BOS}cl?e?a?r?{EOS} {
    /* clear everything */
    clear_halt(breakpoints, 0);
    sprintf(second_buf[num_buffer_lines++], "cleared all");
    return CLEAR;
}

{BOS}cl?e?a?r?({WHITESPACE}+{DIGIT}+)+{EOS} {
    /* clear <breakpt> ... */
    while (!isspace(*yytext)) ++yytext;

    while (*yytext) {
	arg1 = strtol(yytext, &yytext, 10);
	if ((arg1 < 1) || (arg1 > MAX_BRKPTS)) {
	    sprintf(second_buf[num_buffer_lines++], "invalid breakpoint");
	    return CLEAR;
	}
	clear_halt(breakpoints, arg1);
	sprintf(second_buf[num_buffer_lines++], "cleared %d", arg1);
    }
    return CLEAR;
}

{BOS}stat?u?s?{EOS} {
    /* show breakpoints and watchpoints */
    show_breakpts();
    return STATUS;
}

{BOS}gra?p?h?i?c?s?{EOS} {
    /* toggle graphics drawing to screen while in debugger */
    debug_no_graphics = !debug_no_graphics;
    
    sprintf(second_buf[num_buffer_lines++], "%s",
	    debug_no_graphics ? "graphics disabled" : "graphics enabled");
    return GRAPHICS;
}

{BOS}sou?n?d?{EOS} {
    int i;

    /* toggle sound output while in debugger */
    debug_no_sound = !debug_no_sound;
    if (sound_mode) {
	for (i = 0xC030; i < 0xC040; i++)
	    table_read_memory[i] = table_write_memory[ i ] =
		(debug_no_sound) ? ram_nop : read_speaker_toggle_pc;
    }
    sprintf(second_buf[num_buffer_lines++], "%s",
	    (debug_no_sound) ? "sound disabled" : "sound enabled");
    return SOUND;
}

{BOS}sea?r?c?h?{WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* search main memory for <bytes> */
    while (!isspace(*yytext)) ++yytext;
    while (isspace(*yytext)) ++yytext;

    search_mem(yytext, 0);

    return SEARCH;
}

{BOS}sea?r?c?h?{WHITESPACE}*(lc?1|lc?2){WHITESPACE}+{HEXDIGIT}+{EOS} {
    /* search LC<bank> for <bytes> */
    int lc;

    while (tolower(*yytext) != 'l') ++yytext; ++yytext;
    if (tolower(*yytext) == 'c') ++yytext;
    lc = strtol(yytext, &yytext, 10);

    while (!isspace(*yytext)) ++yytext;
    while (isspace(*yytext)) ++yytext;

    search_mem(yytext, lc);

    return SEARCH;
}

{BOS}la?n?g?{EOS} {
    /* display language card settings */
    show_lc_info();
    return LC;
}

{BOS}dri?v?e?{EOS} {
    /* show disk settings */
    show_disk_info();
    return DRIVE;
}

{BOS}(\?|he?l?p?){EOS} {
    display_help();
    return HELP;
}

{BOS}log{EOS} {
    /* log debugger output to file - not implemented */
    return LOG;
}

{BOS}save{EOS} {
    /* save apple2 state to a .img file - not implemented
     * I'd like to be compatible with the applePC's .img format
     * anyone have documentation on this?  -ASC
     **/
    return SAVE;
}


\n	/* ignore newlines */

.	/* ignore extraneous characters */


%%


int yywrap () {
    return 1;
}

/* initialize the buffer - needed each time through */
void init_lex (char *str, int size) {
    if (buffer) yy_delete_buffer(buffer);
    buffer = yy_scan_buffer(str, size);

    if (!buffer) {	/* oops */
	c_interface_normal_keyboard_off();
	c_mouse_close();
	vga_setmode(TEXT);
	printf("lex buffer not big enough\n");
	exit(1);
    }	       
}
